package ohd.hseb.charter.parameters;

import java.awt.Color;

import org.jfree.chart.block.BlockBorder;
import org.jfree.ui.RectangleInsets;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Parameters object records attributes of a border, specifically line widths for four sides and color.
 * 
 * @author hank.herr
 */
public class BorderParameters extends DefaultChartParameters
{
    public final static RectangleInsets DEFAULT_LINE_WIDTHS = new RectangleInsets(0, 0, 0, 0);

    /**
     * The line widths for each border, stored in a {@link RectangleInsets}, which is a JFreeChart class.
     */
    private RectangleInsets _lineWidths = null;

    /**
     * The border color.
     */
    private Color _color = null;

    public BorderParameters(final String xmlTagName)
    {
        setXMLTagName(xmlTagName);
    }

    public BlockBorder createBlockBorder()
    {
        return new BlockBorder(getLineWidths().getTop(),
                               getLineWidths().getLeft(),
                               getLineWidths().getBottom(),
                               getLineWidths().getRight(),
                               getColor());
    }

    public RectangleInsets getLineWidths()
    {
        return _lineWidths;
    }

    public void setLineWidths(final RectangleInsets lineWidths)
    {
        _lineWidths = lineWidths;
    }

    public Color getColor()
    {
        return _color;
    }

    public void setColor(final Color color)
    {
        _color = color;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        throw new IllegalStateException("BorderParameters should be applied by the using subclass.  This method should not be called.");
    }

    @Override
    public void clearParameters()
    {
        _lineWidths = null;
        _color = null;
    }

    @Override
    public Object clone()
    {
        final BorderParameters cloneParms = new BorderParameters(getXMLTagName());
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final BorderParameters base = (BorderParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final BorderParameters base = (BorderParameters)override;
        if(base.getLineWidths() != null)
        {
            setLineWidths(new RectangleInsets(base.getLineWidths().getTop(),
                                              base.getLineWidths().getLeft(),
                                              base.getLineWidths().getBottom(),
                                              base.getLineWidths().getRight()));
        }
        if(base.getColor() != null)
        {
            setColor(ColorTools.deepCopy(base.getColor()));
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.BorderParameters))
        {
            return false;
        }
        final BorderParameters other = (BorderParameters)parameters;
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_lineWidths, other.getLineWidths()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_color, other.getColor()))
        {
            return false;
        }
        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        //called after reading xml for a label sitting by itself.
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(_lineWidths == null)
        {
            throw new ChartParametersException("Border line widths not specified.");
        }
        if(_color == null)
        {
            throw new ChartParametersException("Border color not specified.");
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }

        else if(elementName.equals("lineWidths"))
        {
            try
            {
                _lineWidths = XMLTools.extractInsetsFromAttributes(attr);
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException(e.getMessage());
            }
        }

        else if(elementName.equals("color"))
        {
            try
            {
                _color = XMLTools.extractColorFromAttributes(attr);
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException(e.getMessage());
            }
        }
        else
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }

        return null;
    }

    @Override
    public void setupDefaultParameters()
    {
        this._lineWidths = new RectangleInsets(DEFAULT_LINE_WIDTHS.getTop(),
                                               DEFAULT_LINE_WIDTHS.getLeft(),
                                               DEFAULT_LINE_WIDTHS.getBottom(),
                                               DEFAULT_LINE_WIDTHS.getRight());
        this._color = Color.BLACK;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public String toString()
    {
        String results = "BorderParameters: ";
        results += "lineWidths ='" + _lineWidths + "'; ";
        results += "color = " + _color + ".";
        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());
        if(_lineWidths != null)
        {
            mainElement.appendChild(XMLTools.createInsetsElement("lineWidths", request, _lineWidths));
        }
        if(_color != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("color", request, _color));
        }
        return mainElement;
    }
}
