package ohd.hseb.charter.parameters;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.XYPlot;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import com.google.common.base.Strings;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.charter.datasource.XYChartDataSource;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * This is the top level {@link ChartDrawingParameters}. It stores instances of all the needed parameter objects and
 * provides basic tools for acquiring and manipulating those instances.<br>
 * <br>
 * Some methods are provided to set axis visibility based on axis usage. These are used within
 * {@link #setupDefaultParameters()} and {@link #clearAllParametersOtherThanDataSources()} in order to set default
 * values for axis visibility.<br>
 * <br>
 * To any developer reading this. The XXX comments indicate important points where changes must be made for new
 * parameter containers. When working with lower level parameter classes, it is recommended that you do a search on an
 * existing parameter and let that guide you on what methods need updating.
 * 
 * @author herrhd
 */
public class ChartDrawingParameters extends DefaultChartParameters
{
    private static final Logger LOG = LogManager.getLogger(ChartDrawingParameters.class);

    /**
     * The list of {@link DataSourceDrawingParameters}, one per data source, which specify how to draw sources.
     */
    private final List<DataSourceDrawingParameters> _dataSourceParameters =
                                                                          new ArrayList<DataSourceDrawingParameters>();

    /**
     * Maps the index of the data source within the list provided to the constructor to the parameters corresponding to
     * it.
     */
    private final LinkedHashMap<Integer, DataSourceDrawingParameters> _dataSourceIndexToParametersMap =
                                                                                                      new LinkedHashMap<Integer, DataSourceDrawingParameters>();

    /**
     * List of all sub-plot indices used sorted into ascending order (0, 1, ...). Note that the indices may include gaps
     * depending on user settings.
     */
    private final List<Integer> _subPlotIndices = new ArrayList<Integer>();

    /**
     * List of all subplot parameters in no particular order.
     */
    private final List<SubPlotParameters> _subPlotParameters = new ArrayList<SubPlotParameters>();

    /**
     * Map of the subplot index to the parameters that correspond.
     */
    private final LinkedHashMap<Integer, SubPlotParameters> _subPlotIndexToParametersMap =
                                                                                         new LinkedHashMap<Integer, SubPlotParameters>();

    //XXX Other parameters should be declared, and constructed, here!
    //XXX This could probably be made more efficient by sticking these in a list and then, down below, calling stuff for everyhing
    //in the list instead of one object at a time.  HOWEVER, since the list will not be changed often, I won't make the updates now.
    private final TitleParameters _plotTitleParameters = new TitleParameters("plotTitle");
    private final LegendParameters _legendParameters = new LegendParameters();
    private final ThresholdListParameters _thresholdListParameters = new ThresholdListParameters();
    private final AxisParameters _domainAxisParameters = new AxisParameters(-1,
                                                                            ChartConstants.DOMAIN_AXIS,
                                                                            false);
    private final GeneralParameters _generalParameters = new GeneralParameters();
    private final SubtitleListParameters _subtitleListParameters = new SubtitleListParameters();

    /**
     * Provides a clean slate for adding data source parameters or other parameters as needed. This is only useful if
     * the parameters will be immediately read from XML. In other words, construct then, and then pass it into
     * {@link XMLTools#readXMLFromFile(java.io.File, ohd.hseb.hefs.utils.xml.XMLReadable)} to read it in (or another of
     * the readers provided therein).
     */
    public ChartDrawingParameters()
    {
        this.setXMLTagName("chartDrawingParameters");
    }

    /**
     * Use this to initialize data source parameters by extracting initial parameters from the passed in data sources.
     * It will ensure that the list of data source parameters is complete, one per data source, and will initialize the
     * mapping. Further data sources can be added as needed via the other methods.<br>
     * <br>
     * This constructor ensures that every data source has corresponding parameters, as well as every subplot plotted
     * against by a data source.
     * 
     * @param dataSources No entry can be null!
     */
    public ChartDrawingParameters(final List<XYChartDataSource> dataSources)
    {
        this.setXMLTagName("chartDrawingParameters");

        //Data sources
        for(int i = 0; i < dataSources.size(); i++)
        {
            if (dataSources.get(i) == null)
            {
                throw new IllegalArgumentException("Provided data source is null, which is not allowed.");
            }
            final DataSourceDrawingParameters params =
                                                     dataSources.get(i)
                                                                .getDefaultFullySpecifiedDataSourceDrawingParameters();
            final DataSourceDrawingParameters existingParms =
                                                            _dataSourceIndexToParametersMap.get(params.getDataSourceOrderIndex());
            if(existingParms != null)
            {
                LOG.debug("WARNING -- ChartDrawingParameters were initialized from data sources at least two of which have the same data source index!");
                _dataSourceParameters.remove(existingParms);
            }
            _dataSourceParameters.add(params);
            _dataSourceIndexToParametersMap.put(params.getDataSourceOrderIndex(), params);

        }

        addParametersForEachSubPlot();
    }

    /**
     * Clears all maps and lists.
     */
    @Override
    public void clearParameters()
    {
        this._dataSourceIndexToParametersMap.clear();
        this._dataSourceParameters.clear();
        this._subPlotIndexToParametersMap.clear();
        this._subPlotIndices.clear();
        this._subPlotParameters.clear();

        clearAllParametersOtherThanDataSources();
    }

    /**
     * Clears parameters, excluding data source parameters. For the contents of _subPlotParameters, it clears the
     * individual parameters, but leaves the list intact.
     */
    public void clearAllParametersOtherThanDataSources()
    {
        for(int i = 0; i < this._subPlotParameters.size(); i++)
        {
            this._subPlotParameters.get(i).clearParameters();
        }

        //XXX Add to this for any new parameters.
        _plotTitleParameters.clearParameters();
        _domainAxisParameters.clearParameters();
        _legendParameters.clearParameters();
        _thresholdListParameters.clearParameters();
        _generalParameters.clearParameters();
        _subtitleListParameters.clearParameters();
    }

    /**
     * Removes all subplot parameters.
     */
    public void removeAllSubPlotParameters()
    {
        this._subPlotParameters.clear();
        this._subPlotIndexToParametersMap.clear();
        this._subPlotIndices.clear();
    }

    /**
     * Adds on set of parameters.
     * 
     * @param parameters Add data source parameters and update index to parameters map.
     */
    public void addDataSourceParameters(final DataSourceDrawingParameters parameters)
    {
        final DataSourceDrawingParameters existingParameters =
                                                             this.getDataSourceParameters(parameters.getDataSourceOrderIndex());
        if(existingParameters != null)
        {
            this._dataSourceParameters.remove(existingParameters);
        }
        this._dataSourceParameters.add(parameters);
        this._dataSourceIndexToParametersMap.put(parameters.getDataSourceOrderIndex(), parameters);
    }

    /**
     * Adds all passed in parameters via addDataSourceParameters.
     * 
     * @param parms DataSourceDrawingParameters instances to add.
     */
    public void addAllDataSourceParameters(final List<DataSourceDrawingParameters> parms)
    {
        for(int i = 0; i < parms.size(); i++)
        {
            addDataSourceParameters(parms.get(i));
        }
    }

    /**
     * Removes the passed in DataSourceDrawingParameters from these parameters, renumbering the index of the following
     * drawing parameters accordingly.
     * 
     * @param parms The parameters to remove.
     */
    public void removeDataSourceParameters(final DataSourceDrawingParameters parms)
    {
        _dataSourceParameters.remove(parms);
        _dataSourceIndexToParametersMap.remove(parms.getDataSourceOrderIndex());

        for(int i = 0; i < _dataSourceParameters.size(); i++)
        {
            if(_dataSourceParameters.get(i).getDataSourceOrderIndex() > parms.getDataSourceOrderIndex())
            {
                _dataSourceParameters.get(i)
                                     .setDataSourceOrderIndex(_dataSourceParameters.get(i).getDataSourceOrderIndex()
                                         - 1);
            }
        }

        populateDataSourceHashMapFromList();
    }

    /**
     * Changes the order index for the passed in parameters. It adjusts the order index for all data sources between the
     * old order index and the new one accordingly, under the assumption that the change is reflected in the data
     * sources to which the parameters apply.
     * 
     * @param parms Parameters to change.
     * @param newOrderIndex The new order index, which forces a change to other order indices.
     */
    public void changeOrderForDataSourceParameters(final DataSourceDrawingParameters parms, final int newOrderIndex)
    {
        final int oldIndex = parms.getDataSourceOrderIndex();
        _dataSourceParameters.remove(parms);
        _dataSourceIndexToParametersMap.remove(parms.getDataSourceOrderIndex());
        parms.setDataSourceOrderIndex(newOrderIndex);

        //Is there a better way to handles these two loops... perhaps in a combined manner?
        if(oldIndex < newOrderIndex)
        {
            for(int i = 0; i < _dataSourceParameters.size(); i++)
            {
                if((_dataSourceParameters.get(i).getDataSourceOrderIndex() > oldIndex)
                    && (_dataSourceParameters.get(i).getDataSourceOrderIndex() <= newOrderIndex))
                {
                    _dataSourceParameters.get(i)
                                         .setDataSourceOrderIndex(_dataSourceParameters.get(i).getDataSourceOrderIndex()
                                             - 1);
                }
            }
        }
        else
        {
            for(int i = 0; i < _dataSourceParameters.size(); i++)
            {
                if((_dataSourceParameters.get(i).getDataSourceOrderIndex() >= newOrderIndex)
                    && (_dataSourceParameters.get(i).getDataSourceOrderIndex() < oldIndex))
                {
                    _dataSourceParameters.get(i)
                                         .setDataSourceOrderIndex(_dataSourceParameters.get(i).getDataSourceOrderIndex()
                                             + 1);
                }
            }
        }

        _dataSourceParameters.add(parms);
        _dataSourceIndexToParametersMap.put(newOrderIndex, parms);

        addDataSourceParameters(parms);
        populateDataSourceHashMapFromList();
    }

    /**
     * Clears the {@link #_dataSourceIndexToParametersMap} and the list {@link #_dataSourceParameters}.
     */
    public void clearDataSourceParameters()
    {
        this._dataSourceIndexToParametersMap.clear();
        this._dataSourceParameters.clear();
    }

    /**
     * Update {@link #_dataSourceIndexToParametersMap} to match the list.
     */
    private void populateDataSourceHashMapFromList()
    {
        _dataSourceIndexToParametersMap.clear();

        for(int i = 0; i < this._dataSourceParameters.size(); i++)
        {
            _dataSourceIndexToParametersMap.put(_dataSourceParameters.get(i).getDataSourceOrderIndex(),
                                                _dataSourceParameters.get(i));
        }
    }

    /**
     * Updates both {@link #_subPlotParameters} and {@link #_subPlotIndexToParametersMap}.
     */
    private void addSubPlotParameters(final SubPlotParameters parameters)
    {
        this._subPlotParameters.add(parameters);
        this._subPlotIndexToParametersMap.put(parameters.getSubPlotIndex(), parameters);
    }

    /**
     * Updates {@link #_subPlotIndexToParametersMap} based on {@link #_subPlotParameters}.
     */
    private void populateSubPlotHashMapFromList()
    {
        this._subPlotIndexToParametersMap.clear();
        for(int i = 0; i < this._subPlotParameters.size(); i++)
        {
            _subPlotIndexToParametersMap.put(_subPlotParameters.get(i).getSubPlotIndex(), _subPlotParameters.get(i));
        }
    }

    /**
     * Moves a subplot to a new index. It changes the index of the subplot parameters and modifies any data source that
     * pointed to the old subplot to point to the new one. If any parameters already exist for that index, they will be
     * removed. Any data sources that point to the new subplot will continue to point to the new subplot.
     * 
     * @param originalSubPlotIndex Current index
     * @param newSubPlotIndex New index
     */
    public void moveSubPlotAndAllSeriesDrawnOnIt(final int originalSubPlotIndex, final int newSubPlotIndex)
    {
        //Do nothing if they are equal.  If the are equal and we do not return, the calls below
        //will result in the subplot parms being erased... weird.
        if(originalSubPlotIndex == newSubPlotIndex)
        {
            return;
        }
        SubPlotParameters subPlotParms = null;

        //Remove existing
        if(this.doSubPlotParametersExist(newSubPlotIndex))
        {
            subPlotParms = getSubPlot(newSubPlotIndex);
            getSubPlotParameters().remove(subPlotParms);
        }

        subPlotParms = getSubPlot(originalSubPlotIndex);
        subPlotParms.setSubPlotIndex(newSubPlotIndex);
        for(final DataSourceDrawingParameters dsParms: _dataSourceParameters)
        {
            if(dsParms.getSubPlotIndex() == originalSubPlotIndex)
            {
                dsParms.setSubPlotIndex(newSubPlotIndex);
            }
        }
        populateSubPlotHashMapFromList();
    }

    /**
     * Calls synchronizeSubPlotParametersListWithDataSourceParameters(false).
     */
    public void synchronizeSubPlotParametersListWithDataSourceParameters()
    {
        synchronizeSubPlotParametersListWithDataSourceParameters(false);
    }

    /**
     * Looks at the other parameters and adds any subplot parameters implied by its data sources that do not currently
     * exist in this set of parameters. The axis visibility will also be set, if the callSetupForNewParameters is true.
     * This is useful with override parameters, because it allows you to add any missing subplot parameters implied by a
     * default set of parameters without removing anything. The synchronize methods will remove unused subplot
     * parameters based on the other set.
     * 
     * @param other
     * @param callSetupForNewParameters True if defaults are to be set for any added parameters.
     */
    public void addMissingSubPlotParametersBasedOtherChartsDataSources(final ChartDrawingParameters other,
                                                                       final boolean callSetupForNewParameters)
    {
        //Adjust the calculated parameters to contain defaults relative to the managed parameters.
        for(int i = 0; i < other.getDataSourceParametersCount(); i++)
        {
            final DataSourceDrawingParameters parms = other.getDataSourceParameters().get(i);
            SubPlotParameters subPlotParms = getSubPlot(parms.getSubPlotIndex());
            if(subPlotParms == null)
            {
                subPlotParms = new SubPlotParameters(parms.getSubPlotIndex());
                subPlotParms.setArguments(getArguments());
                if(callSetupForNewParameters)
                {
                    subPlotParms.setupDefaultParameters();
                }
                addSubPlotParameters(subPlotParms);
            }
        }
    }

    /**
     * Determines what subplot parameters are required, based on data source parameters within this and any data source
     * parameters within default parameters provided. Both this and the default parameters must have the same number of
     * data source parameter objects.
     * 
     * @param callSetupDefaultsForAnyNewParameters If true, then any new parameters that are setup will have their
     *            defaults set.
     */
    public void synchronizeSubPlotParametersListWithDataSourceParameters(final boolean callSetupDefaultsForAnyNewParameters)
    {
        //For each set of data source parameters...
        for(int i = 0; i < getDataSourceParametersCount(); i++)
        {
            //Get the parameters for that data source from this.  If the subplot index is not found,
            //then get it from the default parameters, which is required to specify it.
            final DataSourceDrawingParameters sourceParms = getDataSourceParameters(i);

            //So, sourceParms now specifies a subplot index.  Check for the sub plot parameters, and,
            //if not found, add one.
            if(_subPlotIndexToParametersMap.get(sourceParms.getSubPlotIndex()) == null)
            {
                final SubPlotParameters newParms = new SubPlotParameters(sourceParms.getSubPlotIndex());
                newParms.setArguments(getArguments());
                addSubPlotParameters(newParms);
                if(callSetupDefaultsForAnyNewParameters)
                {
                    newParms.setupDefaultParameters();
                    setupDefaultRangeAxisParametersBasedOnDataSources(newParms.getSubPlotIndex());
                }
            }
            else
            {
                if(callSetupDefaultsForAnyNewParameters)
                {
                    setupDefaultRangeAxisParametersBasedOnDataSources(sourceParms.getSubPlotIndex());
                }
            }
        }

        //Ensure that the subplot indices list is recomputed.
        _subPlotIndices.clear();

        //Now get all the keys in the sub plot to parameter map. For each key...
        final List<Integer> keyList = new ArrayList<Integer>(_subPlotIndexToParametersMap.keySet());
        for(int i = 0; i < keyList.size(); i++)
        {
            //Check if the subplot indices implied by both this and default parameters includes
            //the key. If not, remove the subplot parameters.
            if(this.getUsedSubPlotIndices().indexOf(keyList.get(i)) < 0)
            {
                final SubPlotParameters parmsToRemove = _subPlotIndexToParametersMap.get(keyList.get(i));
                _subPlotIndexToParametersMap.remove(keyList.get(i));
                _subPlotParameters.remove(parmsToRemove);
            }
        }

        //Ensure that the subplot indices list is recomputed next time its needed.
        _subPlotIndices.clear();
    }

    /**
     * Given another set of drawing parameters, this ensures that one set of subplot parameters exists for each required
     * subplot specified in the other's list of data sources. The subplot parameters will be empty (all null), including
     * visibility. This is useful for setting up override subplot parameters based on a baseline calculated set of
     * parameters that is ready to be overridden by users. However, note that this method REMOVES any parameters from
     * this that are not needed based on the other set.
     * 
     * @param other
     * @param setupDefaultParmsForNewSubPlotParms If true, then setupDefaultParameters() will be called for any new
     *            subplot parameters.
     */
    public void synchronizeSubPlotParametersListRelativeToOtherParameters(final ChartDrawingParameters other,
                                                                          final boolean setupDefaultParmsForNewSubPlotParms)
    {
        this.addMissingSubPlotParametersBasedOtherChartsDataSources(other, setupDefaultParmsForNewSubPlotParms);

        final List<Integer> keyList = new ArrayList<Integer>(_subPlotIndexToParametersMap.keySet());
        for(int i = 0; i < keyList.size(); i++)
        {
            if(other.getUsedSubPlotIndices().indexOf(keyList.get(i)) < 0)
            {
                final SubPlotParameters parmsToRemove = _subPlotIndexToParametersMap.get(keyList.get(i));
                _subPlotIndexToParametersMap.remove(keyList.get(i));
                _subPlotParameters.remove(parmsToRemove);
            }
        }

        _subPlotIndices.clear();
    }

    /**
     * @param subPlotIndex Sub plot to check, based on _dataSourceParameters.
     * @return True if the left range axis has data plotted against it.
     */
    public boolean isLeftRangeAxisPresentInSubPlot(final int subPlotIndex)
    {
        for(int i = 0; i < this._dataSourceParameters.size(); i++)
        {
            if((_dataSourceParameters.get(i).getSubPlotIndex() == subPlotIndex)
                && ((_dataSourceParameters.get(i).getYAxisIndex() == ChartConstants.LEFT_YAXIS_INDEX)
                    || (_dataSourceParameters.get(i).getYAxisIndex() == null)))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @param subPlotIndex Sub plot to check.
     * @return True if the right range axis has data plotted against it, based on _dataSourceParameters.
     */
    public boolean isRightRangeAxisPresentInSubPlot(final int subPlotIndex)
    {
        for(int i = 0; i < this._dataSourceParameters.size(); i++)
        {
            if((_dataSourceParameters.get(i).getSubPlotIndex() == subPlotIndex)
                && (_dataSourceParameters.get(i).getYAxisIndex() == ChartConstants.RIGHT_YAXIS_INDEX))
            {
                return true;
            }
        }
        return false;
    }

    private void addParametersForEachSubPlot()
    {
        for(int i = 0; i < this.getUsedSubPlotIndices().size(); i++)
        {
            final SubPlotParameters parms = new SubPlotParameters(this.getUsedSubPlotIndices().get(i));
            parms.setArguments(getArguments());
            addSubPlotParameters(parms);
        }
    }

    /**
     * Applies each set of subplot parameters to the appropriate XYPlot.
     * 
     * @param chart
     * @param subPlotIndexToXYPlot
     */
    public void applyAllSettings(final JFreeChart chart,
                                 final HashMap<Integer, XYPlot> subPlotIndexToXYPlot,
                                 final int domainAxisType,
                                 final HashMap<Integer, int[]> subPlotIndexToComputedRangeAxisTypes) throws ChartParametersException
    {
        //Prepare components.
        _thresholdListParameters.setSubPlotIndexToXYPlot(subPlotIndexToXYPlot);
        _thresholdListParameters.setSubPlotIndexToComputedRangeAxisTypesMap(subPlotIndexToComputedRangeAxisTypes);
        _thresholdListParameters.setDomainAxisType(domainAxisType);
        _thresholdListParameters.applyParametersToChart(chart); //Must be done before axis parameters are applied.

        //General parameters must be applied before subplot parameters!
        _generalParameters.applyParametersToChart(chart);

        for(int i = 0; i < getUsedSubPlotIndices().size(); i++)
        {
            final int workingIndex = getUsedSubPlotIndices().get(i);
            final AxisAutoRangeHelper autorangehelper = new AxisAutoRangeHelper(subPlotIndexToXYPlot.get(workingIndex),
                                                                                _thresholdListParameters);
            getSubPlot(workingIndex).applyParametersToChart(autorangehelper);
        }

        //XXX Apply other parameters here, such as plot title, domain axis, legend parameters, and subtitle.
        _plotTitleParameters.applyParametersToChart(chart);
        final AxisAutoRangeHelper autorangehelper =
                                                  new AxisAutoRangeHelper(chart.getXYPlot(), _thresholdListParameters);
        autorangehelper.setAxisIndex(ChartConstants.DOMAIN_AXIS);
        _domainAxisParameters.applyParametersToChart(autorangehelper);
        _legendParameters.applyParametersToChart(chart);
        _subtitleListParameters.applyParametersToChart(chart);

    }

    public boolean doSubPlotParametersExist(final int subPlotIndex)
    {
        return (getSubPlot(subPlotIndex) != null);
    }

    public DataSourceDrawingParameters getDataSourceParameters(final int dataSourceOrderIndex)
    {
        return this._dataSourceIndexToParametersMap.get(dataSourceOrderIndex);
    }

    /**
     * @return Sorted list of Integers providing a full list of subplot indices implied by the data source parameters
     *         within this object.
     */
    public List<Integer> getUsedSubPlotIndices()
    {
        if(_subPlotIndices.size() == 0)
        {
            for(int i = 0; i < this._dataSourceParameters.size(); i++)
            {
                Integer index = _dataSourceParameters.get(i).getSubPlotIndex();
                if(index == null)
                {
                    index = 0;
                }
                if(_subPlotIndices.indexOf(index) < 0)
                {
                    _subPlotIndices.add(index);
                }
            }
            Collections.sort(_subPlotIndices);
        }
        return _subPlotIndices;
    }

    public List<DataSourceDrawingParameters> getDataSourceParameters()
    {
        return _dataSourceParameters;
    }

    public int getDataSourceParametersCount()
    {
        return _dataSourceParameters.size();
    }

    public SubPlotParameters getSubPlot(final int subPlotIndex)
    {
        return _subPlotIndexToParametersMap.get(subPlotIndex);
    }

    public List<SubPlotParameters> getSubPlotParameters()
    {
        return _subPlotParameters;
    }

    public int getSubPlotParametersCount()
    {
        return _subPlotParameters.size();
    }

    //XXX Add get methods for new parameters so external user can edit.
    public GeneralParameters getGeneralParameters()
    {
        return _generalParameters;
    }

    public TitleParameters getPlotTitle()
    {
        return this._plotTitleParameters;
    }

    public SubtitleListParameters getSubtitleList()
    {
        return this._subtitleListParameters;
    }

    public AxisParameters getDomainAxis()
    {
        return this._domainAxisParameters;
    }

    public LegendParameters getLegend()
    {
        return this._legendParameters;
    }

    public ThresholdListParameters getThresholdList()
    {
        return this._thresholdListParameters;
    }

    /**
     * @param subPlotIndex Subplot for which to setup the axis visibility based on _dataSourceParameters plotting
     *            instructions.
     */
    private void setupDefaultRangeAxisParametersBasedOnDataSources(final int subPlotIndex)
    {
        //Visibility
        getSubPlot(subPlotIndex).getLeftAxis()
                                .setVisible(isRangeAxisPlottedAgainstWithinAnyDataSource(subPlotIndex,
                                                                                         ChartConstants.LEFT_YAXIS_INDEX));
        getSubPlot(subPlotIndex).getRightAxis()
                                .setVisible(isRangeAxisPlottedAgainstWithinAnyDataSource(subPlotIndex,
                                                                                         ChartConstants.RIGHT_YAXIS_INDEX));

        //Title
        getSubPlot(subPlotIndex).getLeftAxis()
                                .getLabel()
                                .setText(determineDefaultRangeAxisTitleBasedOnDataSources(subPlotIndex,
                                                                                          ChartConstants.LEFT_YAXIS_INDEX));
        getSubPlot(subPlotIndex).getRightAxis()
                                .getLabel()
                                .setText(determineDefaultRangeAxisTitleBasedOnDataSources(subPlotIndex,
                                                                                          ChartConstants.RIGHT_YAXIS_INDEX));
    }

    /**
     * @return For the subplot and axis of interest, determine the default range axis title from the data sources.
     */
    private String determineDefaultRangeAxisTitleBasedOnDataSources(final int subPlotIndex, final int rangeAxisIndex)
    {
        String title = null;
        for(int i = 0; i < this._dataSourceParameters.size(); i++)
        {
            //Must account for possibility of null values, because this may be called against override
            //parameters.  I'm not sure, but it may be necessary to assum a 0 for both values if null.
            //I'll wait to see if an error crops up before accepting that as fact.
            if((_dataSourceParameters.get(i).getSubPlotIndex() != null)
                && (_dataSourceParameters.get(i).getYAxisIndex() != null)
                && (_dataSourceParameters.get(i).getSubPlotIndex() == subPlotIndex)
                && (_dataSourceParameters.get(i).getYAxisIndex() == rangeAxisIndex)
                && (_dataSourceParameters.get(i).getDefaultRangeAxisTitle() != null))
            {
                final String labelFound = _dataSourceParameters.get(i).getDefaultRangeAxisTitle();
                if(title == null)
                {
                    title = labelFound;
                }
                else if( !Strings.isNullOrEmpty(labelFound) && (!title.equalsIgnoreCase(labelFound)))
                {
                    title =
                          ChartDrawingParameters.combineDefaultLabelWithOtherDefault(title,
                                                                                    _dataSourceParameters.get(i)
                                                                                                         .getDefaultRangeAxisTitle());
                    break;
                }
            }
        }
        if(title == null)
        {
            title = "No Default Label Specified";
        }
        return title;
    }

    /**
     * @return Checks to see if the indicated axis is plotted directly against by any data source.
     */
    private boolean isRangeAxisPlottedAgainstWithinAnyDataSource(final int subPlotIndex, final int rangeAxisIndex)
    {
        for(int i = 0; i < this._dataSourceParameters.size(); i++)
        {
            //Must account for possibility of null values, because this may be called against override
            //parameters.  I'm not sure, but it may be necessary to assum a 0 for both values if null.
            //I'll wait to see if an error crops up before accepting that as fact.
            if((_dataSourceParameters.get(i).getSubPlotIndex() != null)
                && (_dataSourceParameters.get(i).getYAxisIndex() != null)
                && (_dataSourceParameters.get(i).getSubPlotIndex() == subPlotIndex)
                && ((_dataSourceParameters.get(i).getYAxisIndex() == rangeAxisIndex)))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Sets up default domain axis parameters based on data sources.
     */
    private void setupDefaultDomainAxisParametersBasedOnDataSources()
    {
        String title = null;
        for(int i = 0; i < this._dataSourceParameters.size(); i++)
        {
            //Must account for possibility of null values, because this may be called against override
            //parameters.  I'm not sure, but it may be necessary to assum a 0 for both values if null.
            //I'll wait to see if an error crops up before accepting that as fact.
            if(_dataSourceParameters.get(i).getDefaultDomainAxisTitle() != null)
            {
                final String labelFound = _dataSourceParameters.get(i).getDefaultDomainAxisTitle();
                if(title == null)
                {
                    title = labelFound;
                }
                else if( (!Strings.isNullOrEmpty(labelFound)) && !title.equalsIgnoreCase(labelFound))
                {
                    title = "Multiple Default Labels Found";
                    break;
                }
            }
        }
        if(title == null)
        {
            title = "No Default Label Specified";
        }
        this._domainAxisParameters.getLabel().setText(title);
    }

    /**
     * This method partially reverses the {@link #setupDefaultRangeAxisParametersBasedOnDataSources(int)} method.
     * Basically, if the user has overridden parameters and then synchronizes the subplots, which may call the other
     * method, it is possible that the default range axis parameters will not be set to the user overrides anymore. This
     * will allow for undoing those changes if given the original override parameters that should have been preserved.
     * <br>
     * <br>
     * Now years later, its unclear to me why this couldn't have been handled more elegantly.
     * 
     * @param override {@link ChartDrawingParameters} representing the original overrides before changes were made.
     */
    public void recoverChangesRangeAxisParametersBasedOnOverrideParameters(final ChartDrawingParameters override)
    {
        final List<Integer> usedIndices = override.getUsedSubPlotIndices();
        for(int i = 0; i < usedIndices.size(); i++)
        {
            final SubPlotParameters currentParameters = getSubPlot(usedIndices.get(i));
            final SubPlotParameters overrideParameters = override.getSubPlot(usedIndices.get(i));

            if(overrideParameters == null)
            {
                continue;
            }
            if(overrideParameters.getLeftAxis().getVisible() != null)
            {
                currentParameters.getLeftAxis().setVisible(overrideParameters.getLeftAxis().getVisible());
            }
            if(overrideParameters.getRightAxis().getVisible() != null)
            {
                currentParameters.getRightAxis().setVisible(overrideParameters.getRightAxis().getVisible());
            }

            if(overrideParameters.getLeftAxis().getLabel().getText() != null)
            {
                currentParameters.getLeftAxis()
                                 .getLabel()
                                 .setText(overrideParameters.getLeftAxis().getLabel().getText());
            }
            if(overrideParameters.getRightAxis().getLabel().getText() != null)
            {
                currentParameters.getRightAxis()
                                 .getLabel()
                                 .setText(overrideParameters.getRightAxis().getLabel().getText());
            }
        }
    }

    /**
     * Sets up axis parameter defaults based on the data sources, calling
     * setupDefaultRangeAxisParametersBasedOnDataSources and setupDefaultDomainAxisParametersBasedOnDataSources.
     */
    public void setupAllAxisDefaultsBasedOnDataSources()
    {
        for(int i = 0; i < this._subPlotParameters.size(); i++)
        {
            final int subPlotIndex = _subPlotParameters.get(i).getSubPlotIndex();
            this.setupDefaultRangeAxisParametersBasedOnDataSources(subPlotIndex);
        }
        this.setupDefaultDomainAxisParametersBasedOnDataSources();
    }

    /**
     * Copies the overridden data source parameters, only.
     * 
     * @param override
     */
    public void copyOverriddenDataSourceParametersOnly(final ChartDrawingParameters override)
    {
        final ChartDrawingParameters overrideParms = override;
        for(int i = 0; i < overrideParms.getDataSourceParameters().size(); i++)
        {
            final DataSourceDrawingParameters baseParams =
                                                         this._dataSourceIndexToParametersMap.get(overrideParms.getDataSourceParameters()
                                                                                                               .get(i)
                                                                                                               .getDataSourceOrderIndex());

            if(baseParams != null)
            {
                baseParams.copyOverriddenParameters(overrideParms.getDataSourceParameters().get(i));
            }
        }
    }

    /**
     * Copies all parameters other than the data source parameters, if overridden.
     * 
     * @param override The override parameters to copy from.
     * @param setupDefaultParametersIfNew If true, then if a subplot parameters object does not exist in this, but does
     *            in the override, then when it is made it will have default parameters setup. Pass in true if this is a
     *            set of parameters that cannot have any null entries, meaning it is used for plotting. Pass in false if
     *            this is a set of parameters that are intended as override parameters, so that null entries are
     *            reasonable and expected.
     * @param copySubtitleListParameters If true, then the subtitles are also copied. If false, it is not copied. This
     *            is needed because the subtitles, when copied, always create new subtitles; they do not copy over
     *            existing ones. So, if this method is called more than once somehow, then each call will result in new
     *            subtitles being added even though it is the same set of subtitles. Pass in false to avoid that.
     */
    public void copyOverriddenParametersExceptDataSource(final ChartDrawingParameters override,
                                                         final boolean setupDefaultParametersIfNew,
                                                         final boolean copySubtitleListParameters)
    {
        for(int i = 0; i < override.getSubPlotParameters().size(); i++)
        {
            final int subPlotIndexToCheck = override.getSubPlotParameters().get(i).getSubPlotIndex();
            SubPlotParameters baseParms = this._subPlotIndexToParametersMap.get(subPlotIndexToCheck);

            //If baseParms is null, then the subplot parameters are new for this one.  So,
            //add a new set of parameters and setup default values prior to copying.
            if(baseParms == null)
            {
                baseParms = new SubPlotParameters(subPlotIndexToCheck);
                baseParms.setArguments(getArguments());
                if(setupDefaultParametersIfNew)
                {
                    baseParms.setupDefaultParameters();
                }
                addSubPlotParameters(baseParms);
            }
            baseParms.copyOverriddenParameters(override.getSubPlotParameters().get(i));
        }

        //XXX Add lines for new parameters.
        this._plotTitleParameters.copyOverriddenParameters(override.getPlotTitle());
        this._domainAxisParameters.copyOverriddenParameters(override.getDomainAxis());
        this._legendParameters.copyOverriddenParameters(override.getLegend());
        this._thresholdListParameters.copyOverriddenParameters(override.getThresholdList());
        this._generalParameters.copyOverriddenParameters(override.getGeneralParameters());
        if(copySubtitleListParameters)
        {
            this._subtitleListParameters.copyOverriddenParameters(override.getSubtitleList());
        }
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        for(int i = 0; i < this._dataSourceParameters.size(); i++)
        {
            _dataSourceParameters.get(i).setArguments(arguments);
        }
        for(int i = 0; i < this._subPlotParameters.size(); i++)
        {
            _subPlotParameters.get(i).setArguments(arguments);
        }

        //XXX Add new parameters object below.
        _plotTitleParameters.setArguments(arguments);
        _domainAxisParameters.setArguments(arguments);
        _legendParameters.setArguments(arguments);
        _thresholdListParameters.setArguments(arguments);
        _generalParameters.setArguments(arguments);
        _subtitleListParameters.setArguments(arguments);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final ChartDrawingParameters overrideParms = (ChartDrawingParameters)override;
        copyOverriddenDataSourceParametersOnly(overrideParms);
        copyOverriddenParametersExceptDataSource(overrideParms, false, true);
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        clearParameters();
        final ChartDrawingParameters parms = (ChartDrawingParameters)parameters;

        //Handle the data source drawing parameters
        for(int i = 0; i < parms.getDataSourceParameters().size(); i++)
        {
            addDataSourceParameters((DataSourceDrawingParameters)parms.getDataSourceParameters().get(i).clone());
        }

        //Handle the subplot parameters.
        for(int i = 0; i < parms.getSubPlotParameters().size(); i++)
        {
            addSubPlotParameters((SubPlotParameters)parms.getSubPlotParameters().get(i).clone());
        }

        //XXX Add lines for new parameters.
        _plotTitleParameters.copyFrom(parms.getPlotTitle());
        _domainAxisParameters.copyFrom(parms.getDomainAxis());
        _legendParameters.copyFrom(parms.getLegend());
        _thresholdListParameters.copyFrom(parms.getThresholdList());
        _generalParameters.copyFrom(parms.getGeneralParameters());
        _subtitleListParameters.copyFrom(parms.getSubtitleList());
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        populateDataSourceHashMapFromList();
        populateSubPlotHashMapFromList();
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        final DataSourceDrawingParameters newParameters = new DataSourceDrawingParameters();
        final SubPlotParameters newSubPlotParms = new SubPlotParameters(-1);
        newSubPlotParms.setArguments(getArguments());
        //XXX For other parameters, make sure clearParameters clears them and use the declared one.
        //No constructing necessary!
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }
        else if(elementName.equals(newParameters.getXMLTagName()))
        {
            _dataSourceParameters.add(newParameters);
            return newParameters;
        }
        else if(elementName.equals(newSubPlotParms.getXMLTagName()))
        {
            this._subPlotParameters.add(newSubPlotParms);
            return newSubPlotParms;
        }
        else if(elementName.equals(_plotTitleParameters.getXMLTagName()))
        {
            return this._plotTitleParameters;
        }
        else if(elementName.equals(_domainAxisParameters.getXMLTagName()))
        {
            return this._domainAxisParameters;
        }
        else if(elementName.equals(_legendParameters.getXMLTagName()))
        {
            return this._legendParameters;
        }
        else if(elementName.equals(_thresholdListParameters.getXMLTagName()))
        {
            return this._thresholdListParameters;
        }
        else if(elementName.equals(_generalParameters.getXMLTagName()))
        {
            return this._generalParameters;
        }
        else if(elementName.equals(_subtitleListParameters.getXMLTagName()))
        {
            return this._subtitleListParameters;
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());

        for(int i = 0; i < _dataSourceParameters.size(); i++)
        {
            XMLTools.appendElementIfNotEmpty(mainElement,
                                             _dataSourceParameters.get(i).writePropertyToXMLElement(request));
        }

        for(int i = 0; i < _subPlotParameters.size(); i++)
        {
            XMLTools.appendElementIfNotEmpty(mainElement, _subPlotParameters.get(i).writePropertyToXMLElement(request));
        }

        //XXX Add new parameter writing here.
        XMLTools.appendElementIfNotEmpty(mainElement, _generalParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _plotTitleParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _domainAxisParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _legendParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _thresholdListParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _subtitleListParameters.writePropertyToXMLElement(request));

        return mainElement;
    }

    @Override
    public Object clone()
    {
        final ChartDrawingParameters cloneParms = new ChartDrawingParameters();
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public String toString()
    {
        String results = "ChartDrawingParameters: ";

        for(int i = 0; i < _dataSourceParameters.size(); i++)
        {
            results += "Data source " + i + " = {" + _dataSourceParameters.get(i) + "}; ";
        }

        for(int i = 0; i < _subPlotParameters.size(); i++)
        {
            results += "Subplot " + i + " = {" + _subPlotParameters.get(i) + "}; ";
        }

        //XXX Add to string here.
        results += "Plot Title = {" + _plotTitleParameters + "}; ";
        results += "Domain Axis = {" + _domainAxisParameters + "}; ";
        results += this._legendParameters.toString();
        results += this._thresholdListParameters.toString();
        results += this._generalParameters.toString();
        results += this._subtitleListParameters.toString();
        return results;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof ohd.hseb.charter.parameters.ChartDrawingParameters))
        {
            return false;
        }
        final ChartDrawingParameters other = (ChartDrawingParameters)obj;

        //Order is not important when checking the series.  What is important is that we loop over both lists.
        for(int i = 0; i < this.getDataSourceParametersCount(); i++)
        {
            final DataSourceDrawingParameters thisParms = this.getDataSourceParameters().get(i);
            if(!thisParms.equals(other.getDataSourceParameters(thisParms.getDataSourceOrderIndex())))
            {
                return false;
            }
        }
        for(int i = 0; i < other.getDataSourceParametersCount(); i++)
        {
            final DataSourceDrawingParameters otherParms = other.getDataSourceParameters().get(i);
            if(!otherParms.equals(getDataSourceParameters(otherParms.getDataSourceOrderIndex())))
            {
                return false;
            }
        }

        //Order is not important when checking the subplots.  What is important is that we loop over both lists.
        for(int i = 0; i < this.getSubPlotParametersCount(); i++)
        {
            final SubPlotParameters thisParms = this.getSubPlotParameters().get(i);
            if(!thisParms.equals(other.getSubPlot(thisParms.getSubPlotIndex())))
            {
                return false;
            }
        }
        for(int i = 0; i < other.getSubPlotParametersCount(); i++)
        {
            final SubPlotParameters otherParms = other.getSubPlotParameters().get(i);
            if(!otherParms.equals(this.getSubPlot(otherParms.getSubPlotIndex())))
            {
                return false;
            }
        }

        //XXX Insert other parameters here.
        if(!_plotTitleParameters.equals(other.getPlotTitle()))
        {
            return false;
        }
        if(!_domainAxisParameters.equals(other.getDomainAxis()))
        {
            return false;
        }
        if(!_legendParameters.equals(other.getLegend()))
        {
            return false;
        }
        if(!_thresholdListParameters.equals(other.getThresholdList()))
        {
            return false;
        }
        if(!_generalParameters.equals(other.getGeneralParameters()))
        {
            return false;
        }
        if(!_subtitleListParameters.equals(other.getSubtitleList()))
        {
            return false;
        }

        return true;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        throw new IllegalStateException("Do not apply parameters through this method.  Call applyAllSettings");
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        //ASSUMPTION... The two ensure* methods in this instance have been called prior to 
        //calling this, so that I don't have to make sure I have enough data source or subplot
        //parameter objects.
        for(int i = 0; i < this.getDataSourceParametersCount(); i++)
        {
            this._dataSourceParameters.get(i).haveAllParametersBeenSet();
        }
        for(int i = 0; i < this.getSubPlotParametersCount(); i++)
        {
            this._subPlotParameters.get(i).haveAllParametersBeenSet();
        }

        //XXX Insert other parameters here.
        _plotTitleParameters.haveAllParametersBeenSet();
        _domainAxisParameters.haveAllParametersBeenSet();
        _legendParameters.haveAllParametersBeenSet();
        //_thresholdListParameters.haveAllParametersBeenSet(); Don't call this, let it skip incomplete thresholds
        _generalParameters.haveAllParametersBeenSet();
        //_subtitleListParameters.haveAllParametersBeenSet(); Don't call this, let it skip incomplete subtitles (apply handles them)
    }

    @Override
    public void setupDefaultParameters()
    {
        //DO NOT SETUP THE DATASOURCE DEFAULT PARAMETERS.  THAT IS UP TO THE DATASOURCES TO DO!!!

        for(int i = 0; i < this.getSubPlotParametersCount(); i++)
        {
            this._subPlotParameters.get(i).setupDefaultParameters();
            this.setupDefaultRangeAxisParametersBasedOnDataSources(_subPlotParameters.get(i).getSubPlotIndex());
        }

        _domainAxisParameters.setupDefaultParameters();
        this.setupDefaultDomainAxisParametersBasedOnDataSources();

        //XXX Setup other parameters here.
        _plotTitleParameters.setupDefaultParameters();
        _legendParameters.setupDefaultParameters();
        _thresholdListParameters.setupDefaultParameters();
        _generalParameters.setupDefaultParameters();
        _subtitleListParameters.setupDefaultParameters();
    }

    @Override
    public void argumentsChanged()
    {
        for(int i = 0; i < this.getDataSourceParametersCount(); i++)
        {
            this._dataSourceParameters.get(i).argumentsChanged();
        }
        for(int i = 0; i < this.getSubPlotParametersCount(); i++)
        {
            this._subPlotParameters.get(i).argumentsChanged();
        }

        //XXX Insert other parameters here.
        _plotTitleParameters.argumentsChanged();
        _domainAxisParameters.argumentsChanged();
        _legendParameters.argumentsChanged();
        _thresholdListParameters.argumentsChanged();
        _generalParameters.argumentsChanged();
        _subtitleListParameters.argumentsChanged();
    }

    /**
     * Companion to buildDefaultLabelFromTimeSeriesArrays. It can be used to combine multiple labels into one, assuming
     * each of them is of the form 'type [unit]'. It breaks down the string on [] and attempts to match each of the two
     * components.
     * 
     * @param original The original label.
     * @param other The new label to be combined.
     * @return A label that will either equal the original, "Value [orig unit]", "orig type [multiple units]",
     *         "Value [mutliple units]" or "Multiple Default Values Found".
     */
    public static String combineDefaultLabelWithOtherDefault(final String original, final String other)
    {
        final SegmentedLine segLine1 = new SegmentedLine(original, "[]", SegmentedLine.MODE_NO_EMPTY_SEGS);
        final SegmentedLine segLine2 = new SegmentedLine(other, "[]", SegmentedLine.MODE_NO_EMPTY_SEGS);
    
        if((segLine1.getNumberOfSegments() != 2) || (segLine2.getNumberOfSegments() != 2))
        {
            return "Multiple Default Values Found";
        }
        String result = segLine1.getSegment(0);
        if(!segLine1.getSegment(0).equalsIgnoreCase(segLine2.getSegment(0)))
        {
            result = "Value ";
        }
        String unit = segLine1.getSegment(1);
        if(!segLine1.getSegment(1).equalsIgnoreCase(segLine2.getSegment(1)))
        {
            unit = "multiple units";
        }
        return result + "[" + unit + "]";
    
    }

}
