package ohd.hseb.charter.parameters;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import com.google.common.base.Function;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.effect.Effect;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.ListXMLReader;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.hefs.utils.xml.vars.XMLColor;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;

/**
 * A {@link DefaultChartParameters} subclass that is for a color palette definition, which includes a list of base
 * colors as well as the number of colors to generate in the palette.
 * 
 * @author Hank.Herr
 */
public class ColorPaletteParameters extends DefaultChartParameters
{
    private static final String COLOR_XML_TAG = "color";

    /**
     * List of {@link XMLColor} instances defining the colors to be used to determine the palette.
     */
    private final List<XMLColor> _colorsDefiningPalette = new ArrayList<>();

    /**
     * The number of colors to compute for the palette. If this number equal to or less than the size of
     * {@link #_colorsDefiningPalette}, then the colors, themselves, define the palette. If the value is null, which
     * only happens if it is not set at all, then one color is created per series. Otherwise, the colors will be
     * computed using {@link ColorTools#buildColorPalette(int, Color...)}.
     */
    private final XMLInteger _numberOfColorsInPalette = new XMLInteger("numberOfColors", null);

    public ColorPaletteParameters(final String xmlTagName)
    {
        setXMLTagName(xmlTagName);
    }

    /**
     * Call to add a single color to the palette.
     * 
     * @param c Color to add.
     */
    public void addColor(final Color c)
    {
        _colorsDefiningPalette.add(new XMLColor(COLOR_XML_TAG, c));
    }

    public Color[] getColorsDefiningPalette()
    {
        final Color[] colors = ListTools.mapToArrayList(new Function<XMLColor, Color>()
        {
            @Override
            public Color apply(final XMLColor arg0)
            {
                return arg0.get();
            }
        }, _colorsDefiningPalette).toArray(new Color[]{});

        return colors;
    }

    public void setNumberOfColorsInPalette(final int num)
    {
        _numberOfColorsInPalette.set(num);
    }

    public int getNumberOfColorsInPalette()
    {
        return _numberOfColorsInPalette.get();
    }

    /**
     * @return True if the {@link #_numberOfColorsInPalette} value is NOT null.
     */
    public boolean isNumberOfColorsInPaletteSpecified()
    {
        return _numberOfColorsInPalette.get() != null;
    }

    /**
     * Resets {@link #_numberOfColorsInPalette} to be null, so that default number of colors is used in
     * {@link #buildPalette(int)}.
     */
    public void resetNumberOfColorsInPaletteToDefault()
    {
        _numberOfColorsInPalette.set(null);
    }

    /**
     * @param defaultNumber The default number of colors to generate if {@link #_numberOfColorsInPalette} is unspecified
     *            (i.e., is null).
     * @return Colors to use, either specified in this directory or via
     *         {@link ColorTools#buildColorPalette(int, Color...)} depending on the number of colors to determine; see
     *         comment for {@link #_numberOfColorsInPalette}.
     */
    public Color[] buildPalette(final int defaultNumber)
    {
        final Color[] colors = getColorsDefiningPalette();

        //When the number is not specified; i.e., is the max value, then either return the colors
        //directory or build the palette depending on the default value relative to the number of colors.
        if(!isNumberOfColorsInPaletteSpecified())
        {
            if(defaultNumber <= colors.length)
            {
                return colors;
            }
            return ColorTools.buildColorPalette(defaultNumber, colors);
        }

        //If it is specified and is more than the number of colors in the palette, build the colors.
        if(_numberOfColorsInPalette.get() > _colorsDefiningPalette.size())
        {
            return ColorTools.buildColorPalette(_numberOfColorsInPalette.get(), colors);
        }

        //Otherwise, use them directly.
        return colors;
    }

    @Override
    public XMLWriter getWriter()
    {
        //The list of colors is employed directly when writing.
        final CollectionXMLWriter writer = new CollectionXMLWriter(getXMLTagName(), _colorsDefiningPalette);
        if(isNumberOfColorsInPaletteSpecified())
        {
            writer.addAttribute(_numberOfColorsInPalette, false);
        }
        return writer;
    }

    @Override
    public XMLReader getReader()
    {
        //Clear all parameters as soon as the XML is to be read.
        clearParameters();

        //The reader makes use of ListXMLReader to read in a bunch of XMLColor instances and then copies
        //those colors into _colorsDefiningPalette once the list is completely read in.  In other words, the list
        //is read indirectly and then copied over.  The validate method, herein, is called before the final Effect
        //is called.  
        final ListXMLReader<XMLColor> reader = new ListXMLReader<XMLColor>(getXMLTagName(),
                                                                           new XMLReaderFactory<XMLColor>()
                                                                           {
                                                                               @Override
                                                                               public XMLColor get()
                                                                               {
                                                                                   return new XMLColor(COLOR_XML_TAG);
                                                                               }
                                                                           },
                                                                           new Effect<List<XMLColor>>()
                                                                           {
                                                                               @Override
                                                                               public void perform(final List<XMLColor> input)
                                                                               {
                                                                                   _colorsDefiningPalette.addAll(input);
                                                                               }
                                                                           })
        {
            @Override
            public void validate() throws XMLReaderException
            {
                if(isNumberOfColorsInPaletteSpecified() && (_numberOfColorsInPalette.get() > 0)
                    && (this.size() < _numberOfColorsInPalette.get()))
                {
                    throw new XMLReaderException("The number of colors specified in attribute "
                        + _numberOfColorsInPalette.getXMLTagName()
                        + " is larger than the number of colors provided, which is just weird.");
                }
            }
        };

        //The number of colors in the palette is read in as an attribute.
        reader.addAttribute(_numberOfColorsInPalette, false);

        return reader;
    }

    @Override
    public ColorPaletteParameters clone()
    {
        final ColorPaletteParameters result = new ColorPaletteParameters(getXMLTagName());
        for(final XMLColor color: _colorsDefiningPalette)
        {
            result._colorsDefiningPalette.add(color.clone());
        }
        result._numberOfColorsInPalette.set(_numberOfColorsInPalette.get());
        return result;
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        clearParameters();
        _numberOfColorsInPalette.set(((ColorPaletteParameters)override).getNumberOfColorsInPalette());

        final Color[] colors = ((ColorPaletteParameters)override).getColorsDefiningPalette();
        for(final Color c: colors)
        {
            addColor(c);
        }
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo) throws ChartParametersException
    {
        throw new IllegalStateException("Never call this directly.");
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        //Nothing to check. This can be empty.
    }

    @Override
    public void setupDefaultParameters()
    {
        clearParameters();
    }

    @Override
    public void clearParameters()
    {
        _colorsDefiningPalette.clear();
        resetNumberOfColorsInPaletteToDefault();
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return getXMLTagName();
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        throw new IllegalStateException("Never call this directly.");
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        throw new IllegalStateException("Never call this directly.");
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        throw new IllegalStateException("Never call this directly.");
    }

    /**
     * Note that this validate method is not called during XML reading, but is made available for other uses. During XML
     * reading, a {@link #validate()} method is specifically designed for the designated {@link ListXMLReader} that
     * handles the reading.
     */
    @Override
    public void validate() throws XMLReaderException
    {
        if(isNumberOfColorsInPaletteSpecified() && (_numberOfColorsInPalette.get() > 0)
            && (_colorsDefiningPalette.size() < _numberOfColorsInPalette.get()))
        {
            throw new XMLReaderException("The number of colors specified in attribute "
                + _numberOfColorsInPalette.getXMLTagName()
                + " is larger than the number of colors provided, which is just weird.");
        }
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        throw new IllegalStateException("Never call this directly.");
    }

    @Override
    public boolean equals(final Object obj)
    {
        final ColorPaletteParameters other = (ColorPaletteParameters)obj;
        return (GeneralTools.checkForFullEqualityOfObjects(_numberOfColorsInPalette, other._numberOfColorsInPalette)
            && Arrays.equals(getColorsDefiningPalette(), other.getColorsDefiningPalette()));
    }

}
