package ohd.hseb.charter.parameters;

import java.awt.Color;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Provides parameters guiding a plotter on how to draw a chart. Call the determine* methods to get the value of the
 * parameter to use for a series index. If a determine* method returns null, that should be treated as an error and an
 * exception should be thrown.<br>
 * <br>
 * This includes (1) a set of series parameters, each one corresponding to a series index; (2) default parameters which
 * are used if a series does not have a set of series parameters specified; and (3) the name of the plotter to use.
 * 
 * @author herrhd
 */
public class DataSourceDrawingParameters extends DefaultChartParameters
{
    private final static String LINE_COLOR_PALETTE_XML_TAG = "lineColorPalette";
    private final static String FILL_COLOR_PALETTE_XML_TAG = "fillColorPalette";
    private final static int DEFAULT_DEFINING_SERIES_INDEX = -1;

    /**
     * {@link List} of {@link SeriesDrawingParameters}, each corresponding to a series index.
     */
    private final List<SeriesDrawingParameters> _seriesDrawingParameters = new ArrayList<SeriesDrawingParameters>();

    /**
     * Map of series index to {@link SeriesDrawingParameters}.
     */
    //XXX This is redundant with _seriesDrawingParameters.  Remove the list, keep the map.  (later...) This may not be true.  The map uses indices, whereas
    //the list follows the order specified in the XML.  Need to think about this more.
    private final LinkedHashMap<Integer, SeriesDrawingParameters> _seriesIndexToParametersMap =
                                                                                              new LinkedHashMap<Integer, SeriesDrawingParameters>();

    /**
     * The name of the plotter to use, which defaults to "LineAndScatter".
     */
    private String _plotterName = "LineAndScatter";

    /**
     * The subplot index on which the data source is plotted; default 0.
     */
    private Integer _subPlotIndex = 0;

    /**
     * The range axis index against which it is plotted; default 0.
     */
    private Integer _yAxisIndex = 0;

    /**
     * The chart parameters which own this data source drawing parameters.
     */
    //XXX Wow does this seem like a horrible idea!  There should be an event thrown or something when the synchronize method should be called (see below).
    private ChartDrawingParameters _owningChartParameters = null;

    /**
     * This is not part of the {@link #copyOverriddenParameters(ChartParameters)}; it is assumed that the copy from
     * overridden would not get called unless the order indices matched. Clone and copyFrom, however, will handle it.
     */
    private int _dataSourceOrderIndex = -1;

    /**
     * This is not part of the {@link #copyOverriddenParameters(ChartParameters)} or
     * {@link #copyFrom(GeneralPlugInParameters)}. {@link #clone()}, however, will handle it. It is also not output or
     * input from an XML. It is only set via tools that build the data source parameters. It cannot be overridden. This
     * is a default range axis title, as specified by the corresponding data source.
     */
    private String _defaultRangeAxisTitle = null;

    /**
     * This is not part of the {@link #copyOverriddenParameters(ChartParameters)} or
     * {@link #copyFrom(GeneralPlugInParameters)}. {@link #clone()}, however, will handle it. It is also not output or
     * input from an XML. It is only set via tools that build the data source parameters. It cannot be overridden. This
     * is a default domain axis title, as specified by the corresponding data source.
     */
    private String _defaultDomainAxisTitle = null;

    /**
     * Stores a color palette definition if the lineColorPalette XML element is specified.
     */
    private ColorPaletteParameters _lineColorPaletteDefinition = null;

    /**
     * Stores a color palette definition if the fillColorPalette XML element is specified.
     */
    private ColorPaletteParameters _fillColorPaletteDefinition = null;

    /**
     * Stores parameters that are to be applied to all series in order to define the "defaults" that the individual
     * series drawing parameters may then override.
     */
    private SeriesDrawingParameters _defaultDefiningSeriesParameters = null;

    /**
     * Empty constructor sets the tag and clears the parameters.
     */
    public DataSourceDrawingParameters()
    {
        setXMLTagName("datasourceDrawingParameters");
        clearParameters();
    }

    /**
     * Calls {@link #DataSourceDrawingParameters()} and then sets the {@link #_dataSourceOrderIndex}.
     */
    public DataSourceDrawingParameters(final int dataSourceOrderIndex)
    {
        this();
        _dataSourceOrderIndex = dataSourceOrderIndex;
    }

    /**
     * If the number of series plotted is altered, this will synchronize the {@link #_seriesIndexToParametersMap}
     * accordingly, creating new, blank instances of {@link SeriesDrawingParameters} if necessary. It will also call
     * {@link #removeSeriesDrawingParametersForSeriesIndex(int)} for any series for which the key exceeds the new number
     * of series.
     * 
     * @param numberOfSeries The new number of chart series to plot.
     */
    public void synchronizeSeriesParametersListWithNumberOfSeriesPlotted(final int numberOfSeries)
    {
        for(int i = 0; i < numberOfSeries; i++)
        {
            if(_seriesIndexToParametersMap.get(i) == null)
            {
                final SeriesDrawingParameters params = new SeriesDrawingParameters(i);
                this._seriesDrawingParameters.add(params);
                this._seriesIndexToParametersMap.put(i, params);
            }
        }

        final List<Integer> keyList = new ArrayList<Integer>(_seriesIndexToParametersMap.keySet());
        for(int i = 0; i < keyList.size(); i++)
        {
            if(keyList.get(i) >= numberOfSeries)
            {
                removeSeriesDrawingParametersForSeriesIndex(keyList.get(i));
            }
        }
    }

    /**
     * @return A {@link ColorPaletteParameters} instance with the appropriate tag name to which colors can be added and
     *         the number of colors set.
     */
    public ColorPaletteParameters addLineColorPalette()
    {
        _lineColorPaletteDefinition = new ColorPaletteParameters(LINE_COLOR_PALETTE_XML_TAG);
        return _lineColorPaletteDefinition;
    }

    /**
     * @return A {@link ColorPaletteParameters} instance with the appropriate tag name to which colors can be added and
     *         the number of colors set.
     */
    public ColorPaletteParameters addFillColorPalette()
    {
        _fillColorPaletteDefinition = new ColorPaletteParameters(FILL_COLOR_PALETTE_XML_TAG);
        return _fillColorPaletteDefinition;
    }

    /**
     * @return A {@link SeriesDrawingParameters} instance representing override parameters that will be applied before
     *         individual series parameters are applied. In other words, these will override the system defaults to
     *         define a new set of series defaults which will then be overridden on an individual series basis by those
     *         in {@link #_seriesDrawingParameters}.
     */
    public SeriesDrawingParameters addDefaultDefiningSeriesParameters()
    {
        _defaultDefiningSeriesParameters = new SeriesDrawingParameters(DEFAULT_DEFINING_SERIES_INDEX);
        return _defaultDefiningSeriesParameters;
    }

    public SeriesDrawingParameters getDefaultDefiningSeriesParameters()
    {
        return _defaultDefiningSeriesParameters;
    }

    /**
     * Method cleans up {@link #_seriesDrawingParameters} and populates the map {@link #_seriesIndexToParametersMap}. By
     * clean up, I mean if any series parameters are included with negative indices, then they are assumed to be for the
     * {@link #_defaultDefiningSeriesParameters}. In general, there should only be one such
     * {@link SeriesDrawingParameters}, and only one will be kept (the last one read).
     */
    private void populateHashMapFromList()
    {
        _seriesIndexToParametersMap.clear();
        final List<SeriesDrawingParameters> toRemove = new ArrayList<>();
        for(final SeriesDrawingParameters parms: _seriesDrawingParameters)
        {
            if(parms.getCreatedIndexOfSeriesToModify() < 0)
            {
                _defaultDefiningSeriesParameters = parms;
                parms.setCreatedIndexOfSeriesToModify(DEFAULT_DEFINING_SERIES_INDEX);
                toRemove.add(parms);
            }
            _seriesIndexToParametersMap.put(parms.getCreatedIndexOfSeriesToModify(), parms);
        }
        _seriesDrawingParameters.removeAll(toRemove);
    }

    public List<SeriesDrawingParameters> getSeriesDrawingParameters()
    {
        return this._seriesDrawingParameters;
    }

    public void addSeriesDrawingParameters(final SeriesDrawingParameters parms)
    {
        if(parms == null)
        {
            throw new IllegalArgumentException("SeriesDrawingParameters to add cannot be null.");
        }
        this._seriesDrawingParameters.add(parms);
        this._seriesIndexToParametersMap.put(parms.getCreatedIndexOfSeriesToModify(), parms);
    }

    public boolean doSeriesDrawingParametersExistForSeriesIndex(final int seriesIndex)
    {
        return (_seriesIndexToParametersMap.get(seriesIndex) != null);
    }

    public int getSeriesParametersCount()
    {
        return this._seriesDrawingParameters.size();
    }

    public List<SeriesDrawingParameters> getSeriesDrawingParametersList()
    {
        return this._seriesDrawingParameters;
    }

    public SeriesDrawingParameters getSeriesDrawingParametersForSeriesIndex(final int seriesIndex)
    {
        final SeriesDrawingParameters parms = _seriesIndexToParametersMap.get(seriesIndex);
        return parms;
    }

    public void removeSeriesDrawingParametersForSeriesIndex(final int seriesIndex)
    {
        final SeriesDrawingParameters paramsToRemove = _seriesIndexToParametersMap.get(seriesIndex);
        this._seriesDrawingParameters.remove(paramsToRemove);
        this._seriesIndexToParametersMap.remove(seriesIndex);
    }

    public void removeAllSeriesDrawingParameters()
    {
        _seriesDrawingParameters.clear();
        _seriesIndexToParametersMap.clear();
    }

    public String getPlotterName()
    {
        return this._plotterName;
    }

    public void setPlotterName(final String name)
    {
        this._plotterName = name;
    }

    public int getDataSourceOrderIndex()
    {
        return this._dataSourceOrderIndex;
    }

    public void setDataSourceOrderIndex(final int index)
    {
        _dataSourceOrderIndex = index;
    }

    public Integer getSubPlotIndex()
    {
        return this._subPlotIndex;
    }

    /**
     * This calls {@link ChartDrawingParameters#synchronizeSubPlotParametersListWithDataSourceParameters()} for the
     * {@link #_owningChartParameters}. (Bad idea.)
     * 
     * @param index The new subplot index.
     */
    public void setSubPlotIndex(final Integer index)
    {
        _subPlotIndex = index;

        //XXX This should throw some kind of event or call a listener to pass the needed command up the chain to the owning parameters.
        if(this._owningChartParameters != null)
        {
            _owningChartParameters.synchronizeSubPlotParametersListWithDataSourceParameters();
        }
    }

    public Integer getYAxisIndex()
    {
        return this._yAxisIndex;
    }

    public void setYAxisIndex(final Integer index)
    {
        this._yAxisIndex = index;
    }

    //XXX Again, this is a horrible concept.
    public void setOwningChartParameters(final ChartDrawingParameters parms)
    {
        this._owningChartParameters = parms;
    }

    public void setDefaultRangeAxisTitle(final String titleStr)
    {
        _defaultRangeAxisTitle = titleStr;
    }

    public String getDefaultRangeAxisTitle()
    {
        return _defaultRangeAxisTitle;
    }

    public void setDefaultDomainAxisTitle(final String titleStr)
    {
        _defaultDomainAxisTitle = titleStr;
    }

    public String getDefaultDomainAxisTitle()
    {
        return _defaultDomainAxisTitle;
    }

    /**
     * Applies the color palettes to the {@link SeriesDrawingParameters} contained within
     * {@link #_defaultDefiningSeriesParameters}. This calls
     * {@link ChartTools#applyRotatingColorSchemeToSeries(Color[], Color[], DataSourceDrawingParameters)}.
     */
    public void applyColorPalette()
    {
        Color[] lineColors = null;
        Color[] fillColors = null;
        if(_lineColorPaletteDefinition != null)
        {
            lineColors = _lineColorPaletteDefinition.buildPalette(_seriesDrawingParameters.size());
        }
        if(_fillColorPaletteDefinition != null)
        {
            fillColors = _fillColorPaletteDefinition.buildPalette(_seriesDrawingParameters.size());
        }
        ChartTools.applyRotatingColorSchemeToSeries(lineColors, fillColors, this);
    }

    /**
     * Applies the overrides stored in {@link #_defaultDefiningSeriesParameters} to all parameters within
     * {@link #getSeriesDrawingParameters()}.
     */
    public void applyDefaultDefiningSeriesParameters()
    {
        if(_defaultDefiningSeriesParameters != null)
        {
            for(final SeriesDrawingParameters parms: this.getSeriesDrawingParameters())
            {
                parms.copyOverriddenParameters(_defaultDefiningSeriesParameters);
            }
        }
    }

    /**
     * This copies over {@link #_lineColorPaletteDefinition}, {@link #_fillColorPaletteDefinition}, and
     * {@link #_defaultDefiningSeriesParameters} attributes only since those are involved with deterimining the default
     * appearance of the series drawing parameters, {@link #_seriesDrawingParameters}.
     */
    public void copyOverriddenDefaultSeriesSpecifyParametersOnly(final GeneralPlugInParameters parameters)
    {
        final DataSourceDrawingParameters base = (DataSourceDrawingParameters)parameters;
        if(base._lineColorPaletteDefinition != null)
        {
            _lineColorPaletteDefinition = base._lineColorPaletteDefinition.clone();
        }
        if(base._fillColorPaletteDefinition != null)
        {
            _fillColorPaletteDefinition = base._fillColorPaletteDefinition.clone();
        }
        if(base._defaultDefiningSeriesParameters != null)
        {
            _defaultDefiningSeriesParameters = (SeriesDrawingParameters)base._defaultDefiningSeriesParameters.clone();
        }
    }

    @Override
    public void clearParameters()
    {
        _dataSourceOrderIndex = -1;
        _seriesDrawingParameters.clear();
        _seriesIndexToParametersMap.clear();
        _plotterName = null;
        _subPlotIndex = null;
        _yAxisIndex = null;
        _lineColorPaletteDefinition = null;
        _fillColorPaletteDefinition = null;
        _defaultDefiningSeriesParameters = null;
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        for(int i = 0; i < this._seriesDrawingParameters.size(); i++)
        {
            _seriesDrawingParameters.get(i).setArguments(arguments);
        }
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final DataSourceDrawingParameters base = (DataSourceDrawingParameters)parameters;
        clearParameters();

        //Is this necessary?  None of the other parameters objects do it.
        setXMLTagName(parameters.getXMLTagName());

        _dataSourceOrderIndex = base.getDataSourceOrderIndex();
        setSubPlotIndex(base.getSubPlotIndex());
        _yAxisIndex = base.getYAxisIndex();

        this._defaultDomainAxisTitle = base.getDefaultDomainAxisTitle();
        this._defaultRangeAxisTitle = base.getDefaultRangeAxisTitle();

        //Copy the list of ChartSingleSeriesParameters.
        for(int i = 0; i < base._seriesDrawingParameters.size(); i++)
        {
            _seriesDrawingParameters.add((SeriesDrawingParameters)base._seriesDrawingParameters.get(i).clone());
        }
        populateHashMapFromList();

        //Miscellaneous
        _plotterName = base.getPlotterName();
        copyOverriddenDefaultSeriesSpecifyParametersOnly(base);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        copyOverriddenParameters(override, true);
    }

    /**
     * @param override The parameters to copy.
     * @param setupDefaultsForOverrideParametersNotAlreadyExisting If true, then, when copying over series parameters
     *            that do not correspond to something that exists, it will create a blank set of series parameters
     *            initialized to default. Otherwise, a new set of series parameters will be created, but it will not
     *            have its parameters initialized to default (useful for the GUI).
     */
    public void copyOverriddenParameters(final ChartParameters override,
                                         final boolean setupDefaultsForOverrideParametersNotAlreadyExisting)
    {

        final DataSourceDrawingParameters base = (DataSourceDrawingParameters)override;

        if(base == null)
        {
            return;
        }
        if(base.getPlotterName() != null)
        {
            this.setPlotterName(base.getPlotterName());
        }
        if(base.getSubPlotIndex() != null)
        {
            this.setSubPlotIndex(base.getSubPlotIndex());
        }
        if(base.getYAxisIndex() != null)
        {
            this.setYAxisIndex(base.getYAxisIndex());
        }

        //For every series in _seriesDrawingParameters, try to find a matching one in the base.  If found,
        //copy the overridden parameters.
        for(int i = 0; i < this._seriesDrawingParameters.size(); i++)
        {
            final SeriesDrawingParameters parms =
                                                base.getSeriesDrawingParametersForSeriesIndex(_seriesDrawingParameters.get(i)
                                                                                                                      .getCreatedIndexOfSeriesToModify());
            if(parms != null)
            {
                _seriesDrawingParameters.get(i).copyOverriddenParameters(parms);
            }
        }

        //Now go through the base and check to see if any series do not exist in this _seriesDrawingParameters.
        //If so, add a copy to this, but making sure it is fully specified.  This is because this method should 
        //only be used to override default parameters based on overrides.
        for(int i = 0; i < base.getSeriesParametersCount(); i++)
        {
            if(!doSeriesDrawingParametersExistForSeriesIndex(base.getSeriesDrawingParametersList()
                                                                 .get(i)
                                                                 .getCreatedIndexOfSeriesToModify()))
            {
                final SeriesDrawingParameters workingParameters =
                                                                new SeriesDrawingParameters(base.getSeriesDrawingParametersList()
                                                                                                .get(i)
                                                                                                .getCreatedIndexOfSeriesToModify());
                if(setupDefaultsForOverrideParametersNotAlreadyExisting)
                {
                    workingParameters.setupDefaultParameters();
                }
                workingParameters.copyOverriddenParameters(base.getSeriesDrawingParametersList().get(i));
                addSeriesDrawingParameters(workingParameters);
            }
        }

        this.populateHashMapFromList();

        //Do the palettes and all series parameters.  Note that if not null they are copied wholesale, never piecewise, just as is done in copyFrom.
        if(base._lineColorPaletteDefinition != null)
        {
            _lineColorPaletteDefinition = _lineColorPaletteDefinition.clone();
        }
        if(base._fillColorPaletteDefinition != null)
        {
            _fillColorPaletteDefinition = _fillColorPaletteDefinition.clone();
        }
        if(base._defaultDefiningSeriesParameters != null)
        {
            if(_defaultDefiningSeriesParameters == null)
            {
                _defaultDefiningSeriesParameters =
                                                 (SeriesDrawingParameters)base._defaultDefiningSeriesParameters.clone();
            }
            else
            {
                _defaultDefiningSeriesParameters.copyOverriddenParameters(base._defaultDefiningSeriesParameters);
            }
        }
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return "";
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        populateHashMapFromList();
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        final SeriesDrawingParameters parameters = new SeriesDrawingParameters();
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();

            final String indexStr = attr.getValue("sourceIndex");
            if(indexStr == null)
            {
                throw new XMLReaderException("The sourceIndex attribute is not provided.");
            }
            this._dataSourceOrderIndex = XMLTools.processInt(indexStr);
            if(_dataSourceOrderIndex < 0)
            {
                throw new XMLReaderException("The index of the data source cannot be negative, but is.");
            }
        }

        //Reads in the series.  Note that the finalize method is called later, and that calls the populate map method which will
        //spot a series designated for the all series parameters, which has a negative series index associated with it.
        else if(elementName.equals(parameters.getXMLTagName()))
        {
            _seriesDrawingParameters.add(parameters);
            return parameters;
        }

        //Fill color palette.
        else if(elementName.equals(FILL_COLOR_PALETTE_XML_TAG))
        {
            _fillColorPaletteDefinition = new ColorPaletteParameters(FILL_COLOR_PALETTE_XML_TAG);
            return _fillColorPaletteDefinition.getReader();
        }

        //Line color palette.
        else if(elementName.equals(LINE_COLOR_PALETTE_XML_TAG))
        {
            _lineColorPaletteDefinition = new ColorPaletteParameters(LINE_COLOR_PALETTE_XML_TAG);
            return _lineColorPaletteDefinition.getReader();
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("plotter"))
        {
            _plotterName = value.trim();
        }
        else if(elementName.equals("subPlot"))
        {
            try
            {
                this.setSubPlotIndex(Integer.parseInt(value));
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Subplot index must be an integer, but was '" + value + "'.");
            }
        }
        else if(elementName.equals("yAxis"))
        {
            if(value.equalsIgnoreCase(ChartConstants.YAXIS_XML_STRINGS[ChartConstants.LEFT_YAXIS_INDEX]))
            {
                this._yAxisIndex = ChartConstants.LEFT_YAXIS_INDEX;
            }
            else if(value.equalsIgnoreCase(ChartConstants.YAXIS_XML_STRINGS[ChartConstants.RIGHT_YAXIS_INDEX]))
            {
                this._yAxisIndex = ChartConstants.RIGHT_YAXIS_INDEX;
            }
        }
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());
        mainElement.setAttribute("sourceIndex", "" + this._dataSourceOrderIndex);

        if(_plotterName != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "plotter", _plotterName));
        }

        if(_subPlotIndex != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "subPlot", "" + _subPlotIndex));
        }

        if(_yAxisIndex != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "yAxis",
                                                                   ChartConstants.YAXIS_XML_STRINGS[_yAxisIndex]));
        }

        if(_lineColorPaletteDefinition != null)
        {
            XMLTools.appendElementIfNotEmpty(mainElement,
                                             _lineColorPaletteDefinition.getWriter()
                                                                        .writePropertyToXMLElement(request));
        }

        if(_fillColorPaletteDefinition != null)
        {
            XMLTools.appendElementIfNotEmpty(mainElement,
                                             _fillColorPaletteDefinition.getWriter()
                                                                        .writePropertyToXMLElement(request));
        }

        if(_defaultDefiningSeriesParameters != null)
        {
            XMLTools.appendElementIfNotEmpty(mainElement,
                                             _defaultDefiningSeriesParameters.writePropertyToXMLElement(request));
        }

        for(int i = 0; i < _seriesDrawingParameters.size(); i++)
        {
            XMLTools.appendElementIfNotEmpty(mainElement,
                                             _seriesDrawingParameters.get(i).writePropertyToXMLElement(request));
        }

        return mainElement;
    }

    @Override
    public Object clone()
    {
        final DataSourceDrawingParameters result = new DataSourceDrawingParameters(this._dataSourceOrderIndex);
        result.copyFrom(this);
        result.setDefaultDomainAxisTitle(_defaultDomainAxisTitle);
        result.setDefaultRangeAxisTitle(_defaultRangeAxisTitle);
        return result;
    }

    @Override
    public String toString()
    {
        //TODO Add line and fill color palette and all series at a later time!!!
        String results = "";
        results += "Data source: order index = " + this._dataSourceOrderIndex + "; plotterName = " + this._plotterName
            + "; subPlotIndex = " + this._subPlotIndex + "; yAxisIndex = " + this._yAxisIndex + "; ";
        for(int i = 0; i < this._seriesDrawingParameters.size(); i++)
        {
            results += "Series " + i + " = {" + _seriesDrawingParameters.get(i) + "}; ";
        }
        return results;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof ohd.hseb.charter.parameters.DataSourceDrawingParameters))
        {
            return false;
        }
        final DataSourceDrawingParameters other = (DataSourceDrawingParameters)obj;

        if(this.getDataSourceOrderIndex() != other.getDataSourceOrderIndex())
        {
            return false;
        }
        if(!StringTools.checkForFullEqualityOfStrings(this.getPlotterName(), other.getPlotterName(), true))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(this.getSubPlotIndex(), other.getSubPlotIndex()))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(this.getYAxisIndex(), other.getYAxisIndex()))
        {
            return false;
        }

        //Order is not important when checking the series.  What is important is that we loop over both lists.
        for(int i = 0; i < this.getSeriesParametersCount(); i++)
        {
            final SeriesDrawingParameters thisParms = this.getSeriesDrawingParametersList().get(i);
            if(!thisParms.equals(other.getSeriesDrawingParametersForSeriesIndex(thisParms.getCreatedIndexOfSeriesToModify())))
            {
                return false;
            }
        }
        for(int i = 0; i < other.getSeriesParametersCount(); i++)
        {
            final SeriesDrawingParameters otherParms = other.getSeriesDrawingParametersList().get(i);
            if(!otherParms.equals(getSeriesDrawingParametersForSeriesIndex(otherParms.getCreatedIndexOfSeriesToModify())))
            {
                return false;
            }
        }

        //New stuff added for wres-vis.
        if(!GeneralTools.checkForFullEqualityOfObjects(this._lineColorPaletteDefinition,
                                                       other._lineColorPaletteDefinition))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(this._fillColorPaletteDefinition,
                                                       other._fillColorPaletteDefinition))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(this._defaultDefiningSeriesParameters,
                                                       other._defaultDefiningSeriesParameters))
        {
            return false;
        }

        return true;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        throw new IllegalStateException("Never call this directly.  These parameters are applied in a different way via the plotters.");
    }

    /**
     * If the default drawing parameters are set, then all of its parameters must be set for this to pass. Otherwise,
     * every single series drawing parameters must be full set. This does not, however, check that all the series have
     * parameters for them. That is up to the checkForValidity method within DefaultXYChartDataSource.
     */
    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(this._plotterName == null)
        {
            throw new ChartParametersException("For data source with order index " + this._dataSourceOrderIndex
                + ": Plotter name not specified.");
        }
        if(this._subPlotIndex == null)
        {
            throw new ChartParametersException("For data source with order index " + this._dataSourceOrderIndex
                + ": Subplot index not specified.");
        }
        if(this._yAxisIndex == null)
        {
            throw new ChartParametersException("For data source with order index " + this._dataSourceOrderIndex
                + ": Range axis to use index not specified.");
        }

        for(int i = 0; i < this._seriesDrawingParameters.size(); i++)
        {
            try
            {
                this._seriesDrawingParameters.get(i).haveAllParametersBeenSet();
            }
            catch(final ChartParametersException e)
            {
                throw new ChartParametersException("For data source with order index " + this._dataSourceOrderIndex
                    + ": " + e.getMessage());
            }
        }

        //the line and fill color palette and _allSeriesParameters need not be specified.
    }

    @Override
    public void setupDefaultParameters()
    {
        _plotterName = "LineAndScatter";
        _subPlotIndex = 0;
        _yAxisIndex = ChartConstants.LEFT_YAXIS_INDEX;
        for(int i = 0; i < this._seriesDrawingParameters.size(); i++)
        {
            this._seriesDrawingParameters.get(i).setupDefaultParameters();
        }
        _lineColorPaletteDefinition = null;
        _fillColorPaletteDefinition = null;
        _defaultDefiningSeriesParameters = null;
    }

    @Override
    public void argumentsChanged()
    {
        //Note that _allSeriesParameters is not affected by this since arguments embedded in the
        //parameters are never processed.  Rather, the parameters are copied to other series drawing 
        //parameters and those are processed.
        for(int i = 0; i < this._seriesDrawingParameters.size(); i++)
        {
            this._seriesDrawingParameters.get(i).argumentsChanged();
        }
    }
}
