package ohd.hseb.charter.parameters;

import java.text.SimpleDateFormat;
import java.util.Calendar;

import org.jfree.chart.axis.DateTickUnit;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.charter.tools.DateAxisPlus;
import ohd.hseb.hefs.utils.datetime.HEFSDateTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.hefs.utils.xml.vars.XMLTimeStep;
import ohd.hseb.util.misc.HCalendar;

/**
 * Stores parameters related to a date axis. Other axis parameters are handled through {@link AxisParameters}.
 * 
 * @author Hank.Herr
 */
public class DateAxisParameters extends DefaultChartParameters
{
    public static final String DEFAULT_DATE_TICK_FORMAT = "MM/dd\nHH z";
    public static final String AUTO_TICK_SPACING_VALUE = "auto";

    /**
     * Indicates if the range should be auto-computed.
     */
    private Boolean _autoRange = null;

    /**
     * The lower bound to use.
     */
    private String _lowerBound = null;

    /**
     * The upper bound to use.
     */
    private String _upperBound = null;

    /**
     * The tick spacing. This is passed through to a {@link DateTickUnit} and must be of the form
     * "<quantity> <time unit>" such as "1 day" or "3 weeks".
     */
    private String _tickSpacing = null;

    /**
     * The tick format string. This is passed into a {@link SimpleDateFormat} in order to format the tick marks.
     */
    private String _tickFormat = null;

    /**
     * The tick start hour, specifying a fixed start hour for the very first tick mark.
     */
    private Integer _tickStartHour = null;

    /**
     * Empty constructor.
     */
    public DateAxisParameters()
    {
        setXMLTagName("dateAxis");
    }

    /**
     * This constructor is useful for overriding the parameters through a single method call. The only parameter that
     * must not be null the auto-range flag.
     * 
     * @param autoRange The only required parameter; it cannot be null!
     * @param lowerBound The bound is processed through
     *            {@link HEFSDateTools#computeTime(long, nl.wldelft.util.timeseries.TimeSeriesArray, String)}
     * @param upperBound The bound is processed through
     *            {@link HEFSDateTools#computeTime(long, nl.wldelft.util.timeseries.TimeSeriesArray, String)}
     * @param tickSpacing Passed through to a JFreeChart {@link DateTickUnit} and must be of the form
     *            "<quantity> <time unit>" such as "1 day" or "3 weeks".
     * @param tickFormat Uses {@link SimpleDateFormat}.
     * @param tickStartHour Dictates the hour of day for the first tick mark on the first day. The remaining tick marks
     *            are then computed using the tick spacing.
     */
    public DateAxisParameters(final boolean autoRange,
                              final String lowerBound,
                              final String upperBound,
                              final String tickSpacing,
                              final String tickFormat,
                              final Integer tickStartHour)
    {
        _autoRange = autoRange;
        _lowerBound = lowerBound;
        _upperBound = upperBound;
        _tickFormat = tickFormat;
        _tickSpacing = tickSpacing;
        _tickStartHour = tickStartHour;
    }

    public Boolean getAutoRange()
    {
        return this._autoRange;
    }

    public void setAutoRange(final Boolean b)
    {
        this._autoRange = b;
    }

    public String getTickSpacingUnit()
    {
        if(getTickSpacing() == null)
        {
            return null;
        }
        final String[] dateTicArray = this.getTickSpacing().split(" ");
        return dateTicArray[1].trim();
    }

    public String getLowerBound()
    {
        return _lowerBound;
    }

    public void setLowerBound(final String lowerBound)
    {
        this._lowerBound = lowerBound;
    }

    public String getTickFormat()
    {
        return this._tickFormat;
    }

    public void setTickFormat(final String format)
    {
        this._tickFormat = format;
    }

    public String getTickSpacing()
    {
        return _tickSpacing;
    }

    public void setTickSpacing(final String tickSpacing)
    {
        this._tickSpacing = tickSpacing;
    }

    public Integer getTickSpacingQuantity()
    {
        if(getTickSpacing() == null)
        {
            return null;
        }
        final String[] dateTicArray = this.getTickSpacing().split(" ");
        return Integer.parseInt(dateTicArray[0].trim());
    }

    public Integer getTickStartHour()
    {
        return this._tickStartHour;
    }

    public void setTickStartHour(final Integer hr)
    {
        this._tickStartHour = hr;
    }

    public String getUpperBound()
    {
        return _upperBound;
    }

    public void setUpperBound(final String upperBound)
    {
        this._upperBound = upperBound;
    }

    public boolean isAutoTickSpacing()
    {
        return getTickSpacing().equals(AUTO_TICK_SPACING_VALUE);
    }

    /**
     * @return The {@link DateTickUnit} corresponding to {@link #_tickSpacing}. Null is returned if the
     *         {@link #_tickSpacing} is invalid.
     */
    private DateTickUnit computeDateTickUnit()
    {
        DateTickUnit ticUnit = null;
        final String[] dateTicArray = this.getTickSpacing().split(" ");
        if(dateTicArray.length == 2)
        {
            final int quantity = Integer.parseInt(dateTicArray[0]);
            final String strUnit = dateTicArray[1].trim().toLowerCase();
            if(strUnit.contains("year"))
            {
                ticUnit = new DateTickUnit(DateTickUnit.YEAR, quantity);
            }
            else if(strUnit.contains("month"))
            {
                ticUnit = new DateTickUnit(DateTickUnit.MONTH, quantity);
            }
            else if(strUnit.contains("week"))
            {
                ticUnit = new DateTickUnit(DateTickUnit.DAY, quantity * 7);
            }
            else if(strUnit.contains("day"))
            {
                ticUnit = new DateTickUnit(DateTickUnit.DAY, quantity);
            }
            else if(strUnit.contains("hour"))
            {
                ticUnit = new DateTickUnit(DateTickUnit.HOUR, quantity);
            }
            return ticUnit;
        }
        return null;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        final AxisAutoRangeHelper helper = (AxisAutoRangeHelper)objectAppliedTo;
        helper.setAxisIndex(ChartConstants.DOMAIN_AXIS);
        if(helper.getAxis() instanceof DateAxisPlus)
        {
            final DateAxisPlus dateAxis = (DateAxisPlus)(helper.getAxis());

            dateAxis.setAutoRange(true);

            //Theoretically, this if check should really not be necessary, but since there is a 
            //possibility that someone can force the bound strings to be empty (programmer
            //error), I'll keep it.
            if(!_autoRange)
            {
                dateAxis.setAutoRange(false);

                final Calendar systemTime = ohd.hseb.hefs.utils.tools.GeneralTools.getSystemTime(getArguments());

                final Long longDateLower = HEFSDateTools.computeTime(systemTime, null, getLowerBound());
                final Long longDateUpper = HEFSDateTools.computeTime(systemTime, null, getUpperBound());
                dateAxis.setRange(HCalendar.computeCalendarFromMilliseconds(longDateLower).getTime(),
                                  HCalendar.computeCalendarFromMilliseconds(longDateUpper).getTime());
            }
            else
            {
                helper.autoRangeWithExcludingThresholds();
            }

            //The date tick spacing may be forced to be zero length (programmer error), so 
            //check for it.
            dateAxis.setAutoTickUnitSelection(true);
            if(!isAutoTickSpacing())
            {
                final DateTickUnit ticUnit = computeDateTickUnit();
                if(ticUnit != null)
                {
                    dateAxis.setAutoTickUnitSelection(false);
                    dateAxis.setTickUnit(ticUnit);
                }
            }

            if(dateAxis.getDateFormatOverride() == null)
            {
                dateAxis.setDateFormatOverride(new SimpleDateFormat(DEFAULT_DATE_TICK_FORMAT));
            }

            if(dateAxis.getDateFormatOverride() instanceof SimpleDateFormat)
            {
                final SimpleDateFormat format = (SimpleDateFormat)dateAxis.getDateFormatOverride();

                //update time zone to the current axis
                format.setTimeZone(dateAxis.getTimeZone());
                format.applyPattern(getTickFormat());
            }

            dateAxis.setTickStartHour(getTickStartHour());
        }
    }

    @Override
    public void clearParameters()
    {
        _lowerBound = null;
        _upperBound = null;
        _tickSpacing = null;
        _tickFormat = null;
        _tickStartHour = null;
        _autoRange = null;
    }

    @Override
    public Object clone()
    {
        final DateAxisParameters cloneParms = new DateAxisParameters();
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final DateAxisParameters base = (DateAxisParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final DateAxisParameters base = (DateAxisParameters)override;
        if(base.getLowerBound() != null)
        {
            _lowerBound = base.getLowerBound();
        }
        if(base.getUpperBound() != null)
        {
            _upperBound = base.getUpperBound();
        }
        if(base.getTickSpacing() != null)
        {
            _tickSpacing = base.getTickSpacing();
        }
        if(base.getTickFormat() != null)
        {
            _tickFormat = base.getTickFormat();
        }
        if(base.getTickStartHour() != null)
        {
            _tickStartHour = base.getTickStartHour();
        }
        if(base.getAutoRange() != null)
        {
            _autoRange = base.getAutoRange();
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof DateAxisParameters))
        {
            return false;
        }
        final DateAxisParameters other = (DateAxisParameters)parameters;
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._lowerBound,
                                                                                 other.getLowerBound()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._upperBound,
                                                                                 other.getUpperBound()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._tickSpacing,
                                                                                 other.getTickSpacing()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._tickFormat,
                                                                                other.getTickFormat(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._tickStartHour,
                                                                                 other.getTickStartHour()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_autoRange, other.getAutoRange()))
        {
            return false;
        }

        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
        //check if both of the lower bound and upper bound are set
        if(((this._lowerBound != null) && (this._upperBound == null))
            || ((this._lowerBound == null) && (this._upperBound != null)))
        {
            throw new XMLReaderException("The axis lower/upper bound is not set simultaneously.");
        }
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(_tickFormat == null)
        {
            throw new ChartParametersException("Date tick mark not specified.");
        }
        else
        {
            try
            {
                new SimpleDateFormat(_tickFormat);
            }
            catch(final IllegalArgumentException e)
            {
                throw new ChartParametersException("Date tick mark format '" + _tickFormat + "' is invalid.");
            }
        }

        if(_tickStartHour == null)
        {
            throw new ChartParametersException("Date tick start hour not specified.");
        }

        if(_autoRange == false)
        {
            if((this._lowerBound == null) || (this._upperBound == null))
            {
                throw new ChartParametersException("Axis auto range calculation is turned off, "
                    + "but lower and upper bounds are not set.");
            }
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }
        else if(elementName.equals("lowerBound"))
        {
            try
            {
                setLowerBound(XMLTools.computeDateStringFromAttributes(attr));
            }
            catch(final XMLReaderException e)
            {
                throw new XMLReaderException("startDate element invalid: " + e.getMessage());
            }
        }
        else if(elementName.equals("upperBound"))
        {
            try
            {
                setUpperBound(XMLTools.computeDateStringFromAttributes(attr));
            }
            catch(final XMLReaderException e)
            {
                throw new XMLReaderException("endDate element invalid: " + e.getMessage());
            }
        }
        else if(elementName.equals("tickSpacing"))
        {
            try
            {
                final String unitStr = attr.getValue("unit");
                if(unitStr.equals(AUTO_TICK_SPACING_VALUE))
                {
                    setTickSpacing(AUTO_TICK_SPACING_VALUE);
                }
                else
                {
                    setTickSpacing(XMLTools.computeTimeStepStringFromAttributes(attr,
                                                                                XMLTimeStep.PERIOD_SPECIAL_UNITS));
                }
            }
            catch(final XMLReaderException e)
            {
                throw new XMLReaderException("timeStep element invalid: " + e.getMessage());
            }
        }
        else if((!elementName.equals("lowerBound")) && (!elementName.equals("upperBound"))
            && (!elementName.equals("tickSpacing")) && (!elementName.equals("tickFormat"))
            && (!elementName.equals("tickStartHour")) && (!elementName.equals("autoRange")))
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }

        return null;
    }

    @Override
    public void setupDefaultParameters()
    {
        _lowerBound = null;
        _upperBound = null;
        _tickSpacing = AUTO_TICK_SPACING_VALUE;
        _tickFormat = DEFAULT_DATE_TICK_FORMAT;
        _tickStartHour = 0;
        _autoRange = true;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("tickFormat"))
        {
            setTickFormat(value);
        }
        else if(elementName.equals("tickStartHour"))
        {
            try
            {
                this._tickStartHour = Integer.parseInt(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Element tickStartHour must be an integer, but is '" + value + "'.");
            }
        }
        else if(elementName.equals("autoRange"))
        {
            this._autoRange = Boolean.parseBoolean(value);
        }
    }

    @Override
    public String toString()
    {
        String results = "DateAxisParameters: ";
        results += "lowerBound = '" + _lowerBound + "'; ";
        results += "upperBound = '" + _upperBound + "'; ";
        results += "tickSpacing = '" + _tickSpacing + "'; ";
        results += "tickFormat = '" + _tickFormat + "'; ";
        results += "tickStartHour = " + _tickStartHour + "; ";
        results += "autoRange = " + _autoRange + "; ";

        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());

        if((getLowerBound() != null) && (getUpperBound() != null))
        {
            final Element startDate = XMLTools.createDateElement(request, "lowerBound", getLowerBound());
            final Element endDate = XMLTools.createDateElement(request, "upperBound", getUpperBound());
            mainElement.appendChild(startDate);
            mainElement.appendChild(endDate);
        }
        if(getTickSpacing() != null)
        {
            final Element timeStep = XMLTools.createTimeStepElement(request, "tickSpacing", getTickSpacing());
            mainElement.appendChild(timeStep);
        }
        if(getTickFormat() != null)
        {
            //TODO May need to replace '\n' with html stuff here.
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "tickFormat", getTickFormat()));
        }
        if(getTickStartHour() != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "tickStartHour", "" + getTickStartHour()));
        }
        if(_autoRange != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "autoRange", _autoRange.toString()));
        }
        return mainElement;
    }
}
