package ohd.hseb.charter.parameters;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Image;
import java.util.TimeZone;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.XYPlot;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.datetime.HEFSTimeZoneTools;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceProvider;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Stores general parameters that can be applied to a chart.
 * 
 * @author Hank.Herr
 */
public class GeneralParameters extends DefaultChartParameters
{
    private static final Logger LOG = LogManager.getLogger(SubPlotParameters.class);

    public static final Color DEFAULT_CHART_BACKGROUND_COLOR = Color.WHITE;
    public static final Color DEFAULT_PLOT_BACKGROUND_COLOR = new Color(255, 255, 255, 0);
    public static final Image DEFAULT_BACKGROUND_IMAGE = null;
    public static final Float DEFAULT_GRIDLINE_WIDTH = 0.5f;
    public static final Boolean DEFAULT_GRIDLINE_ISDASHED = true;
    public static final String[] DEFAULT_ISDASHED_DISPLAYED_STRING = {"Yes (Default)", "No", "Yes"};
    public static final Boolean[] DEFAULT_ISDASHED_BOOLEAN = new Boolean[]{null, false, true};
    public static final String TIME_ZONE_TAG_NAME = "timeZone";

    /**
     * The chart background color.
     */
    private Color _chartBackgroundColor = null;

    /**
     * The plot background color, unless it is overridden by subplot.
     */
    private Color _plotBackgroundColor = null;

    /**
     * The background image parameters.
     */
    private final ImageParameters _backgroundImageParameters = new ImageParameters("backgroundImage");

    /**
     * The width of grid lines, if present.
     */
    private Float _gridLineWidth = null;

    /**
     * Flag indicating if lines are dashed or not.
     */
    private Boolean _dashedGridLine = null;

    /**
     * The time zone id that applies to any time based item in the plot, including, for example, date tick mark labels.
     * This is processed by {@link HEFSTimeZoneTools#isTimeZoneValid(String)} AFTER argument replacement.
     */
    private String _timeZoneId = null;

    /**
     * Empty constructor.
     */
    public GeneralParameters()
    {
        setXMLTagName("generalParameters");
    }

    public ImageParameters getBackgroundImageParameters()
    {
        return this._backgroundImageParameters;
    }

    public Color getChartBackgroundColor()
    {
        return _chartBackgroundColor;
    }

    private String getDefaultTimeZone()
    {
        String timeZoneID = HEFSTimeZoneTools.DEFAULT_TIMEZONE;
        try
        {
            if(FewsPiServiceProvider.isDefaultServiceConnected())
            {
                timeZoneID = FewsPiServiceProvider.getTimeZoneId();
            }
        }
        catch(final Throwable t)
        {
            //Keep the original id and ignore the problem.  This is almost certainly because the FEWS PI-service is not connected.
            LOG.debug("Attempted to load the time zone from the FEWS PI-service that is connected and failed: "
                + t.getMessage());
        }
        return timeZoneID;
    }

    public Float getGridLineWidth()
    {
        return _gridLineWidth;
    }

    public Color getPlotBackgroundColor()
    {
        return _plotBackgroundColor;
    }

    public String getTimeZoneId()
    {
        return _timeZoneId;
    }

    public Boolean isDashedGridLine()
    {
        return _dashedGridLine;
    }

    public String getTimeZoneUsedInGraphic()
    {
        if((getTimeZoneId() != null)
            && (HEFSTimeZoneTools.isTimeZoneValid(getArguments().replaceArgumentsInString(getTimeZoneId()))))
        {
            return getArguments().replaceArgumentsInString(getTimeZoneId());
        }
        return getDefaultTimeZone();
    }

    public TimeZone getTimeZone()
    {
        return HEFSTimeZoneTools.retrieveTimeZone(getTimeZoneUsedInGraphic());
    }

    public void setChartBackgroundColor(final Color chartBackgroundColor)
    {
        _chartBackgroundColor = chartBackgroundColor;
    }

    public void setDashedGridLine(final Boolean dashedGridLine)
    {
        _dashedGridLine = dashedGridLine;
    }

    public void setGridLineWidth(final Float gridLineWidth)
    {
        _gridLineWidth = gridLineWidth;
    }

    public void setPlotBackgroundColor(final Color c)
    {
        _plotBackgroundColor = c;
    }

    public void setTimeZoneId(final String timeZoneId)
    {
        this._timeZoneId = timeZoneId;
    }

    public void setupDefaultValuesForNonOverriddenParameters()
    {
        if(_chartBackgroundColor == null)
        {
            this._chartBackgroundColor = DEFAULT_CHART_BACKGROUND_COLOR;
        }
        if(_plotBackgroundColor == null)
        {
            this._plotBackgroundColor = DEFAULT_PLOT_BACKGROUND_COLOR;
        }
        if(_dashedGridLine == null)
        {
            this._dashedGridLine = DEFAULT_GRIDLINE_ISDASHED;
        }
        if(_gridLineWidth == null)
        {
            this._gridLineWidth = DEFAULT_GRIDLINE_WIDTH;
        }
        if(_timeZoneId == null)
        {
            this._timeZoneId = getDefaultTimeZone();
        }
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        this._backgroundImageParameters.setArguments(arguments);
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(this._gridLineWidth == null)
        {
            throw new ChartParametersException("For general parameter, grid line width not specified.");
        }
        if(this._dashedGridLine == null)
        {
            throw new ChartParametersException("For general parameter, grid line style not specified.");
        }
        if(this._chartBackgroundColor == null)
        {
            throw new ChartParametersException("For general parameter, chart background color not specified.");
        }
        if(this._plotBackgroundColor == null)
        {
            throw new ChartParametersException("For general parameter, plot area background color not specified.");
        }
        if(this._timeZoneId == null)
        {
            throw new ChartParametersException("For general parameter, time zone id not specified.");
        }
    }

    @Override
    public void setupDefaultParameters()
    {
        clearParameters();

        this._chartBackgroundColor = DEFAULT_CHART_BACKGROUND_COLOR;
        this._plotBackgroundColor = DEFAULT_PLOT_BACKGROUND_COLOR;
        this._backgroundImageParameters.setupDefaultParameters();
        this._dashedGridLine = DEFAULT_GRIDLINE_ISDASHED;
        this._gridLineWidth = DEFAULT_GRIDLINE_WIDTH;

        this._timeZoneId = getDefaultTimeZone();
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo) throws ChartParametersException
    {
        final JFreeChart chart = (JFreeChart)objectAppliedTo;

        //Load the background image based on parameter
        try
        {
            this._backgroundImageParameters.loadImageIfNecessary();
        }
        catch(final Exception e)
        {
            LOG.warn("Unable to load image with path '" + this.getBackgroundImageParameters().getImagePath()
                + "'; assuming no background image.");
            LOG.warn("Exception message: " + e.getMessage());
        }

        //Background image
        chart.setBackgroundImage(getBackgroundImageParameters().getImage());
        chart.setBackgroundImageAlignment(getBackgroundImageParameters().getImageAlignment());
        /* FB 1835 */
        chart.setBackgroundImageAlpha(getBackgroundImageParameters().getImageTransparency().floatValue());

        //Chart color
        if(getChartBackgroundColor() != null)
        {
            chart.setBackgroundPaint(getChartBackgroundColor());
        }
        else
        {
            chart.setBackgroundPaint(DEFAULT_CHART_BACKGROUND_COLOR);
        }

        //Plot background color, which is applied int he for loop later.
        Color plotColor = getPlotBackgroundColor();
        if(plotColor == null)
        {
            plotColor = DEFAULT_PLOT_BACKGROUND_COLOR;
        }

        final Boolean dashedLine = isDashedGridLine();

        final Float width = getGridLineWidth();

        BasicStroke origDomainStroke = null;
        BasicStroke origRangeStroke = null;
        BasicStroke domainStroke = null;
        BasicStroke rangeStroke = null;

        //Loops through all of the subplots within the overall combined domain plot.
        final CombinedDomainXYPlot combinedPlot = (CombinedDomainXYPlot)chart.getXYPlot();
        ValueAxis axis = null;
        for(int i = 0; i < combinedPlot.getSubplots().size(); i++)
        {
            final XYPlot subPlot = (XYPlot)combinedPlot.getSubplots().get(i);

            //set user-specified time zone to all instances of DateAxis
            for(int j = 0; j < subPlot.getDomainAxisCount(); j++)
            {
                axis = subPlot.getDomainAxis(j);
                if(axis instanceof DateAxis)
                {
                    ((DateAxis)axis).setTimeZone(getTimeZone());
                }
            }
            for(int j = 0; j < subPlot.getRangeAxisCount(); j++)
            {
                axis = subPlot.getRangeAxis(j);
                if(axis instanceof DateAxis)
                {
                    ((DateAxis)axis).setTimeZone(getTimeZone());
                }
            }

            subPlot.setBackgroundPaint(plotColor);

            //For a 0 width, do not show the grid lines.
            if(width == 0)
            {
                subPlot.setDomainGridlinesVisible(false);
                subPlot.setRangeGridlinesVisible(false);
            }
            //Otherwise, setup the stroke appropriately.
            else
            {
                origDomainStroke = (BasicStroke)subPlot.getDomainGridlineStroke();
                if(dashedLine)
                {
                    domainStroke = new BasicStroke(width,
                                                   origDomainStroke.getEndCap(),
                                                   origDomainStroke.getLineJoin(),
                                                   origDomainStroke.getMiterLimit(),
                                                   origDomainStroke.getDashArray(),
                                                   origDomainStroke.getDashPhase());
                }
                else
                {
                    domainStroke = new BasicStroke(width);
                }
                subPlot.setDomainGridlineStroke(domainStroke);

                origRangeStroke = (BasicStroke)subPlot.getRangeGridlineStroke();
                if(dashedLine)
                {
                    rangeStroke = new BasicStroke(width,
                                                  origRangeStroke.getEndCap(),
                                                  origRangeStroke.getLineJoin(),
                                                  origRangeStroke.getMiterLimit(),
                                                  origRangeStroke.getDashArray(),
                                                  origRangeStroke.getDashPhase());
                }
                else
                {
                    rangeStroke = new BasicStroke(width);
                }
                subPlot.setRangeGridlineStroke(rangeStroke);
            }
        }
    }

    @Override
    public void clearParameters()
    {
        getBackgroundImageParameters().clearParameters();
        this._chartBackgroundColor = null;
        this._plotBackgroundColor = null;
        this._gridLineWidth = null;
        this._dashedGridLine = null;
        this._timeZoneId = null;
    }

    @Override
    public Object clone()
    {
        final GeneralParameters results = new GeneralParameters();
        results.copyFrom(this);
        return results;
    }

    /**
     * Does a deep copy.
     */
    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);

        //By clearing and then copying overridden, this becomes equivalent to a deep copy.
        this.clearParameters();
        copyOverriddenParameters((ChartParameters)parameters);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final GeneralParameters base = (GeneralParameters)override;

        getBackgroundImageParameters().copyOverriddenParameters(base.getBackgroundImageParameters());

        if(base.getChartBackgroundColor() != null) //Only copy if not null.
        {
            _chartBackgroundColor = ColorTools.deepCopy(base.getChartBackgroundColor());
        }
        if(base.getPlotBackgroundColor() != null) //Only copy if not null.
        {
            _plotBackgroundColor = ColorTools.deepCopy(base.getPlotBackgroundColor());
        }
        if(base.getGridLineWidth() != null)
        {
            _gridLineWidth = base.getGridLineWidth();
        }
        if(base.isDashedGridLine() != null)
        {
            _dashedGridLine = base.isDashedGridLine();
        }
        if(base.getTimeZoneId() != null)
        {
            _timeZoneId = base.getTimeZoneId();
        }
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof GeneralParameters))
        {
            return false;
        }

        final GeneralParameters base = (GeneralParameters)obj;

        if(!this._backgroundImageParameters.equals(base.getBackgroundImageParameters()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._chartBackgroundColor,
                                                                                 base.getChartBackgroundColor()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._plotBackgroundColor,
                                                                                 base.getPlotBackgroundColor()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._gridLineWidth,
                                                                                 base.getGridLineWidth()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._dashedGridLine,
                                                                                 base.isDashedGridLine()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._timeZoneId,
                                                                                 base.getTimeZoneId()))
        {
            return false;
        }

        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public String toString()
    {
        String result = "GeneralParameters: ";
        result += "xmlTagName = '" + this.getXMLTagName() + "'; ";
        if(_chartBackgroundColor != null)
        {
            result += "chartBackgroundColor = " + this._chartBackgroundColor.toString() + "; ";
        }
        else
        {
            result += "chartBackgroundColor = null; ";
        }

        if(_plotBackgroundColor != null)
        {
            result += "plotBackgroundColor = " + this._plotBackgroundColor.toString() + "; ";
        }
        else
        {
            result += "plotBackgroundColor = null; ";
        }

        result += "Background Image = {" + this._backgroundImageParameters + "}; ";

        if(_dashedGridLine != null)
        {
            result += "dashedGridLine = " + this._dashedGridLine.toString() + "; ";
        }
        else
        {
            result += "dashedGridLine = null; ";
        }

        if(_gridLineWidth != null)
        {
            result += "gridLineWidth = " + this._gridLineWidth.toString() + "; ";
        }
        else
        {
            result += "gridLineWidth = null; ";
        }

        if(_timeZoneId != null)
        {
            result += "timeZoneId = " + this._timeZoneId + "; ";
        }

        return result;
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(this.getXMLTagName()))
        {
            clearParameters();
        }
        else if(elementName.equals("chartBackgroundColor"))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _chartBackgroundColor = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert color attributes to a color: " + e.getMessage());
            }
        }
        else if(elementName.equals("plotBackgroundColor"))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _plotBackgroundColor = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert color attributes to a color: " + e.getMessage());
            }
        }
        else if(elementName.equals("gridLineStyle"))
        {
            this._dashedGridLine = null;
            String val = attr.getValue("isDashed");
            if(val != null)
            {
                try
                {
                    _dashedGridLine = Boolean.parseBoolean(val);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("Grid line attribute, '" + val + "', is INVALID.");
                }
            }

            val = attr.getValue("width");
            if(val != null)
            {
                try
                {
                    _gridLineWidth = Float.parseFloat(val);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("Grid line attribute, '" + val + "', is INVALID.");
                }
            }
        }
        else if(elementName.equals(_backgroundImageParameters.getXMLTagName()))
        {
            return this._backgroundImageParameters;
        }
        else if(!elementName.equals(TIME_ZONE_TAG_NAME))
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }

        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals(TIME_ZONE_TAG_NAME))
        {
            this._timeZoneId = value;
        }
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());
        Element childElement = null;

        if(this.getChartBackgroundColor() != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("chartBackgroundColor",
                                                                request,
                                                                this.getChartBackgroundColor()));
        }

        if(this._plotBackgroundColor != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("plotBackgroundColor",
                                                                request,
                                                                this._plotBackgroundColor));
        }

        XMLTools.appendElementIfNotEmpty(mainElement,
                                         this._backgroundImageParameters.writePropertyToXMLElement(request));

        if(this._dashedGridLine != null || this._gridLineWidth != null)
        {
            childElement = request.createElement("gridLineStyle");
            if(this._dashedGridLine != null)
            {
                childElement.setAttribute("isDashed", String.valueOf(isDashedGridLine()));
            }
            if(this._gridLineWidth != null)
            {
                childElement.setAttribute("width", String.valueOf(getGridLineWidth()));
            }
            mainElement.appendChild(childElement);
        }

        if(_timeZoneId != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, TIME_ZONE_TAG_NAME, _timeZoneId));
        }

        return mainElement;
    }

    /**
     * Converts the provided {@link Boolean} into a Yes/No string using {@link #DEFAULT_ISDASHED_BOOLEAN} and
     * {@link #DEFAULT_ISDASHED_DISPLAYED_STRING}
     */
    public static String getGridLineIsDashedDisplayedString(final Boolean isDashed)
    {
        for(int index = 0; index < DEFAULT_ISDASHED_BOOLEAN.length; index++)
        {
            if(isDashed == DEFAULT_ISDASHED_BOOLEAN[index])
            {
                return DEFAULT_ISDASHED_DISPLAYED_STRING[index];
            }
        }

        return null;
    }

    /**
     * Converts the provided Yes/No string into a {@link Boolean} using {@link #DEFAULT_ISDASHED_BOOLEAN} and
     * {@link #DEFAULT_ISDASHED_DISPLAYED_STRING}
     */
    public static Boolean getGridLineIsDashedValue(final String strIsDashed)
    {
        for(int index = 0; index < DEFAULT_ISDASHED_DISPLAYED_STRING.length; index++)
        {
            if(strIsDashed.equalsIgnoreCase(DEFAULT_ISDASHED_DISPLAYED_STRING[index]))
            {
                return DEFAULT_ISDASHED_BOOLEAN[index];
            }
        }

        return null;
    }
}
