package ohd.hseb.charter.parameters;

import java.awt.Image;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import javax.imageio.ImageIO;

import org.jfree.ui.Align;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Generic parameters class records an image path and alignment and provides a tool to load the image and remember it.
 * 
 * @author hank.herr
 */
public class ImageParameters extends DefaultChartParameters
{
    public static final String DEFAULT_IMAGE_ALIGNMENT_DISPLAY_STRING = "Default (FIT)";
    public static final Integer DEFAULT_IMAGE_ALIGNMENT = Align.FIT;
    /* FB 1835 */
    public static final String DEFAULT_IMAGE_TRANSPARENCY_STRING = "Default (0.5)";
    public static final String DEFAULT_SIGNIFICANT_DIGITS_STRING = "ALL";
    public static final Integer DEFAULT_SIGNIFICANT_DIGITS = 0;
    public static final Double DEFAULT_IMAGE_TRANSPARENCY = 0.5;
    public static final String[] DEFAULT_IMAGE_PATHS = {"noaaWaterMark"};
    public static final String[] IMAGE_PATH_IN_JAR = {"hefsImages/noaaWaterMark.png"};
    public static final String DEFAULT_NOAA_WATER_MARK_PATH_STR = "noaaWaterMark";

    /**
     * The image corresponding to the parameters after loading.
     */
    private Image _image;

    /**
     * The path of the image.
     */
    private String _imagePath;

    /**
     * The alignment of the image within its plotted area.
     */
    private Integer _imageAlignment;
    /**
     * FB 1835 - The transparency associated with the image.
     */
    private Double _imageTransparency;

    /**
     * @param tagName Since this is used in various places, the outside called specifies the XML tag name.
     */
    public ImageParameters(final String tagName)
    {
        this.setXMLTagName(tagName);
        clearParameters();
    }

    /**
     * Loads the image only if {@link #_image} is null AND the image path is specified and is non-empty.
     * 
     * @throws Exception
     */
    public void loadImageIfNecessary() throws Exception
    {
        if((this._image == null) && !Strings.isNullOrEmpty(getImagePath()))
        {
            _image = loadImageBasedOnPath(getImagePath(), getArguments());
        }
    }

    public Image getImage()
    {
        return _image;
    }

    public void setImage(final Image image)
    {
        _image = image;
    }

    public String getImagePath()
    {
        return _imagePath;
    }

    public void setImagePath(final String imagePath)
    {
        _imagePath = imagePath;
        _image = null;
    }

    public Integer getImageAlignment()
    {
        return _imageAlignment;
    }

    public void setImageAlignment(final Integer imageAlignment)
    {
        _imageAlignment = imageAlignment;
    }

    public Double getImageTransparency()
    {
        return _imageTransparency;
    }

    public void setImageTransparency(final Double imageTransparency)
    {
        _imageTransparency = imageTransparency;
    }

    @Override
    public Object clone() throws CloneNotSupportedException
    {
        final ImageParameters copy = new ImageParameters(getXMLTagName());
        copy.copyFrom(this);
        return copy;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo) throws ChartParametersException
    {
        throw new IllegalStateException("Do not call this method.  The parameters are used to load an image which the"
            + " outside caller then must apply to the chart.");
    }

    @Override
    public void clearParameters()
    {
        this._image = null;
        this._imageAlignment = null;
        this._imagePath = null;
        this._imageTransparency = null;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final ImageParameters base = (ImageParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
    }

    /**
     * Note that all is deep copied EXCEPT the image returned by {@link #getImage()}; that is only pointed to so that it
     * need not be reloaded after copying.
     */
    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final ImageParameters base = (ImageParameters)override;
        if(base.getImage() != null)
        {
            setImage(base.getImage());
        }
        if(base.getImagePath() != null)
        {
            setImagePath(base.getImagePath());
        }
        if(base.getImageAlignment() != null)
        {
            setImageAlignment(base.getImageAlignment());
        }
        if(base.getImageTransparency() != null)
        {
            setImageTransparency(base.getImageTransparency());
        }
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(getImagePath() == null)
        {
            throw new ChartParametersException("Image source path not specified.");
        }
        if(getImageAlignment() == null)
        {
            throw new ChartParametersException("Image alignment not specified.");
        }
        if(getImageTransparency() == null)
        {
            throw new ChartParametersException("Image Transparency not specified.");
        }
    }

    @Override
    public void setupDefaultParameters()
    {
        this._image = null;
        this._imagePath = null;
        this._imageAlignment = DEFAULT_IMAGE_ALIGNMENT;
        this._imageTransparency = DEFAULT_IMAGE_TRANSPARENCY;
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
            final String path = attr.getValue("path");
            if(path == null)
            {
                throw new XMLReaderException("Required attribute 'path' not provided with " + getXMLTagName()
                    + " element.");
            }
            setImagePath(path);
            final String strAlignment = attr.getValue("alignment");
            setImageAlignment(determineAlignmentForAlignmentString(strAlignment));
            /* FB 1835 */
            final String strTransparency = attr.getValue("transparency");
            setImageTransparency(determineTransparencyForTransparencyString(strTransparency));
        }
        else
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public String toString()
    {
        String result = "ImageParameters: ";
        result += "xmlTagName = '" + this.getXMLTagName() + "'; ";
        result += "imagePath = " + _imagePath + "; ";
        result += "imageAlignment = " + _imageAlignment + "; ";
        result += "imageTransparency = " + _imageTransparency + "; ";

        return result;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());
        if(getImagePath() != null)
        {
            mainElement.setAttribute("path", getImagePath());

            //Only write the alignment if there is a path.
            if(getImageAlignment() != null)
            {
                mainElement.setAttribute("alignment", determineAlignmentStringForAlignment(getImageAlignment()));
            }
            if(this.getImageTransparency() != null)
            {
                mainElement.setAttribute("transparency",
                                         determineTransparencyStringForTransparency(getImageTransparency()));
            }
        }
        return mainElement;
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.ImageParameters))
        {
            return false;
        }
        final ImageParameters other = (ImageParameters)parameters;

        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._imagePath, other.getImagePath()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._imageAlignment,
                                                                                 other.getImageAlignment()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._imageTransparency,
                                                                                 other.getImageTransparency()))
        {
            return false;
        }

        return true;
    }

    /**
     * @return The item within {@link #IMAGE_PATH_IN_JAR} that corresponds in index to the location of the path provided
     *         within {@link #DEFAULT_IMAGE_PATHS}. The provided path is returned if the path is not within
     *         {@link #DEFAULT_IMAGE_PATHS}.
     */
    public static String findImagePathFromDefaultPath(final String path)
    {
        final int index = Lists.newArrayList(DEFAULT_IMAGE_PATHS).indexOf(path);
        if(index < 0)
        {
            return path;
        }
        return IMAGE_PATH_IN_JAR[index];
    }

    /**
     * This depends upon the arguments being known.
     * 
     * @param path The path to assume. If path is null, then use the getBackgroundImagePath() return value.
     * @param argProc ArgumentsProcessor to use to evaluate the path
     * @throws XMLReaderException If a problem is encountered.
     * @return Image loaded
     */
    public static Image loadImageBasedOnPath(final String path,
                                             final ArgumentsProcessor argProc) throws XMLReaderException
    {
        if(Strings.isNullOrEmpty(path))
        {
            return null;
        }

        String evaluatedPath = argProc.replaceArgumentsInFileName(path.trim());
        evaluatedPath = findImagePathFromDefaultPath(evaluatedPath);

        //Make the replacement for default image paths.
        try
        {
            //check if the image is from jar file
            Image img;
            final URL url =
                          Thread.currentThread()
                                .getContextClassLoader()
                                .getResource(path) == null ? ClassLoader.getSystemResource(evaluatedPath) : Thread.currentThread()
                                                                                                                  .getContextClassLoader()
                                                                                                                  .getResource(path);

            if(url != null)
            {
                img = SwingTools.loadImage(url);
            }
            else
            {
                final File imageFile = new File(evaluatedPath);
                if(!imageFile.isFile() || !imageFile.exists() || !imageFile.canRead())
                {
                    throw new XMLReaderException("Background image file cannot be read or does not exist:\n"
                        + evaluatedPath);
                }

                img = ImageIO.read(imageFile);
                if(img == null)
                {
                    throw new XMLReaderException("Cannot load background image from file:\n" + evaluatedPath);
                }
            }

            return img;
        }
        catch(final MalformedURLException e)
        {
            throw new XMLReaderException("URL to load background image is malformed: " + e.getMessage());
        }
        catch(final IOException e)
        {
            throw new XMLReaderException("Unable to load background image file from file system with path "
                + evaluatedPath + ": " + e.getMessage());
        }
    }

    /**
     * @return Convert alignment {@link Integer} to {@link String}. Either
     *         {@link ChartConstants#determineImageAlignmentStringForAlignment(Integer)} or
     *         {@link #DEFAULT_IMAGE_ALIGNMENT_DISPLAY_STRING} is argument is null.
     */
    public static String determineAlignmentStringForAlignment(final Integer alignment)
    {
        if(alignment == null)
        {
            return DEFAULT_IMAGE_ALIGNMENT_DISPLAY_STRING;
        }
        return ChartConstants.determineImageAlignmentStringForAlignment(alignment);
    }

    /**
     * @return Convert alignment {@link String} to {@link Integer}. Either null if its default or
     *         {@link ChartConstants#determineImageAlignmentForAlignmentString(String)}.
     */
    public static Integer determineAlignmentForAlignmentString(final String strAlignment)
    {
        if(DEFAULT_IMAGE_ALIGNMENT_DISPLAY_STRING.equalsIgnoreCase(strAlignment))
        {
            return null;
        }
        return ChartConstants.determineImageAlignmentForAlignmentString(strAlignment);
    }

    /**
     * @return Convert a transparency fraction to {@link String}. If transparency is null,
     *         {@link #DEFAULT_IMAGE_TRANSPARENCY} is returned.
     */
    public static String determineTransparencyStringForTransparency(final Double transparency)
    {
        if(transparency == null)
        {
            return DEFAULT_IMAGE_TRANSPARENCY_STRING;
        }
        return ChartConstants.determineImageTransparencyStringForTransparency(transparency);
    }

    /**
     * @return Convert {@link String} transparency to fraction. Either {@link #DEFAULT_IMAGE_TRANSPARENCY} or
     *         {@link ChartConstants#determineImageTransparencyForTransparencyString(String)}.
     */
    public static Double determineTransparencyForTransparencyString(final String strTransparency)
    {
        if(DEFAULT_IMAGE_TRANSPARENCY_STRING.equalsIgnoreCase(strTransparency))
        {
            return DEFAULT_IMAGE_TRANSPARENCY;
        }
        return ChartConstants.determineImageTransparencyForTransparencyString(strTransparency);
    }

    /**
     * @return Convert significant digits {@link Integer} to {@link String}. Returns
     *         {@link #DEFAULT_SIGNIFICANT_DIGITS_STRING} if null.
     */
    public static String determineSignificantDigitsStringForSignificantDigits(final Integer significantDigits)
    {
        if(significantDigits == null)
        {
            return DEFAULT_SIGNIFICANT_DIGITS_STRING;
        }
        return ChartConstants.determineSignificantDigitsStringForSignificantDigits(significantDigits);
    }

    /**
     * @return Convert significant digits {@link String} to {@link Integer}. Returns {@link #DEFAULT_SIGNIFICANT_DIGITS}
     *         if its the {@link #DEFAULT_SIGNIFICANT_DIGITS_STRING}. Otherwise, it calls
     *         {@link ChartConstants#determineSignificantDigitsForSignificantDigitsString(String)}.
     */
    public static Integer determineSignificantDigitsForSignificantDigitsString(final String strSignificantDigits)
    {
        if(DEFAULT_SIGNIFICANT_DIGITS_STRING.equalsIgnoreCase(strSignificantDigits))
        {
            return DEFAULT_SIGNIFICANT_DIGITS;
        }
        return ChartConstants.determineSignificantDigitsForSignificantDigitsString(strSignificantDigits);
    }

    /**
     * @return Array of Strings that can be placed in a combo box for selection. Includes a default string as the first
     *         item.
     */
    public static String[] buildSignificantDigitsDisplayedStrings()
    {
        final String[] results = new String[ChartConstants.SIGNIFICANT_DIGITS_STRINGS.length + 1];
        results[0] = DEFAULT_SIGNIFICANT_DIGITS_STRING;
        for(int i = 0; i < ChartConstants.SIGNIFICANT_DIGITS_STRINGS.length; i++)
        {
            results[i + 1] = ChartConstants.SIGNIFICANT_DIGITS_STRINGS[i];
        }
        return results;
    }

    /**
     * @return Array of Strings that can be placed in a combo box for selection. Includes a default string as the first
     *         item.
     */
    public static String[] buildAlignmentDisplayedStrings()
    {
        final String[] results = new String[ChartConstants.IMAGE_ALIGNMENT_STRINGS.length + 1];
        results[0] = DEFAULT_IMAGE_ALIGNMENT_DISPLAY_STRING;
        for(int i = 0; i < ChartConstants.IMAGE_ALIGNMENT_STRINGS.length; i++)
        {
            results[i + 1] = ChartConstants.IMAGE_ALIGNMENT_STRINGS[i];
        }
        return results;
    }

    /**
     * @return Array of Strings that can be placed in a combo box for selection. Includes a default string as the first
     *         item.
     */
    public static String[] buildTansparencyDisplayedStrings()
    {
        final String[] results = new String[ChartConstants.IMAGE_TRANSPARENCY_STRINGS.length + 1];
        results[0] = DEFAULT_IMAGE_TRANSPARENCY_STRING;
        for(int i = 0; i < ChartConstants.IMAGE_TRANSPARENCY_STRINGS.length; i++)
        {
            results[i + 1] = ChartConstants.IMAGE_TRANSPARENCY_STRINGS[i];
        }
        return results;
    }

}
