package ohd.hseb.charter.parameters;

import java.awt.Color;
import java.awt.Font;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Parameters for drawing chart labels.
 * 
 * @author Hank.Herr
 */
public class LabelChartParameters extends DefaultChartParameters
{

    /**
     * The text to display. This can include new-lines.
     */
    private String _text = null;

    /**
     * The font in which to display the text.
     */
    private Font _font = null;

    /**
     * The color of the font.
     */
    private Color _color = null;

    //TODO add horizontal alignment and text alignment, though they won't always be used.

    /**
     * Empty constructor assumes an XML tag name of "label".
     */
    public LabelChartParameters()
    {
        setXMLTagName("label");
    }

    /**
     * Allows for customizing the tag name.
     */
    public LabelChartParameters(final String xmlTagName)
    {
        setXMLTagName(xmlTagName);
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        throw new IllegalStateException("Do not call this method directly.  The parameters of a label are applied through the owning class.");
    }

    @Override
    public void clearParameters()
    {
        _text = null;
        _font = null;
        _color = null;
    }

    @Override
    public Object clone()
    {
        final LabelChartParameters cloneParms = new LabelChartParameters();
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final LabelChartParameters base = (LabelChartParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final LabelChartParameters parameters = (LabelChartParameters)override;
        if(parameters.getText() != null)
        {
            this._text = parameters.getText();
        }
        if(parameters.getFont() != null)
        {
            setFont(new Font(parameters.getFont().getName(),
                             parameters.getFont().getStyle(),
                             parameters.getFont().getSize()));
        }
        if(parameters.getColor() != null)
        {
            this._color = ColorTools.deepCopy(parameters.getColor());
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.LabelChartParameters))
        {
            return false;
        }
        final LabelChartParameters other = (LabelChartParameters)parameters;
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(_text, other.getText(), false))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_font, other.getFont()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_color, other.getColor()))
        {
            return false;
        }
        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    public String getArgumentReplacedText()
    {
        return getArguments().replaceArgumentsInString(getText());
    }

    public Color getColor()
    {
        return _color;
    }

    public Font getFont()
    {
        return _font;
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    public String getText()
    {
        return _text;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(_text == null)
        {
            throw new ChartParametersException("Label text not specified.");
        }
        if(_font == null)
        {
            throw new ChartParametersException("Label font not specified.");
        }
        if(_color == null)
        {
            throw new ChartParametersException("Label color not specified.");
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }

        //For font...
        else if(elementName.equals("font"))
        {
            try
            {
                _font = XMLTools.extractFontFromAttributes(attr);
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException(e.getMessage());
            }
        }

        else if(elementName.equals("color"))
        {
            try
            {
                _color = XMLTools.extractColorFromAttributes(attr);
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException(e.getMessage());
            }
        }

        else if(!elementName.equals("text") && !elementName.equals("rectangleAnchor")
            && !elementName.equals("textAnchor"))
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }

        return null;
    }

    public void setColor(final Color color)
    {
        _color = color;
    }

    public void setFont(final Font font)
    {
        _font = font;
    }

    public void setText(final String text)
    {
        _text = text;
    }

    @Override
    public void setupDefaultParameters()
    {
        this._text = "LABEL";
        this._font = new Font("SanSerif", Font.PLAIN, 12);
        this._color = Color.BLACK;
    }

    public void setupDefaultValuesForNonOverriddenParameters()
    {
        if(_text == null)
        {
            _text = "LABEL";
        }
        if(_font == null)
        {
            this._font = new Font("SanSerif", Font.PLAIN, 12);
        }
        if(_color == null)
        {
            this._color = Color.BLACK;
        }
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("text"))
        {
            _text = value;
        }
    }

    @Override
    public String toString()
    {
        String results = "LabelChartParameters: ";
        results += "text ='" + _text + "'; ";
        results += "font = " + _font + "; ";
        results += "color = " + _color + ".";
        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());
        if(_font != null)
        {
            mainElement.appendChild(XMLTools.createFontElement("font", request, _font));
        }
        if(_color != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("color", request, _color));
        }
        if(_text != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "text", _text));
        }

        return mainElement;
    }
}
