package ohd.hseb.charter.parameters;

import java.awt.BasicStroke;
import java.awt.Color;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.block.BlockContainer;
import org.jfree.chart.block.LabelBlock;
import org.jfree.ui.RectangleEdge;
import org.jfree.ui.RectangleInsets;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Parameters of the chart legend.
 * 
 * @author Hank.Herr
 */
public class LegendParameters extends DefaultChartParameters
{
    public static final String DEFAULT_POSITION_DISPLAY_STRING = "Default (BOTTOM)";
    public static final String DEFAULT_POSITION_STRING = "BOTTOM";

    /**
     * Position string is processed through {@link ChartConstants#determineRectangleEdgeForPosition(String)}.
     */
    private String _positionString = null;

    /**
     * True for visible, false for not.
     */
    private Boolean _visible = null;

    /**
     * Stores parameters of the legend title.
     */
    private final LabelChartParameters _legendTitle = new LabelChartParameters("title");

    /**
     * Stores parameters of the legend entries.
     */
    private final LabelChartParameters _legendEntryFont = new LabelChartParameters("entries");

    /**
     * Fill color for the legend.
     */
    private Color _fillColor;

    /**
     * Border color. Note that this does not use {@link BorderParameters} because of rules for the legend border not
     * matching those for other, arbitrary borders. Blame JFreeChart.
     */
    private Color _borderLineColor;

    /**
     * The border line width. Note that this does not use {@link BorderParameters} because of rules for the legend
     * border not matching those for other, arbitrary borders. Blame JFreeChart.
     */
    private Float _borderLineWidth;

    /**
     * Empty constructor sets the XML tag to be "legend".
     */
    public LegendParameters()
    {
        setXMLTagName("legend");
    }

    public String getPositionString()
    {
        return _positionString;
    }

    public void setPositionString(final String pos)
    {
        this._positionString = pos;
    }

    public Boolean getVisible()
    {
        return _visible;
    }

    public void setVisible(final Boolean b)
    {
        _visible = b;
    }

    public LabelChartParameters getLegendTitle()
    {
        return this._legendTitle;
    }

    public LabelChartParameters getLegendEntryFont()
    {
        return this._legendEntryFont;
    }

    public Color getFillColor()
    {
        return this._fillColor;
    }

    public void setFillColor(final Color c)
    {
        this._fillColor = c;
    }

    public Color getBorderLineColor()
    {
        return this._borderLineColor;
    }

    public void setBorderLineColor(final Color c)
    {
        this._borderLineColor = c;
    }

    public Float getBorderLineWidth()
    {
        return this._borderLineWidth;
    }

    public void setBorderLineWidth(final Float w)
    {
        this._borderLineWidth = w;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        final RectangleEdge edge = ChartConstants.determineRectangleEdgeForPosition(_positionString);

        final JFreeChart chart = (JFreeChart)objectAppliedTo;

        if(!_visible)
        {
            chart.removeLegend();
        }
        else
        {
            final BlockContainer legendContainer = new BlockContainer();

            //Legend title
            if(this.getLegendTitle().getText().length() > 0)
            {
                final String legendText = getArguments().replaceArgumentsInString(getLegendTitle().getText());
                final LabelBlock title = new LabelBlock(legendText,
                                                        getLegendTitle().getFont(),
                                                        getLegendTitle().getColor());
                if (edge == RectangleEdge.TOP || edge == RectangleEdge.BOTTOM)
                {
                    title.setPadding(1, 2, 1, 2);
                    legendContainer.add(title, RectangleEdge.LEFT);
                }
                else
                {
                    legendContainer.add(title, RectangleEdge.TOP);
                }
            }

            //Border
            final org.jfree.chart.block.LineBorder border =
                                                          new org.jfree.chart.block.LineBorder(this._borderLineColor,
                                                                                               new BasicStroke(_borderLineWidth),
                                                                                               new RectangleInsets(_borderLineWidth,
                                                                                                                   _borderLineWidth,
                                                                                                                   _borderLineWidth,
                                                                                                                   _borderLineWidth));
            chart.getLegend().setFrame(border);

            //Legend entry font and color
            chart.getLegend().setItemFont(this._legendEntryFont.getFont());
            chart.getLegend().setItemPaint(_legendEntryFont.getColor());

            legendContainer.add(chart.getLegend().getItemContainer());
            chart.getLegend().setWrapper(legendContainer);
            /* Start FB 1835 */
            chart.getLegend().setBackgroundPaint(this.getFillColor());
            /* End FB 1835 */

            //Legend Position
            chart.getLegend().setPosition(edge);
        }
    }

    @Override
    public void clearParameters()
    {
        _positionString = null;
        _visible = null;
        _borderLineColor = null;
        _fillColor = null;
        _borderLineWidth = null;
        _legendEntryFont.clearParameters();
        _legendTitle.clearParameters();
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final LegendParameters base = (LegendParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final LegendParameters base = (LegendParameters)override;
        if(base.getPositionString() != null)
        {
            this._positionString = base.getPositionString();
        }
        if(base.getVisible() != null)
        {
            this._visible = base.getVisible();
        }
        if(base.getBorderLineColor() != null)
        {
            this._borderLineColor = ColorTools.deepCopy(base.getBorderLineColor());
        }
        if(base.getFillColor() != null)
        {
            this._fillColor = ColorTools.deepCopy(base.getFillColor());
        }
        if(base.getBorderLineWidth() != null)
        {
            this._borderLineWidth = base.getBorderLineWidth();
        }
        this._legendEntryFont.copyOverriddenParameters(base.getLegendEntryFont());
        this._legendTitle.copyOverriddenParameters(base.getLegendTitle());
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(_positionString == null)
        {
            throw new ChartParametersException("Legend position string not set.");
        }
        if(_visible == null)
        {
            throw new ChartParametersException("Legend visibility not set.");
        }
        if(this._borderLineColor == null)
        {
            throw new ChartParametersException("Legend border line color not set.");
        }
        if(this._fillColor == null)
        {
            throw new ChartParametersException("Legend fill color not set.");
        }
        if(this._borderLineWidth == null)
        {
            throw new ChartParametersException("Legend border line width not set.");
        }
        try
        {
            _legendTitle.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            throw new ChartParametersException("For legend title: " + e.getMessage());
        }
        try
        {
            this._legendEntryFont.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            throw new ChartParametersException("For legend entry font: " + e.getMessage());
        }
    }

    @Override
    public void setupDefaultParameters()
    {
        _positionString = DEFAULT_POSITION_STRING;
        _visible = true;
        _borderLineColor = Color.black;
        _fillColor = ColorTools.TRANSPARENT_WHITE;
        _borderLineWidth = 1.0f;
        _legendEntryFont.setupDefaultParameters();
        _legendTitle.setupDefaultParameters();
        _legendTitle.setText("");
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
        if(this._legendEntryFont.getText() != null)
        {
            throw new XMLReaderException("Element legendEntryFont includes text, which is not allowed.  "
                + "Only font and color can be included.");
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(this.getXMLTagName()))
        {
            clearParameters();

            final String visStr = attr.getValue("visible");
            if(visStr != null)
            {
                try
                {
                    _visible = Boolean.parseBoolean(visStr);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("Visible attribute is neither true nor false.");
                }
            }
        }
        else if(elementName.equals("fillColor"))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _fillColor = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert attributes to fillColor: " + e.getMessage());
            }
        }
        else if(elementName.equals("borderLineColor"))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _borderLineColor = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert attributes to borderLineColor: " + e.getMessage());
            }
        }
        else if(elementName.equals(this._legendTitle.getXMLTagName()))
        {
            return this._legendTitle;
        }
        else if(elementName.equals(this._legendEntryFont.getXMLTagName()))
        {
            return _legendEntryFont;
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("borderLineWidth"))
        {
            try
            {
                this._borderLineWidth = Float.parseFloat(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Border line width, '" + value + "', is not a number.");
            }
        }
        else if(elementName.equals("position"))
        {
            if(ChartConstants.determineRectangleEdgeForPosition(value) == null)
            {
                throw new XMLReaderException("Legend position string '" + value + "' is not recognized.");
            }
            this._positionString = value;
        }
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());

        if(_visible != null)
        {
            mainElement.setAttribute("visible", "" + _visible);
        }

        if(_positionString != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "position", _positionString));
        }

        XMLTools.appendElementIfNotEmpty(mainElement, _legendTitle.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement, _legendEntryFont.writePropertyToXMLElement(request));
        if(this._fillColor != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("fillColor", request, _fillColor));
        }
        if(this._borderLineColor != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("borderLineColor", request, _borderLineColor));
        }
        if(this._borderLineWidth != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "borderLineWidth", "" + _borderLineWidth));
        }

        return mainElement;
    }

    @Override
    public Object clone()
    {
        final LegendParameters results = new LegendParameters();
        results.copyFrom(this);
        return results;
    }

    @Override
    public String toString()
    {
        String results = "LegendParameters: ";
        results += "xmlTagName = '" + this.getXMLTagName() + "'; ";
        results += "visible = " + this.getVisible() + "; ";
        results += "positionString = '" + this.getPositionString() + "'; ";
        results += "legendTitle = {" + this.getLegendTitle() + "}; ";
        results += "legendEntryFont = {" + this.getLegendEntryFont() + "}; ";
        results += "fillColor = " + this.getFillColor() + "; ";
        results += "borderLineColor = " + this.getBorderLineColor() + "; ";
        results += "borderLineWidth = " + this.getBorderLineWidth() + "; ";
        return results;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof LegendParameters))
        {
            return false;
        }
        final LegendParameters base = (LegendParameters)obj;

        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_visible, base.getVisible()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(_positionString,
                                                                                base.getPositionString(),
                                                                                false))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_borderLineColor,
                                                                                 base.getBorderLineColor()))
        {
            return false;
        }
        /* Start FB 1835 */
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_fillColor, base.getFillColor()))
        {
            return false;
        }
        /* End FB 1835 */
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_borderLineWidth,
                                                                                 base.getBorderLineWidth()))
        {
            return false;
        }
        if(!this._legendEntryFont.equals(base.getLegendEntryFont()))
        {
            return false;
        }
        if(!this._legendTitle.equals(base.getLegendTitle()))
        {
            return false;
        }
        return true;
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        this._legendEntryFont.setArguments(arguments);
        this._legendTitle.setArguments(arguments);
    }

    /**
     * @return Array of Strings that can be placed in a combo box for selection. Includes a default string as the first
     *         item.
     */
    public static String[] buildPositionDisplayedStrings()
    {
        final String[] results = new String[ChartConstants.POSITION_STRINGS.length + 1];
        results[0] = DEFAULT_POSITION_DISPLAY_STRING;
        for(int i = 0; i < ChartConstants.POSITION_STRINGS.length; i++)
        {
            results[i + 1] = ChartConstants.POSITION_STRINGS[i];
        }
        return results;
    }
}
