package ohd.hseb.charter.parameters;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberTickUnit;
import org.jfree.chart.axis.TickUnits;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.charter.tools.NumberAxisOverride;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.hefs.utils.xml.vars.XMLString;
import ohd.hseb.util.misc.HString;

/**
 * Stores parameters related to a date axis. Other axis parameters are handled through {@link AxisParameters}.
 * 
 * @author Hank.Herr
 */
public class NumericAxisParameters extends DefaultChartParameters
{
    public final static String CATEGORY_LABEL_TAG = "label";
    
    public static double AUTO_TICK_SPACING_VALUE = Double.POSITIVE_INFINITY;
    public static String AUTO_TICK_SPACING_TEXT = "auto";
    public static double AUTO_BOUND_SPACING_VALUE = Double.NaN;
    public static String AUTO_BOUND_SPACING_TEXT = "auto";

    /**
     * Indicates if the range should be auto-computed.
     */
    private Boolean _autoRange = null;

    /**
     * The lower bound to use.
     */
    private Double _lowerBound = null;

    /**
     * The upper bound to use.
     */
    private Double _upperBound = null;

    /**
     * The tick spacing.
     */
    private Double _tickSpacing = null;

    /**
     * The tick format string. This is passed into a {@link DecimalFormat#applyPattern(String)}.
     */
    private String _tickFormat = null;

    /**
     * Indicates if the range, when auto-computed, should include zero always.
     */
    private Boolean _rangeIncludesZero = null;

    /**
     * Indicates if the default tick spacing should be integer only.
     */
    private Boolean _intOnlyTickSpacing = null;

    /**
     * For WRES: Records the label to output for a specific tick.  The numerical value corresponding to the label
     * is (index in list + 1).  So, be sure to create the data such that first category values are plotted against value 1.0,
     * second against 2.0, etc.
     */
    private final List<XMLString> _categoryLabels = new ArrayList<>();

    /**
     * Empty constructor sets the XML tag to numericAxis.
     */
    public NumericAxisParameters()
    {
        setXMLTagName("numericAxis");
    }

    /**
     * This constructor is useful for overriding the parameters through a single method call. The only parameter that
     * must not be null the auto-range flag.
     * 
     * @param autoRange The only required parameter; it cannot be null!
     * @param lowerBound Pass in {@link Double#NaN} if the bound is to be auto-computed. Otherwise, provide a number.
     * @param upperBound Pass in {@link Double#NaN} if the bound is to be auto-computed. Otherwise, provide a number.
     * @param tickFormat {@link String} is processed via {@link DecimalFormat#applyPattern(String)}.
     * @param rangeIncludesZero True to force any auto-range computations to include 0.
     * @param intOnlyTickSpacing True to force tick spacing to be integers.
     */
    public NumericAxisParameters(final boolean autoRange,
                                 final Double lowerBound,
                                 final Double upperBound,
                                 final Double tickSpacing,
                                 final String tickFormat,
                                 final Boolean rangeIncludesZero,
                                 final Boolean intOnlyTickSpacing)
    {
        setAutoRange(autoRange);
        setLowerBound(lowerBound);
        setUpperBound(upperBound);
        setTickSpacing(tickSpacing);
        setTickFormat(tickFormat);
        setRangeIncludesZero(rangeIncludesZero);
        setIntOnlyTickSpacing(intOnlyTickSpacing);
    }

    public Double getLowerBound()
    {
        return _lowerBound;
    }

    /**
     * @return True if the {@link #_lowerBound} is either null or equals {@link #AUTO_BOUND_SPACING_VALUE}.
     */
    public boolean isLowerBoundAuto()
    {
        if(_lowerBound != null)
            return (_lowerBound.equals(AUTO_BOUND_SPACING_VALUE));
        else
            return true;
    }

    public void setLowerBound(final Double lowerBound)
    {
        this._lowerBound = lowerBound;
    }

    public Double getUpperBound()
    {
        return _upperBound;
    }

    /**
     * @return True if the {@link #upperBound} is either null or equals {@link #AUTO_BOUND_SPACING_VALUE}.
     */
    public boolean isUpperBoundAuto()
    {
        if(_upperBound != null)
            return (_upperBound.equals(AUTO_BOUND_SPACING_VALUE));
        else
            return true;
    }

    public void setUpperBound(final Double upperBound)
    {
        this._upperBound = upperBound;
    }

    public Boolean getRangeIncludesZero()
    {
        return this._rangeIncludesZero;
    }

    public void setRangeIncludesZero(final Boolean b)
    {
        this._rangeIncludesZero = b;
    }

    public Boolean getAutoRange()
    {
        return this._autoRange;
    }

    public void setAutoRange(final Boolean b)
    {
        this._autoRange = b;
    }

    public String getTickFormat()
    {
        return _tickFormat;
    }

    public void setTickFormat(final String format)
    {
        this._tickFormat = format;
    }

    public Double getTickSpacing()
    {
        return _tickSpacing;
    }

    public void setTickSpacing(final Double tickSpacing)
    {
        this._tickSpacing = tickSpacing;
    }

    /**
     * @return True if {@link #_tickSpacing} is {@link #AUTO_TICK_SPACING_VALUE}.
     */
    public boolean isAutoTickSpacing()
    {
        return getTickSpacing() == AUTO_TICK_SPACING_VALUE;
    }

    public void setIntOnlyTickSpacing(final Boolean b)
    {
        _intOnlyTickSpacing = b;
    }

    public Boolean getIntOnlyTickSpacing()
    {
        return _intOnlyTickSpacing;
    }

    /**
     * @return The flag {@link #_intOnlyTickSpacing}. If that is not defined (is null), it returns false (no restriction
     *         to tick spacing).
     */
    public Boolean isIntOnlyTickSpacing()
    {
        if(_intOnlyTickSpacing != null)
        {
            return _intOnlyTickSpacing;
        }
        return false;
    }

    /**
     * @return True if the axis is a categorical axis, as determined by whether {@link #_categoryLabels} has anything in it.
     */
    public boolean isCategoricalAxis()
    {
        return !_categoryLabels.isEmpty();
    }
    
    /**
     * Sets the category labels, with the first label being for number value 1.0, second for 2.0, etc.  
     * @param labels Labels in order, starting with that corresponding to number 1.0.  
     */
    public void setCategoryLabels(final String ... labels)
    {
        for (final String label : labels)
        {
            _categoryLabels.add(new XMLString(CATEGORY_LABEL_TAG, label));
        }
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        final AxisAutoRangeHelper helper = (AxisAutoRangeHelper)objectAppliedTo;

        if(helper.getAxis() instanceof NumberAxis)
        {
            //NumberAxis numberAxis = (NumberAxis)objectAppliedTo;
            final NumberAxis numberAxis = (NumberAxis)(helper.getAxis());

            //The auto range flag is checked and bounds set if false.

            //For WRES: Category axis
            if(isCategoricalAxis())
            {
                //XXX May want to autorange here for categorical data, but allow for overrides. Hmmm.... Think!!!
                ((NumberAxisOverride)numberAxis).setCategoricalLabels(_categoryLabels);
            }
            //This is where the old non-WRES stuff kicks in...
            
            //FB 1989  The if is trigger if the bounds are manually specified, then set them without computing the automatic bounds.
            if(!_autoRange && !isLowerBoundAuto() && !isUpperBoundAuto())
            {
                numberAxis.setAutoRange(false);
                numberAxis.setRange(getLowerBound(), getUpperBound());
            }
            //In any other case (where both or one are automatically computed), we need to compute the automatic bounds and then override as needed.
            else
            {
                //Handle the range include zero flag.
                if(getRangeIncludesZero())
                {
                    numberAxis.setAutoRangeIncludesZero(getRangeIncludesZero());
                }

                //Compute the automatic bounds through the helper.
                helper.autoRangeWithExcludingThresholds();

                //If the lower bound is manually specified, override the lower bound computed above with the manual specification.
                if(!isLowerBoundAuto())
                {
                    numberAxis.setRange(getLowerBound(), numberAxis.getRange().getUpperBound());
                }

                //Do the same thing for the upper bound if its manually specified.
                if(!isUpperBoundAuto())
                {
                    numberAxis.setRange(numberAxis.getRange().getLowerBound(), getUpperBound());
                }
            }

            if(getTickSpacing() != AUTO_TICK_SPACING_VALUE)
            {
                numberAxis.setAutoTickUnitSelection(false);
                numberAxis.setTickUnit(new NumberTickUnit(getTickSpacing()));
            }
            else
            {
                numberAxis.setAutoTickUnitSelection(true);
            }

            //set tick format
            if(!getTickFormat().isEmpty())
            {
                final NumberFormat formatter = NumberFormat.getNumberInstance();
                ((DecimalFormat)formatter).applyPattern(_tickFormat);
                //formatter.setMaximumFractionDigits(getDecimalPlaces());
                //formatter.setMinimumFractionDigits(getDecimalPlaces());
                numberAxis.setNumberFormatOverride(formatter);
            }
            else
            {
                numberAxis.setNumberFormatOverride(null);
            }

            //Int only tick spacing.
            if(isIntOnlyTickSpacing() || isCategoricalAxis())
            {
                //Copies over the old units into the new units, but only if their "size" attributes are integers.  
                //This will result in forcing integer tick spacing to be used if auto tick unit selection is used.
                final TickUnits oldUnits = (TickUnits)numberAxis.getStandardTickUnits();
                final TickUnits newUnits = new TickUnits();
                for(int i = 0; i < oldUnits.size(); i++)
                {
                    final NumberTickUnit oldUnit = (NumberTickUnit)oldUnits.get(i);
                    if(oldUnit.getSize() == ((int)oldUnit.getSize()))
                    {
                        newUnits.add(oldUnit);
                    }
                }
                numberAxis.setStandardTickUnits(newUnits);
            }
        }
    }

    @Override
    public void clearParameters()
    {
        _lowerBound = null;
        _upperBound = null;
        _tickSpacing = null;
        _tickFormat = null;
        _rangeIncludesZero = null;
        _autoRange = null;
        _intOnlyTickSpacing = null;
        _categoryLabels.clear();
    }

    @Override
    public Object clone()
    {
        final NumericAxisParameters cloneParms = new NumericAxisParameters();
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final NumericAxisParameters base = (NumericAxisParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final NumericAxisParameters base = (NumericAxisParameters)override;
        if(base.getLowerBound() != null)
        {
            _lowerBound = base.getLowerBound();
        }
        if(base.getUpperBound() != null)
        {
            _upperBound = base.getUpperBound();
        }
        if(base.getTickSpacing() != null)
        {
            _tickSpacing = base.getTickSpacing();
        }
        if(base.getTickFormat() != null)
        {
            _tickFormat = base.getTickFormat();
        }
        if(base.getRangeIncludesZero() != null)
        {
            _rangeIncludesZero = base.getRangeIncludesZero();
        }
        if(base.getAutoRange() != null)
        {
            this._autoRange = base.getAutoRange();
        }
        if(base.getIntOnlyTickSpacing() != null)
        {
            _intOnlyTickSpacing = base.getIntOnlyTickSpacing();
        }
        if(base.isCategoricalAxis())
        {
            for (final XMLString catLabel : base._categoryLabels)
            {
                _categoryLabels.add(new XMLString(catLabel.getXMLTagName(), catLabel.get()));
            }
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof NumericAxisParameters))
        {
            return false;
        }
        final NumericAxisParameters other = (NumericAxisParameters)parameters;
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_lowerBound, other.getLowerBound()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_upperBound, other.getUpperBound()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_tickSpacing, other.getTickSpacing()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(_tickFormat,
                                                                                other.getTickFormat(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_rangeIncludesZero,
                                                                                 other.getRangeIncludesZero()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_autoRange, other.getAutoRange()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_intOnlyTickSpacing,
                                                                                 other.getIntOnlyTickSpacing()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_categoryLabels,
                                                                                 other._categoryLabels))
        {
            return false;
        }

        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
        //check if both of the lower bound and upper bound are set
        if(((this._lowerBound != null) && (this._upperBound == null))
            || ((this._lowerBound == null) && (this._upperBound != null)))

        {
            throw new XMLReaderException("The axis lower/upper bounds are not set to the same value.  "
                + "GraphGen does not support limits where one end is defined and the other is not.");
        }
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(!_autoRange)
        {
            if((this._lowerBound == null) || (_upperBound == null))
            {
                throw new ChartParametersException("Axis auto range calculation is turned off, "
                    + "but lower and upper bounds are not set.");
            }
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }
        
        //For WRES: Category labels read in from XML!
        else if(elementName.equals("categoryLabels"))
        {
            return new CollectionXMLReader<XMLString>("categoryLabels",
                                                      _categoryLabels,
                                                      new XMLReaderFactory<XMLString>()
                                                      {
                                                          @Override
                                                          public XMLString get()
                                                          {
                                                              return new XMLString(CATEGORY_LABEL_TAG);
                                                          }
                                                      });
        }
        else if((!elementName.equals("lowerBound")) && (!elementName.equals("upperBound"))
            && (!elementName.equals("tickSpacing")) && (!elementName.equals("tickFormat"))
            && (!elementName.equals("rangeIncludesZero")) && (!elementName.equals("autoRange"))
            && (!elementName.equals("intOnlyTickSpacing")))
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }

        return null;
    }

    @Override
    public void setupDefaultParameters()
    {
        _autoRange = true;
        _lowerBound = null;
        _upperBound = null;
        _rangeIncludesZero = false;
        _tickSpacing = AUTO_TICK_SPACING_VALUE;
        _tickFormat = "";
        _intOnlyTickSpacing = null;
        _categoryLabels.clear();
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        //FB 1969, if in the XML a bound of NaN is to be displayed as "auto" or "Auto".
        if(elementName.equals("lowerBound"))
        {
            try
            {
                if(value.equalsIgnoreCase(AUTO_BOUND_SPACING_TEXT))
                    this._lowerBound = AUTO_BOUND_SPACING_VALUE;
                else
                    this._lowerBound = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Element lowerBound must be a float, but is '" + value + "'.");
            }
        }
        else if(elementName.equals("upperBound"))
        {
            try
            {
                if(value.equalsIgnoreCase(AUTO_BOUND_SPACING_TEXT))
                    this._upperBound = AUTO_BOUND_SPACING_VALUE;
                else
                    this._upperBound = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Element upperBound must be a float, but is '" + value + "'.");
            }
        }

        else if(elementName.equals("tickSpacing"))
        {
            try
            {
                if(value.equals(AUTO_TICK_SPACING_TEXT))
                {
                    _tickSpacing = AUTO_TICK_SPACING_VALUE;
                }
                else
                {
                    _tickSpacing = Double.parseDouble(value);
                }
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Element tickSpacing must be a float, but is '" + value + "'.");
            }
        }
        else if(elementName.equals("tickFormat"))
        {
            _tickFormat = value;
        }
        else if(elementName.equals("rangeIncludesZero"))
        {
            //This will never throw an exception, even if the string is neither true nor false (false is assumed).
            this._rangeIncludesZero = Boolean.parseBoolean(value);
        }
        else if(elementName.equals("autoRange"))
        {
            this._autoRange = Boolean.parseBoolean(value);
        }
        else if(elementName.equals("intOnlyTickSpacing"))
        {
            this._intOnlyTickSpacing = Boolean.parseBoolean(value);
        }
    }

    @Override
    public String toString()
    {
        String results = "NumericAxisParameters: ";
        results += "lowerBound = " + _lowerBound + "; ";
        results += "upperBound = " + _upperBound + "; ";
        results += "tickSpacing = " + _tickSpacing + "; ";
        results += "tickFormat = '" + _tickFormat + "'; ";
        results += "rangeIncludesZero = " + _rangeIncludesZero + "; ";
        results += "autoRange = " + _autoRange + "; ";
        if(_intOnlyTickSpacing != null)
        {
            results += "intOnlyTickSpacing = " + _intOnlyTickSpacing + "; ";
        }
        if (isCategoricalAxis())
        {
            results += "categoryLabels = ";
            results += HString.buildStringFromList(_categoryLabels, ",");
            results += "; ";
        }

        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());

        //FB 1969, if in the XML a bound of NaN is to be displayed as "auto" or "Auto".
        if((_lowerBound != null) && (_upperBound != null))
        {
            if(isLowerBoundAuto())
            {
                mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                       "lowerBound",
                                                                       "" + AUTO_BOUND_SPACING_TEXT));
            }
            else
            {
                mainElement.appendChild(XMLTools.createTextNodeElement(request, "lowerBound", "" + _lowerBound));
            }
            if(isUpperBoundAuto())
            {
                mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                       "upperBound",
                                                                       "" + AUTO_BOUND_SPACING_TEXT));
            }
            else
            {
                mainElement.appendChild(XMLTools.createTextNodeElement(request, "upperBound", "" + _upperBound));
            }
        }

        if(_tickSpacing != null)
        {
            String value = AUTO_TICK_SPACING_TEXT;
            if(_tickSpacing != AUTO_TICK_SPACING_VALUE)
            {
                value = "" + _tickSpacing;
            }
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "tickSpacing", value));
        }
        if(_tickFormat != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "tickFormat", _tickFormat));
        }
        if(_rangeIncludesZero != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "rangeIncludesZero",
                                                                   _rangeIncludesZero.toString()));
        }
        if(_autoRange != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "autoRange", _autoRange.toString()));
        }
        if(_intOnlyTickSpacing != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "intOnlyTickSpacing",
                                                                   _intOnlyTickSpacing.toString()));
        }
        //For WRES: Category labels written to XML.
        if(isCategoricalAxis())
        {
            final CollectionXMLWriter listWriter = new CollectionXMLWriter("categoryLabels",  _categoryLabels);
            mainElement.appendChild(listWriter.writePropertyToXMLElement(request));
        }
        return mainElement;
    }
}
