package ohd.hseb.charter.parameters;

import java.awt.Color;
import java.util.Arrays;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * The drawing parameters for a single chart series. Note that this one class records all possible drawing parameters,
 * even though it may be more appropriate to have subclass devoted to individual plot types (scatter, box, etc.).
 * 
 * @author Hank.Herr
 */
public class SeriesDrawingParameters extends DefaultChartParameters
{
    /**
     * Stores the names of the different parameters in the order in which they should be displayed in a GUI editor.
     */
    public static String[] ELEMENT_NAMES = {"lineColor", "fillColor", "nameInLegend", "boxWidth", "showInLegend",
        "includeInPlot", "shape", "shapeSize", "lineWidth", "shapeFilled", "barWidth"};

    public static int INDEX_OF_DEFAULT_PARAMETERS = Integer.MIN_VALUE;
    private static String DEFAULT_PARAMETERS_XML_STRING = "default";
    private int _createdIndexOfSeriesToModify = -1;
    private Color _lineColor = null;
    private Color _fillColor = null;
    private String _nameInLegend = null;
    private Boolean _showInLegend = null;
    private String _shapeName = null;
    private Double _shapeSize = null;
    private Boolean _includeInPlot = null; //TODO What do I do with this?
    private Float _lineWidth = null;
    private Boolean _shapeFilled = null;
    private Double _boxWidth = null;
    private Float _barWidth = null;

    /**
     * Empty constructor sets XML tag to seriesDrawingParameters.
     */
    public SeriesDrawingParameters()
    {
        setXMLTagName("seriesDrawingParameters");
    }

    /**
     * Calls empty constructor and then sets {@link #_createdIndexOfSeriesToModify} to the passed in value.
     */
    public SeriesDrawingParameters(final int createdIndexOfSeriesToModify)
    {
        this();
        this._createdIndexOfSeriesToModify = createdIndexOfSeriesToModify;
    }

    /**
     * Empties out the parameters, except for {@link #_createdIndexOfSeriesToModify}.
     */
    public void setAllToEmpty()
    {
        _lineColor = null;
        _fillColor = null;
        _nameInLegend = null;
        _showInLegend = null;
        _includeInPlot = null;
        _boxWidth = null;
        _shapeName = null;
        _shapeSize = null;
        _lineWidth = null;
        _shapeFilled = null;
        _barWidth = null;
    }

    public boolean isLineColorOverridden()
    {
        return (_lineColor != null);
    }

    public boolean isFillColorOverridden()
    {
        return (_fillColor != null);
    }

    public boolean isNameInLegendOverridden()
    {
        return (_nameInLegend != null);
    }

    public boolean isBoxWidthOverridden()
    {
        return (_boxWidth != null);
    }

    public boolean isBarWidthOverridden()
    {
        return (_barWidth != null);
    }

    public boolean isShowInLegendOverridden()
    {
        return (_showInLegend != null);
    }

    public boolean isIncludeInPlotOverridden()
    {
        return (_includeInPlot != null);
    }

    public boolean isShapeNameOverridden()
    {
        return (_shapeName != null);
    }

    public boolean isShapeSizeOverridden()
    {
        return (this._shapeSize != null);
    }

    public boolean isLineWidthOverridden()
    {
        return (this._lineWidth != null);
    }

    public boolean isShapeFilledOverridden()
    {
        return (this._shapeFilled != null);
    }

    public void setCreatedIndexOfSeriesToModify(final int index)
    {
        this._createdIndexOfSeriesToModify = index;
    }

    public int getCreatedIndexOfSeriesToModify()
    {
        return this._createdIndexOfSeriesToModify;
    }

    public void setLineColor(final Color color)
    {
        if(color == null)
        {
            _lineColor = null;
        }
        else
        {
            _lineColor = new Color(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha());
        }
    }

    public Color getLineColor()
    {
        return _lineColor;
    }

    public void setFillColor(final Color color)
    {
        if(color == null)
        {
            _fillColor = null;
        }
        else
        {
            _fillColor = new Color(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha());
        }
    }

    public Color getFillColor()
    {
        return _fillColor;
    }

    public void setNameInLegend(final String name)
    {
        this._nameInLegend = name;
    }

    public String getNameInLegend()
    {
        return _nameInLegend;
    }

    public String getArgumentReplacedNameInLegend()
    {
        return this.getArguments().replaceArgumentsInString(_nameInLegend);
    }

    public void setBoxWidth(final Double width)
    {
        this._boxWidth = width;
    }

    public Double getBoxWidth()
    {
        return this._boxWidth;
    }

    public void setBarWidth(final Float width)
    {
        this._barWidth = width;
    }

    public Float getBarWidth()
    {
        return this._barWidth;
    }

    public void setShowInLegend(final Boolean b)
    {
        this._showInLegend = b;
    }

    public Boolean getShowInLegend()
    {
        return this._showInLegend;
    }

    public void setIncludeInPlot(final Boolean b)
    {
        this._includeInPlot = b;
    }

    public Boolean getIncludeInPlot()
    {
        return this._includeInPlot;
    }

    public void setShapeName(final String name)
    {
        _shapeName = name;
    }

    public String getShapeName()
    {
        return _shapeName;
    }

    public void setShapeSize(final Double shapeSize)
    {
        _shapeSize = shapeSize;
    }

    public Double getShapeSize()
    {
        return _shapeSize;
    }

    public void setLineWidth(final Float width)
    {
        _lineWidth = width;
    }

    public Float getLineWidth()
    {
        return _lineWidth;
    }

    public void setShapeFilled(final Boolean b)
    {
        this._shapeFilled = b;
    }

    public Boolean getShapeFilled()
    {
        return this._shapeFilled;
    }

    /**
     * Turns {@link #ELEMENT_NAMES} as a {@link List} for a contains check.
     */
    public List<String> getElementNames()
    {
        return Arrays.asList(ELEMENT_NAMES);
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            this.clearParameters();

            String indexStr = attr.getValue("createdIndexOfSeriesToModify");
            if(indexStr == null)
            {
                indexStr = DEFAULT_PARAMETERS_XML_STRING;
            }
            try
            {
                if(indexStr.trim().equalsIgnoreCase(DEFAULT_PARAMETERS_XML_STRING))
                {
                    _createdIndexOfSeriesToModify = INDEX_OF_DEFAULT_PARAMETERS;
                }
                else
                {
                    this._createdIndexOfSeriesToModify = Integer.parseInt(indexStr);
                    if(_createdIndexOfSeriesToModify < 0)
                    {
                        _createdIndexOfSeriesToModify = INDEX_OF_DEFAULT_PARAMETERS;
                    }
                }
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("The attribution createdIndexOfSeriesToModify must be an integer but is '"
                    + indexStr + "'.");
            }
        }
        else if(elementName.equals(ELEMENT_NAMES[0]))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _lineColor = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert attributes to color: " + e.getMessage());
            }
        }
        else if(elementName.equals(ELEMENT_NAMES[1]))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _fillColor = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert attributes to area color: " + e.getMessage());
            }
        }
        else if(!getElementNames().contains(elementName))
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }
        return null;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());

        if(_lineColor != null)
        {
            mainElement.appendChild(XMLTools.createColorElement(ELEMENT_NAMES[0], request, this._lineColor));
        }
        if(_fillColor != null)
        {
            mainElement.appendChild(XMLTools.createColorElement(ELEMENT_NAMES[1], request, this._fillColor));
        }
        if(this._nameInLegend != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[2], this._nameInLegend));
        }
        if(this._boxWidth != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[3], "" + _boxWidth));
        }
        if(this._showInLegend != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[4], "" + this._showInLegend));
        }
        if(this._includeInPlot != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   ELEMENT_NAMES[5],
                                                                   "" + this._includeInPlot));
        }
        if(this._shapeName != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[6], "" + this._shapeName));
        }
        if(this._shapeSize != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[7], "" + this._shapeSize));
        }
        if(this._lineWidth != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[8], "" + this._lineWidth));
        }
        if(this._shapeFilled != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[9], "" + this._shapeFilled));
        }
        if(this._barWidth != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, ELEMENT_NAMES[10], "" + _barWidth));
        }

        if(XMLTools.doesElementHaveAttributesOrChildren(mainElement))
        {
            if(_createdIndexOfSeriesToModify == INDEX_OF_DEFAULT_PARAMETERS)
            {
                mainElement.setAttribute("createdIndexOfSeriesToModify", DEFAULT_PARAMETERS_XML_STRING);
            }
            else
            {
                mainElement.setAttribute("createdIndexOfSeriesToModify", "" + this._createdIndexOfSeriesToModify);
            }
        }
        return mainElement;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals(ELEMENT_NAMES[2]))
        {
            this._nameInLegend = value;
        }
        else if(elementName.equals(ELEMENT_NAMES[3]))
        {
            try
            {
                this._boxWidth = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Box width '" + value.trim() + "' is not an integer.");
            }
        }
        else if(elementName.equals(ELEMENT_NAMES[4]))
        {
            _showInLegend = XMLTools.processBoolean(value);
        }
        else if(elementName.equals(ELEMENT_NAMES[5]))
        {
            _includeInPlot = XMLTools.processBoolean(value);
        }
        else if(elementName.equals(ELEMENT_NAMES[6]))
        {
            if(value.length() == 0)
            {
                _shapeName = ChartConstants.NO_SHAPE;
            }
            else
            {
                _shapeName = value.trim();
            }
        }
        else if(elementName.equals(ELEMENT_NAMES[7]))
        {
            try
            {
                this._shapeSize = Double.parseDouble(value.trim());
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Shape size '" + value.trim() + "' is not a number.");
            }
        }
        else if(elementName.equals(ELEMENT_NAMES[8]))
        {
            try
            {
                this._lineWidth = Float.parseFloat(value.trim());
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Line width '" + value.trim() + "' is not a number.");
            }
        }
        else if(elementName.equals(ELEMENT_NAMES[9]))
        {
            _shapeFilled = XMLTools.processBoolean(value);
        }
        else if(elementName.equals(ELEMENT_NAMES[10]))
        {
            try
            {
                final float width = Float.parseFloat(value.trim());
                if((width < 0) || (width > 1))
                {
                    throw new XMLReaderException("Bar width '" + value.trim()
                        + "' must be between 0 and 1 (inclusive).");
                }
                this._barWidth = Float.parseFloat(value.trim());
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Bar width '" + value.trim() + "' is not a number.");
            }
        }
    }

    @Override
    public String toString()
    {
        String result = "ChartSingleSeriesParameters: ";
        result += "xmlTagName = '" + this.getXMLTagName() + "'; ";
        result += "indexOfSeriesToModify = " + this._createdIndexOfSeriesToModify + "; ";

        if(_lineColor != null)
        {
            result += "lineColor = " + this._lineColor.toString() + "; ";
        }
        else
        {
            result += "lineColor = null; ";
        }

        if(_fillColor != null)
        {
            result += "fillColor = " + this._fillColor.toString() + "; ";
        }
        else
        {
            result += "fillColor = null; ";
        }

        result += "nameInLegend = " + this._nameInLegend + "; ";
        result += "boxWidth = " + this._boxWidth + "; ";
        result += "showInLegend = " + this._showInLegend + "; ";
        result += "includeInPlot = " + this._includeInPlot + "; ";
        result += "shapeName = " + this._shapeName + "; ";
        result += "shapeSize = " + this._shapeSize + "; ";
        result += "lineWidth = " + this._lineWidth + "; ";
        result += "barWidth = " + this._barWidth + "; ";
        result += "shapeFilled = " + this._shapeFilled + ".";
        return result;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof ohd.hseb.charter.parameters.SeriesDrawingParameters))
        {
            return false;
        }
        final SeriesDrawingParameters other = (SeriesDrawingParameters)obj;

        //Deal with color separately  as it is tricky because it can be null.
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(getLineColor(), other.getLineColor()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(getFillColor(), other.getFillColor()))
        {
            return false;
        }

        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(getNameInLegend(),
                                                                                other.getNameInLegend(),
                                                                                true))
        {
            return false;
        }

        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(getShapeName(),
                                                                                other.getShapeName(),
                                                                                true))
        {
            return false;
        }

        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this.getShapeSize(),
                                                                                 other.getShapeSize()))
        {
            return false;
        }

        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this.getLineWidth(),
                                                                                 other.getLineWidth()))
        {
            return false;
        }

        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this.getBarWidth(),
                                                                                 other.getBarWidth()))
        {
            return false;
        }

        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this.getBoxWidth(),
                                                                                 other.getBoxWidth()))
        {
            return false;
        }

        if((this.getCreatedIndexOfSeriesToModify() != other.getCreatedIndexOfSeriesToModify())
            || (this.getShowInLegend() != other.getShowInLegend()) || (this.getShapeFilled() != other.getShapeFilled())
            || (this.getIncludeInPlot() != other.getIncludeInPlot()))
        {
            return false;
        }

        return true;
    }

    @Override
    public Object clone()
    {
        final SeriesDrawingParameters results = new SeriesDrawingParameters();
        results.copyFrom(this);
        return results;
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return toString();
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        this.clearParameters();
        final SeriesDrawingParameters base = (SeriesDrawingParameters)parameters;
        setCreatedIndexOfSeriesToModify(base.getCreatedIndexOfSeriesToModify());
        copyOverriddenParameters(base);
//        if(base.getLineColor() != null)
//        {
//            setLineColor(new Color(base.getLineColor().getRed(),
//                                   base.getLineColor().getGreen(),
//                                   base.getLineColor().getBlue(),
//                                   base.getLineColor().getAlpha()));
//        }
//        setIncludeInPlot(base.getIncludeInPlot());
//        setNameInLegend(base.getNameInLegend());
//        setShowInLegend(base.getShowInLegend());
//        if(base.getFillColor() != null)
//        {
//            setFillColor(new Color(base.getFillColor().getRed(),
//                                   base.getFillColor().getGreen(),
//                                   base.getFillColor().getBlue(),
//                                   base.getFillColor().getAlpha()));
//        }
//        setShapeName(base.getShapeName());
//        setShapeSize(base.getShapeSize());
//        setLineWidth(base.getLineWidth());
//        setShapeFilled(base.getShapeFilled());
//        setBoxWidth(base.getBoxWidth());
//        setBarWidth(base.getBarWidth());
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final SeriesDrawingParameters base = (SeriesDrawingParameters)override;
        if(base == null)
        {
            return;
        }
        if(base.isLineColorOverridden())
        {
            this._lineColor = ColorTools.deepCopy(base.getLineColor());
        }
        if(base.isFillColorOverridden())
        {
            this._fillColor = ColorTools.deepCopy(base.getFillColor());
        }
        if(base.isIncludeInPlotOverridden())
        {
            this._includeInPlot = base.getIncludeInPlot();
        }
        if(base.isNameInLegendOverridden())
        {
            this._nameInLegend = base.getNameInLegend();
        }
        if(base.isBoxWidthOverridden())
        {
            this._boxWidth = base.getBoxWidth();
        }
        if(base.isShowInLegendOverridden())
        {
            this._showInLegend = base.getShowInLegend();
        }
        if(base.isShapeNameOverridden())
        {
            this._shapeName = base.getShapeName();
        }
        if(base.isShapeSizeOverridden())
        {
            this._shapeSize = base.getShapeSize();
        }
        if(base.isLineWidthOverridden())
        {
            this._lineWidth = base.getLineWidth();
        }
        if(base.isShapeFilledOverridden())
        {
            this._shapeFilled = base.getShapeFilled();
        }
        if(base.isBarWidthOverridden())
        {
            this._barWidth = base.getBarWidth();
        }
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        throw new IllegalStateException("This method should never be called.  The parameters are applied via a plotter, not directly.");
    }

    @Override
    public void setupDefaultParameters()
    {
        setNameInLegend("Series");
        setShowInLegend(true);
        setLineColor(Color.RED);
        setShapeName(ChartConstants.NO_SHAPE);
        setShapeSize(ChartConstants.DEFAULT_SHAPE_SIZE);
        setBoxWidth(-1.0d);
        setFillColor(Color.red);
        setIncludeInPlot(true);
        setLineWidth(0.5f);
        setShapeFilled(true);
        setBarWidth(0.9f);
    }

    @Override
    public void clearParameters()
    {
        _createdIndexOfSeriesToModify = -1;
        setAllToEmpty();
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        if(_lineColor == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Line color not specified.");
        }
        if(_fillColor == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Fill color not specified.");
        }
        if(_nameInLegend == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Name in legend not specified.");
        }
        if(_showInLegend == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Show in legend not specified.");
        }
        if(_includeInPlot == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Include in plot not specified.");
        }
        if(_boxWidth == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Box width not specified.");
        }
        if(_shapeName == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Shape name not specified.");
        }
        if(_shapeSize == null)
        {
            ohd.hseb.hefs.utils.tools.GeneralTools.dumpStackTrace();
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Shape size not specified.");
        }
        if(_lineWidth == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Line width not specified.");
        }
        if(_shapeFilled == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Shape filled not specified.");
        }
        if(_barWidth == null)
        {
            throw new ChartParametersException("For series with created index of " + this._createdIndexOfSeriesToModify
                + ": Bar width not specified.");
        }
    }
}
