package ohd.hseb.charter.parameters;

import java.awt.Color;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.chart.plot.DatasetRenderingOrder;
import org.jfree.chart.plot.SeriesRenderingOrder;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.charter.jfreechartoverride.ExtendedXYPlot;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Record parameters pertaining to a subplot, including left and right range axis and plot weight. Note that _plotWeight
 * can never be null and is an attribute within the subPlotParameters XML element. Every instance of this element in XML
 * must specify a plot weight. Also note that _subPlotIndex cannot is a required parameter for the constructor and must
 * not be changed by XML.
 * 
 * @author herrhd
 */
public class SubPlotParameters extends DefaultChartParameters
{

    private static final Logger LOG = LogManager.getLogger(SubPlotParameters.class);

    /**
     * When reading XML, the subplot index must be read in on-the-fly, meaning that when a sub plot element is found,
     * the index must be specified as an attribute, otherwise we'll have no way of knowing what the subplot is.
     */
    private int _subPlotIndex = -1;

    /**
     * The weight to apply.
     */
    private int _plotWeight = 1;

    /**
     * Left range axis parameters.
     */
    private final AxisParameters _leftRangeAxisParameters;

    /**
     * Right range axis parameters.
     */
    private final AxisParameters _rightRangeAxisParameters;

    /**
     * Subplot background color.
     */
    private Color _plotBackgroundColor = null;

    /**
     * Subplot background image parameters.
     */
    private final ImageParameters _backgroundImageParameters = new ImageParameters("backgroundImage");

    /**
     * True if the rendering order of series plotted in the subplot is to be in reverse order.
     */
    private Boolean _reverseRendering = null;

    /**
     * True if the legend order for series plotted in teh subplot is to be reversed.
     */
    private Boolean _reverseLegendItemOrder = null;

    /**
     * Constructor takes teh subplot index and initializes {@link #_leftRangeAxisParameters} and
     * {@link #_rightRangeAxisParameters}. The XML tag is set to "subPlot".
     * 
     * @param subPlotIndex The index corresponding to the subplot.
     */
    public SubPlotParameters(final int subPlotIndex)
    {
        setXMLTagName("subPlot");
        _subPlotIndex = subPlotIndex;
        _leftRangeAxisParameters = new AxisParameters(subPlotIndex, ChartConstants.LEFT_YAXIS_INDEX, true);
        _rightRangeAxisParameters = new AxisParameters(subPlotIndex, ChartConstants.RIGHT_YAXIS_INDEX, true);
    }

    public int getSubPlotIndex()
    {
        return _subPlotIndex;
    }

    protected void setSubPlotIndex(int index)
    {
        if(index < 0)
        {
            index = 0;
        }
        _subPlotIndex = index;
    }

    public int getPlotWeight()
    {
        return this._plotWeight;
    }

    public void setPlotWeight(int weight)
    {
        if(weight <= 0)
        {
            weight = 1;
        }
        this._plotWeight = weight;
    }

    public AxisParameters getLeftAxis()
    {
        return this._leftRangeAxisParameters;
    }

    public AxisParameters getRightAxis()
    {
        return this._rightRangeAxisParameters;
    }

    public AxisParameters getAxis(final int axisIndex)
    {
        if(axisIndex == ChartConstants.LEFT_YAXIS_INDEX)
        {
            return this._leftRangeAxisParameters;
        }
        else if(axisIndex == ChartConstants.RIGHT_YAXIS_INDEX)
        {
            return this._rightRangeAxisParameters;
        }
        return null;
    }

    public ImageParameters getBackgroundImageParameters()
    {
        return this._backgroundImageParameters;
    }

    public Color getPlotBackgroundColor()
    {
        return _plotBackgroundColor;
    }

    public void setPlotBackgroundColor(final Color c)
    {
        _plotBackgroundColor = c;
    }

    public Boolean getReverseRendering()
    {
        return this._reverseRendering;
    }

    public void setReverseRendering(final Boolean b)
    {
        this._reverseRendering = b;
    }

    public Boolean getReverseLegendItemOrder()
    {
        return this._reverseLegendItemOrder;
    }

    public void setReverseLegendItemOrder(final Boolean b)
    {
        this._reverseLegendItemOrder = b;
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        this._leftRangeAxisParameters.setArguments(arguments);
        this._rightRangeAxisParameters.setArguments(arguments);
        this._backgroundImageParameters.setArguments(arguments);
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        //XYPlot plotAffected = (XYPlot)objectAppliedTo;
        final ExtendedXYPlot plotAffected = (ExtendedXYPlot)((AxisAutoRangeHelper)objectAppliedTo).getXYPlot();
        plotAffected.setSeriesRenderingOrder(SeriesRenderingOrder.REVERSE); //JFreeChart default!
        plotAffected.setDatasetRenderingOrder(DatasetRenderingOrder.REVERSE);
        if(_reverseRendering)
        {
            plotAffected.setSeriesRenderingOrder(SeriesRenderingOrder.FORWARD);
            plotAffected.setDatasetRenderingOrder(DatasetRenderingOrder.FORWARD);
        }
        if(_reverseLegendItemOrder)
        {
            plotAffected.setReverseLegendItemOrder(_reverseLegendItemOrder);
        }

        //Load the background image based on parameter
        try
        {
            this._backgroundImageParameters.loadImageIfNecessary();
        }
        catch(final Exception e)
        {
            LOG.warn("Unable to load image with path '" + this.getBackgroundImageParameters().getImagePath()
                + "'; assuming no background image.");
            LOG.warn("Exception message: " + e.getMessage());
        }

        //Background image
        plotAffected.setBackgroundImage(_backgroundImageParameters.getImage());
        plotAffected.setBackgroundImageAlignment(_backgroundImageParameters.getImageAlignment());

        //Background color -- Override what is already there, which may have been set in General Parameters
        if(this._plotBackgroundColor != null)
        {
            plotAffected.setBackgroundPaint(_plotBackgroundColor);
        }
        plotAffected.setBackgroundImageAlpha(_backgroundImageParameters.getImageTransparency().floatValue());

        //Apply the left and right axis parameters.  The domain axis is handled elsewhere.
        ((AxisAutoRangeHelper)objectAppliedTo).setAxisIndex(ChartConstants.LEFT_YAXIS_INDEX);
        _leftRangeAxisParameters.applyParametersToChart(objectAppliedTo);
        ((AxisAutoRangeHelper)objectAppliedTo).setAxisIndex(ChartConstants.RIGHT_YAXIS_INDEX);
        _rightRangeAxisParameters.applyParametersToChart(objectAppliedTo);

    }

    @Override
    public void clearParameters()
    {
        this._subPlotIndex = 0;
        this._plotWeight = 1;
        this._leftRangeAxisParameters.clearParameters();
        this._rightRangeAxisParameters.clearParameters();
        this._backgroundImageParameters.clearParameters();
        this._plotBackgroundColor = null;
        this._reverseRendering = null;
        this._reverseLegendItemOrder = null;
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final SubPlotParameters base = (SubPlotParameters)override;
        _subPlotIndex = base.getSubPlotIndex();
        _plotWeight = base.getPlotWeight();
        _leftRangeAxisParameters.copyOverriddenParameters(base.getLeftAxis());
        _rightRangeAxisParameters.copyOverriddenParameters(base.getRightAxis());
        _backgroundImageParameters.copyOverriddenParameters(base.getBackgroundImageParameters());
        if(base.getPlotBackgroundColor() != null)
        {
            this._plotBackgroundColor = ColorTools.deepCopy(base.getPlotBackgroundColor());
        }
        if(base.getReverseRendering() != null)
        {
            this._reverseRendering = base.getReverseRendering();
        }
        if(base.getReverseLegendItemOrder() != null)
        {
            this._reverseLegendItemOrder = base.getReverseLegendItemOrder();
        }
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final SubPlotParameters base = (SubPlotParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
//        this._subPlotIndex = base.getSubPlotIndex();
//        this._plotWeight = base.getPlotWeight();
//        this._leftRangeAxisParameters.copyFrom(base.getLeftAxis());
//        this._rightRangeAxisParameters.copyFrom(base.getRightAxis());
//        this._backgroundImageParameters.copyFrom(base.getBackgroundImageParameters());
//        if(base.getPlotBackgroundColor() != null)
//        {
//            this._plotBackgroundColor = ColorTools.deepCopy(base.getPlotBackgroundColor());
//        }
//        if(base.getReverseRendering() != null)
//        {
//            this._reverseRendering = base.getReverseRendering();
//        }
//        if(base.getReverseLegendItemOrder() != null)
//        {
//            this._reverseLegendItemOrder = base.getReverseLegendItemOrder();
//        }
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            //Subplot index
            final String indexStr = attr.getValue("subPlotIndex");
            if(indexStr == null)
            {
                throw new XMLReaderException("The subPlotIndex attribute is not provided.");
            }
            _subPlotIndex = XMLTools.processInt(indexStr);
            if(_subPlotIndex < 0)
            {
                throw new XMLReaderException("The index of the data source cannot be negative, but is.");
            }
            this._leftRangeAxisParameters.setSubPlotIndex(_subPlotIndex);
            this._rightRangeAxisParameters.setSubPlotIndex(_subPlotIndex);

            //Plot weight
            final String weightStr = attr.getValue("weight");
            if(weightStr == null)
            {
                this._plotWeight = 1;
            }
            else
            {
                _plotWeight = XMLTools.processInt(weightStr);
                if(_plotWeight < 0)
                {
                    throw new XMLReaderException("The weight of the subplot cannot be negative.");
                }
            }
        }
        else if(elementName.equals(this._leftRangeAxisParameters.getXMLTagName()))
        {
            return this._leftRangeAxisParameters;
        }
        else if(elementName.equals(_rightRangeAxisParameters.getXMLTagName()))
        {
            return this._rightRangeAxisParameters;
        }
        else if(elementName.equals("backgroundColor"))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _plotBackgroundColor = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert color attributes to a color: " + e.getMessage());
            }
        }
        else if(elementName.equals(_backgroundImageParameters.getXMLTagName()))
        {
            return this._backgroundImageParameters;
        }
        else if((!elementName.equals("reverseRendering")) && (!elementName.equals("reverseLegendItemOrder")))
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("reverseRendering"))
        {
            try
            {
                this._reverseRendering = Boolean.parseBoolean(value);
            }
            catch(final Exception e)
            {
                throw new XMLReaderException("Value of element reverseRendering cannot be parsed: " + e.getMessage());
            }
        }
        else if(elementName.equals("reverseLegendItemOrder"))
        {
            try
            {
                this._reverseLegendItemOrder = Boolean.parseBoolean(value);
            }
            catch(final Exception e)
            {
                throw new XMLReaderException("Value of element reverseLegendItemOrder cannot be parsed: "
                    + e.getMessage());
            }
        }
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());

        mainElement.setAttribute("subPlotIndex", "" + this._subPlotIndex);
        mainElement.setAttribute("weight", "" + this._plotWeight);

        XMLTools.appendElementIfNotEmpty(mainElement, this._leftRangeAxisParameters.writePropertyToXMLElement(request));
        XMLTools.appendElementIfNotEmpty(mainElement,
                                         this._rightRangeAxisParameters.writePropertyToXMLElement(request));

        if(this._plotBackgroundColor != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("backgroundColor", request, this._plotBackgroundColor));
        }
        if(this._reverseRendering != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "reverseRendering",
                                                                   _reverseRendering.toString()));
        }
        if(this._reverseLegendItemOrder != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "reverseLegendItemOrder",
                                                                   _reverseLegendItemOrder.toString()));
        }

        XMLTools.appendElementIfNotEmpty(mainElement,
                                         this._backgroundImageParameters.writePropertyToXMLElement(request));

        return mainElement;
    }

    @Override
    public Object clone()
    {
        final SubPlotParameters result = new SubPlotParameters(this.getSubPlotIndex());
        result.copyFrom(this);
        return result;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        try
        {
            _leftRangeAxisParameters.haveAllParametersBeenSet();
            _rightRangeAxisParameters.haveAllParametersBeenSet();
        }
        catch(final ChartParametersException e)
        {
            throw new ChartParametersException("For subplot " + _subPlotIndex + ": " + e.getMessage());
        }
    }

    @Override
    public void setupDefaultParameters()
    {
        this._plotWeight = 1;
        this._leftRangeAxisParameters.setupDefaultParameters();
        this._rightRangeAxisParameters.setupDefaultParameters();
        this._backgroundImageParameters.setupDefaultParameters();
        this._plotBackgroundColor = null;
        this._reverseRendering = false;
        this._reverseLegendItemOrder = false;
    }

    @Override
    public String toString()
    {
        String result = "SubPlotParameters: ";
        result += "xmlTagName = '" + this.getXMLTagName() + "'; ";
        result += "subPlotIndex = " + this._subPlotIndex + "; ";
        result += "plotWeight = " + this._plotWeight + "; ";
        result += "Left Axis = {" + this._leftRangeAxisParameters + "}; ";
        result += "Right Axis = {" + this._rightRangeAxisParameters + "}; ";

        if(_plotBackgroundColor != null)
        {
            result += "plotBackgroundColor = " + this._plotBackgroundColor.toString() + "; ";
        }
        else
        {
            result += "plotBackgroundColor = null; ";
        }

        result += "reverseRendering = " + this._reverseRendering + "; ";
        result += "reverseLegendItemOrder = " + this._reverseLegendItemOrder + "; ";
        result += "Background Image = {" + this._backgroundImageParameters + "}; ";

        return result;
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.SubPlotParameters))
        {
            return false;
        }
        final SubPlotParameters other = (SubPlotParameters)parameters;

        if((this._plotWeight != other._plotWeight) || (this._subPlotIndex != other._subPlotIndex))
        {
            return false;
        }
        if(!this._leftRangeAxisParameters.equals(other.getLeftAxis()))
        {
            return false;
        }
        if(!this._rightRangeAxisParameters.equals(other.getRightAxis()))
        {
            return false;
        }
        if(!this._backgroundImageParameters.equals(other.getBackgroundImageParameters()))
        {
            return false;
        }

        if(!GeneralTools.checkForFullEqualityOfObjects(this._plotBackgroundColor, other.getPlotBackgroundColor()))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(_reverseRendering, other.getReverseRendering()))
        {
            return false;
        }
        if(!GeneralTools.checkForFullEqualityOfObjects(_reverseLegendItemOrder, other.getReverseLegendItemOrder()))
        {
            return false;
        }

        return true;
    }

    @Override
    public void argumentsChanged()
    {
        this._leftRangeAxisParameters.argumentsChanged();
        this._rightRangeAxisParameters.argumentsChanged();
    }
}
