package ohd.hseb.charter.parameters;

import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Parameters for subtitles within a chart. This maintains a list of {@link SubtitleParameters}.
 * 
 * @author hank.herr
 */
public class SubtitleListParameters extends DefaultChartParameters
{
    /**
     * The list of subtitles being maintained.
     */
    private List<SubtitleParameters> _subtitleList = new ArrayList<SubtitleParameters>();

    /**
     * Empty constructor sets XML tag to subtitleList.
     */
    public SubtitleListParameters()
    {
        setXMLTagName("subtitleList");
    }

    /**
     * Adds a subtitle with default parameters set and returning its {@link SubtitleParameters}. This calls
     * {@link #addSubtitle(SubtitleParameters)}.
     * 
     * @return The added {@link SubtitleParameters} so that the caller can then modify them.
     */
    public SubtitleParameters addSubtitle()
    {
        final SubtitleParameters parms = new SubtitleParameters();
        addSubtitle(parms);
        parms.setupDefaultParameters();
        return parms;
    }

    /**
     * Adds a subtitle with the provided {@link SubtitleParameters}, forcing its arguments to match
     * {@link #getArguments()} and its XML tag to be "subtitle".
     */
    public void addSubtitle(final SubtitleParameters parms)
    {
        parms.setXMLTagName("subtitle");
        parms.setArguments(getArguments());
        this._subtitleList.add(parms);
    }

    public void removeSubtitle(final SubtitleParameters parms)
    {
        this._subtitleList.remove(parms);
    }

    public void removeSubtitle(final int index)
    {
        this._subtitleList.remove(index);
    }

//XXX If we use identifiers for subtitles; see header comment in subtitle.
//      public void prefixSubtitleIdentifiers(String prefix)
//      {
//          for(SubtitleParameters parms: this._subtitleList)
//          {
//              parms.setIdentifier(prefix + parms.getIdentifier());
//          }
//      }

    public List<SubtitleParameters> getSubtitleList()
    {
        return _subtitleList;
    }

    public void setSubtitleList(final List<SubtitleParameters> subtitleList)
    {
        _subtitleList = subtitleList;
    }

    public SubtitleParameters getSubtitle(final int index)
    {
        return this._subtitleList.get(index);
    }

    public int getSubtitleCount()
    {
        return this._subtitleList.size();
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        for(final SubtitleParameters parms: _subtitleList)
        {
            parms.setArguments(arguments);
        }
    }

    /**
     * This will call the apply method for the contained {@link SubtitleParameters}.
     */
    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        //objectAppliedTo must be a JFreeChart.  
        for(int i = _subtitleList.size() - 1; i >= 0; i--)
        {
            final SubtitleParameters parms = _subtitleList.get(i);
            final SubtitleParameters fullySpecified = new SubtitleParameters();
            fullySpecified.setArguments(getArguments());
            fullySpecified.setupDefaultParameters();
            fullySpecified.copyOverriddenParameters(parms);
            fullySpecified.applyParametersToChart(objectAppliedTo);
        }
    }

    @Override
    public void clearParameters()
    {
        _subtitleList.clear();
    }

    @Override
    public Object clone()
    {
        final SubtitleListParameters cloneParms = new SubtitleListParameters();
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        clearParameters();
        copyOverriddenParameters((SubtitleListParameters)parameters);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final SubtitleListParameters base = (SubtitleListParameters)override;
        for(final SubtitleParameters parms: base.getSubtitleList())
        {
            this.addSubtitle((SubtitleParameters)parms.clone());
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.SubtitleListParameters))
        {
            return false;
        }
        if(!ListTools.twoSidedListCheck(this.getSubtitleList(), ((SubtitleListParameters)parameters).getSubtitleList()))
        {
            return false;
        }
        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        //Does nothing.  The parameters are never fully set until it is applied.
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        final SubtitleParameters parameters = new SubtitleParameters();
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }
        else if(elementName.equals(parameters.getXMLTagName()))
        {
            this.addSubtitle(parameters);
            return parameters;
        }
        else
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '" + elementName
                + "'.");
        }

        return null;
    }

    @Override
    public void setupDefaultParameters()
    {
        //When this is called, the list should be empty, because it will be called before the override parameters
        //are applied.  The default list parameters is always empty.
        clearParameters();
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public String toString()
    {
        String results = "SubtitleListParameters: [";
        for(final SubtitleParameters parms: _subtitleList)
        {
            results += "{" + parms + "}; ";
        }
        results += "].";
        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());
        for(final SubtitleParameters parms: _subtitleList)
        {
            XMLTools.appendElementIfNotEmpty(mainElement, parms.writePropertyToXMLElement(request));
        }
        return mainElement;
    }
}
