package ohd.hseb.charter.parameters;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.title.TextTitle;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Parameters for an individual subtitle within a chart, which are contained within a {@link SubtitleListParameters}
 * object. Same as title parameters, but positioning is allowed. <br>
 * <br>
 * This class used to allow for an identifier written as an attribute. I'm not sure what purpose it served, but, in the
 * past, I thought that the user's might ask for it. Compare this with an earlier revision to see what code to added for
 * subtitle identifiers.
 * 
 * @author hank.herr
 */
public class SubtitleParameters extends TitleParameters
{
    public static final String DEFAULT_POSITION_DISPLAY_STRING = "Default (TOP)";
    public static final String DEFAULT_POSITION_STRING = "TOP";

    /**
     * The position of the subtitle. This is processed through
     * {@link ChartConstants#determineRectangleEdgeForPosition(String)}.
     */
    private String _position = null;

    /**
     * Empty constructor sets the XML tag to be "subtitle".
     */
    public SubtitleParameters()
    {
        setXMLTagName("subtitle");
    }

    public String getPosition()
    {
        return _position;
    }

    public void setPosition(final String positionString)
    {
        _position = positionString;
    }

    @Override
    public TextTitle createTitle()
    {
        final TextTitle subTitle = super.createTitle();
        subTitle.setPosition(ChartConstants.determineRectangleEdgeForPosition(_position));
        return subTitle;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        final JFreeChart chart = (JFreeChart)objectAppliedTo;

        //Putting it at index 0 places it before the legend, so that bottom and top subtitles are centered correctly.
        //Left and right will still potentially look ugly because it will be centered to an area no including the 
        //plot title... plot title is always drawn first.
        //
        //Note that if a subtitle has already been added, then this will force it to add BEFORE that subtitle.
        chart.addSubtitle(0, createTitle());
    }

    @Override
    public void clearParameters()
    {
        super.clearParameters();
        this._position = null;
    }

    @Override
    public Object clone()
    {
        final SubtitleParameters cloneParms = new SubtitleParameters();
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final SubtitleParameters base = (SubtitleParameters)parameters;
        this._position = base.getPosition();
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        super.copyOverriddenParameters(override); //Copies the title parameters.
        final SubtitleParameters base = (SubtitleParameters)override;
        if(base.getPosition() != null)
        {
            this._position = base.getPosition();
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.SubtitleParameters))
        {
            return false;
        }
        if(!super.equals(parameters))
        {
            return false;
        }
        final SubtitleParameters other = (SubtitleParameters)parameters;
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(_position, other.getPosition(), true))
        {
            return false;
        }
        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        super.haveAllParametersBeenSet();
        if(this._position == null)
        {
            throw new ChartParametersException("Subtitle position not specified.");
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }
        else if(!elementName.equals("position"))
        {
            return super.readInPropertyFromXMLElement(elementName, attr);
        }

        return null;
    }

    @Override
    public void setupDefaultParameters()
    {
        super.setupDefaultParameters();
        this._position = DEFAULT_POSITION_STRING;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("position"))
        {
            if(ChartConstants.determineRectangleEdgeForPosition(value) == null)
            {
                throw new XMLReaderException("The value of position, '" + value + "', is invalid.");
            }
            this._position = value;
        }
        else
        {
            super.setValueOfElement(elementName, value);
        }
    }

    @Override
    public String toString()
    {
        String results = "SubtitleParameters: ";
        results += super.toString() + "; ";
        results += "position = " + this._position + ".";
        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = super.writePropertyToXMLElement(request);
        if(this._position != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "position", _position));
        }
        return mainElement;
    }

    /**
     * @return Array of Strings that can be placed in a combo box for selection. Includes a default string as the first
     *         item.
     */
    public static String[] buildPositionDisplayedStrings()
    {
        final String[] results = new String[ChartConstants.POSITION_STRINGS.length + 1];
        results[0] = DEFAULT_POSITION_DISPLAY_STRING;
        for(int i = 0; i < ChartConstants.POSITION_STRINGS.length; i++)
        {
            results[i + 1] = ChartConstants.POSITION_STRINGS[i];
        }
        return results;
    }
}
