package ohd.hseb.charter.parameters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.XYPlot;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Maintains a list of {@link ThresholdParameters} instances.
 * 
 * @author Hank.Herr
 */
public class ThresholdListParameters extends DefaultChartParameters
{

    private static final Logger LOG = LogManager.getLogger(ThresholdListParameters.class);

    /**
     * Maps identifiers to thresholds.
     */
    private final LinkedHashMap<String, ThresholdParameters> _idToThresholdMap =
                                                                               new LinkedHashMap<String, ThresholdParameters>();

    /**
     * List of the threshold parameters (redundant with the above map).
     */
    private final List<ThresholdParameters> _thresholdList = new ArrayList<ThresholdParameters>();

    /**
     * Map of subplot index to the {@link XYPlot} corresponding to it. Useful when applying thresholds. This is passed
     * through to {@link ThresholdParameters}.
     */
    private HashMap<Integer, XYPlot> _subPlotIndexToXYPlot = null;

    /**
     * Map of subplot index to computed range axis types; again useful when applying thresholds.
     */
    private HashMap<Integer, int[]> _subPlotIndexToComputedRangeAxisTypes = new HashMap<Integer, int[]>();

    /**
     * Indicates if the domain axis is of type time or number.
     */
    private int _domainAxisType = -1;

    /**
     * This flag is NOT a parameter. It merely tracks if there were issues related to the display of any thresholds
     * within this list.
     */
    private boolean _issuesEncounteredProcessingThresholds = false;

    /**
     * Empty constructor sets XML tag to "thresholdList".
     */
    public ThresholdListParameters()
    {
        setXMLTagName("thresholdList");
    }

    /**
     * @return The {@link ThresholdParameters} for the given identifier.
     */
    public ThresholdParameters retrieveThresholdParameters(final String identifier)
    {
        return _idToThresholdMap.get(getArguments().replaceArgumentsInString(identifier));
    }

    /**
     * Adds the given {@link ThresholdParameters} at the specified index, inserting it into {@link #_thresholdList}.
     * This is only done if the identifier is non-empty.<br>
     * <br>
     * Potential problem: Code does not check for existing identifier. So, when a threshold is added with an identifier
     * that already is used, then it will be added to the list, but will override the current threshold in the list,
     * causing them to become out of sink. That's not good.
     */
    //TODO fix the problem mentioned in the javadoc above when we have time to fully test its impact.
    public void addThreshold(final ThresholdParameters params, final int index)
    {
        params.setArguments(getArguments());

        //Check for validity of identifier
        if((params.getIdentifier() != null)
            && (getArguments().replaceArgumentsInString(params.getIdentifier()).length() > 0))
        {
            this._thresholdList.add(index, params);
            this.addThresholdToMap(params);
        }
    }

    /**
     * Calls {@link #addThreshold(ThresholdParameters, int)} so that it adds it at the end.
     */
    public void addThreshold(final ThresholdParameters params)
    {
        addThreshold(params, _thresholdList.size());
    }

    /**
     * @return An {@link ThresholdParameters} instance for which {@link ThresholdParameters#setupDefaultParameters()}
     *         has been called and which has been added to this through {@link #addThreshold(ThresholdParameters)}.
     */
    public ThresholdParameters addThreshold(final String identifier)
    {
        final ThresholdParameters parms = new ThresholdParameters();
        parms.setupDefaultParameters();
        parms.setIdentifier(identifier);
        addThreshold(parms);
        return parms;
    }

    /**
     * Convenience wrapper on {@link #_idToThresholdMap} put method. Note that the identifier will have any arguments
     * replaced when added. The identifier, at that point, will be free of arguments. (Not really sure if this is
     * needed.)
     */
    private void addThresholdToMap(final ThresholdParameters params)
    {
        this._idToThresholdMap.put(getArguments().replaceArgumentsInString(params.getIdentifier()), params);
    }

    /**
     * Adds all thresholds from the provided {@link ThresholdListParameters} in order, making sure to set the arguments
     * to be consistent with this.
     */
    public void addThresholds(final ThresholdListParameters params)
    {
        for(int i = 0; i < params.getThresholdParametersList().size(); i++)
        {
            addThreshold(params.getThresholdParametersList().get(i));
            params.getThresholdParametersList().get(i).setArguments(getArguments());
        }
    }

    public void removeThreshold(final String identifier)
    {
        final ThresholdParameters parms = _idToThresholdMap.get(getArguments().replaceArgumentsInString(identifier));
        this._idToThresholdMap.remove(getArguments().replaceArgumentsInString(identifier));
        this._thresholdList.remove(parms);
    }

    /**
     * Removes, renames, and adds.
     */
    public void renameThreshold(final String oldIdentifier, final String newIdentifier)
    {
        final ThresholdParameters parms = _idToThresholdMap.get(getArguments().replaceArgumentsInString(oldIdentifier));
        this._idToThresholdMap.remove(getArguments().replaceArgumentsInString(oldIdentifier));
        parms.setIdentifier(newIdentifier);
        addThresholdToMap(parms);
    }

    public void setSubPlotIndexToXYPlot(final HashMap<Integer, XYPlot> subPlotIndexToXYPlot)
    {
        this._subPlotIndexToXYPlot = subPlotIndexToXYPlot;
    }

    public void setSubPlotIndexToComputedRangeAxisTypesMap(final HashMap<Integer, int[]> subPlotIndexToComputedRangeAxisTypes)
    {
        _subPlotIndexToComputedRangeAxisTypes = subPlotIndexToComputedRangeAxisTypes;
    }

    public void setDomainAxisType(final int type)
    {
        this._domainAxisType = type;
    }

    /**
     * Reconstructs {@link #_idToThresholdMap} based on {@link #_thresholdList}.
     */
    public void rebuildMap()
    {
        this._idToThresholdMap.clear();
        for(int i = 0; i < this._thresholdList.size(); i++)
        {
            this.addThresholdToMap(_thresholdList.get(i));
        }
    }

    public List<ThresholdParameters> getThresholdParametersList()
    {
        return this._thresholdList;
    }

    /**
     * @return Searches {@link #_thresholdList} for the threshold that most recently yielded the provided {@link Marker}
     *         . If found, it is returned.
     */
    public ThresholdParameters findParametersForMarker(final Marker marker)
    {
        for(final ThresholdParameters parm: _thresholdList)
        {
            if(parm.getThresholdMarker() == marker)
            {
                return parm;
            }
        }
        return null;
    }

    /**
     * @return True if the latest call to {@link #applyParametersToChart(Object)} resulted in a warning message
     *         associated with any threshold.
     */
    public boolean issuesEncounteredProcessingThresholds()
    {
        return _issuesEncounteredProcessingThresholds;
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);

        //Set arguments for all list components.
        for(int i = 0; i < this._thresholdList.size(); i++)
        {
            _thresholdList.get(i).setArguments(arguments);
        }

        rebuildMap();
    }

    /**
     * @return A general header to use for all threshold parameter warning messages.
     */
    private String generateThresholdWarningMessageHeader(final ThresholdParameters threshParms)
    {
        return "Problem processing threshold with identifier '" + threshParms.getEvaluatedIdentifier()
            + "' (evaluated from '" + threshParms.getIdentifier() + "') - threshold will be ignored:";
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo) throws ChartParametersException
    {
        _issuesEncounteredProcessingThresholds = false;

        //System.out.println("####>> Applying parameters: " + this);
        //System.out.println("####>> Number of parameters being applied: " + this.getThresholdParametersList().size());

        //No argument replacements here, as the key set contains already-evaluated identifiers.
        final Iterator<String> iter = _idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            final String id = iter.next();
            final ThresholdParameters threshParms = _idToThresholdMap.get(id);

            try
            {
                threshParms.haveAllParametersBeenSet();
            }
            catch(final ChartParametersException e)
            {
                _issuesEncounteredProcessingThresholds = true;
                LOG.info(generateThresholdWarningMessageHeader(threshParms) + " it is incomplete; "
                    + "perhaps it is an override threshold with no corresponding default.");
                LOG.debug("Incomplete threshold message: " + e.getMessage());
                continue;
            }

            int axisType = -1;
            if(ChartConstants.DOMAIN_AXIS_STRING.equalsIgnoreCase(threshParms.getAxisIdString()))
            {
                axisType = _domainAxisType;
            }
            else
            {
                final int index = ChartConstants.determineAxisIndexFromString(threshParms.getAxisIdString());
                if(_subPlotIndexToComputedRangeAxisTypes.get(threshParms.getSubPlotIndex()) == null) //Subplot index for threshold not valid relative to plotted data
                {
                    _issuesEncounteredProcessingThresholds = true;
                    LOG.info(generateThresholdWarningMessageHeader(threshParms) + "The subplot index, "
                        + threshParms.getSubPlotIndex() + ", "
                        + " is a subplot that has no data and will not be displayed.");
                    continue;
                }
                axisType = _subPlotIndexToComputedRangeAxisTypes.get(threshParms.getSubPlotIndex())[index];
            }

            threshParms.setSubPlotIndexToXYPlot(_subPlotIndexToXYPlot);
            threshParms.setAxisType(axisType);
            try
            {
                threshParms.applyParametersToChart(objectAppliedTo);
            }
            catch(final Throwable t)
            {
                _issuesEncounteredProcessingThresholds = true;
                LOG.warn(generateThresholdWarningMessageHeader(threshParms) + " " + t.getMessage());
                continue;
            }
        }
    }

    @Override
    public void clearParameters()
    {
        _idToThresholdMap.clear();
        _thresholdList.clear();
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final ThresholdListParameters base = (ThresholdListParameters)override;

        //First, loop through this map overwriting parameters based on base.
        //No argument replacements here, as the key set contains already-evaluated identifiers.
        Iterator<String> iter = _idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            final String id = iter.next();
            if(base.retrieveThresholdParameters(id) != null)
            {
                this.retrieveThresholdParameters(id).copyOverriddenParameters(base.retrieveThresholdParameters(id));
            }
        }

        //Now add any parameters not included in this.
        //No argument replacements here, as the key set contains already-evaluated identifiers.
        iter = base._idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            final String id = iter.next();
            if(retrieveThresholdParameters(id) == null)
            {
                this.addThreshold((ThresholdParameters)base.retrieveThresholdParameters(id).clone());
            }
        }
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        //No argument replacements here, as the key set contains already-evaluated identifiers.
        final Iterator<String> iter = _idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            final String id = iter.next();
            _idToThresholdMap.get(id).haveAllParametersBeenSet();
        }
    }

    @Override
    public void setupDefaultParameters()
    {
        this.clearParameters();
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final ThresholdListParameters base = (ThresholdListParameters)parameters;
        clearParameters();

        //No argument replacements here, as the key set contains already-evaluated identifiers.
        final Iterator<ThresholdParameters> iter = base._thresholdList.iterator();
        while(iter.hasNext())
        {
            final ThresholdParameters parms = iter.next();
            addThreshold((ThresholdParameters)parms.clone());
        }
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        this.rebuildMap();
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        final ThresholdParameters newParameters = new ThresholdParameters();
        if(elementName.equals(this.getXMLTagName()))
        {
            this.clearParameters();
        }
        if(elementName.equals(newParameters.getXMLTagName()))
        {
            //Force it to do an initial read of the parameters to acquire the identifier and visibility... but  mainly
            //the identifier because I need it for the map.
            newParameters.readInPropertyFromXMLElement(elementName, attr);
            _thresholdList.add(newParameters);
            //this.addThreshold(newParameters);
            return newParameters;
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());

        //No argument replacements here, as the key set contains already-evaluated identifiers.
        final Iterator<String> iter = this._idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            mainElement.appendChild(_idToThresholdMap.get(iter.next()).writePropertyToXMLElement(request));
        }

        return mainElement;
    }

    @Override
    public Object clone()
    {
        final ThresholdListParameters results = new ThresholdListParameters();
        results.copyFrom(this);
        return results;
    }

    public String printParameters()
    {
        final Iterator<String> iter = this._idToThresholdMap.keySet().iterator();
        String result = "ThresholdList: \n";

        while(iter.hasNext())
        {
            result += "    {" + _idToThresholdMap.get(iter.next()).toString() + "};\n ";
        }
        return result;
    }

    @Override
    public String toString()
    {
        String result = "ThresholdList: ";

        result += "xmlTagName = '" + this.getXMLTagName() + "'; ";

        //No argument replacements here, as the key set contains already-evaluated identifiers.
        final Iterator<String> iter = this._idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            result += "{" + _idToThresholdMap.get(iter.next()).toString() + "}; ";
        }
        return result;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof ThresholdListParameters))
        {
            return false;
        }

        final ThresholdListParameters base = (ThresholdListParameters)obj;

        //First, check this against base.
        //No argument replacements here, as the key set contains already-evaluated identifiers.
        Iterator<String> iter = _idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            final String id = iter.next();
            if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this.retrieveThresholdParameters(id),
                                                                                     base.retrieveThresholdParameters(id)))
            {
                return false;
            }
        }

        //Now check base against this.
        //No argument replacements here, as the key set contains already-evaluated identifiers.
        iter = base._idToThresholdMap.keySet().iterator();
        while(iter.hasNext())
        {
            final String id = iter.next();
            if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this.retrieveThresholdParameters(id),
                                                                                     base.retrieveThresholdParameters(id)))
            {
                return false;
            }
        }

        return true;

    }

    @Override
    public void argumentsChanged()
    {
        for(int i = 0; i < this._thresholdList.size(); i++)
        {
            _thresholdList.get(i).argumentsChanged();
        }
        this.rebuildMap();
    }

}
