package ohd.hseb.charter.parameters;

import java.awt.BasicStroke;
import java.awt.Color;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.CombinedDomainXYPlot;
import org.jfree.chart.plot.IntervalMarker;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.xy.XYSeriesCollection;
import org.jfree.ui.Layer;
import org.jfree.ui.RectangleAnchor;
import org.jfree.ui.TextAnchor;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.DefaultChartParameters;
import ohd.hseb.charter.jfreechartoverride.GraphGenXYLineAndShapeRenderer;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.arguments.DefaultPredefinedArguments;
import ohd.hseb.hefs.utils.datetime.HEFSDateTools;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.plugins.GenericParameter;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Provides user control over a single threshold to plot. The threshold can be a line or interval and can apply to the
 * range or domain axes. If the subPlotIndex provided is negative, the threshold will be added to all plots and, for a
 * domain threshold, will only have it label displayed on the bottom most plot. <br>
 * <br>
 * Added marks are rendererd using {@link GraphGenXYLineAndShapeRenderer}, which has the ability to remember added marks
 * and add entities for them. This class includes one attribute which is not a user controllable parameter:
 * {@link #_showLabelInPlot}. This is only used by tools that must allow for thresholds to have tool tips without a
 * corresponding label on the plot.
 * 
 * @author herrhd
 */
public class ThresholdParameters extends DefaultChartParameters
{
    private static final Logger LOG = LogManager.getLogger(ThresholdParameters.class);

    public static String UNBOUNDED_STRING = "unbounded";

    public static final String LABEL_ANCHOR_AUTO = "auto";
    public static final String[] LABEL_ANCHOR_ARRAY = {"auto", "top_left", "top_right", "bottom_left", "bottom_right"};
    public static final RectangleAnchor[] LABEL_RECTAGLE_ANCHOR_ARRAY = new RectangleAnchor[]{null,
        RectangleAnchor.TOP_LEFT, RectangleAnchor.TOP_RIGHT, RectangleAnchor.BOTTOM_LEFT, RectangleAnchor.BOTTOM_RIGHT};

    /**
     * Threshold identifier.
     */
    private String _identifier = "";

    /**
     * The system time to be used for dates when positioning marks.
     */
    private long _systemTimeUsedForApplication = Long.MIN_VALUE;

    /**
     * Visible or not.
     */
    private Boolean _visible;

    /**
     * True to include the threshold value when computing axis limits in order to guarantee it is drawn.
     */
    private Boolean _includeInAxisLimitsComputation;

    /**
     * Parameters for the label of the threshold.
     */
    private final LabelChartParameters _labelParameters = new LabelChartParameters();

    /**
     * Color of the threshold, either the line if its a mark or the fill if its a zone.
     */
    private Color _color = null;

    /**
     * Line width.
     */
    private Float _lineWidth = null;

    /**
     * The index of the subplot on which the threshold is plotted.
     */
    private Integer _subPlotIndex = null;

    /**
     * The id of the axis against which it is plotted. Either "domain", "left", or "right"; see
     * {@link ChartConstants#DOMAIN_AXIS_STRING} and {@link ChartConstants#YAXIS_XML_STRINGS}.
     */
    private String _axisIdString = null;

    /**
     * Numerical axis value start as a {@link String}.
     */
    private String _numericalAxisValueStartStr = null;

    /**
     * Numerical axis value end as a {@link String}.
     */
    private String _numericalAxisValueEndStr = null;

    /**
     * Date axis value start as a {@link String}.
     */
    private String _dateAxisValueStart = null;

    /**
     * Date axis value end as a {@link String}.
     */
    private String _dateAxisValueEnd = null;

    /**
     * True if this is a zone, false for a mark/line.
     */
    private Boolean _isZone = null;

    /**
     * Tracks the type of the axis. Processed through {@link ChartConstants#isAxisTypeTime(int)} and
     * {@link ChartConstants#isAxisTypeTranslated(int)}.
     */
    private int _axisType = -1;

    /**
     * The label anchor, which must correspond to an entry in {@link #LABEL_ANCHOR_ARRAY}.
     */
    private String _labelAnchor = null;

    /**
     * This is NOT! a parameter. It is used to store the mark that is added to the chart for this threshold.
     */
    private Marker _thresholdMarker;

    /**
     * This is NOT! a parameter. It is only currently used for some special plots that need to display a tool tip, but
     * not necessary a label on the plot.
     */
    private boolean _showLabelInPlot = true;

    /**
     * This is used for application, only.
     */
    private HashMap<Integer, XYPlot> _subPlotIndexToXYPlot = null;

    /**
     * Empty constructor sets the XML tag to "threshold" and {@link #_identifier} to null.
     */
    public ThresholdParameters()
    {
        setXMLTagName("threshold");
        _identifier = null;
    }

    /**
     * Constructor allows for setting the id.
     * 
     * @param id
     */
    public ThresholdParameters(final String id)
    {
        this();
        _identifier = id;
    }

    /**
     * This method calls {@link #buildMark()} and sets {@link #_thresholdMarker} when done in order to record what was
     * built.
     */
    @SuppressWarnings("unchecked")
    @Override
    //TODO This is a long method and perhaps should be broken up and/or placed inside a class dedicated to apply threshold parameters.
    public void applyParametersToChart(final Object objectAppliedTo) throws ChartParametersException
    {
//        System.out.println("Applying parameters: " + this);
        if(!_visible)
        {
            return;
        }

        final GenericParameter systemTimeArg = this.getArguments().getArgument(DefaultPredefinedArguments.SYSTEM_TIME);
        final Calendar presentTime = HEFSDateTools.computeFixedDate(systemTimeArg.getValue());
        _systemTimeUsedForApplication = presentTime.getTimeInMillis();

        final JFreeChart chart = (JFreeChart)objectAppliedTo;

        //Build the mark and position the text label.
        final Marker mark = buildMark();

        //Mark2 and mark3 are used to force the range axis to include the thresholds in computation
        //of min and max.  For zones, they are in the background and are, therefore, not counted toward
        //the axis limits.
        Marker mark2 = null;
        Marker mark3 = null;

        //Turn off the outline for the zone mark.  Default is a light gray, thin line drawn around each
        //zone.  ESPADP does not include that line, so I'll make the outline paint equal the color.
        if(getIsZone())
        {
            ((IntervalMarker)mark).setOutlinePaint(getColor());
        }

        //If its a zone and is to be included in the axis limits, then add dummy markers for axis limits 
        //computation purposes.
        if(getIsZone() && this._includeInAxisLimitsComputation)
        {
            if((((IntervalMarker)mark).getStartValue() != Double.MAX_VALUE)
                && (((IntervalMarker)mark).getStartValue() != Double.MIN_VALUE))
            {
                mark2 = new ValueMarker(((IntervalMarker)mark).getStartValue());
                mark2.setPaint(ColorTools.TRANSPARENT_WHITE);
                mark2.setStroke(new BasicStroke(0));
            }
            if((((IntervalMarker)mark).getEndValue() != Double.MAX_VALUE)
                && (((IntervalMarker)mark).getEndValue() != Double.MIN_VALUE))
            {
                mark3 = new ValueMarker(((IntervalMarker)mark).getEndValue());
                mark3.setPaint(ColorTools.TRANSPARENT_WHITE);
                mark3.setStroke(new BasicStroke(0));
            }
        }

        RectangleAnchor rectAnchor = null;
        if(!isLabelAnchorAuto())
        {
            rectAnchor = getRectangleAnchorByString(getLabelAnchor());
        }
        if(rectAnchor == null)
        {
            determineAutoLabelAnchor(mark);
        }
        else
        {
            mark.setLabelAnchor(rectAnchor);
            determineLabelTextAnchor(mark, rectAnchor);
        }

        //Set other mark parameters based on settings.
        mark.setPaint(_color);
        mark.setLabel(this._labelParameters.getArgumentReplacedText());
        mark.setLabelFont(this._labelParameters.getFont());
        mark.setLabelPaint(this._labelParameters.getColor());
        mark.setStroke(new BasicStroke(_lineWidth));

        //Label must not be shown in the plot, so make its color transparent.  
        if(!_showLabelInPlot)
        {
            mark.setLabelPaint(ColorTools.TRANSPARENT_WHITE); //Indicates a label that is to not be shown.  See GraphGenXYLineAndShapeRenderer.
        }

        //Obtain a list of plots to apply the marker to.  This assumes the main plot is a combined domain plot,
        //since that is what the ChartEngine always uses.
        final CombinedDomainXYPlot plot = (CombinedDomainXYPlot)chart.getXYPlot();
        final List<XYPlot> affectedPlots = new ArrayList<XYPlot>();
        if(_subPlotIndex < 0)
        {
            affectedPlots.addAll(plot.getSubplots());
        }
        else
        {
            final XYPlot foundPlot = this._subPlotIndexToXYPlot.get(this._subPlotIndex);
            if(foundPlot == null)
            {
                throw new ChartParametersException("For threshold with id " + this._identifier + ", subplot index of "
                    + this._subPlotIndex + " does not have a corresponding plot.");
            }
            affectedPlots.add(foundPlot);
        }

        //Get the axis index and create a dummy renderer added to the subplot for that axis.
        final int axisIndex = this.convertAxisIdStringToInteger();

        //Decide on the layer.  Marks are in the foreground, zones in background.
        Layer drawnLayer = Layer.FOREGROUND;
        if(getIsZone())
        {
            drawnLayer = Layer.BACKGROUND;
        }

        //Add the marker to each of the affected plots.
        for(int i = 0; i < affectedPlots.size(); i++)
        {
            final XYPlot subPlot = affectedPlots.get(i);
            if(axisIndex == ChartConstants.DOMAIN_AXIS)
            {
                //For the domain axis, I only want the label to show up on the last subplot, at the bottom.
                Marker usedMark = mark;
                if(i < affectedPlots.size() - 1)
                {
                    try
                    {
                        usedMark = (Marker)mark.clone();
                    }
                    catch(final CloneNotSupportedException e)
                    {
                        e.printStackTrace();
                    }
                    usedMark.setLabel("");
                }

                subPlot.addDomainMarker(usedMark, drawnLayer);
                if(mark2 != null)
                {
                    subPlot.addDomainMarker(mark2, Layer.FOREGROUND);
                }
                if(mark3 != null)
                {
                    subPlot.addDomainMarker(mark3, Layer.FOREGROUND);
                }
            }
            else
            {
                //For some stupid reason, JFreeChart maps thresholds to renderers instead of axes.
                //Hence, I need to create a dummy renderer for any range axis threshold since there is
                //no guarantee that a range axis will have a dataset plotted against it.  
                //I don't need to do this for the domain axis, because there is only one domain axis and
                //everything plots against it!
                final XYLineAndShapeRenderer renderer = new GraphGenXYLineAndShapeRenderer();
                final int dataSetIndex = subPlot.getDatasetCount();
                final XYSeriesCollection dataset = new XYSeriesCollection();
                subPlot.setDataset(dataSetIndex, dataset);
                subPlot.mapDatasetToRangeAxis(dataSetIndex, axisIndex);
                subPlot.setRenderer(dataSetIndex, renderer);

                //Adding the marker for the dummy data set.
                subPlot.addRangeMarker(dataSetIndex, mark, drawnLayer);
                renderer.setSeriesVisible(dataSetIndex, false);
                if(mark2 != null)
                {
                    subPlot.addRangeMarker(dataSetIndex, mark2, Layer.FOREGROUND);
                }
                if(mark3 != null)
                {
                    subPlot.addRangeMarker(dataSetIndex, mark3, Layer.FOREGROUND);
                }
            }
        }

        //Record the mark used
        _thresholdMarker = mark;

    }

    /**
     * @return A constructed {@link Marker} based on the parameters defined herein.
     */
    private Marker buildMark() throws ChartParametersException
    {
        //Determine if the mark is an interval or value marker.
        Marker mark = null;

        if(ChartConstants.isAxisTypeTime(_axisType))
        {
            double dateValueStart;
            double dateValueEnd;

            dateValueStart = Double.MIN_VALUE;
            if((_dateAxisValueStart != null) && (!this.isDateStartValueUnbounded()))
            {
                final Long dateValue = HEFSDateTools.computeTime(this._systemTimeUsedForApplication,
                                                                 null,
                                                                 _dateAxisValueStart);
                if(dateValue == null)
                {
                    throw new ChartParametersException("For threshold with id " + this._identifier
                        + ", date value start is not a valid date.");
                }
                dateValueStart = dateValue;
            }

            if(_isZone)
            {
                dateValueEnd = Double.MAX_VALUE;
                if((_dateAxisValueEnd != null) && (!this.isDateEndValueUnbounded()))
                {
                    final Long dateValue = HEFSDateTools.computeTime(this._systemTimeUsedForApplication,
                                                                     null,
                                                                     _dateAxisValueEnd);
                    if(dateValue == null)
                    {
                        throw new ChartParametersException("For threshold with id " + this._identifier
                            + ", date value end is not a valid date.");
                    }
                    dateValueEnd = dateValue;
                }
                if((dateValueStart == Double.MIN_VALUE) && (dateValueEnd == Double.MAX_VALUE))
                {
                    throw new ChartParametersException("For threshold with id " + this._identifier
                        + ", both start and end dates are not specified or are unbounded; "
                        + "at least one must be specified and not unbounded.");
                }
                if(dateValueEnd < dateValueStart)
                {
                    throw new ChartParametersException("For threshold with id " + this._identifier
                        + ", the start date is after than the end date, which is not allowed.");
                }
                mark = new IntervalMarker(dateValueStart, dateValueEnd);
            }
            else
            {
                if(dateValueStart == Double.MIN_VALUE)
                {
                    throw new ChartParametersException("For threshold with id " + this._identifier
                        + ", the start date is not specified, is empty, or is unbounded; for non-zone thresholds, the start must be specified.");
                }
                mark = new ValueMarker(dateValueStart);
            }
        }
        else if(!ChartConstants.isAxisTypeTranslated(_axisType))
        {
            double valueStart = Double.MIN_VALUE;
            if((getNumericalAxisValueStartStr() != null) && (!this.isNumericalStartValueUnbounded())
                && (getNumericalAxisValueStartValue() != null))
            {
                valueStart = getNumericalAxisValueStartValue();
            }
            if(_isZone)
            {
                double valueEnd = Double.MAX_VALUE;
                if((getNumericalAxisValueEndStr() != null) && (!this.isNumericalEndValueUnbounded())
                    && (getNumericalAxisValueEndValue() != null))
                {
                    valueEnd = getNumericalAxisValueEndValue();
                }
                if((valueStart == Double.MIN_VALUE) && (valueEnd == Double.MAX_VALUE))
                {
                    throw new ChartParametersException("For threshold with id " + this._identifier
                        + ", both start and end values are not specified or unbounded; at least one must be specified and bounded.");
                }
                if(valueEnd < valueStart)
                {
                    throw new ChartParametersException("For threshold with id " + this._identifier
                        + ", the start value is larger than the end value, which is not allowed.");
                }
                mark = new IntervalMarker(valueStart, valueEnd);
            }
            else
            {
                if(valueStart == Double.MIN_VALUE)
                {
                    throw new ChartParametersException("For threshold with id " + this._identifier
                        + ", the start value is not specified, is empty, or is unbounded; for non-zone thresholds, the start must be specified.");
                }
                mark = new ValueMarker(valueStart);
            }
        }
        else
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", it is plotted against a translated axis which is not allowed.");
        }
        return mark;
    }

    @Override
    public void clearParameters()
    {
        this._visible = null;
        this._includeInAxisLimitsComputation = null;
        this._isZone = null;
        this._axisIdString = null;
        this._dateAxisValueStart = null;
        this._dateAxisValueEnd = null;
        this._color = null;
        this._lineWidth = null;
        this._numericalAxisValueStartStr = null;
        this._numericalAxisValueEndStr = null;
        this._subPlotIndex = null;
        this._labelAnchor = null;
        _showLabelInPlot = true;

        _labelParameters.clearParameters();
    }

    @Override
    public Object clone()
    {
        final ThresholdParameters results = new ThresholdParameters();
        results.copyFrom(this);
        return results;
    }

    /**
     * Convenience wrapper on tool {@link ChartConstants#determineAxisIndexFromString(String)}
     */
    public Integer convertAxisIdStringToInteger()
    {
        return ChartConstants.determineAxisIndexFromString(_axisIdString);
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final ThresholdParameters base = (ThresholdParameters)parameters;
        this.clearParameters();

        //TODO Due to the complexity, for now I am not going to confirm that the below is equivalent to copyOverridden*.  
        this._identifier = base.getIdentifier();
        this._visible = base.getVisible();
        this._includeInAxisLimitsComputation = base.getIncludeInAxisLimitsComputation();
        this._isZone = base.getIsZone();
        this._axisIdString = base.getAxisIdString();
        this._dateAxisValueEnd = base.getDateAxisValueEnd();
        this._dateAxisValueStart = base.getDateAxisValueStart();
        this._labelParameters.copyFrom(base.getLabel());
        if(base.getColor() != null)
        {
            this._color = ColorTools.deepCopy(base.getColor());
        }
        this._lineWidth = base.getLineWidth();
        this._numericalAxisValueEndStr = base.getNumericalAxisValueEndStr();
        this._numericalAxisValueStartStr = base.getNumericalAxisValueStartStr();
        this._subPlotIndex = base.getSubPlotIndex();
        this._labelAnchor = base.getLabelAnchor();
        _showLabelInPlot = base.getShowLabelInPlot();

    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        final ThresholdParameters base = (ThresholdParameters)override;

        if(base.getVisible() != null)
        {
            _visible = base.getVisible();
        }
        if(base.getIncludeInAxisLimitsComputation() != null)
        {
            _includeInAxisLimitsComputation = base.getIncludeInAxisLimitsComputation();
        }
        if(base.getIsZone() != null)
        {
            _isZone = base.getIsZone();
        }
        if(base.getAxisIdString() != null)
        {
            this._axisIdString = base.getAxisIdString();
        }
        if(base.getDateAxisValueStart() != null)
        {
            this._dateAxisValueStart = base.getDateAxisValueStart();
        }
        if(base.getDateAxisValueEnd() != null)
        {
            this._dateAxisValueEnd = base.getDateAxisValueEnd();
        }
        if(base.getColor() != null)
        {
            this._color = ColorTools.deepCopy(base.getColor());
        }
        if(base.getLineWidth() != null)
        {
            this._lineWidth = base.getLineWidth();
        }
        if(base.getNumericalAxisValueStartStr() != null)
        {
            this._numericalAxisValueStartStr = base.getNumericalAxisValueStartStr();
        }
        if(base.getNumericalAxisValueEndStr() != null)
        {
            this._numericalAxisValueEndStr = base.getNumericalAxisValueEndStr();
        }
        if(base.getSubPlotIndex() != null)
        {
            this._subPlotIndex = base.getSubPlotIndex();
        }
        if(base.getLabelAnchor() != null)
        {
            this._labelAnchor = base.getLabelAnchor();
        }
        _labelParameters.copyOverriddenParameters(base.getLabel());
    }

    /**
     * Calls {@link Marker#setLabelAnchor(RectangleAnchor)} and {@link Marker#setLabelTextAnchor(TextAnchor)} for the
     * passed in {@link Marker}, setting the anchors based on the return of {@link #convertAxisIdStringToInteger()}.
     * This sets up the default anchor for the created {@link Marker} prior to overriding based on parameters.
     */
    private void determineAutoLabelAnchor(final Marker mark)
    {
        //Determine label position.
        //Zone marker...
        if(mark instanceof IntervalMarker)
        {
            if(this.convertAxisIdStringToInteger() == ChartConstants.LEFT_YAXIS_INDEX)
            {
                if(isIntervalMarkerStartMinimum((IntervalMarker)mark))
                {
                    mark.setLabelAnchor(RectangleAnchor.TOP_LEFT);
                    mark.setLabelTextAnchor(TextAnchor.TOP_LEFT);
                }
                else
                {
                    mark.setLabelAnchor(RectangleAnchor.BOTTOM_LEFT);
                    mark.setLabelTextAnchor(TextAnchor.BASELINE_LEFT);
                }
            }
            else if(this.convertAxisIdStringToInteger() == ChartConstants.RIGHT_YAXIS_INDEX)
            {
                if(this.isIntervalMarkerEndMaximum((IntervalMarker)mark))
                {
                    mark.setLabelAnchor(RectangleAnchor.TOP_RIGHT);
                    mark.setLabelTextAnchor(TextAnchor.TOP_RIGHT);
                }
                else
                {
                    mark.setLabelAnchor(RectangleAnchor.BOTTOM_RIGHT);
                    mark.setLabelTextAnchor(TextAnchor.BASELINE_RIGHT);
                }
            }
            else
            //DOMAIN
            {
                if(this.isIntervalMarkerStartMinimum((IntervalMarker)mark))
                {
                    mark.setLabelAnchor(RectangleAnchor.BOTTOM_RIGHT);
                    mark.setLabelTextAnchor(TextAnchor.BASELINE_RIGHT);
                }
                else
                {
                    mark.setLabelAnchor(RectangleAnchor.BOTTOM_LEFT);
                    mark.setLabelTextAnchor(TextAnchor.BASELINE_LEFT);
                }
            }

            //The text is black or white depending upon the shade of the zone color.
            if(ColorTools.isColorDark(getColor()))
            {
                mark.setLabelPaint(Color.white);
            }
        }
        //Line marker.
        else
        {
            if(this.convertAxisIdStringToInteger() == ChartConstants.LEFT_YAXIS_INDEX)
            {
                mark.setLabelAnchor(RectangleAnchor.TOP_LEFT);
                mark.setLabelTextAnchor(TextAnchor.BASELINE_LEFT);
            }
            else if(this.convertAxisIdStringToInteger() == ChartConstants.RIGHT_YAXIS_INDEX)
            {
                mark.setLabelAnchor(RectangleAnchor.TOP_RIGHT);
                mark.setLabelTextAnchor(TextAnchor.BASELINE_RIGHT);
            }
            else
            //DOMAIN
            {
                mark.setLabelAnchor(RectangleAnchor.BOTTOM_LEFT);
                mark.setLabelTextAnchor(TextAnchor.BASELINE_RIGHT);
            }
        }

    }

    /**
     * For the {@link Marker} provided, sets calls {@link Marker#setLabelTextAnchor(TextAnchor)} based on the provided
     * {@link RectangleAnchor}.
     */
    private void determineLabelTextAnchor(final Marker mark, final RectangleAnchor rectAnchor)
    {
        //Determine label position.
        if(mark instanceof IntervalMarker)
        {
            //for zoom yes option
            if(rectAnchor.equals(RectangleAnchor.TOP_LEFT))
            {
                mark.setLabelTextAnchor(TextAnchor.TOP_LEFT);
            }
            else if(rectAnchor.equals(RectangleAnchor.TOP_RIGHT))
            {
                mark.setLabelTextAnchor(TextAnchor.TOP_RIGHT);
            }
            else if(rectAnchor.equals(RectangleAnchor.BOTTOM_LEFT))
            {
                mark.setLabelTextAnchor(TextAnchor.BASELINE_LEFT);
            }
            else if(rectAnchor.equals(RectangleAnchor.BOTTOM_RIGHT))
            {
                mark.setLabelTextAnchor(TextAnchor.BASELINE_RIGHT);
            }
        }
        else
        {
            //for zoom no option
            if(rectAnchor.equals(RectangleAnchor.TOP_LEFT))
            {
                mark.setLabelTextAnchor(TextAnchor.BASELINE_LEFT);
            }
            else if(rectAnchor.equals(RectangleAnchor.TOP_RIGHT))
            {
                mark.setLabelTextAnchor(TextAnchor.BASELINE_RIGHT);
            }
            else if(rectAnchor.equals(RectangleAnchor.BOTTOM_LEFT))
            {
                mark.setLabelTextAnchor(TextAnchor.TOP_LEFT);
            }
            else if(rectAnchor.equals(RectangleAnchor.BOTTOM_RIGHT))
            {
                mark.setLabelTextAnchor(TextAnchor.TOP_RIGHT);
            }

            //for domain axis
            if(this.convertAxisIdStringToInteger() == ChartConstants.DOMAIN_AXIS)
            {
                if(rectAnchor.equals(RectangleAnchor.TOP_LEFT))
                {
                    mark.setLabelTextAnchor(TextAnchor.TOP_RIGHT);
                }
                else if(rectAnchor.equals(RectangleAnchor.TOP_RIGHT))
                {
                    mark.setLabelTextAnchor(TextAnchor.TOP_LEFT);
                }
                else if(rectAnchor.equals(RectangleAnchor.BOTTOM_LEFT))
                {
                    mark.setLabelTextAnchor(TextAnchor.BASELINE_RIGHT);
                }
                else if(rectAnchor.equals(RectangleAnchor.BOTTOM_RIGHT))
                {
                    mark.setLabelTextAnchor(TextAnchor.BASELINE_LEFT);
                }
            }
        }
    }

    /**
     * Checks for equality based on the axis being numerical.
     */
    //TODO HORRIBLE! There should be a numerical axis threshold and date axis threshold subclasses of threshold parameters.  
    public boolean equalParametersWithNumericalAxis(final Object obj)
    {
        if(!(obj instanceof ThresholdParameters))
        {
            return false;
        }

        final ThresholdParameters base = (ThresholdParameters)obj;

        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._identifier,
                                                                                base.getIdentifier(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._axisIdString,
                                                                                base.getAxisIdString(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._color, base.getColor()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._lineWidth, base.getLineWidth()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._numericalAxisValueStartStr,
                                                                                base.getNumericalAxisValueStartStr(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._numericalAxisValueEndStr,
                                                                                base.getNumericalAxisValueEndStr(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._subPlotIndex,
                                                                                 base.getSubPlotIndex()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._visible, base.getVisible()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._includeInAxisLimitsComputation,
                                                                                 base.getIncludeInAxisLimitsComputation()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._isZone, base.getIsZone()))
        {
            return false;
        }
        if(!this._labelParameters.equals(base.getLabel()))
        {
            return false;
        }
        return true;
    }

    /**
     * Checks for equality based on the axis being time-based.
     */
    //TODO HORRIBLE! There should be a numerical axis threshold and date axis threshold subclasses of threshold parameters.  
    public boolean equalParametersWithTimeAxis(final Object obj)
    {
        if(!(obj instanceof ThresholdParameters))
        {
            return false;
        }

        final ThresholdParameters base = (ThresholdParameters)obj;

        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._identifier,
                                                                                base.getIdentifier(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._axisIdString,
                                                                                base.getAxisIdString(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._dateAxisValueStart,
                                                                                base.getDateAxisValueStart(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._dateAxisValueEnd,
                                                                                base.getDateAxisValueEnd(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._color, base.getColor()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._lineWidth, base.getLineWidth()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._subPlotIndex,
                                                                                 base.getSubPlotIndex()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._visible, base.getVisible()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._includeInAxisLimitsComputation,
                                                                                 base.getIncludeInAxisLimitsComputation()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._isZone, base.getIsZone()))
        {
            return false;
        }
        if(!this._labelParameters.equals(base.getLabel()))
        {
            return false;
        }
        return true;
    }

    /**
     * Checks for full equality.
     */
    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof ThresholdParameters))
        {
            return false;
        }

        final ThresholdParameters base = (ThresholdParameters)obj;

        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._identifier,
                                                                                base.getIdentifier(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._axisIdString,
                                                                                base.getAxisIdString(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._dateAxisValueStart,
                                                                                base.getDateAxisValueStart(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._dateAxisValueEnd,
                                                                                base.getDateAxisValueEnd(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._color, base.getColor()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._lineWidth, base.getLineWidth()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._numericalAxisValueStartStr,
                                                                                base.getNumericalAxisValueStartStr(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(this._numericalAxisValueEndStr,
                                                                                base.getNumericalAxisValueEndStr(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._subPlotIndex,
                                                                                 base.getSubPlotIndex()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._visible, base.getVisible()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._includeInAxisLimitsComputation,
                                                                                 base.getIncludeInAxisLimitsComputation()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(this._isZone, base.getIsZone()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(_labelAnchor,
                                                                                base.getLabelAnchor(),
                                                                                false))
        {
            return false;
        }
        if(!this._labelParameters.equals(base.getLabel()))
        {
            return false;
        }
        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    /**
     * @return The {@link Marker} most recently drawn for this threshold.
     */
    protected Marker getThresholdMarker()
    {
        return _thresholdMarker;
    }

    public String getAxisIdString()
    {
        return this._axisIdString;
    }

    public Color getColor()
    {
        return _color;
    }

    public String getDateAxisValueEnd()
    {
        return this._dateAxisValueEnd;
    }

    public String getDateAxisValueStart()
    {
        return this._dateAxisValueStart;
    }

    public boolean isDateStartValueUnbounded()
    {
        return this.getDateAxisValueStart().equals(UNBOUNDED_STRING);
    }

    public boolean isDateEndValueUnbounded()
    {
        return this.getDateAxisValueEnd().equals(UNBOUNDED_STRING);
    }

    public String getEvaluatedIdentifier()
    {
        if(getArguments() == null)
        {
            return _identifier;
        }
        return this.getArguments().replaceArgumentsInString(_identifier);
    }

    public String getIdentifier()
    {
        return this._identifier;
    }

    public Boolean getIsZone()
    {
        return _isZone;
    }

    public LabelChartParameters getLabel()
    {
        return this._labelParameters;
    }

    public String getLabelAnchor()
    {
        return _labelAnchor;
    }

    public boolean getShowLabelInPlot()
    {
        return _showLabelInPlot;
    }

    public Float getLineWidth()
    {
        return _lineWidth;
    }

    /**
     * @throws ChartParametersException If {@link String} parameter cannot be parsed by
     *             {@link #getNumericalValueForString(String, String)}.
     */
    public Double getNumericalAxisValueEndValue() throws ChartParametersException
    {
        try
        {
            return getNumericalValueForString(_numericalAxisValueEndStr,
                                              "Getting numerical threshold end value from '"
                                                  + _numericalAxisValueEndStr);
        }
        catch(final Exception e)
        {
            throw new ChartParametersException("Numerical axis value end invalid: " + e.getMessage());
        }
    }

    /**
     * @throws ChartParametersException If {@link String} parameter cannot be parsed by
     *             {@link #getNumericalValueForString(String, String)}.
     */
    public Double getNumericalAxisValueStartValue() throws ChartParametersException
    {
        try
        {
            return getNumericalValueForString(_numericalAxisValueStartStr,
                                              "Getting numerical threshold start value from '"
                                                  + _numericalAxisValueStartStr);
        }
        catch(final Exception e)
        {
            throw new ChartParametersException("Numerical axis value start invalid: " + e.getMessage());
        }
    }

    /**
     * Logs a warning if a problem occurs.
     * 
     * @param str String to check.
     * @param messagePrefix Prefix for a message to generate if a problem occurs.
     * @return The parsed value of the numerical string AFTER arguments are replaced. Will return null if a problem
     *         occurs so that the threshold value is unbounded if a zone or undefined if a line.
     */
    private Double getNumericalValueForString(final String str, final String messagePrefix) throws Exception
    {
        final String evaluatedStr = this.getArguments().replaceArgumentsInString(str);
        if((evaluatedStr != null) && (!evaluatedStr.isEmpty()))
        {
            try
            {
                final NumberFormat format = NumberFormat.getInstance(Locale.US);
                final Number number = format.parse(evaluatedStr);//XXX I'm not sure why this was used instead of a normal parser.
                return number.doubleValue();
            }
            catch(final Exception e)
            {
                LOG.warn(messagePrefix + ": Value after evaluating arguments, '" + evaluatedStr
                    + "', is not a number (may indicate invalid or non-existant arguments); "
                    + "unbounded threshold value is assumed, which may cause other errors.");
                //null will be returned below, yielding unbounded intervals.
            }
        }
        return null;
    }

    public boolean isNumericalStartValueUnbounded()
    {
        return (getNumericalAxisValueStartStr() != null) && (getNumericalAxisValueStartStr().equals(UNBOUNDED_STRING));
    }

    public boolean isNumericalEndValueUnbounded()
    {
        return (getNumericalAxisValueEndStr() != null) && (getNumericalAxisValueEndStr().equals(UNBOUNDED_STRING));
    }

    public String getNumericalAxisValueEndStr()
    {
        return this._numericalAxisValueEndStr;
    }

    public String getNumericalAxisValueStartStr()
    {
        return this._numericalAxisValueStartStr;
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    public Integer getSubPlotIndex()
    {
        return this._subPlotIndex;
    }

    public Boolean getVisible()
    {
        return _visible;
    }

    public Boolean getIncludeInAxisLimitsComputation()
    {
        return _includeInAxisLimitsComputation;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        //Are the other parameters known?
        if(this._visible == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", visibility flag is not specified.");
        }
        if(this._includeInAxisLimitsComputation == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", includeInAxisLimitsComputation flag is not specified.");
        }
        if(this._isZone == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", isZone flag is not specified.");
        }
        if(this._axisIdString == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", axis identifying string not specified.");
        }
        if(this.convertAxisIdStringToInteger() == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", axis identifying string, '" + this._axisIdString + "', is not valid.");
        }
        if(this._color == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", mark color not specified.");
        }
        if(this._identifier == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", identifier for the mark is not specified.");
        }
        if(this._lineWidth == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", line width is not specified.");
        }
        if(this._subPlotIndex == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", subplot index is not specified.");
        }
        if(this._labelAnchor == null)
        {
            throw new ChartParametersException("For threshold with id " + this._identifier
                + ", label anchor is not specified.");
        }
        this._labelParameters.haveAllParametersBeenSet();
    }

    /**
     * @return True if all parameters are undefined. It does not check the identifier.
     */
    public boolean isEmpty()
    {
        if((_visible == null) && (_includeInAxisLimitsComputation == null) && (_isZone == null)
            && (_axisIdString == null) && (_dateAxisValueStart == null) && (_dateAxisValueEnd == null)
            && (_color == null) && (_lineWidth == null) && (_numericalAxisValueStartStr == null)
            && (_numericalAxisValueEndStr == null) && (_subPlotIndex == null) && (_labelParameters.getFont() == null)
            && (_labelParameters.getText() == null) && (_labelParameters.getColor() == null))
        {
            return true;
        }
        return false;
    }

    /**
     * @return For the passed in {@link IntervalMarker}, returns true if the end value value is
     *         {@value Double#MAX_VALUE}.
     */
    private boolean isIntervalMarkerEndMaximum(final IntervalMarker mark)
    {
        return (mark.getEndValue() == Double.MAX_VALUE);
    }

    /**
     * @return For the passed in {@link IntervalMarker}, returns true if the start value value is
     *         {@value Double#MIN_VALUE}.
     */
    private boolean isIntervalMarkerStartMinimum(final IntervalMarker mark)
    {
        return (mark.getStartValue() == Double.MIN_VALUE);
    }

    /**
     * Checks for validity of these parameters if they are to be override parameters. Check is based on provided axis
     * type.
     * 
     * @param axisType Processed through {@link ChartConstants#isAxisTypeTime(int)}.
     */
    public boolean isValidOverRideParameter(final int axisType)
    {
        boolean valid = true;

        if(this._identifier == null)
        {
            valid = false;
        }
        else if(this._visible == null)
        {
            valid = false;
        }
        else if(this._includeInAxisLimitsComputation == null)
        {
            valid = false;
        }
        else if(this._isZone == null)
        {
            valid = false;
        }
        else if(this._axisIdString == null)
        {
            valid = false;
        }
        else if(this._labelParameters == null)
        {
            valid = false;
        }
        else if(this._color == null)
        {
            valid = false;
        }
        else if(this._lineWidth == null)
        {
            valid = false;
        }
        else if(this._subPlotIndex == null)
        {
            valid = false;
        }
        else
        {
            if(ChartConstants.isAxisTypeTime(axisType))
            {
                if(this._dateAxisValueStart == null)
                {
                    valid = false;
                }
            }
            else
            {
                if(this._numericalAxisValueStartStr == null)
                {
                    valid = false;
                }
            }
        }

        return valid;
    }

    /**
     * Wrapper on {@link String#equals(Object)} making a comparison with {@link #LABEL_ANCHOR_AUTO}.
     */
    public boolean isLabelAnchorAuto()
    {
        return (_labelAnchor.equals(LABEL_ANCHOR_AUTO));
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(this.getXMLTagName()))
        {
            clearParameters();

            //Visible attribute
            this._visible = null;
            final String visStr = attr.getValue("visible");
            if(visStr != null)
            {
                try
                {
                    _visible = Boolean.parseBoolean(visStr);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("Visible attribute, '" + visStr + "', is neither true nor false.");
                }
            }
            //is included in axis limits computation attribute
            this._includeInAxisLimitsComputation = null;
            final String includeInAxisLimitsComputationString = attr.getValue("includeInAxisLimitsComputation");
            if(includeInAxisLimitsComputationString != null)
            {
                try
                {
                    _includeInAxisLimitsComputation = Boolean.parseBoolean(includeInAxisLimitsComputationString);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("includeInAxisLimitsComputation attribute, '"
                        + includeInAxisLimitsComputationString + "', is neither true nor false.");
                }
            }
            //isZone attribute
            this._isZone = null;
            final String zoneStr = attr.getValue("isZone");
            if(zoneStr != null)
            {
                try
                {
                    _isZone = Boolean.parseBoolean(zoneStr);
                }
                catch(final NumberFormatException e)
                {
                    throw new XMLReaderException("isZone attribute, '" + zoneStr + "', is neither true nor false.");
                }
            }

            //Identifier
            final String idString = attr.getValue("id");
            if(idString == null)
            {
                throw new XMLReaderException("Identifier is required for threshold, but not specified.");
            }
            this.setIdentifier(idString);
        }
        else if(elementName.equals("color"))
        {
            try
            {
                final Color newColor = XMLTools.extractColorFromAttributes(attr);
                _color = newColor;
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException("Unable to convert color attributes to a color: " + e.getMessage());
            }
        }
        else if(elementName.equals("dateStartValue"))
        {
            final String unboundedValue = attr.getValue("unbounded");
            if(unboundedValue != null)
            {
                try
                {
                    final boolean b = Boolean.parseBoolean(unboundedValue);
                    if(b == true)
                    {
                        this.setDateAxisValueStart(UNBOUNDED_STRING);
                        return null;
                    }
                }
                catch(final Exception e)
                {
                    throw new XMLReaderException("Unbounded attribute must be true or false, but was '" + unboundedValue
                        + "'.");
                }
            }

            //At this point, we know the bound is not unbounded, so process the rest of the date.
            final String dateStr = XMLTools.computeDateStringFromAttributes(attr);
            if(HEFSDateTools.isDateStringValid(dateStr))
            {
                this.setDateAxisValueStart(dateStr);
            }
            else
            {
                throw new XMLReaderException("Element specifies a bad date string, '" + dateStr + "'.");
            }
        }
        else if(elementName.equals("dateEndValue"))
        {
            final String unboundedValue = attr.getValue("unbounded");
            if(unboundedValue != null)
            {
                try
                {
                    final boolean b = Boolean.parseBoolean(unboundedValue);
                    if(b == true)
                    {
                        this.setDateAxisValueEnd(UNBOUNDED_STRING);
                        return null;
                    }
                }
                catch(final Exception e)
                {
                    throw new XMLReaderException("Unbounded attribute must be true or false, but was '" + unboundedValue
                        + "'.");
                }
            }

            //At this point, we know the bound is not unbounded, so process the rest of the date.
            final String dateStr = XMLTools.computeDateStringFromAttributes(attr);
            if(HEFSDateTools.isDateStringValid(dateStr))
            {
                this.setDateAxisValueEnd(dateStr);
            }
            else
            {
                throw new XMLReaderException("Element specifies a bad date string, '" + dateStr + "'.");
            }
        }
        else if(elementName.equals(this._labelParameters.getXMLTagName()))
        {
            return this._labelParameters;
        }
        return null;
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        super.setArguments(arguments);
        this._labelParameters.setArguments(arguments);

    }

    public void setAxisIdString(final String id)
    {
        this._axisIdString = id;
    }

    public void setColor(final Color c)
    {
        _color = c;
    }

    public void setDateAxisValueEnd(final String value)
    {
        this._dateAxisValueEnd = value;
    }

    public void setDateAxisValueStart(final String value)
    {
        this._dateAxisValueStart = value;
    }

    public void setIdentifier(final String id)
    {
        this._identifier = id;
    }

    public void setIsZone(final Boolean b)
    {
        _isZone = b;
    }

    public void setLabelAnchor(final String labelAnchor)
    {
        _labelAnchor = labelAnchor;
    }

    public void setShowLabelInPlot(final boolean b)
    {
        _showLabelInPlot = b;
    }

    public void setLineWidth(final Float wid)
    {
        _lineWidth = wid;
    }

    public void setNumericalAxisValueEndStr(final String value)
    {
        this._numericalAxisValueEndStr = value;
    }

    public void setNumericalAxisValueStartStr(final String value)
    {
        this._numericalAxisValueStartStr = value;
    }

    public void setSubPlotIndex(final Integer i)
    {
        this._subPlotIndex = i;
    }

    /**
     * This should only be called prior to applying the threshold from within the list application method.
     * 
     * @param type
     */
    protected void setAxisType(final int type)
    {
        this._axisType = type;
    }

    /**
     * This should only be called prior to applying the threshold from within the list application method.
     * 
     * @param type
     */
    protected void setSubPlotIndexToXYPlot(final HashMap<Integer, XYPlot> subPlotIndexToXYPlot)
    {
        this._subPlotIndexToXYPlot = subPlotIndexToXYPlot;
    }

    @Override
    public void setupDefaultParameters()
    {
        clearParameters();
        this._visible = true;
        this._includeInAxisLimitsComputation = true;
        this._isZone = false;
        this._numericalAxisValueStartStr = "0.0";
        this._axisIdString = "LEFT";
        this._labelParameters.setupDefaultParameters();
        this._color = Color.BLACK;
        this._lineWidth = 1.0F;
        this._subPlotIndex = 0;
        this._labelAnchor = LABEL_ANCHOR_AUTO;
        _showLabelInPlot = true;
    }

    /**
     * Populates any non-overridden parameter with a default value, essentially filling in the empty parameters. Useful
     * when a threshold is provided as an override, but there is nothing corresponding to override it when the user
     * tries to edit it.
     */
    public void setupDefaultValuesForNonOverriddenParameters()
    {
        if(_visible == null)
        {
            this._visible = true;
        }
        if(_includeInAxisLimitsComputation == null)
        {
            this._includeInAxisLimitsComputation = true;
        }
        if(_isZone == null)
        {
            this._isZone = false;
        }
        if(_numericalAxisValueStartStr == null)
        {
            this._numericalAxisValueStartStr = "0.0";
        }
        if(_axisIdString == null)
        {
            this._axisIdString = "LEFT";
        }
        if(_color == null)
        {
            this._color = Color.RED;
        }
        if(_lineWidth == null)
        {
            this._lineWidth = 1.0F;
        }
        if(_subPlotIndex == null)
        {
            this._subPlotIndex = 0;
        }
        if(_labelAnchor == null)
        {
            this._labelAnchor = LABEL_ANCHOR_AUTO;
        }
        this._labelParameters.setupDefaultValuesForNonOverriddenParameters();
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("subPlot"))
        {
            try
            {
                setSubPlotIndex(Integer.parseInt(value));
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Subplot index must be an integer, but was '" + value + "'.");
            }
        }
        else if(elementName.equals("axis"))
        {
            if(ChartConstants.determineAxisIndexFromString(value) == null)
            {
                throw new XMLReaderException("Axis string, '" + value + "', is invalid.");
            }
            this.setAxisIdString(value);
        }
        else if(elementName.equals("numericalStartValue"))
        {
            this.setNumericalAxisValueStartStr(value);
        }
        else if(elementName.equals("numericalEndValue"))
        {
            this.setNumericalAxisValueEndStr(value);
        }
        else if(elementName.equals("lineWidth"))
        {
            try
            {
                this._lineWidth = Float.parseFloat(value.trim());
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Line width '" + value.trim() + "' is not a number.");
            }
        }
        else if(elementName.equals("labelAnchor"))
        {
            _labelAnchor = value;
        }
    }

    public void setVisible(final Boolean b)
    {
        _visible = b;
    }

    public void setIncludeInAxisLimitsComputation(final Boolean b)
    {
        _includeInAxisLimitsComputation = b;
    }

    @Override
    public String toString()
    {
        String result = "ThresholdParameters: ";
        result += "xmlTagName = '" + this.getXMLTagName() + "'; ";
        result += "identifier = '" + this.getIdentifier() + "'; ";
        result += "Label:font = " + this.getLabel().getFont() + ";";
        result += "Label:color = " + this.getLabel().getColor() + ";";
        result += "Label:text = " + this.getLabel().getText() + ";";
        result += "visible = " + this.getVisible() + "; ";
        result += "includeInAxisLimitsComputation = " + this.getIncludeInAxisLimitsComputation() + "; ";
        result += "isZone = " + this.getIsZone() + "; ";
        result += "subPlotIndex = " + this.getSubPlotIndex() + "; ";
        result += "axisIdString = '" + this.getAxisIdString() + "'; ";
        result += "numericalAxisValueStart = " + this.getNumericalAxisValueStartStr() + ";";
        result += "numericalAxisValueEnd = " + this.getNumericalAxisValueEndStr() + ";";
        result += "dateAxisValueStart = " + this.getDateAxisValueStart() + ";";
        result += "dateAxisValueEnd = " + this.getDateAxisValueEnd() + ";";
        result += "labelAnchor = " + this.getLabelAnchor() + ";";

        if(_color != null)
        {
            result += "color = " + this._color.toString() + "; ";
        }
        else
        {
            result += "color = null; ";
        }

        result += "lineWidth = " + this._lineWidth + "; ";
        return result;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(this.getXMLTagName());
        mainElement.setAttribute("id", this._identifier);
        if(_visible != null)
        {
            mainElement.setAttribute("visible", "" + _visible);
        }
        if(_includeInAxisLimitsComputation != null)
        {
            mainElement.setAttribute("includeInAxisLimitsComputation", "" + _includeInAxisLimitsComputation);
        }
        if(_isZone != null)
        {
            mainElement.setAttribute("isZone", "" + _isZone);
        }

        if(this._axisIdString != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "axis", this._axisIdString));
        }
        if(this._subPlotIndex != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "subPlot", "" + _subPlotIndex));
        }

        if(this._dateAxisValueStart != null)
        {
            Element startElement;
            if(this._dateAxisValueStart.equals(UNBOUNDED_STRING))
            {
                startElement = request.createElement("dateStartValue");
                startElement.setAttribute("unbounded", "true");
            }
            else
            {
                startElement = XMLTools.createDateElement(request, "dateStartValue", _dateAxisValueStart);
            }
            mainElement.appendChild(startElement);
        }
        if(this._dateAxisValueEnd != null)
        {
            Element startElement;
            if(this._dateAxisValueEnd.equals(UNBOUNDED_STRING))
            {
                startElement = request.createElement("dateEndValue");
                startElement.setAttribute("unbounded", "true");
            }
            else
            {
                startElement = XMLTools.createDateElement(request, "dateEndValue", _dateAxisValueEnd);
            }
            mainElement.appendChild(startElement);
        }
        if(this._numericalAxisValueStartStr != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "numericalStartValue",
                                                                   "" + _numericalAxisValueStartStr));
        }
        if(this._numericalAxisValueEndStr != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "numericalEndValue",
                                                                   "" + _numericalAxisValueEndStr));
        }

        XMLTools.appendElementIfNotEmpty(mainElement, this._labelParameters.writePropertyToXMLElement(request));

        if(_color != null)
        {
            mainElement.appendChild(XMLTools.createColorElement("color", request, _color));
        }
        if(_lineWidth != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "lineWidth", "" + _lineWidth));
        }
        if(_labelAnchor != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "labelAnchor", _labelAnchor));
        }

        return mainElement;
    }

    /**
     * @return The {@link RectangleAnchor} corresponding to the provided anchro name. See {@link #LABEL_ANCHOR_ARRAY}
     *         and {@link #LABEL_RECTAGLE_ANCHOR_ARRAY}.
     */
    public static RectangleAnchor getRectangleAnchorByString(final String anchorName)
    {
        if(anchorName == null || anchorName.trim().length() == 0)
        {
            return null;
        }

        for(int i = 0; i < LABEL_ANCHOR_ARRAY.length; i++)
        {
            if(anchorName.equals(LABEL_ANCHOR_ARRAY[i]))
            {
                return LABEL_RECTAGLE_ANCHOR_ARRAY[i];
            }
        }

        return null;
    }
}
