package ohd.hseb.charter.parameters;

import org.jfree.chart.JFreeChart;
import org.jfree.chart.title.TextTitle;
import org.jfree.ui.RectangleInsets;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLToolsException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

/**
 * Parameters for the title within a chart.
 * 
 * @author hank.herr
 */
public class TitleParameters extends LabelChartParameters
{
    public static final String DEFAULT_ALIGNMENT_DISPLAY_STRING = "Default (CENTER)";
    public static final String DEFAULT_ALIGNMENT_STRING = "CENTER";
    public static final RectangleInsets DEFAULT_TEXT_INSETS = new RectangleInsets(0.0, 0.0, 0.0, 0.0);

    /**
     * The horizontal alignment if the text box within the chart. Processed through
     * {@link ChartConstants#determineHorizontalAlignmentFromAlignmentString(String)}.
     */
    private String _horizontalAlignment = null;

    /**
     * The text alignment within the box. Processed through
     * {@link ChartConstants#determineHorizontalAlignmentFromAlignmentString(String)}.
     */
    private String _textAlignment = null;

    /**
     * Insets of the text within the title box.
     */
    private RectangleInsets _textInsets = null;

    /**
     * Parameters of the border for the title box.
     */
    private BorderParameters _border = new BorderParameters("border");

    /**
     * Empty constructor initializes XML tag to "title".
     */
    public TitleParameters()
    {
        setXMLTagName("title");
    }

    /**
     * Constructor allows for specifying the XML tag.
     */
    public TitleParameters(final String xmlTagName)
    {
        setXMLTagName(xmlTagName);
    }

    public String getHorizontalAlignment()
    {
        return _horizontalAlignment;
    }

    public void setHorizontalAlignment(final String horizontalAlignment)
    {
        _horizontalAlignment = horizontalAlignment;
    }

    public String getTextAlignment()
    {
        return _textAlignment;
    }

    public void setTextAlignment(final String textAlignment)
    {
        _textAlignment = textAlignment;
    }

    public RectangleInsets getTextInsets()
    {
        return _textInsets;
    }

    public void setTextInsets(final RectangleInsets textInsets)
    {
        _textInsets = textInsets;
    }

    public BorderParameters getBorder()
    {
        return _border;
    }

    public void setBorder(final BorderParameters border)
    {
        _border = border;
        _border.setXMLTagName("border");
    }

    /**
     * Creates a {@link TextTitle} object for insert into a JFreeChart and returns it.
     */
    public TextTitle createTitle()
    {
        final TextTitle title = new TextTitle(getArgumentReplacedText(), getFont());
        title.setPaint(getColor());
        title.setHorizontalAlignment(ChartConstants.determineHorizontalAlignmentFromAlignmentString(this._horizontalAlignment));
        title.setTextAlignment(ChartConstants.determineHorizontalAlignmentFromAlignmentString(this._textAlignment));
        title.setFrame(_border.createBlockBorder());
        title.setMargin(this._textInsets);
        return title;
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo)
    {
        final JFreeChart chart = (JFreeChart)objectAppliedTo;
        final TextTitle chartTitle = createTitle();
        chart.setTitle(chartTitle);
    }

    @Override
    public void clearParameters()
    {
        super.clearParameters();
        this._border.clearParameters();
        this._horizontalAlignment = null;
        this._textAlignment = null;
        this._textInsets = null;
    }

    @Override
    public Object clone()
    {
        final TitleParameters cloneParms = new TitleParameters();
        cloneParms.copyFrom(this);
        return cloneParms;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        super.copyFrom(parameters);
        final TitleParameters base = (TitleParameters)parameters;
        clearParameters();
        copyOverriddenParameters(base);
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        super.copyOverriddenParameters(override);
        final TitleParameters base = (TitleParameters)override;
        _border.copyOverriddenParameters(base.getBorder());
        if(base.getHorizontalAlignment() != null)
        {
            this._horizontalAlignment = base.getHorizontalAlignment();
        }
        if(base.getTextAlignment() != null)
        {
            this._textAlignment = base.getTextAlignment();
        }
        if(base.getTextInsets() != null)
        {
            this._textInsets = new RectangleInsets(base.getTextInsets().getTop(),
                                                   base.getTextInsets().getLeft(),
                                                   base.getTextInsets().getBottom(),
                                                   base.getTextInsets().getRight());
        }
    }

    @Override
    public boolean equals(final Object parameters)
    {
        if(!(parameters instanceof ohd.hseb.charter.parameters.TitleParameters))
        {
            return false;
        }
        if(!super.equals(parameters))
        {
            return false;
        }
        final TitleParameters other = (TitleParameters)parameters;
        if(!_border.equals(other.getBorder()))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(_horizontalAlignment,
                                                                                other.getHorizontalAlignment(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(_textAlignment,
                                                                                other.getTextAlignment(),
                                                                                true))
        {
            return false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(_textInsets, other.getTextInsets()))
        {
            return false;
        }
        return true;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        return null;
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
        super.haveAllParametersBeenSet();
        _border.haveAllParametersBeenSet();
        if(this._horizontalAlignment == null)
        {
            throw new ChartParametersException("Horizontal alignment not specified.");
        }
        if(this._textAlignment == null)
        {
            throw new ChartParametersException("Text alignment not specified.");
        }
        if(this._textInsets == null)
        {
            throw new ChartParametersException("Text insets (interior margin) not specified.");
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }
        else if(elementName.equals(_border.getXMLTagName()))
        {
            return _border;
        }
        else if(elementName.equals("textInsets"))
        {
            try
            {
                _textInsets = XMLTools.extractInsetsFromAttributes(attr);
            }
            catch(final XMLToolsException e)
            {
                throw new XMLReaderException(e.getMessage());
            }
        }
        else if((!elementName.equals("horizontalAlignment")) && (!elementName.equals("textAlignment")))
        {
            return super.readInPropertyFromXMLElement(elementName, attr);
        }

        return null;
    }

    @Override
    public void setupDefaultParameters()
    {
        super.setupDefaultParameters();
        _horizontalAlignment = DEFAULT_ALIGNMENT_STRING;
        _textAlignment = DEFAULT_ALIGNMENT_STRING;
        _textInsets = DEFAULT_TEXT_INSETS;
        _border.setupDefaultParameters();
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        if(elementName.equals("horizontalAlignment"))
        {
            if(ChartConstants.determineHorizontalAlignmentFromAlignmentString(value) == null)
            {
                throw new XMLReaderException("The value of horizontalAlignment, '" + value + "', is invalid.");
            }
            this._horizontalAlignment = value;
        }
        else if(elementName.equals("textAlignment"))
        {
            if(ChartConstants.determineHorizontalAlignmentFromAlignmentString(value) == null)
            {
                throw new XMLReaderException("The value of textAlignment, '" + value + "', is invalid.");
            }
            this._textAlignment = value;
        }
        else
        {
            super.setValueOfElement(elementName, value);
        }
    }

    @Override
    public String toString()
    {
        String results = "TitleParameters: ";
        results += super.toString() + "; ";
        results += "Border = {" + _border.toString() + "}; ";
        results += "horizontalAlignment = " + this._horizontalAlignment + "; ";
        results += "textAlignment = " + this._textAlignment + "; ";
        results += "textInsets = " + this._textInsets + ".";
        return results;
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = super.writePropertyToXMLElement(request);
        XMLTools.appendElementIfNotEmpty(mainElement, _border.writePropertyToXMLElement(request));
        if(this._horizontalAlignment != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request,
                                                                   "horizontalAlignment",
                                                                   _horizontalAlignment));
        }
        if(this._textAlignment != null)
        {
            mainElement.appendChild(XMLTools.createTextNodeElement(request, "textAlignment", _textAlignment));
        }
        if(this._textInsets != null)
        {
            mainElement.appendChild(XMLTools.createInsetsElement("textInsets", request, _textInsets));
        }
        return mainElement;
    }

    /**
     * @return Array of Strings that can be placed in a combo box for selection. Includes a default string as the first
     *         item.
     */
    public static String[] buildAlignmentDisplayedStrings()
    {
        final String[] results = new String[ChartConstants.ALIGNMENT_STRINGS.length + 1];
        results[0] = DEFAULT_ALIGNMENT_DISPLAY_STRING;
        for(int i = 0; i < ChartConstants.ALIGNMENT_STRINGS.length; i++)
        {
            results[i + 1] = ChartConstants.ALIGNMENT_STRINGS[i];
        }
        return results;
    }
}
