package ohd.hseb.charter.parameters.panels;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import ohd.hseb.charter.ChartEngine;
import ohd.hseb.charter.ChartTools;
import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.parameters.AxisParameters;
import ohd.hseb.charter.translator.AxisTranslator;
import ohd.hseb.charter.translator.AxisTranslatorException;
import ohd.hseb.charter.translator.AxisTranslatorFactory;
import ohd.hseb.hefs.utils.arguments.ArgumentInsertingTextFieldComboBoxEditor;
import ohd.hseb.hefs.utils.arguments.DefaultArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.components.BufferJPanel;
import ohd.hseb.hefs.utils.gui.tools.ComponentHelper;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInPanelListener;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;

import org.jfree.chart.JFreeChart;

public class AxisParametersPanel extends ChartParametersPanel implements GeneralPlugInPanelListener, ActionListener
{

    private static final long serialVersionUID = 1703884066250891487L;

    public static final int DEFAULT_DECIMAL_NUMBER = 2;

    private JComboBox _visibleChoices;
    private final JComboBox _invertedChoices = new JComboBox(AxisParameters.DEFAULT_YES_NO_DISPLAY_STRINGS_NO_DEFAULT);
    private TextFontSelectionPanel _labelPanel;
    private DateAxisParametersPanel _dateAxisParametersPanel;
    private NumericAxisParametersPanel _numericAxisParametersPanel;
    private JComboBox _axisTypeChoices;
    private JPanel _translationTypePanel;
    private JComboBox _translatedChoices;
    private final JPanel _axisTypeEditingPanelWrapper = new JPanel(new BorderLayout());
    private int _calculatedAxisType = ChartConstants.AXIS_IS_NUMERICAL;
    private ChartParametersPanel _translatorPanel = null;
    private DefaultArgumentsProcessor _argumentsProcessor = null;
    private AxisParameters _parametersLastTimeChangeWasMade = null;

    public AxisParametersPanel(AxisParameters managedParameters,
                               AxisParameters defaultParameters,
                               int calculatedAxisType,
                               DefaultArgumentsProcessor argumentsProcessor)
    {
        setManagedParameters(managedParameters);
        setDefaultParameters(defaultParameters);
        _calculatedAxisType = calculatedAxisType;
        setArgumentsProcessor(argumentsProcessor);
        createDisplay();
        makePanelReflectParameters();
    }

    private void addListeners()
    {
        //Axis type choices uses an action listener because it is editable, capable of displaying 
        //invalid choices.
        _axisTypeChoices.addActionListener(this);

        _invertedChoices.addActionListener(this);
        _translatedChoices.addActionListener(this);
        _visibleChoices.addActionListener(this);

        _labelPanel.addGraphicsGeneratorPlugInPanelListener(this);
        _dateAxisParametersPanel.addGraphicsGeneratorPlugInPanelListener(this);
        _numericAxisParametersPanel.addGraphicsGeneratorPlugInPanelListener(this);
    }

    private void createDisplay()
    {
        _visibleChoices = HSwingFactory.createJComboBox(determineVisibleChoices(), null);

        _axisTypeChoices = HSwingFactory.createJComboBox(determineAxisTypeChoices(), null);

        //This combobox is editable with the component turned off in order to allow it to be turned
        //pink if there is an error in its setting relative to the axis.  
        //A mouse listener is added to the editor so that clicking on the editor opens up the choices.
        _axisTypeChoices.setEditable(true);
        _axisTypeChoices.setEditor(new ArgumentInsertingTextFieldComboBoxEditor());
        ((JTextField)_axisTypeChoices.getEditor().getEditorComponent()).setDisabledTextColor(_visibleChoices.getEditor()
                                                                                                            .getEditorComponent()
                                                                                                            .getForeground());
        _axisTypeChoices.getEditor().getEditorComponent().setEnabled(false);
        _axisTypeChoices.getEditor().getEditorComponent().addMouseListener(new MouseAdapter()
        {
            @Override
            public void mouseClicked(MouseEvent e)
            {
                if(_axisTypeChoices.isPopupVisible())
                {
                    _axisTypeChoices.hidePopup();
                }
                else
                {
                    _axisTypeChoices.showPopup();
                }
            }
        });

        _translatedChoices = HSwingFactory.createJComboBox(AxisTranslatorFactory.getListOfRegisteredTranslators(), null);

        //Translation type subpanel.
        _translationTypePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        _translationTypePanel.add(new JLabel("Specify Translation: "));
        _translationTypePanel.add(_translatedChoices);

        //Visible combo box.
        JPanel visiblePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        visiblePanel.add(new JLabel("Select Axis Visibility:"));
        visiblePanel.add(_visibleChoices);

        //Axis type combo box panel.
        JPanel typePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        typePanel.add(new JLabel("Select Axis Type:"));
        typePanel.add(_axisTypeChoices);

        //Inverted combo box.
        JPanel invertedPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        invertedPanel.add(new JLabel("Select Axis Inversion:"));
        invertedPanel.add(_invertedChoices);

        //The combined editing panel for the axis type.
        JPanel translatorPanel = new JPanel(new BorderLayout());
        translatorPanel.add(_translationTypePanel, BorderLayout.NORTH);
        translatorPanel.add(_axisTypeEditingPanelWrapper, BorderLayout.CENTER);

        //Others
        AxisParameters defaultParameters = (AxisParameters)getDefaultParameters();
        _labelPanel = new TextFontSelectionPanel(getParameters().getLabel(), defaultParameters.getLabel(), true, true);

        _dateAxisParametersPanel = new DateAxisParametersPanel(getParameters().getDateAxisParameters(),
                                                               defaultParameters.getDateAxisParameters(),
                                                               getArgumentsProcessor());

        _numericAxisParametersPanel = new NumericAxisParametersPanel(getParameters().getNumericAxisParameters(),
                                                                     defaultParameters.getNumericAxisParameters(),
                                                                     getArgumentsProcessor());

        this.setLayout(new GridBagLayout());
        int gridy = 0;
        GridBagConstraints constraints = SwingTools.returnGridBagConstraints(0,
                                                                                gridy,
                                                                                1,
                                                                                1,
                                                                                1,
                                                                                0,
                                                                                GridBagConstraints.WEST,
                                                                                GridBagConstraints.BOTH,
                                                                                new Insets(2, 2, 2, 2),
                                                                                0,
                                                                                0);
        this.add(visiblePanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                             gridy,
                                                             1,
                                                             1,
                                                             1,
                                                             0,
                                                             GridBagConstraints.WEST,
                                                             GridBagConstraints.BOTH,
                                                             new Insets(2, 2, 2, 2),
                                                             0,
                                                             0);
        this.add(invertedPanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                             gridy,
                                                             1,
                                                             1,
                                                             1,
                                                             0,
                                                             GridBagConstraints.WEST,
                                                             GridBagConstraints.BOTH,
                                                             new Insets(2, 2, 2, 2),
                                                             0,
                                                             0);
        this.add(typePanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                             gridy,
                                                             1,
                                                             1,
                                                             1,
                                                             0,
                                                             GridBagConstraints.WEST,
                                                             GridBagConstraints.BOTH,
                                                             new Insets(0, 0, 0, 0),
                                                             0,
                                                             0);
        this.add(translatorPanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                             gridy,
                                                             1,
                                                             1,
                                                             1,
                                                             0,
                                                             GridBagConstraints.WEST,
                                                             GridBagConstraints.BOTH,
                                                             new Insets(0, 0, 0, 0),
                                                             0,
                                                             0);
        this.add(ComponentHelper.removeEndingBufferJPanelIfAny(_labelPanel), constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                             gridy,
                                                             1,
                                                             1,
                                                             1,
                                                             0,
                                                             GridBagConstraints.WEST,
                                                             GridBagConstraints.BOTH,
                                                             new Insets(0, 0, 0, 0),
                                                             0,
                                                             0);
        this.add(_dateAxisParametersPanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                             gridy,
                                                             1,
                                                             1,
                                                             1,
                                                             0,
                                                             GridBagConstraints.WEST,
                                                             GridBagConstraints.BOTH,
                                                             new Insets(0, 0, 0, 0),
                                                             0,
                                                             0);
        this.add(_numericAxisParametersPanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                             gridy,
                                                             1,
                                                             1,
                                                             1,
                                                             1,
                                                             GridBagConstraints.WEST,
                                                             GridBagConstraints.BOTH,
                                                             new Insets(0, 0, 0, 0),
                                                             0,
                                                             0);
        this.add(new BufferJPanel(), constraints);

        this.determineVisibilityOfPanels();
    }

    private List<String> determineAxisTypeChoices()
    {
        List<Integer> compatibleTypes = ChartTools.obtainListOfCompatibleAxesForAxisType(_calculatedAxisType);
        List<String> choices = new ArrayList<String>();

        if(_calculatedAxisType < 0)
        {
            choices.add("Default (undefined)");
        }
        else
        {
            choices.add("Default (" + ChartConstants.AXIS_CHOICES[this._calculatedAxisType] + ")");
        }

        for(int i = 0; i < compatibleTypes.size(); i++)
        {
            choices.add(ChartConstants.AXIS_CHOICES[compatibleTypes.get(i)]);
        }

        return choices;
    }

    private void determineVisibilityOfPanels()
    {
        //The axis type implied by the current set of parameters.
        int chosenAxisType = getCurrentAxisType();

        //Date/Time
        if(chosenAxisType < 0)
        {
            _dateAxisParametersPanel.setVisible(false);
            _numericAxisParametersPanel.setVisible(false);
            if(_translationTypePanel != null)
            {
                _translationTypePanel.setVisible(false);
            }
        }
        else if(ChartConstants.isAxisTypeTime(chosenAxisType))
        {
            _dateAxisParametersPanel.setVisible(true);
            _numericAxisParametersPanel.setVisible(false);
            if(_translationTypePanel != null)
            {
                _translationTypePanel.setVisible(false);
            }
        }
        //Translated
        else if(ChartConstants.isAxisTypeTranslated(chosenAxisType))
        {
            _dateAxisParametersPanel.setVisible(false);
            _translationTypePanel.setVisible(true);
            _numericAxisParametersPanel.setVisible(true); //Make the entire panel visible, then make subpanels visible
            _numericAxisParametersPanel.displayTickFormatPanelOnly();
        }
        else
        //Numerical, not translated
        {
            _dateAxisParametersPanel.setVisible(false);
            _numericAxisParametersPanel.setVisible(true); //Make the entire panel visible, then make subpanels visible
            _numericAxisParametersPanel.displayAllPanels();
            _translationTypePanel.setVisible(false);
        }
    }

    private List<String> determineVisibleChoices()
    {
        List<String> choices = new ArrayList<String>();
        if(((AxisParameters)this.getDefaultParameters()).getVisible())
        {
            choices.add("Default (Yes)");
        }
        else
        {
            choices.add("Default (No)");
        }
        choices.add("Yes");
        choices.add("No");
        return choices;
    }

    public DefaultArgumentsProcessor getArgumentsProcessor()
    {
        return _argumentsProcessor;
    }

    /**
     * Gets either the overridden axis type int or the default (calculated) axis type.
     */
    private int getCurrentAxisType()
    {
        if(getParameters().getAxisTypeInt() < 0)
        {
            return this._calculatedAxisType;
        }
        return getParameters().getAxisTypeInt();
    }

    public NumericAxisParametersPanel getNumericAxisParametersPanel()
    {
        return _numericAxisParametersPanel;
    }

    public AxisParameters getParameters()
    {
        return (AxisParameters)getManagedParameters();
    }

    private void makeAxisTypePanelReflectParameters()
    {
        if(getParameters().getAxisType() == null)
        {
            _axisTypeChoices.setSelectedIndex(0);

            _axisTypeChoices.getEditor().getEditorComponent().setBackground(this._visibleChoices.getBackground());
        }
        else
        {
            _axisTypeChoices.getEditor().setItem(getParameters().getAxisType());
            if(SwingTools.findStringInJComboBox(getParameters().getAxisType(), _axisTypeChoices) < 0)
            {
                _axisTypeChoices.getEditor().getEditorComponent().setBackground(Color.PINK);
            }
            else
            {
                _axisTypeChoices.getEditor().getEditorComponent().setBackground(_visibleChoices.getBackground());
            }
        }
        updateEditingPanelToBeForAxisType();
    }

    private void makeInvertedPanelReflectParameters()
    {
        _invertedChoices.setSelectedItem(AxisParameters.getYesNoChoiceString(getParameters().getInverted()));
    }

    private void makeVisiblePanelReflectParameters()
    {
        if(this.getParameters().getVisible() == null)
        {
            this._visibleChoices.setSelectedIndex(0);
        }
        else if(this.getParameters().getVisible() == true)
        {
            this._visibleChoices.setSelectedIndex(1);
        }
        else
        {
            this._visibleChoices.setSelectedIndex(2);
        }
    }

    private void reactToTranslationChoiceChange(String translatorName)
    {
        try
        {
            AxisTranslator trans = AxisTranslatorFactory.loadTranslator(translatorName);
            getParameters().getAxisTranslator().setTranslatorName(translatorName);
            trans.clearAllButUsedParameters();
            trans.ensureRequiredParametersArePresent();

            _axisTypeEditingPanelWrapper.setVisible(false);
            _axisTypeEditingPanelWrapper.removeAll();
            _translatorPanel = trans.buildEditingPanel(getParameters(), getParameters().getAxisTranslator(), null);
            _translatorPanel.addGraphicsGeneratorPlugInPanelListener(this);
            _axisTypeEditingPanelWrapper.add(_translatorPanel, BorderLayout.CENTER);
            _axisTypeEditingPanelWrapper.setVisible(true);

            this.revalidate();

        }
        catch(AxisTranslatorException e)
        {
            //This should never happen!!!
            e.printStackTrace();
        }
    }

    private void removeListeners()
    {
        _axisTypeChoices.removeActionListener(this);
        _invertedChoices.removeActionListener(this);
        _translatedChoices.removeActionListener(this);
        _visibleChoices.removeActionListener(this);
        _labelPanel.removeGraphicsGeneratorPlugInPanelListener(this);
        _dateAxisParametersPanel.removeGraphicsGeneratorPlugInPanelListener(this);
        _numericAxisParametersPanel.removeGraphicsGeneratorPlugInPanelListener(this);
    }

    public void setArgumentsProcessor(DefaultArgumentsProcessor _argumentsProcessor)
    {
        this._argumentsProcessor = _argumentsProcessor;
    }

    public void setNumericAxisParametersPanel(NumericAxisParametersPanel _numericAxisParametersPanel)
    {
        this._numericAxisParametersPanel = _numericAxisParametersPanel;
    }

    private void updateEditingPanelToBeForAxisType()
    {
        if(ChartConstants.isAxisTypeTranslated(getParameters().getAxisTypeInt()))
        {
            _translatedChoices.removeActionListener(this);
            _translatedChoices.setSelectedItem(getParameters().getAxisTranslator().getTranslatorName());
            reactToTranslationChoiceChange(getParameters().getAxisTranslator().getTranslatorName());
            _translationTypePanel.setVisible(true);
            _translatedChoices.addActionListener(this);
        }
        else
        {
            _translatorPanel = null;
            _translationTypePanel.setVisible(false);
            _axisTypeEditingPanelWrapper.setVisible(false);
            _axisTypeEditingPanelWrapper.removeAll();
            _axisTypeEditingPanelWrapper.add(new BufferJPanel(), BorderLayout.CENTER);
            _axisTypeEditingPanelWrapper.setVisible(true);
            this.revalidate();
        }
    }

    protected boolean makeParametersReflectPanel()
    {
        boolean changeMade = false;
        changeMade = makeAxisTypeParameterReflectPanel() || changeMade;
        changeMade = this.makeInvertedParameterReflectPanel() || changeMade;
        changeMade = this.makeVisibleParameterReflectPanel() || changeMade;
        changeMade = this.makeInvertedParameterReflectPanel() || changeMade;
        changeMade = this.makeTranslationChoiceParameterReflectPanel() || changeMade;
        changeMade = this._numericAxisParametersPanel.makeParametersReflectPanel() || changeMade;
        changeMade = this._dateAxisParametersPanel.makeParametersReflectPanel() || changeMade;
        return changeMade;
    }

    private boolean makeAxisTypeParameterReflectPanel()
    {
        //If there is no change in the type, then just return and do nothing.
        if(((getParameters().getAxisType() == null) && (_axisTypeChoices.getSelectedIndex() == 0))
            || ((getParameters().getAxisType() != null) && (getParameters().getAxisType().equals(_axisTypeChoices.getSelectedItem()))))
        {
            return false;
        }

        //Unspecified... use calculated
        int selectedAxisType = ChartConstants.determineAxisTypeIntFromAxisTypeString((String)_axisTypeChoices.getSelectedItem());
        if(_axisTypeChoices.getSelectedIndex() == 0) //Default chosen
        {
            getParameters().setAxisType(null);
            getParameters().removeAxisTranslator();
        }
        //Translated...
        else if(ChartConstants.isAxisTypeTranslated(selectedAxisType))
        {
            getParameters().setAxisType((String)_axisTypeChoices.getSelectedItem());
            getParameters().addAxisTranslator();
        }
        else
        {
            getParameters().setAxisType((String)_axisTypeChoices.getSelectedItem());
            getParameters().removeAxisTranslator();
        }
        return true;
    }

    private boolean makeVisibleParameterReflectPanel()
    {
        //Check for a change, first.
        if(((_visibleChoices.getSelectedIndex() == 0) && (getParameters().getVisible() == null))
            && ((_visibleChoices.getSelectedIndex() == 1) && getParameters().getVisible())
            && ((_visibleChoices.getSelectedIndex() == 1) && !getParameters().getVisible()))
        {
            return false;
        }
        if(_visibleChoices.getSelectedIndex() == 0)
        {
            getParameters().setVisible(null);
        }
        else if(_visibleChoices.getSelectedIndex() == 1)
        {
            getParameters().setVisible(true);
        }
        else
        {
            getParameters().setVisible(false);
        }
        return true;
    }

    private boolean makeInvertedParameterReflectPanel()
    {
        Boolean selectedValue = AxisParameters.getYesNoChoiceValue((String)_invertedChoices.getSelectedItem());
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(getParameters().getInverted(), selectedValue))
        {
            getParameters().setInverted(selectedValue);
            return true;
        }
        return false;
    }

    private boolean makeTranslationChoiceParameterReflectPanel()
    {
        String selectedValue = (String)_translatedChoices.getSelectedItem();

        //Theoretically, the translator will never be null if the choices box is able to react.  Hence, no null
        //check on getParameters().getAxisTranslator().
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(getParameters().getAxisTranslator()
                                                                                         .getTranslatorName(),
                                                                          selectedValue,
                                                                          true))

        {
            reactToTranslationChoiceChange(selectedValue);
            return true;
        }
        return false;
    }

    @Override
    public void actionPerformed(ActionEvent e)
    {
        if(e.getSource() == _axisTypeChoices)
        {
            if(makeAxisTypeParameterReflectPanel())
            {
                makeAxisTypePanelReflectParameters();
                determineVisibilityOfPanels();
                fireParametersChanged();
            }
        }
        if(e.getSource() == _visibleChoices)
        {
            if(makeVisibleParameterReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if(e.getSource() == _invertedChoices)
        {
            if(makeInvertedParameterReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if(e.getSource() == _translatedChoices)
        {
            if(makeTranslationChoiceParameterReflectPanel())
            {
                fireParametersChanged();
            }
        }
    }

    @Override
    public void chartEngineReconstructed(ChartEngine engine)
    {
        //Translator panel, first.
        if(_translatorPanel != null)
        {
            _translatorPanel.chartEngineReconstructed(engine);
        }

        //ChartEngine has a memory, so this should not result in a new build, just a check to make
        //sure the parms have not been changed... I hope.
        JFreeChart chart;
        try
        {
            chart = engine.buildChart();

            //Now find the axis in the chart and hand it to the numeric and date axis subpanels.
            _numericAxisParametersPanel.setCurrentAxis(getParameters().getAxisFromChart(engine, chart));
            _dateAxisParametersPanel.setCurrentAxis(getParameters().getAxisFromChart(engine, chart));
        }
        catch(Exception e)
        {
            e.printStackTrace();
            _numericAxisParametersPanel.setCurrentAxis(null);
            _dateAxisParametersPanel.setCurrentAxis(null);
        }
    }

    @Override
    public void disableSubPanel()
    {
        _labelPanel.disableSubPanel();
        _dateAxisParametersPanel.disableSubPanel();
        _numericAxisParametersPanel.disableSubPanel();
    }

    @Override
    public void enableSubPanel()
    {
        _labelPanel.enableSubPanel();
        _dateAxisParametersPanel.enableSubPanel();
        _numericAxisParametersPanel.enableSubPanel();
    }

    @Override
    public void fireParametersChanged()
    {
        //This object will sometimes fire parameter changed events even though the parameters did not change.
        //This check will ensure that does not happen.
        if((_parametersLastTimeChangeWasMade == null) || (!_parametersLastTimeChangeWasMade.equals(getParameters())))
        {
            super.fireParametersChanged();
            _parametersLastTimeChangeWasMade = (AxisParameters)getParameters().clone();
        }
    }

    @Override
    public void makePanelReflectParameters()
    {
        removeListeners();

        makeVisiblePanelReflectParameters();
        makeInvertedPanelReflectParameters();
        makeAxisTypePanelReflectParameters();

        _labelPanel.makePanelReflectParameters();
        _dateAxisParametersPanel.makePanelReflectParameters();
        _numericAxisParametersPanel.makePanelReflectParameters();

        addListeners();
    }

    @Override
    public void parameterChanged(GeneralPlugInParameters parameters)
    {
        fireParametersChanged();
    }

}
