package ohd.hseb.charter.parameters.panels;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.WindowConstants;
import javax.swing.border.EtchedBorder;

import ohd.hseb.charter.parameters.BorderParameters;
import ohd.hseb.hefs.utils.gui.components.BufferJPanel;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInPanelListener;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;

import org.jfree.ui.RectangleInsets;

/**
 * Used to edit a BorderParameters object. It uses the RectangleInsetsEditingPanel to edit the line widths and standard
 * color components for the color.
 * 
 * @author hank.herr
 */
public class BorderParametersPanel extends ChartParametersPanel implements ActionListener,
RectangleInsetsEditingPanelListener
{
    private static final long serialVersionUID = 1L;

    //line widths
    private final RectangleInsetsEditingPanel _lineWidthsPanel;

    //color
    private final JPanel _colorButtonPanel = new JPanel(new GridBagLayout());
    private final JRadioButton _defaultColorRadioButton = new JRadioButton("Default");
    private final JRadioButton _specifiedColorRadioButton = new JRadioButton("User Specified");
    private final JButton _colorButton = new JButton();

    public BorderParametersPanel(final BorderParameters managedParameters, final BorderParameters defaultParameters)
    {
        this.setManagedParameters(managedParameters);
        this.setDefaultParameters(defaultParameters);

        this._lineWidthsPanel = new RectangleInsetsEditingPanel(this.getDefaults().getLineWidths());

        createDisplay();
        addListeners();
        makePanelReflectParameters();
    }

    private void createDisplay()
    {

        final ButtonGroup colorGroup = new ButtonGroup();
        colorGroup.add(this._defaultColorRadioButton);
        colorGroup.add(this._specifiedColorRadioButton);
        _defaultColorRadioButton.setSelected(true);

        this.createColorPanel();

        this.setLayout(new GridBagLayout());
        _lineWidthsPanel.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(),
                                                                    "Select Border Line Widths",
                                                                    null));
        this.add(this._lineWidthsPanel, SwingTools.returnGridBagConstraints(0,
                                                                            0,
                                                                            1,
                                                                            1,
                                                                            1,
                                                                            0,
                                                                            GridBagConstraints.WEST,
                                                                            GridBagConstraints.BOTH,
                                                                            new Insets(2, 2, 2, 2),
                                                                            0,
                                                                            0));
        this.add(this._colorButtonPanel, SwingTools.returnGridBagConstraints(0,
                                                                             1,
                                                                             1,
                                                                             1,
                                                                             1,
                                                                             0,
                                                                             GridBagConstraints.WEST,
                                                                             GridBagConstraints.BOTH,
                                                                             new Insets(0, 0, 0, 0),
                                                                             0,
                                                                             0));
        this.add(new BufferJPanel(), SwingTools.returnGridBagConstraints(0,
                                                                         2,
                                                                         1,
                                                                         1,
                                                                         1,
                                                                         1,
                                                                         GridBagConstraints.WEST,
                                                                         GridBagConstraints.BOTH,
                                                                         new Insets(0, 0, 0, 0),
                                                                         0,
                                                                         0));
    }

    private void createColorPanel()
    {
        _colorButtonPanel.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(), "Select Border Color", null));
        _colorButtonPanel.add(_defaultColorRadioButton,
                              SwingTools.returnGridBagConstraints(0,
                                                                  0,
                                                                  1,
                                                                  1,
                                                                  0,
                                                                  0,
                                                                  GridBagConstraints.LINE_START,
                                                                  GridBagConstraints.NONE,
                                                                  new Insets(0, 0, 0, 0),
                                                                  0,
                                                                  0));

        final JPanel selectedColorButtonPanel = new JPanel();
        selectedColorButtonPanel.add(_specifiedColorRadioButton);
        selectedColorButtonPanel.add(_colorButton);
        _colorButton.setToolTipText("Select Color");
        _colorButtonPanel.add(selectedColorButtonPanel,
                              SwingTools.returnGridBagConstraints(1,
                                                                  0,
                                                                  1,
                                                                  1,
                                                                  1,
                                                                  0,
                                                                  GridBagConstraints.LINE_START,
                                                                  GridBagConstraints.NONE,
                                                                  new Insets(0, 0, 0, 0),
                                                                  0,
                                                                  0));
    }

    private void addListeners()
    {
        this._defaultColorRadioButton.addActionListener(this);
        this._specifiedColorRadioButton.addActionListener(this);

        this._colorButton.addActionListener(this);

        this._lineWidthsPanel.addListener(this);
    }

    private void removeListeners()
    {
        this._defaultColorRadioButton.removeActionListener(this);
        this._specifiedColorRadioButton.removeActionListener(this);

        this._colorButton.removeActionListener(this);

        this._lineWidthsPanel.removeListener(this);

    }

    public BorderParameters getParameters()
    {
        return (BorderParameters)getManagedParameters();
    }

    public BorderParameters getDefaults()
    {
        return (BorderParameters)this.getDefaultParameters();
    }

    protected boolean makeParametersReflectPanel()
    {
        boolean changeMade = makeLineWidthParametersReflectPanel();
        changeMade = makeColorParameterReflectPanel() || changeMade;
        return changeMade;
    }

    private boolean makeLineWidthParametersReflectPanel()
    {
        final RectangleInsets newValue = this._lineWidthsPanel.buildInsets();
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(newValue,
                                                                                 getParameters().getLineWidths()))
        {
            getParameters().setLineWidths(newValue);
            return true;
        }
        return false;
    }

    private void makeLineWidthsPanelReflectParameters()
    {
        this.removeListeners();
        this._lineWidthsPanel.makePanelReflectInsets(getParameters().getLineWidths());
        this.addListeners();
    }

    private boolean makeColorParameterReflectPanel()
    {
        Color newValue = null;
        if(this._specifiedColorRadioButton.isSelected())
        {
            newValue = ((ColorIcon)this._colorButton.getIcon()).getColor();
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(newValue, getParameters().getColor()))
        {
            getParameters().setColor(newValue);
            return true;
        }
        return false;
    }

    private void makeColorPanelReflectParameters()
    {
        this.removeListeners();
        Color buttonBG = null;

        if(this.getParameters().getColor() == null)
        {
            this._defaultColorRadioButton.setSelected(true);
            buttonBG = this.getDefaults().getColor();
            this._colorButton.setEnabled(false);
        }
        else
        {
            this._specifiedColorRadioButton.setSelected(true);
            buttonBG = this.getParameters().getColor();
            this._colorButton.setEnabled(true);
        }

        this._colorButton.setIcon(new ColorIcon(buttonBG));
        this.addListeners();
    }

    @Override
    public void makePanelReflectParameters()
    {
        this.makeLineWidthsPanelReflectParameters();
        this.makeColorPanelReflectParameters();
    }

    @Override
    public void disableSubPanel()
    {
        _lineWidthsPanel.disableSubPanel();

        _defaultColorRadioButton.setEnabled(false);
        _specifiedColorRadioButton.setEnabled(false);
        _colorButton.setEnabled(false);
    }

    @Override
    public void enableSubPanel()
    {
        _lineWidthsPanel.setEnabledStateOfWidthFields(true);
        this._colorButton.setEnabled(true);
    }

    @Override
    public void actionPerformed(final ActionEvent e)
    {
        if((e.getSource() == this._defaultColorRadioButton) || (e.getSource() == this._specifiedColorRadioButton))
        {
            if(makeColorParameterReflectPanel())
            {
                fireParametersChanged();
            }
            this.makeColorPanelReflectParameters();
        }
        else if(e.getSource() == this._colorButton)
        {
            Color color = getParameters().getColor();
            color = ColorTools.chooseColor(this, color);
            if(color != null)
            {
                this._colorButton.setIcon(new ColorIcon(color));
            }
            if(makeColorParameterReflectPanel())
            {
                fireParametersChanged();
            }
            makeColorPanelReflectParameters();
        }
    }

    @Override
    public void reactToPossibleChange()
    {
        if(makeLineWidthParametersReflectPanel())
        {
            fireParametersChanged();
        }
        this.makeLineWidthsPanelReflectParameters();
    }

    public static void main(final String[] args)
    {
        final BorderParameters parms = new BorderParameters("border");
        final BorderParameters defaultParms = new BorderParameters("border");
        defaultParms.setupDefaultParameters();
        final BorderParametersPanel panel = new BorderParametersPanel(parms, defaultParms);
        final JFrame display = new JFrame();
        display.setContentPane(panel);
        display.setSize(500, 500);
        display.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        display.setVisible(true);

        panel.addGraphicsGeneratorPlugInPanelListener(new GeneralPlugInPanelListener()
        {

            @Override
            public void parameterChanged(final GeneralPlugInParameters parameters)
            {
                System.out.println("####>> PARAMDS -- " + parms);
            }

        });
    }
}
