package ohd.hseb.charter.parameters.panels;

import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.border.EtchedBorder;

import ohd.hseb.charter.parameters.NumericAxisParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.arguments.DefaultArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.components.BufferJPanel;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.ValueAxis;

public class NumericAxisParametersPanel extends ChartParametersPanel implements FocusListener, ActionListener
{
    private static final long serialVersionUID = -5464147185016314999L;

    private static final Logger LOG = LogManager.getLogger(NumericAxisParametersPanel.class);

    public static final String DEFAULT_AUTO_TICK_SPACING_TEXT = "-auto-";
    public static final int DEFAULT_TEXTFIELD_WIDTH = 10;
    public static final int DEFAULT_DECIMAL_NUMBER = 2;
    public static final int DEFAULT_INDEX_NUMBER = 0;
    public static final int AUTO_INDEX_NUMBER = 1;

    private ArgumentsProcessor _argumentsProcessor = null;
    protected ValueAxis _currentAxis = null;

    private JPanel _axisBoundEditPanel = null;
    private JComboBox _autoRangeChoices;
    private final JRadioButton _defaultAxisBoundsRadioButton = new JRadioButton("Default", true);
    private final JRadioButton _selectedAxisBoundsRadioButton = new JRadioButton("User Specified");
    private JComboBox _lowerBoundChoices;
    private JComboBox _upperBoundChoices;
    private JComboBox _rangeIncludesZeroChoices;

    private JPanel _tickSpacingEditPanel = null;
    private final JRadioButton _defaultTickSpacingRadioButton = new JRadioButton("Default", true);
    private final JRadioButton _selectedTickSpacingRadioButton = new JRadioButton("User Specified");
    private final JTextField _tickSpacingTextField = new JTextField(DEFAULT_AUTO_TICK_SPACING_TEXT,
                                                                    DEFAULT_TEXTFIELD_WIDTH);
    private final JButton _autoTickSpacingButton = HSwingFactory.createJButton("Auto", null, this);
    private final JLabel _tickSpacingLabel = new JLabel("Tick Spacing:");

    private JPanel _tickFormatEditPanel = null;
    private final JRadioButton _defaultTickFormatRadioButton = new JRadioButton("Default", true);
    private final JRadioButton _selectedTickFormatRadioButton = new JRadioButton("User Specified");
    private final JTextField _tickFormatTextField = new JTextField(DEFAULT_TEXTFIELD_WIDTH);

    public NumericAxisParametersPanel(final NumericAxisParameters managedParameters,
                                      final NumericAxisParameters defaultParameters,
                                      final DefaultArgumentsProcessor argumentsProcessor)
    {
        setArgumentsProcessor(argumentsProcessor);
        setManagedParameters(managedParameters);
        setDefaultParameters(defaultParameters);
        createDisplay();
        makePanelReflectParameters();
    }

    private void createDisplay()
    {
        _lowerBoundChoices = HSwingFactory.createJComboBox(this.determineLowerRangeChoices(), null);
        _lowerBoundChoices.setEditable(true);
        _lowerBoundChoices.setSize(10, 1);
        _upperBoundChoices = HSwingFactory.createJComboBox(this.determineUpperRangeChoices(), null);
        _upperBoundChoices.setEditable(true);
        _upperBoundChoices.setSize(10, 1);
        _autoRangeChoices = HSwingFactory.createJComboBox(this.determineAutoRangeChoices(), null);
        _rangeIncludesZeroChoices = HSwingFactory.createJComboBox(this.determineRangeIncludeZeroChoices(), null);
        setupAxisBoundsEditingPanel();
        setupTickSpacingEditPanel();
        setupTickFormatEditingPanel();

        final ButtonGroup group = new ButtonGroup();
        group.add(_defaultTickSpacingRadioButton);
        group.add(_selectedTickSpacingRadioButton);

        GridBagConstraints constraints;
        this.setLayout(new GridBagLayout());
        int gridy = 0;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          gridy,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        this.add(this._axisBoundEditPanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          gridy,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        this.add(_tickSpacingEditPanel, constraints);
        gridy++;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          gridy,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        this.add(_tickFormatEditPanel, constraints);
    }

    private void setupAxisBoundsEditingPanel()
    {

        //Lower Bound
        final JPanel lowerBoundPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        lowerBoundPanel.add(new JLabel("Lower:"));
        lowerBoundPanel.add(_lowerBoundChoices);

        //Upper Bound
        final JPanel upperBoundPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        upperBoundPanel.add(new JLabel("Upper:"));
        upperBoundPanel.add(_upperBoundChoices);

        //Auto range
        final JPanel autoRangePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        autoRangePanel.add(new JLabel("Auto Range Calculation:"));
        autoRangePanel.add(_autoRangeChoices);

        //Range includes zero combo box.
        final JPanel rangeIncludesZeroPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        rangeIncludesZeroPanel.add(new JLabel("Select If Auto-range Should Include Zero:"));
        rangeIncludesZeroPanel.add(_rangeIncludesZeroChoices);

        _axisBoundEditPanel = new JPanel(new GridBagLayout());
        GridBagConstraints constraints = null;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          2,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _axisBoundEditPanel.add(autoRangePanel, constraints);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _axisBoundEditPanel.add(lowerBoundPanel, constraints);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _axisBoundEditPanel.add(upperBoundPanel, constraints);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          2,
                                                          2,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _axisBoundEditPanel.add(rangeIncludesZeroPanel, constraints);

        _axisBoundEditPanel.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(),
                                                                       "Select Numerical Axis Limits",
                                                                       null));
    }

    /**
     * Add number format panel
     */
    private void setupTickFormatEditingPanel()
    {
        final ButtonGroup group = new ButtonGroup();
        group.add(_defaultTickFormatRadioButton);
        group.add(_selectedTickFormatRadioButton);

        _tickFormatEditPanel = new JPanel(new GridBagLayout());
        _tickFormatEditPanel.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(),
                                                                        "Select Tick Number Format",
                                                                        null));

        GridBagConstraints constraints = null;

        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _tickFormatEditPanel.add(_defaultTickFormatRadioButton, constraints);

        constraints = SwingTools.returnGridBagConstraints(1,
                                                          0,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _tickFormatEditPanel.add(_selectedTickFormatRadioButton, constraints);

        constraints = SwingTools.returnGridBagConstraints(2,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _tickFormatEditPanel.add(new BufferJPanel(), constraints);

        constraints = SwingTools.returnGridBagConstraints(0,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _tickFormatEditPanel.add(new JLabel("Tick Number Format: "), constraints);

        constraints = SwingTools.returnGridBagConstraints(1,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _tickFormatEditPanel.add(this._tickFormatTextField, constraints);

        constraints = SwingTools.returnGridBagConstraints(2,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);

        _tickFormatEditPanel.add(new BufferJPanel(), constraints);
    }

    /**
     * Add tick spacing panel.
     */
    private void setupTickSpacingEditPanel()
    {
        _tickSpacingEditPanel = new JPanel(new GridBagLayout());
        _tickSpacingEditPanel.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(),
                                                                         "Select Tick Spacing",
                                                                         null));

        final JPanel subPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        subPanel.add(HSwingFactory.createLabeledJTextField(_tickSpacingLabel, _tickSpacingTextField));
        subPanel.add(_autoTickSpacingButton);

        //Button panel
        final JPanel buttonPanel = new JPanel(new GridBagLayout());
        GridBagConstraints constraints = null;
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        buttonPanel.add(_defaultTickSpacingRadioButton, constraints);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          0,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        buttonPanel.add(_selectedTickSpacingRadioButton, constraints);
        constraints = SwingTools.returnGridBagConstraints(2,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        buttonPanel.add(new BufferJPanel(), constraints);

        //Editing panel
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _tickSpacingEditPanel.add(buttonPanel, constraints);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.HORIZONTAL,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _tickSpacingEditPanel.add(subPanel, constraints);

        _tickSpacingTextField.setEnabled(!this._defaultTickSpacingRadioButton.isSelected());
    }

    private List<String> determineRangeIncludeZeroChoices()
    {
        final List<String> choices = new ArrayList<String>();
        if(getDefaults().getRangeIncludesZero())
        {
            choices.add("Default (Yes)");
        }
        else
        {
            choices.add("Default (No)");
        }
        choices.add("Yes");
        choices.add("No");
        return choices;
    }

    private List<String> determineLowerRangeChoices()
    {
        final List<String> choices = new ArrayList<String>();
        if(getDefaults().getLowerBound() != null)
        {
            choices.add(getDefaults().getLowerBound().toString());

        }
        else
        {
            choices.add("0.0");
        }
        choices.add(NumericAxisParameters.AUTO_BOUND_SPACING_TEXT);
        return choices;
    }

    private List<String> determineUpperRangeChoices()
    {
        final List<String> choices = new ArrayList<String>();
        if(getDefaults().getUpperBound() != null)
        {
            choices.add(getDefaults().getUpperBound().toString());
        }
        else
        {
            choices.add("1.0");
        }
        choices.add(NumericAxisParameters.AUTO_BOUND_SPACING_TEXT);
        return choices;
    }

    private void setUpperBoundChoices(final Double boundValue)
    {
        _upperBoundChoices.removeAllItems();
        if(boundValue.isNaN())
        {
            if(getDefaults().getUpperBound() != null)
            {
                _upperBoundChoices.addItem(getDefaults().getUpperBound().toString());
            }
            else
            {
                _upperBoundChoices.addItem("1.0");
            }
        }
        else
        {
            _upperBoundChoices.addItem(boundValue.toString());
        }
        _upperBoundChoices.addItem(NumericAxisParameters.AUTO_BOUND_SPACING_TEXT);
        _upperBoundChoices.setSelectedIndex(DEFAULT_INDEX_NUMBER);
    }

    private void setLowerBoundChoices(final Double boundValue)
    {
        _lowerBoundChoices.removeAllItems();
        if(boundValue.isNaN())
        {
            if(getDefaults().getLowerBound() != null)
            {
                _lowerBoundChoices.addItem(getDefaults().getLowerBound().toString());

            }
            else
            {
                _lowerBoundChoices.addItem("0.0");
            }
        }
        else
        {
            _lowerBoundChoices.addItem(boundValue.toString());
        }
        _lowerBoundChoices.addItem(NumericAxisParameters.AUTO_BOUND_SPACING_TEXT);
        _lowerBoundChoices.setSelectedIndex(DEFAULT_INDEX_NUMBER);
    }

    private List<String> determineAutoRangeChoices()
    {
        final List<String> choices = new ArrayList<String>();
        if(getDefaults().getAutoRange())
        {
            choices.add("Default (Auto)");
        }
        else
        {
            choices.add("Default (Manual)");
        }
        choices.add("Auto");
        choices.add("Manual");
        return choices;
    }

    public void displayTickFormatPanelOnly()
    {
        this._axisBoundEditPanel.setVisible(false);
        this._tickSpacingEditPanel.setVisible(false);
        this._tickFormatEditPanel.setVisible(true);
    }

    public void displayAllPanels()
    {
        this._axisBoundEditPanel.setVisible(true);
        this._tickSpacingEditPanel.setVisible(true);
        this._tickFormatEditPanel.setVisible(true);
    }

    private void disableAxisBoundsEditingPanel()
    {
        _lowerBoundChoices.setEnabled(false);
        _upperBoundChoices.setEnabled(false);
    }

    private void enableAxisBoundEditPanel()
    {
        _lowerBoundChoices.setEnabled(true);
        _upperBoundChoices.setEnabled(true);
    }

    private void disableTickSpacingEditPanel()
    {
        _defaultTickSpacingRadioButton.setEnabled(false);
        _selectedTickSpacingRadioButton.setEnabled(false);
        disableTickSpacingEditing();
    }

    private void enableTickSpacingEditPanel()
    {
        _defaultTickSpacingRadioButton.setEnabled(true);
        _selectedTickSpacingRadioButton.setEnabled(true);
        enableTickSpacingEditing();
    }

    private void disableTickSpacingEditing()
    {
        _tickSpacingLabel.setEnabled(false);
        _tickSpacingTextField.setEnabled(false);
        _autoTickSpacingButton.setEnabled(false);
    }

    private void enableTickSpacingEditing()
    {
        _tickSpacingLabel.setEnabled(true);
        _tickSpacingTextField.setEnabled(true);
        _autoTickSpacingButton.setEnabled(true);
    }

    private void determineAxisBoundsFields()
    {
        Double lb = null;
        Double ub = null;

        //If auto is chosen, have it display the number axis limits in the field.
        //If the default is chosen, then get the bounds to display from default.  This may or may not
        //display the calculated number axis limits, depending on default settings.
        if(getParameters().getAutoRange() == null)
        {
            if(!getDefaults().getAutoRange())
            {
                lb = getDefaults().getLowerBound();
                ub = getDefaults().getUpperBound();
            }
        }
        else if(!getParameters().getAutoRange())
        {
            return;
        }
        if((getCurrentAxis() != null) && (getCurrentAxis() instanceof NumberAxis))
        {
            final NumberAxis numberAxis = (NumberAxis)getCurrentAxis();
            if(lb == null)
            {
                if(numberAxis.getLowerBound() == Double.NaN)
                {
                    _lowerBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
                }
                else
                {
                    this.setLowerBoundChoices(numberAxis.getLowerBound());
                }

            }
            else
            {
                if(getDefaults().getLowerBound().isNaN())
                {
                    _lowerBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
                }
                else
                {
                    this.setLowerBoundChoices(lb);
                }
            }
            if(ub == null)
            {
                if(numberAxis.getUpperBound() == Double.NaN)
                {
                    _upperBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
                }
                else
                {
                    this.setUpperBoundChoices(numberAxis.getUpperBound());
                }
            }
            else
            {
                if(getDefaults().getUpperBound().isNaN())
                {
                    _upperBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
                }
                else
                {
                    this.setUpperBoundChoices(ub);
                }
            }
        }
    }

    protected boolean makeParametersReflectPanel()
    {
        boolean changeMade = makeAxisBoundParametersReflectPanel();
        changeMade = makeTickSpacingParametersReflectPanel() || changeMade;
        changeMade = makeTickFormatParametersReflectPanel() || changeMade;
        changeMade = makeRangeIncludesZeroParameterReflectPanel() || changeMade;
        return changeMade;
    }

    private void makeAxisBoundPanelReflectParameters()
    {
        //Means that both are null.
        if(getParameters().getAutoRange() == null)
        {
            _autoRangeChoices.setSelectedIndex(0);
            determineAxisBoundsFields();
            disableAxisBoundsEditingPanel();
        }
        else if(getParameters().getAutoRange())
        {
            _autoRangeChoices.setSelectedIndex(1);
            determineAxisBoundsFields();
            disableAxisBoundsEditingPanel();
        }
        else
        {
            _autoRangeChoices.setSelectedIndex(2);
            // FB 1969
            if(getParameters().getLowerBound().isNaN())
            {
                _lowerBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
            }
            else
            {
                this.setLowerBoundChoices(getParameters().getLowerBound());
            }
            if(getParameters().getUpperBound().isNaN())
            {
                _upperBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
            }
            else
            {
                this.setUpperBoundChoices(getParameters().getUpperBound());
            }
            enableAxisBoundEditPanel();
        }
    }

    private boolean makeAxisBoundParametersReflectPanel()
    {
        boolean changeMade = false;

        //Default is selected; make sure the bounds are both null
        if(_autoRangeChoices.getSelectedIndex() == 0)
        {
            disableAxisBoundsEditingPanel();
            if(getParameters().getAutoRange() != null)
            {
                changeMade = true;
                getParameters().setAutoRange(null);
                getParameters().setLowerBound(null);
                getParameters().setUpperBound(null);
                determineAxisBoundsFields();
            }
        }

        //Auto is selected; don't change the bounds
        else if(_autoRangeChoices.getSelectedIndex() == 1)
        {
            disableAxisBoundsEditingPanel();
            if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(getParameters().getAutoRange(),
                                                                                     Boolean.TRUE))
            {
                changeMade = true;
                getParameters().setAutoRange(true);
                getParameters().setLowerBound(null);
                getParameters().setUpperBound(null);
                determineAxisBoundsFields();
            }
        }

        //Manual is selected; make sure bounds equal the fields
        else
        {
            enableAxisBoundEditPanel();
            if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(getParameters().getAutoRange(),
                                                                                     Boolean.FALSE))
            {
                changeMade = true;
                getParameters().setAutoRange(false);
            }
            changeMade = makeLowerBoundReflectField() || changeMade;
            changeMade = makeUpperBoundReflectField() || changeMade;
        }

        return changeMade;
    }

    private boolean makeUpperBoundReflectField()
    {
        if(_upperBoundChoices.getSelectedItem().toString().trim().length() > 0)
        {
            try
            { //FB 1969
                double upper;
                if(_upperBoundChoices.getSelectedItem()
                                     .toString()
                                     .equals(NumericAxisParameters.AUTO_BOUND_SPACING_TEXT))
                {
                    upper = Double.NaN;
                }
                else
                {
                    upper = Double.parseDouble(_upperBoundChoices.getSelectedItem().toString());

                }
                //check if the input upper value is not greater than the existed upper value
                if(getParameters().getLowerBound() != null && upper != Double.NaN
                    && upper <= getParameters().getLowerBound())
                {
                    JOptionPane.showMessageDialog(this,
                                                  "Axis upper limit value is not greater than the lower value.",
                                                  "input error",
                                                  JOptionPane.ERROR_MESSAGE);
                    if(getParameters().getUpperBound().isNaN())
                        _upperBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
                    else
                    {
                        this.setUpperBoundChoices(getParameters().getUpperBound());
                    }
                    return false;
                }
                if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(getParameters().getUpperBound(),
                                                                                         upper))
                {
                    getParameters().setUpperBound(upper);
                    return true;
                }
                else
                {
                    return false;
                }
            }
            catch(final NumberFormatException e)
            {
                LOG.error("Axis upper limit number format exception caught:" + e.getMessage());
                this.setUpperBoundChoices(getParameters().getUpperBound());
                return false;
            }
        }
        else
        {
            this.setUpperBoundChoices(getParameters().getUpperBound());
            return false;
        }
    }

    private boolean makeLowerBoundReflectField()
    {
        //Lower bound...
        if(_lowerBoundChoices.getSelectedItem().toString().trim().length() > 0)
        {
            try
            { //For FB 1969
                double lower;
                if(_lowerBoundChoices.getSelectedItem()
                                     .toString()
                                     .equals(NumericAxisParameters.AUTO_BOUND_SPACING_TEXT))
                {
                    lower = Double.NaN;
                }
                else
                {
                    lower = Double.parseDouble(_lowerBoundChoices.getSelectedItem().toString());
                }
                //lower = Double.parseDouble(_lowerBoundTextField.getText().trim());
                //check if the input lower value is not less than the existed upper value
                //IF lower is not NAN then do this check
                if(getParameters().getUpperBound() != null && lower != Double.NaN
                    && lower >= getParameters().getUpperBound())
                {
                    JOptionPane.showMessageDialog(this,
                                                  "Axis lower limit value is not less than the upper value.",
                                                  "input error",
                                                  JOptionPane.ERROR_MESSAGE);
                    if(getParameters().getLowerBound().isNaN())
                    {
                        _lowerBoundChoices.setSelectedIndex(AUTO_INDEX_NUMBER);
                    }
                    else
                    {
                        this.setLowerBoundChoices(getParameters().getLowerBound());
                    }
                    return false;
                }
                if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(getParameters().getLowerBound(),
                                                                                         lower))
                {
                    getParameters().setLowerBound(lower);
                    return true;
                }
                else
                {
                    return false;
                }
            }
            catch(final NumberFormatException e)
            {
                LOG.error("Axis lower limit number format exception caught:" + e.getMessage());

                System.out.println("ERROR == " + getParameters().getLowerBound().toString());
                this.setLowerBoundChoices(getParameters().getLowerBound());
                return false;
            }
        }
        else
        {
            this.setLowerBoundChoices(getParameters().getLowerBound());
            return false;
        }
    }

    private void makeTickSpacingPanelReflectParameters()
    {
        if(getParameters().getTickSpacing() == null)
        {
            _defaultTickSpacingRadioButton.setSelected(true);
            updateTickSpacingComponentToReflectParameters(getDefaults());
            disableTickSpacingEditing();
        }
        else
        {
            _selectedTickSpacingRadioButton.setSelected(true);
            updateTickSpacingComponentToReflectParameters(getParameters());
            enableTickSpacingEditing();
        }
    }

    private boolean makeTickSpacingParametersReflectPanel()
    {
        if(_defaultTickSpacingRadioButton.isSelected())
        {
            disableTickSpacingEditing();
            updateTickSpacingComponentToReflectParameters(getDefaults());
            if(getParameters().getTickSpacing() != null)
            {
                getParameters().setTickSpacing(null);
                return true;
            }
            return false;
        }
        else
        {
            this.enableTickSpacingEditing();
            Double newTickSpacing = null;
            if(_tickSpacingTextField.getText().trim().length() > 0)
            {
                if(_tickSpacingTextField.getText().trim().toLowerCase().equals(DEFAULT_AUTO_TICK_SPACING_TEXT))
                {
                    newTickSpacing = NumericAxisParameters.AUTO_TICK_SPACING_VALUE;
                }
                else
                {
                    try
                    {
                        newTickSpacing = Double.parseDouble(_tickSpacingTextField.getText().trim());

                        //check if the input tick spacing value is less than 9
                        if(newTickSpacing < 0.0)
                        {
                            JOptionPane.showMessageDialog(this,
                                                          "Axis tick spacing value is not greater than 0.",
                                                          "input error",
                                                          JOptionPane.ERROR_MESSAGE);
                            updateTickSpacingComponentToReflectParameters(getParameters());
                            return false;
                        }
                    }
                    catch(final NumberFormatException e)
                    {
                        LOG.error("Axis unit spacing, '" + _tickSpacingTextField.getText().trim()
                            + "', is not a number: " + e.getMessage());
                        updateTickSpacingComponentToReflectParameters(getParameters());
                        return false;
                    }
                }
            }
            else
            {
                //if user inputs empty tick unit, then use previous tick unit
                //set to default "-auto-" string if the value is null
                updateTickSpacingComponentToReflectParameters(getParameters());
                return false;
            }

            if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(newTickSpacing,
                                                                                     getParameters().getTickSpacing()))
            {
                getParameters().setTickSpacing(newTickSpacing);
                return true;
            }
            return false;
        }
    }

    private void updateTickSpacingComponentToReflectParameters(final NumericAxisParameters parms)
    {
        if(parms.isAutoTickSpacing())
        {
            _tickSpacingTextField.setText(DEFAULT_AUTO_TICK_SPACING_TEXT);
        }
        else
        {
            _tickSpacingTextField.setText(parms.getTickSpacing().toString());
        }
    }

    private void makeTickFormatPanelReflectParameters()
    {
        if(getParameters().getTickFormat() == null)
        {
            this._defaultTickFormatRadioButton.setSelected(true);
            this._tickFormatTextField.setEnabled(false);
            this._tickFormatTextField.setText("");
        }
        else
        {
            this._selectedTickFormatRadioButton.setSelected(true);
            this._tickFormatTextField.setEnabled(true);
            this._tickFormatTextField.setText(getParameters().getTickFormat());
        }
    }

    private boolean makeTickFormatParametersReflectPanel()
    {
        String newValue = null;
        if(this._defaultTickFormatRadioButton.isSelected())
        {
            _tickFormatTextField.setEnabled(false);
        }
        else
        {
            _tickFormatTextField.setEnabled(true);
            newValue = _tickFormatTextField.getText().trim();
        }
        if(!ohd.hseb.hefs.utils.tools.StringTools.checkForFullEqualityOfStrings(newValue,
                                                                                getParameters().getTickFormat(),
                                                                                true))
        {
            getParameters().setTickFormat(newValue);
            return true;
        }
        return false;
    }

    private void makeRangeIncludesZeroPanelReflectParameters()
    {
        if(getParameters().getRangeIncludesZero() == null)
        {
            this._rangeIncludesZeroChoices.setSelectedIndex(0);
        }
        else if(getParameters().getRangeIncludesZero())
        {
            this._rangeIncludesZeroChoices.setSelectedIndex(1);
        }
        else
        {
            this._rangeIncludesZeroChoices.setSelectedIndex(2);
        }
    }

    private boolean makeRangeIncludesZeroParameterReflectPanel()
    {
        Boolean newValue = null;
        if(_rangeIncludesZeroChoices.getSelectedIndex() == 1)
        {
            newValue = true;
        }
        else if(_rangeIncludesZeroChoices.getSelectedIndex() == 1)
        {
            newValue = false;
        }
        if(!ohd.hseb.hefs.utils.tools.GeneralTools.checkForFullEqualityOfObjects(newValue,
                                                                                 getParameters().getRangeIncludesZero()))
        {
            getParameters().setRangeIncludesZero(newValue);
            fireParametersChanged();
            return true;
        }
        return false;
    }

    private void addListeners()
    {
        _lowerBoundChoices.addFocusListener(this);
        _lowerBoundChoices.addActionListener(this);
        _upperBoundChoices.addFocusListener(this);
        _upperBoundChoices.addActionListener(this);
        _tickSpacingTextField.addFocusListener(this);
        _tickSpacingTextField.addActionListener(this);
        _defaultTickFormatRadioButton.addActionListener(this);
        _selectedTickFormatRadioButton.addActionListener(this);
        _tickFormatTextField.addActionListener(this);
        _tickFormatTextField.addFocusListener(this);
        _rangeIncludesZeroChoices.addActionListener(this);
        _autoRangeChoices.addActionListener(this);
        _defaultTickSpacingRadioButton.addActionListener(this);
        _selectedTickSpacingRadioButton.addActionListener(this);
        _defaultAxisBoundsRadioButton.addActionListener(this);
        _selectedAxisBoundsRadioButton.addActionListener(this);
    }

    private void removeListeners()
    {
        _lowerBoundChoices.removeFocusListener(this);
        _lowerBoundChoices.removeActionListener(this);
        _upperBoundChoices.removeFocusListener(this);
        _upperBoundChoices.removeActionListener(this);
        _tickSpacingTextField.removeFocusListener(this);
        _tickSpacingTextField.removeActionListener(this);
        _defaultTickFormatRadioButton.removeActionListener(this);
        _selectedTickFormatRadioButton.removeActionListener(this);
        _tickFormatTextField.removeActionListener(this);
        _tickFormatTextField.removeFocusListener(this);
        _rangeIncludesZeroChoices.removeActionListener(this);
        _autoRangeChoices.removeActionListener(this);
        _defaultTickSpacingRadioButton.removeActionListener(this);
        _selectedTickSpacingRadioButton.removeActionListener(this);
        _defaultAxisBoundsRadioButton.removeActionListener(this);
        _selectedAxisBoundsRadioButton.removeActionListener(this);
    }

    public ValueAxis getCurrentAxis()
    {
        return _currentAxis;
    }

    public void setCurrentAxis(final ValueAxis currAxis)
    {
        _currentAxis = currAxis;
        determineAxisBoundsFields();
    }

    public void setArgumentsProcessor(final DefaultArgumentsProcessor argumentsProcessor)
    {
        this._argumentsProcessor = argumentsProcessor;
    }

    public ArgumentsProcessor getArgumentsProcessor()
    {
        return this._argumentsProcessor;
    }

    public NumericAxisParameters getParameters()
    {
        return (NumericAxisParameters)this.getManagedParameters();
    }

    public NumericAxisParameters getDefaults()
    {
        return (NumericAxisParameters)this.getDefaultParameters();
    }

    @Override
    public void makePanelReflectParameters()
    {
        removeListeners();

        makeAxisBoundPanelReflectParameters();
        makeTickSpacingPanelReflectParameters();
        makeTickFormatPanelReflectParameters();
        makeRangeIncludesZeroPanelReflectParameters();

        addListeners();
    }

    @Override
    public void disableSubPanel()
    {
        disableAxisBoundsEditingPanel();
        disableTickSpacingEditPanel();
    }

    @Override
    public void enableSubPanel()
    {
        enableAxisBoundEditPanel();
        enableTickSpacingEditPanel();
    }

    public void actionPerformed(final ActionEvent evt)
    {
        if((evt.getSource() == this._defaultTickSpacingRadioButton)
            || (evt.getSource() == this._selectedTickSpacingRadioButton))
        {
            if(this.makeTickSpacingParametersReflectPanel())
            {
                fireParametersChanged();
            }
        }
        // else if((evt.getSource() == this._lowerBoundTextField) || (evt.getSource() == this._upperBoundTextField))
        else if((evt.getSource() == this._lowerBoundChoices) || (evt.getSource() == this._upperBoundChoices))
        {
            if(makeAxisBoundParametersReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if(evt.getSource() == this._tickSpacingTextField)
        {
            if(makeTickSpacingParametersReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if((evt.getSource() == this._defaultTickFormatRadioButton)
            || (evt.getSource() == this._selectedTickFormatRadioButton))
        {
            if(makeTickFormatParametersReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if(evt.getSource() == this._tickFormatTextField)
        {
            if(makeTickFormatParametersReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if(evt.getSource() == this._rangeIncludesZeroChoices)
        {
            if(makeRangeIncludesZeroParameterReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if(evt.getSource() == this._autoTickSpacingButton)
        {
            _tickSpacingTextField.setText(DEFAULT_AUTO_TICK_SPACING_TEXT);
            if(makeTickSpacingParametersReflectPanel())
            {
                fireParametersChanged();
            }
        }
        else if(evt.getSource() == this._autoRangeChoices)
        {
            if(makeAxisBoundParametersReflectPanel())
            {
                fireParametersChanged();
            }
        }
    }

    @Override
    public void focusGained(final FocusEvent arg0)
    {
    }

    @Override
    public void focusLost(final FocusEvent arg0)
    {
        if(makeParametersReflectPanel())
        {
            fireParametersChanged();
        }
    }
}
