package ohd.hseb.charter.parameters.panels;

import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;

import ohd.hseb.hefs.utils.gui.components.RevertingJFormattedTextField;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;

import org.jfree.ui.RectangleInsets;

/**
 * Editing panel designed for RectangleInsets. It requires a default set of insets for when the radio button is clicked.
 * Whenever a change is made or focus is lost by a text field, the method reactToPossibleChange is called for all
 * listeners. The method buildInsets can be called to get insets that reflect the current value, or null if default
 * radio button is clicked (defaults can be acquired via get method). To make the panel reflect a new set of insets,
 * call the makePanelReflectInsets. If null is passed in for the overrideInsets, the default radio button will be
 * selected.
 * 
 * @author hank.herr
 */
public class RectangleInsetsEditingPanel extends JPanel implements ActionListener, FocusListener
{
    private static final long serialVersionUID = 1L;

    private final JRadioButton _defaultLineWidthsRadioButton = new JRadioButton("Default");
    private final JRadioButton _specifiedLineWidthsRadioButton = new JRadioButton("User Specified");
    private final JFormattedTextField _topField = new RevertingJFormattedTextField(new PositiveDecimalFormat("0.0#"));
    private final JFormattedTextField _leftField = new RevertingJFormattedTextField(new PositiveDecimalFormat("0.0#"));
    private final JFormattedTextField _bottomField = new RevertingJFormattedTextField(new PositiveDecimalFormat("0.0#"));
    private final JFormattedTextField _rightField = new RevertingJFormattedTextField(new PositiveDecimalFormat("0.0#"));

    private final List<RectangleInsetsEditingPanelListener> _listeners = new ArrayList<RectangleInsetsEditingPanelListener>();

    private final RectangleInsets _defaultInsets;

    public RectangleInsetsEditingPanel(RectangleInsets defaultInsets)
    {
        this._defaultInsets = defaultInsets;
        createDisplay();
    }

    private void createDisplay()
    {
        ButtonGroup group = new ButtonGroup();
        group.add(this._defaultLineWidthsRadioButton);
        group.add(this._specifiedLineWidthsRadioButton);
        _defaultLineWidthsRadioButton.setSelected(true);

        //TODO At some point, I need to figure out how to get these fields to selectAll when entered.
        //It appears that using focus listener focusGained does not work for JFormattedTextFields.
        //The RevertingJFormattedTextField is supposed to add the needed focusGained, but it doesn't work.
        _topField.setColumns(5);
        _leftField.setColumns(5);
        _bottomField.setColumns(5);
        _rightField.setColumns(5);

        setLayout(new GridBagLayout());
        add(this._defaultLineWidthsRadioButton, SwingTools.returnGridBagConstraints(0,
                                                                                       0,
                                                                                       1,
                                                                                       1,
                                                                                       1,
                                                                                       0,
                                                                                       GridBagConstraints.WEST,
                                                                                       GridBagConstraints.NONE,
                                                                                       new Insets(0, 0, 0, 0),
                                                                                       0,
                                                                                       0));
        add(this._specifiedLineWidthsRadioButton, SwingTools.returnGridBagConstraints(1,
                                                                                         0,
                                                                                         1,
                                                                                         1,
                                                                                         1,
                                                                                         0,
                                                                                         GridBagConstraints.WEST,
                                                                                         GridBagConstraints.NONE,
                                                                                         new Insets(0, 0, 0, 0),
                                                                                         0,
                                                                                         0));

        JPanel textFieldPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        textFieldPanel.add(new JLabel("Top:"));
        textFieldPanel.add(_topField);
        textFieldPanel.add(new JLabel("Left:"));
        textFieldPanel.add(_leftField);
        textFieldPanel.add(new JLabel("Bottom:"));
        textFieldPanel.add(_bottomField);
        textFieldPanel.add(new JLabel("Right:"));
        textFieldPanel.add(_rightField);

        add(textFieldPanel, SwingTools.returnGridBagConstraints(0,
                                                                   1,
                                                                   2,
                                                                   1,
                                                                   1,
                                                                   0,
                                                                   GridBagConstraints.WEST,
                                                                   GridBagConstraints.NONE,
                                                                   new Insets(0, 0, 0, 0),
                                                                   0,
                                                                   0));
    }

    private void addListeners()
    {
        this._defaultLineWidthsRadioButton.addActionListener(this);
        this._specifiedLineWidthsRadioButton.addActionListener(this);

        this._topField.addActionListener(this);
        this._topField.addFocusListener(this);
        this._leftField.addActionListener(this);
        this._leftField.addFocusListener(this);
        this._bottomField.addActionListener(this);
        this._bottomField.addFocusListener(this);
        this._rightField.addActionListener(this);
        this._rightField.addFocusListener(this);
    }

    private void removeListeners()
    {
        this._defaultLineWidthsRadioButton.removeActionListener(this);
        this._specifiedLineWidthsRadioButton.removeActionListener(this);

        this._topField.removeActionListener(this);
        this._topField.removeFocusListener(this);
        this._leftField.removeActionListener(this);
        this._leftField.removeFocusListener(this);
        this._bottomField.removeActionListener(this);
        this._bottomField.removeFocusListener(this);
        this._rightField.removeActionListener(this);
        this._rightField.removeFocusListener(this);
    }

    public RectangleInsets buildInsets()
    {
        RectangleInsets newValue = null;

        //If user specified, change the new value to reflect the text fields.
        if(this._specifiedLineWidthsRadioButton.isSelected())
        {
            double top, left, bottom, right;
            top = returnOrResetFieldValue(_topField, "Top");
            left = returnOrResetFieldValue(_leftField, "Left");
            bottom = returnOrResetFieldValue(_bottomField, "Bottom");
            right = returnOrResetFieldValue(_rightField, "Right");
            newValue = new RectangleInsets(top, left, bottom, right);
        }
        return newValue;
    }

    public void makePanelReflectInsets(RectangleInsets overrideInsets)
    {
        this.removeListeners();
        if(overrideInsets == null)
        {
            _defaultLineWidthsRadioButton.setSelected(true);
            this.makeWidthFieldsValuesReflectParameters(_defaultInsets);
            this.setEnabledStateOfWidthFields(false);
        }
        else
        {
            _specifiedLineWidthsRadioButton.setSelected(true);
            this.makeWidthFieldsValuesReflectParameters(overrideInsets);
            this.setEnabledStateOfWidthFields(true);
        }
        this.addListeners();
    }

    private void makeWidthFieldsValuesReflectParameters(RectangleInsets widths)
    {
        this._topField.setValue(Double.valueOf(widths.getTop()));
        this._leftField.setValue(Double.valueOf(widths.getLeft()));
        this._bottomField.setValue(Double.valueOf(widths.getBottom()));
        this._rightField.setValue(Double.valueOf(widths.getRight()));
    }

    public void setEnabledStateOfWidthFields(boolean b)
    {
        this._topField.setEnabled(b);
        this._leftField.setEnabled(b);
        this._bottomField.setEnabled(b);
        this._rightField.setEnabled(b);
    }

    private double returnOrResetFieldValue(JFormattedTextField field, String fieldId)
    {
        Number oldValue = (Number)field.getValue();
        try
        {
            field.commitEdit();
        }
        catch(ParseException e1)
        {
            field.setValue(oldValue);
        }

        //The getValue cannot be used directly, because it may return a Long.  Instead, this is done:
        double value = ((Number)field.getValue()).doubleValue();
        return value;
    }

    public void addListener(RectangleInsetsEditingPanelListener listener)
    {
        this._listeners.add(listener);
    }

    public void removeListener(RectangleInsetsEditingPanelListener listener)
    {
        this._listeners.remove(listener);
    }

    public void fireReactToPossibleChange()
    {
        for(RectangleInsetsEditingPanelListener listener: _listeners)
        {
            listener.reactToPossibleChange();
        }
    }

    public void disableSubPanel()
    {
        setEnabledStateOfWidthFields(false);
        _defaultLineWidthsRadioButton.setEnabled(false);
        _specifiedLineWidthsRadioButton.setEnabled(false);
    }

    public RectangleInsets getDefaultInsets()
    {
        return this._defaultInsets;
    }

    @Override
    public void actionPerformed(ActionEvent e)
    {
        if((e.getSource() == this._defaultLineWidthsRadioButton)
            || (e.getSource() == this._specifiedLineWidthsRadioButton))
        {
            fireReactToPossibleChange();
        }
        else if((e.getSource() == _topField) || (e.getSource() == _leftField) || (e.getSource() == _bottomField)
            || (e.getSource() == _rightField))
        {
            fireReactToPossibleChange();
        }
    }

    @Override
    public void focusGained(FocusEvent e)
    {
    }

    @Override
    public void focusLost(FocusEvent e)
    {
        if((e.getSource() == _topField) || (e.getSource() == _leftField) || (e.getSource() == _bottomField)
            || (e.getSource() == _rightField))
        {
            fireReactToPossibleChange();
        }
    }

    private class PositiveDecimalFormat extends DecimalFormat
    {
        private static final long serialVersionUID = 1L;

        public PositiveDecimalFormat(String format)
        {
            super(format);

        }

        @Override
        public Number parse(String text, ParsePosition pos)
        {
            Number num = super.parse(text, pos);
            if(num != null)
            {
                //Reset the pos index to 0 if the value is negative, indicating a parse error.
                if(num.doubleValue() < 0)
                {
                    pos.setIndex(0);
                }
            }

            return num;
        }

    }
}
