package ohd.hseb.charter.parameters.panels;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.WindowConstants;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import ohd.hseb.charter.parameters.SubtitleListParameters;
import ohd.hseb.charter.parameters.SubtitleParameters;
import ohd.hseb.hefs.utils.gui.jtable.JTableAddDeleteMouseListener;
import ohd.hseb.hefs.utils.gui.jtable.JTableDeleteProcessor;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.renderers.JLabelTableCellRenderer;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInPanelListener;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;

/**
 * Basic GUI to manage a SubtitleListParameters under the assumption that the default parameters are never overridden or
 * modified.
 * 
 * @author hank.herr
 */
public class SubtitleListParametersPanel extends ChartParametersPanel implements ActionListener,
GeneralPlugInPanelListener, JTableDeleteProcessor, ListSelectionListener
{
    private static final long serialVersionUID = 1L;

    private JTable _subtitleTable;
    private SubtitleTableModel _subtitleTableModel;
    private final JScrollPane _centralPane = new JScrollPane(new JPanel());
    private final JButton _addButton = new JButton("New");

    public SubtitleListParametersPanel(SubtitleListParameters managedParameters,
                                       SubtitleListParameters defaultParameters)
    {
        this.setManagedParameters(managedParameters);
        this.setDefaultParameters(defaultParameters);

        createDisplay();
        addListeners();
        makePanelReflectParameters();
    }

    protected void createDisplay()
    {
        _centralPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        _centralPane.setPreferredSize(new Dimension(300, 140));

        //Button panel
        JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        buttonPanel.add(_addButton);

        //Table stuff
        _subtitleTableModel = new SubtitleTableModel(getParameters(),
                                                     (SubtitleListParameters)this.getDefaultParameters());
        _subtitleTable = new JTable(_subtitleTableModel);
        _subtitleTable.setDefaultRenderer(JLabel.class, new JLabelTableCellRenderer());
        _subtitleTable.addMouseListener(new JTableAddDeleteMouseListener(_subtitleTable, this));
        _subtitleTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        TableTools.setColumnFixedWidth(_subtitleTable.getColumnModel().getColumn(0),
                                           TableTools.getDeleteButtonFixedWidth());

        JScrollPane tablePane = new JScrollPane(_subtitleTable);
        tablePane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        tablePane.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(), "List of Subtitles", null));
        tablePane.setPreferredSize(new Dimension(300, 140));

        //The top panel for the split pane is the tablePane setup so that it fills up the panel (BorderLayout
        //handles this).
        JPanel topPanel = new JPanel(new BorderLayout());
        topPanel.add(tablePane, BorderLayout.CENTER);
        topPanel.add(buttonPanel, BorderLayout.SOUTH);

        //Setup the split pane
        JSplitPane splitPane = HSwingFactory.createJSPlitPane(JSplitPane.VERTICAL_SPLIT, topPanel, _centralPane, true);
        splitPane.setDividerSize(5);
        splitPane.setResizeWeight(0.0);
        splitPane.setDividerLocation(140);

        this.setLayout(new BorderLayout());
        this.add(splitPane, BorderLayout.CENTER);
    }

    protected void addListeners()
    {
        _addButton.addActionListener(this);
        _subtitleTable.getSelectionModel().addListSelectionListener(this);
    }

    protected void removeListeners()
    {
        _addButton.removeActionListener(this);
        _subtitleTable.getSelectionModel().removeListSelectionListener(this);
    }

    private void displaySelectedSubtitleParametersPanel()
    {
        int index = this._subtitleTable.getSelectedRow();
        if(index >= 0)
        {
            SubtitleParameters parms = this._subtitleTableModel.getSubtitleParametersForRow(index);
            SubtitleParameters defaults = new SubtitleParameters();
            defaults.setupDefaultParameters();

            JPanel displayedPanel = new JPanel(new BorderLayout());
            SubtitleParametersPanel subtitlePanel = new SubtitleParametersPanel(parms, defaults);
            subtitlePanel.addGraphicsGeneratorPlugInPanelListener(this);
            displayedPanel.add(subtitlePanel, BorderLayout.CENTER);

            if(this._subtitleTableModel.isDefault(index))
            {
                subtitlePanel.disableSubPanel();
                JPanel messagePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
                messagePanel.add(new JLabel("<html>This subtitle is defined in a referenced template.<br>It cannot be modified!</html>"));
                displayedPanel.add(messagePanel, BorderLayout.NORTH);
            }

            _centralPane.getViewport().setView(displayedPanel);
        }
        else
        {
            _centralPane.getViewport().setView(new JPanel());
        }
    }

    public SubtitleListParameters getParameters()
    {
        return (SubtitleListParameters)this.getManagedParameters();
    }

    @Override
    public void makePanelReflectParameters()
    {
        removeListeners();

        //This will attempt to maintain the current selection, unless the selected index is outside
        //the range of the table.
        int currentSelection = _subtitleTable.getSelectedRow();
        _subtitleTable.clearSelection();
        _subtitleTable.setVisible(false);
        _subtitleTable.setVisible(true);
        if(currentSelection >= getParameters().getSubtitleCount())
        {
            currentSelection = getParameters().getSubtitleCount() - 1;
        }
        _subtitleTable.getSelectionModel().setSelectionInterval(currentSelection, currentSelection);
        displaySelectedSubtitleParametersPanel();

        addListeners();
    }

    @Override
    public void disableSubPanel()
    {
        //Don't work with this for now.
    }

    @Override
    public void enableSubPanel()
    {
        //Don't work with this for now.
    }

    @Override
    public void actionPerformed(ActionEvent e)
    {
        if(e.getSource() == this._addButton)
        {
            SubtitleParameters parms = new SubtitleParameters();
            getParameters().addSubtitle(parms);
            _subtitleTable.clearSelection();
            makePanelReflectParameters();
            _subtitleTable.getSelectionModel().setSelectionInterval(getParameters().getSubtitleCount() - 1,
                                                                    getParameters().getSubtitleCount() - 1);
            fireParametersChanged();
        }
    }

    @Override
    public void parameterChanged(GeneralPlugInParameters parameters)
    {
        _subtitleTable.setVisible(false);
        _subtitleTable.setVisible(true);
        fireParametersChanged();
    }

    @Override
    public void processDeleteAction(Object source, int displayedRowIndex)
    {
        SubtitleParameters deleted = this._subtitleTableModel.getSubtitleParametersForRow(displayedRowIndex);
        getParameters().removeSubtitle(deleted);
        makePanelReflectParameters();
        fireParametersChanged();
    }

    @Override
    public void valueChanged(ListSelectionEvent e)
    {
        displaySelectedSubtitleParametersPanel();
    }

    public static void main(String[] args)
    {
        SubtitleListParameters defaultParms = new SubtitleListParameters();
        defaultParms.setupDefaultParameters();

        final SubtitleListParameters parms = new SubtitleListParameters();
        SubtitleListParametersPanel panel = new SubtitleListParametersPanel(parms, defaultParms);
        JFrame display = new JFrame();
        display.setContentPane(panel);
        display.setSize(500, 900);
        display.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
        display.setVisible(true);

        panel.addGraphicsGeneratorPlugInPanelListener(new GeneralPlugInPanelListener()
        {

            @Override
            public void parameterChanged(GeneralPlugInParameters parameters)
            {
                System.out.println("####>> PARAMDS -- " + parms);
            }

        });
    }
}
