package ohd.hseb.charter.parameters.panels;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.border.Border;
import javax.swing.border.EtchedBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableColumnModel;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.parameters.ThresholdListParameters;
import ohd.hseb.charter.parameters.ThresholdParameters;
import ohd.hseb.hefs.utils.arguments.ArgumentInsertingTextField;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.gui.jtable.JTableAddDeleteMouseListener;
import ohd.hseb.hefs.utils.gui.jtable.JTableDeleteProcessor;
import ohd.hseb.hefs.utils.gui.jtable.TableTools;
import ohd.hseb.hefs.utils.gui.jtable.renderers.JLabelTableCellRenderer;
import ohd.hseb.hefs.utils.gui.tools.DialogHelper;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInPanelListener;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.util.misc.HString;

public class ThresholdParametersListPanel extends ChartParametersPanel implements GeneralPlugInPanelListener,
ThresholdParametersListPanelTableModelListener, ThresholdParameterPanelListener, JTableDeleteProcessor
{
    private static final long serialVersionUID = 1L;

    private final ThresholdListParameters _defaultTLParameters;

    private JSplitPane _splitPanel = null;

    private final JButton _addNewThresholdButton = new JButton("Add New Threshold");

    private ThresholdParametersListPanelTableModel _defaultTableModel = null;
    private JTable _defaultThresholdListTable = null;
    private JScrollPane _defaultThresholdListScrollPane = null;

    private final JPanel _thresholdListPanel = new JPanel(new GridBagLayout());

    private ThresholdParametersListPanelTableModel _overRideTableModel = null;
    private JTable _overRideThresholdListTable = null;
    private JScrollPane _overRideThresholdListScrollPane = null;

    private JScrollPane _selectedThresholdScrollPane = null;

    private HashMap<Integer, int[]> _subPlotIndexToRangeAxisTypes = null;

    private final int _xAxisType;

    @SuppressWarnings("rawtypes")
    private final Map _defaultIdentifierToVisibileBooleanHashMap = new HashMap();

    private final DefaultThresholdListTableSelectionListener _defaultThresholdListTableSelectionListener = new DefaultThresholdListTableSelectionListener();
    private final OverRideThresholdListTableSelectionListener _overRideThresholdListTableSelectionListener = new OverRideThresholdListTableSelectionListener();

    /**
     * @param managedParameters - Override Threshold Parameters
     * @param defaultTLParameters - Default Threshold Parameters
     * @param subPlotIndexToRangeAxisTypes - SubPlotIndexToRangeAxisTypes HashMap
     * @param xAxisType - XAxis Type (int)
     */
    public ThresholdParametersListPanel(final ThresholdListParameters managedParameters,
                                        final ThresholdListParameters defaultTLParameters,
                                        final HashMap<Integer, int[]> subPlotIndexToRangeAxisTypes,
                                        final int xAxisType)
    {
        setManagedParameters(managedParameters);
        _defaultTLParameters = defaultTLParameters;
        _subPlotIndexToRangeAxisTypes = subPlotIndexToRangeAxisTypes;
        _xAxisType = xAxisType;
        buildDefaultIdentifierToBooleanHashMap();

        if(subPlotIndexToRangeAxisTypes != null)
        {
            createDisplay();
            addListeners();
        }
    }

    /**
     * Create the Identifier to default Boolean hashmap. If the ThresholdParameter is a default, the boolean value will
     * be true.
     */
    @SuppressWarnings("unchecked")
    private void buildDefaultIdentifierToBooleanHashMap()
    {
        for(int i = 0; i < _defaultTLParameters.getThresholdParametersList().size(); i++)
        {
            final ThresholdParameters extractedParam = _defaultTLParameters.getThresholdParametersList().get(i);
            _defaultIdentifierToVisibileBooleanHashMap.put(getEvaluatedIdentifier(extractedParam.getIdentifier()),
                                                           extractedParam.getVisible());
        }
    }

    private ThresholdParameters getOverrideParameterFromIdentifier(final String identifier, final boolean tableClicked)
    {

        final String evaluatedIdentifier = this.getEvaluatedIdentifier(identifier);
        ThresholdParameters param = null;

        final List<ThresholdParameters> overRideThresholdParametersList = ((ThresholdListParameters)getManagedParameters()).getThresholdParametersList();

        for(final ThresholdParameters extractedOverrideParam: overRideThresholdParametersList)
        {
            if(extractedOverrideParam.getEvaluatedIdentifier().equalsIgnoreCase(evaluatedIdentifier))
            {
                param = extractedOverrideParam;
                break;
            }
        }

        if((param == null) && (!tableClicked))
        {
            if(_selectedThresholdScrollPane.getViewport().getView() instanceof ThresholdParameterPanel)
            {
                final ThresholdParameterPanel panel = (ThresholdParameterPanel)_selectedThresholdScrollPane.getViewport()
                                                                                                           .getView();
                param = panel.getOverRideParameters();
            }
        }

        return param;
    }

    /**
     * @param identifier - Identifier string
     * @return - ThresholdParameter
     */
    private ThresholdParameters getDefaultParametersFromIdentifier(final String identifier)
    {
        ThresholdParameters param = null;

        final List<ThresholdParameters> defaultThresholdParametersList = _defaultTLParameters.getThresholdParametersList();

        for(final ThresholdParameters extractedDefaultParam: defaultThresholdParametersList)
        {
            if(extractedDefaultParam.getIdentifier().equalsIgnoreCase(getEvaluatedIdentifier(identifier)))
            {
                param = extractedDefaultParam;
                break;
            }
        }
        return param;
    }

    /**
     * Creates the Display
     */
    private void createDisplay()
    {
        setupThresholdListTable();
        setupSelectedThresholdScrollPane();
        setupSplitPanel();
        this.setLayout(new GridBagLayout());
        final GridBagConstraints constraints = SwingTools.returnGridBagConstraints(0,
                                                                                   0,
                                                                                   1,
                                                                                   1,
                                                                                   1,
                                                                                   1,
                                                                                   GridBagConstraints.WEST,
                                                                                   GridBagConstraints.BOTH,
                                                                                   new Insets(0, 0, 0, 0),
                                                                                   0,
                                                                                   0);
        this.add(_splitPanel, constraints);
    }

    /**
     * Setup the Split Panel. Top half is the _thresholdListPanel, bottom half is the _selectedThresholdScrollPane
     */
    private void setupSplitPanel()
    {
        _defaultThresholdListTable.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        _overRideThresholdListTable.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        _selectedThresholdScrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        _selectedThresholdScrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);

        _thresholdListPanel.setPreferredSize(new Dimension(300, 400));
        _splitPanel = new JSplitPane(JSplitPane.VERTICAL_SPLIT, _thresholdListPanel, _selectedThresholdScrollPane);
        _splitPanel.setOneTouchExpandable(true);
        _splitPanel.setDividerLocation(225);
    }

    private void setThresholdPanel(final JPanel panel)
    {
        _selectedThresholdScrollPane.setViewportView(panel);
        _selectedThresholdScrollPane.setVisible(false);
        _selectedThresholdScrollPane.setVisible(true);
    }

    /**
     * Sets up the Selected Threshold Scroll Pane (used in the buttom of the splitpane)
     */
    private void setupSelectedThresholdScrollPane()
    {
        _selectedThresholdScrollPane = new JScrollPane(new JPanel());
        _selectedThresholdScrollPane.setPreferredSize(new Dimension(100, 100)); //If too large, the panel will not display right.
    }

    private void selectDefaultParameterInDefaultParametersTable(final String evaluatedIdentifier)
    {
        for(int i = 0; i < _defaultTableModel.getRowCount(); i++)
        {
            final String identifier = (String)_defaultThresholdListTable.getValueAt(i, 1);
            if(identifier.equalsIgnoreCase(evaluatedIdentifier))
            {
                _defaultThresholdListTable.setRowSelectionInterval(i, i);
            }
        }
    }

    private void selectOverrideParameterInOverrideParametersTable(final String unevaluatedIdentifier)
    {
        for(int i = 0; i < _overRideTableModel.getRowCount(); i++)
        {
            final String identifier = (String)_overRideThresholdListTable.getValueAt(i, 4);
            if(identifier.equalsIgnoreCase(unevaluatedIdentifier))
            {
                _overRideThresholdListTable.setRowSelectionInterval(i, i);
            }
        }
    }

    private void setupColumnModelForDefaultTable()
    {
        final DefaultTableColumnModel model = new DefaultTableColumnModel();
        final TableColumnModel masterCopy = this._defaultThresholdListTable.getColumnModel();
        model.addColumn(masterCopy.getColumn(2));
        model.addColumn(masterCopy.getColumn(3));
        this._defaultThresholdListTable.setColumnModel(model);
    }

    /**
     * Set up the Threshold List Table
     */
    private void setupThresholdListTable()
    {
        final JButton addButton = new JButton("Add New Threshold");
        GridBagConstraints constraints = SwingTools.returnGridBagConstraints(0,
                                                                             0,
                                                                             1,
                                                                             2,
                                                                             1,
                                                                             4,
                                                                             GridBagConstraints.WEST,
                                                                             GridBagConstraints.BOTH,
                                                                             new Insets(0, 0, 0, 0),
                                                                             0,
                                                                             0);

        addButton.setBackground(Color.RED);
        _defaultTableModel = new ThresholdParametersListPanelTableModel(this,
                                                                        getParameters(),
                                                                        _defaultTLParameters,
                                                                        false,
                                                                        getParameters().getArguments());

        _defaultThresholdListTable = new JTable(_defaultTableModel);
        _defaultThresholdListTable.setDefaultRenderer(JLabel.class, new JLabelTableCellRenderer());
        //TableUtilities.setColumnFixedWidth(_defaultThresholdListTable.getColumnModel().getColumn(0),
        //                                   TableUtilities.getDeleteButtonFixedWidth());
        //TableUtilities.setColumnFixedWidth(_defaultThresholdListTable.getColumnModel().getColumn(1),
        //                                   TableUtilities.getStatusColumnFixedWidth());
        TableTools.setColumnFixedWidth(_defaultThresholdListTable.getColumnModel().getColumn(2), 50);
        setupColumnModelForDefaultTable();

        _defaultThresholdListScrollPane = new JScrollPane(_defaultThresholdListTable);

        constraints = SwingTools.returnGridBagConstraints(0,
                                                          2,
                                                          1,
                                                          2,
                                                          1,
                                                          4,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);

        _overRideTableModel = new ThresholdParametersListPanelTableModel(this,
                                                                         getParameters(),
                                                                         _defaultTLParameters,
                                                                         true,
                                                                         getParameters().getArguments());

        _overRideThresholdListTable = new JTable(_overRideTableModel);
        _overRideThresholdListTable.setDefaultRenderer(JLabel.class, new JLabelTableCellRenderer());
        TableTools.setColumnFixedWidth(_overRideThresholdListTable.getColumnModel().getColumn(0),
                                       TableTools.getDeleteButtonFixedWidth());
        TableTools.setColumnFixedWidth(_overRideThresholdListTable.getColumnModel().getColumn(1),
                                       TableTools.getStatusColumnFixedWidth());
        TableTools.setColumnFixedWidth(_overRideThresholdListTable.getColumnModel().getColumn(2), 50);
        _overRideThresholdListTable.addMouseListener(new JTableAddDeleteMouseListener(_overRideThresholdListTable, this));

        _overRideThresholdListScrollPane = new JScrollPane(_overRideThresholdListTable);

        final Border border = BorderFactory.createEmptyBorder(0, 0, 0, 0);
        _defaultThresholdListScrollPane.setViewportBorder(border);
        _defaultThresholdListScrollPane.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(),
                                                                                   "Default Thresholds",
                                                                                   null));

        _overRideThresholdListScrollPane.setViewportBorder(border);
        _overRideThresholdListScrollPane.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(),
                                                                                    "Override Thresholds",
                                                                                    null));

        final JSplitPane tableSplitPanel = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
                                                          _defaultThresholdListScrollPane,
                                                          _overRideThresholdListScrollPane);
        tableSplitPanel.setDividerSize(tableSplitPanel.getDividerSize() - 1);
        tableSplitPanel.setDividerLocation(100);

        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          4,
                                                          1,
                                                          1,
                                                          GridBagConstraints.CENTER,
                                                          GridBagConstraints.BOTH,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);

        _thresholdListPanel.add(tableSplitPanel, constraints);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          4,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          GridBagConstraints.CENTER,
                                                          GridBagConstraints.NONE,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        _thresholdListPanel.add(_addNewThresholdButton, constraints);
    }

    private String getIdentifierFromDialog(final String message, final String initialValue)
    {
        final JOptionPane newIdentifierPane = new JOptionPane();
        final ArgumentsProcessor argProc = getParameters().getArguments();
        final ArgumentInsertingTextField identifierTextField = new ArgumentInsertingTextField(argProc, "");
        identifierTextField.setColumns(15);
        identifierTextField.setText(initialValue);
        final Object[] messageObjectArray = {message, identifierTextField};
        String identifier = null;
        newIdentifierPane.setMessage(messageObjectArray);
        newIdentifierPane.setMessageType(JOptionPane.QUESTION_MESSAGE);
        newIdentifierPane.setOptionType(JOptionPane.OK_CANCEL_OPTION);
        final JDialog dialog = newIdentifierPane.createDialog(this, "Specify Identifier");
        dialog.setVisible(true);
        dialog.dispose();

        //Do nothing if the user closes with the x or hits cancel.
        if((newIdentifierPane.getValue() == null)
            || ((Integer)newIdentifierPane.getValue() == JOptionPane.CANCEL_OPTION))
        {
            identifier = "";
        }
        else
        {
            identifier = identifierTextField.getText().trim();
        }
        return identifier;
    }

    /**
     * Add a new Threshold.
     */
    private void addNewThreshold(String identifier)
    {
        String evaluatedIdentifier = null;
        final ArgumentsProcessor argProc = getParameters().getArguments();

        if(identifier == null)
        {
            identifier = getIdentifierFromDialog("Enter an identifier:", "");
        }
        evaluatedIdentifier = getParameters().getArguments().replaceArgumentsInString(identifier);

        if(!evaluatedIdentifier.isEmpty()) // If the evaluted Identifier is not an empty string.
        {
            if(!_overRideTableModel.doesOverrideExist(identifier))
            {
                if(!_overRideTableModel.doesDefaultExist(identifier))
                {
                    final ThresholdParameterPanel panel = buildThresholdParameterPanelWithIdentifier(identifier, true);
                    panel.addGraphicsGeneratorPlugInPanelListener(this);
                    setThresholdPanel(panel);
                    _defaultThresholdListTable.clearSelection();
                    selectOverrideParameterInOverrideParametersTable(identifier);
                }
                else
                {
                    DialogHelper.displayMessageDialog(null,
                                                      "Default Threshold Parameter already exists with the identifier "
                                                          + argProc.replaceArgumentsInString(identifier));
                    selectDefaultParameterInDefaultParametersTable(argProc.replaceArgumentsInString(identifier));
                    for(int i = 0; i < _defaultTableModel.getRowCount(); i++)
                    {
                        final String defaultidentifier = (String)_defaultThresholdListTable.getValueAt(i, 1);
                        if(defaultidentifier.equalsIgnoreCase(evaluatedIdentifier))
                        {
                            _defaultTableModel.setValueAt(true, i, 2);
                            break;
                        }
                    }

                }
            }
            else
            {
                DialogHelper.displayMessageDialog(null,
                                                  "Override Threshold Parameter already exists with the identifier "
                                                      + argProc.replaceArgumentsInString(identifier));
                selectOverrideParameterInOverrideParametersTable(identifier);
            }
        }
        else
        {
            DialogHelper.displayErrorDialog(null,
                                            "Identifier cannot be empty when arguments are evaluated.\nMake sure the needed argument is defined",
                                            "Empty Identifier");
        }
        this._overRideThresholdListTable.revalidate();
        this._defaultThresholdListTable.revalidate();
        fireParametersChanged();
    }

    private void highlightMatchingOverRideThresholdParameters(final String identifier)
    {
        final Integer row = _overRideTableModel.getRowNumber(identifier);

        if(row != null)
        {
            TableTools.selectRowAndScrollToIt(_overRideThresholdListTable, row);
        }
    }

    private void highlightMatchingDefaultThresholdParameters(final String identifier)
    {
        final Integer row = _defaultTableModel.getRowNumber(identifier);

        if(row != null)
        {
            TableTools.selectRowAndScrollToIt(_defaultThresholdListTable, row);
        }
    }

    private boolean validatedOverRideParameter(final String identifier)
    {
        boolean validParameter = true;

        if(!defaultParametersExist(identifier))
        {
            final ThresholdParameters paramToEval = getOverrideParameterFromIdentifier(identifier, false);
            validParameter = paramToEval.isValidOverRideParameter(getAxisType(paramToEval));
        }
        return validParameter;
    }

    @Override
    public int getAxisType(final ThresholdParameters parms)
    {
        final Integer subPlotIndex = parms.getSubPlotIndex();
        final String axisIdString = parms.getAxisIdString();
        if((subPlotIndex == null) || (subPlotIndex < 0))
        {
            return -1;
        }
        if((axisIdString == null) || (ChartConstants.determineAxisIndexFromString(axisIdString) == null))
        {
            return -1;
        }
        if(axisIdString.equalsIgnoreCase(ChartConstants.DOMAIN_AXIS_STRING)) //domain axis
        {
            return _xAxisType;
        }
        final int[] rangeAxisTypes = _subPlotIndexToRangeAxisTypes.get(subPlotIndex);
        if(rangeAxisTypes == null)
        {
            return -1;
        }
        return rangeAxisTypes[ChartConstants.determineAxisIndexFromString(axisIdString)];
    }

    /**
     * @param identifier - Identifier string
     * @return - true if a default thresholdparameter exists for the given identifier, false otherwise
     */
    private boolean defaultParametersExist(final String identifier)
    {
        boolean defaultExists = false;

        for(int i = 0; i < _defaultTLParameters.getThresholdParametersList().size(); i++)
        {
            final ThresholdParameters params = _defaultTLParameters.getThresholdParametersList().get(i);
            if(params.getIdentifier().equalsIgnoreCase(identifier))
            {
                defaultExists = true;
                break;
            }
        }

        return defaultExists;
    }

    /**
     * Add listeners to the GUI
     */
    private void addListeners()
    {
        addTableListeners();
        _addNewThresholdButton.addActionListener(new AddNewThresholdButtonListener());
    }

    private void addTableListeners()
    {
        _defaultThresholdListTable.getSelectionModel()
                                  .addListSelectionListener(_defaultThresholdListTableSelectionListener);

        _overRideThresholdListTable.getSelectionModel()
                                   .addListSelectionListener(_overRideThresholdListTableSelectionListener);

        //_overRideThresholdListTable.addMouseListener(_overrideThresholdListTableMouseListener);
    }

    private void removeTableListeners()
    {
        _defaultThresholdListTable.getSelectionModel()
                                  .removeListSelectionListener(_defaultThresholdListTableSelectionListener);

        _overRideThresholdListTable.getSelectionModel()
                                   .removeListSelectionListener(_overRideThresholdListTableSelectionListener);

        //_overRideThresholdListTable.removeMouseListener(_overrideThresholdListTableMouseListener);
    }

    /**
     * ActionListener added to the Add Threshold Button
     * 
     * @author SoodG
     */
    private class AddNewThresholdButtonListener implements ActionListener
    {
        @Override
        public void actionPerformed(final ActionEvent arg0)
        {
            addNewThreshold(null);
        }
    }

    /**
     * ListSelectionListener added to the ThresholdListTable
     * 
     * @author SoodG
     */
    private class OverRideThresholdListTableSelectionListener implements ListSelectionListener
    {
        @Override
        public void valueChanged(final ListSelectionEvent arg0)
        {
            if(!arg0.getValueIsAdjusting())
            {
                removeTableListeners();
                _defaultThresholdListTable.clearSelection();
                addTableListeners();

                final int row = _overRideThresholdListTable.getSelectedRow();
                if(row >= 0)
                {
                    String identifier = (String)_overRideThresholdListTable.getValueAt(row, 3);
                    final String evaluatedIdentifier = getEvaluatedIdentifier(identifier);
                    boolean defaultExists = defaultParametersExist(evaluatedIdentifier);
                    final boolean validParameter = validatedOverRideParameter(evaluatedIdentifier);
                    if(!validParameter)
                    {
                        final String newIdentifier = getIdentifierFromDialog(HString.wordWrap("Invalid Threshold:\n\n"
                                                                                                  + "Identifier: '"
                                                                                                  + identifier
                                                                                                  + "'\nEvaluated Identifier: '"
                                                                                                  + evaluatedIdentifier
                                                                                                  + "'\n\nThreshold appears to be an override for a default threshold that no longer "
                                                                                                  + "exists.  Since an override threshold only includes overridden parameters, the threshold "
                                                                                                  + "parameters are not complete.  In order to edit the parameters, they must be complete.  Please provide an "
                                                                                                  + "identifier that allows a default threshold to be matched or just click OK and the missing "
                                                                                                  + "parameters will be filled in with software defaults.\n \n"
                                                                                                  + "Clicking Cancel (or closing the dialog by clicking 'X' in the upper corner) will "
                                                                                                  + "result in the row being deselected so that the threshold's "
                                                                                                  + "parameters are not edited.\n\nNew Identifier:",
                                                                                              100),
                                                                             identifier);

                        //If cancel is clicked or "" manually entered...
                        if(newIdentifier.isEmpty())
                        {
                            removeTableListeners();
                            _overRideThresholdListTable.clearSelection();
                            addTableListeners();
                            setThresholdPanel(new JPanel());
                            return;
                        }

                        //The new identifier matches the already evaluated identifier, which we know corresponds
                        //to no default, so don't bother to do anything with it.
                        else if(!newIdentifier.equalsIgnoreCase(evaluatedIdentifier))
                        {
                            final ThresholdParameters params = getOverrideParameterFromIdentifier(evaluatedIdentifier,
                                                                                                  false);
                            identifier = newIdentifier;
                            reactToIdentifierChange(params, newIdentifier);
                            removeTableListeners(); //The above react may have re-added them.
                            defaultExists = defaultParametersExist(newIdentifier);
                            parameterChanged(params);
                        }
                    }
                    if(defaultExists)
                    {
                        highlightMatchingDefaultThresholdParameters(evaluatedIdentifier);
                    }
                    final ThresholdParameterPanel panel = buildThresholdParameterPanelWithIdentifier(identifier, true);

                    setThresholdPanel(panel);
                }
                else
                {
                    setThresholdPanel(new JPanel());
                }

                _overRideThresholdListTable.setVisible(false);
                _overRideThresholdListTable.setVisible(true);
            }
        }
    }

    /**
     * ListSelectionListener added to the ThresholdListTable
     * 
     * @author SoodG
     */
    private class DefaultThresholdListTableSelectionListener implements ListSelectionListener
    {
        @Override
        public void valueChanged(final ListSelectionEvent event)
        {
            if(!event.getValueIsAdjusting())
            {
                removeTableListeners();
                _overRideThresholdListTable.clearSelection();
                final int row = _defaultThresholdListTable.getSelectedRow();
                if(row >= 0)
                {
                    final String identifier = _defaultTLParameters.getThresholdParametersList()
                                                                  .get(row)
                                                                  .getIdentifier();
                    final ThresholdParameterPanel panel = buildThresholdParameterPanelWithIdentifier(getEvaluatedIdentifier(identifier),
                                                                                                     true);

                    final boolean overRideExists = _overRideTableModel.doesOverrideExist(identifier);
                    if(overRideExists)
                    {
                        highlightMatchingOverRideThresholdParameters(identifier);
                    }
                    setThresholdPanel(panel);
                }
                addTableListeners();
            }
        }
    }

    @Override
    public void parameterChanged(final GeneralPlugInParameters parameters)
    {
        final ThresholdParameters params = (ThresholdParameters)parameters;
        final ThresholdParameterPanel panel = buildThresholdParameterPanelWithIdentifier(params.getIdentifier(), false);
        final Point originalPosition = _selectedThresholdScrollPane.getViewport().getViewPosition();

        if(panel != null)
        {
            final boolean newOverRide = panel.isOverRideParametersDifferentFromDefaultParameters();

            if(newOverRide)
            {
                panel.setManagedParameters(params);
                getParameters().addThreshold(params);
            }

            removeTableListeners();
            _defaultThresholdListTable.clearSelection();
            addTableListeners();
            highlightMatchingOverRideThresholdParameters(params.getIdentifier());
            highlightMatchingDefaultThresholdParameters(params.getIdentifier());
            panel.makePanelReflectParameters();
            setThresholdPanel(panel);
            _selectedThresholdScrollPane.getViewport().setViewPosition(originalPosition);
            setVisible(false);
            setVisible(true);
            fireParametersChanged();
        }
    }

    /**
     * @param identifier - Unevaluated Identifier
     * @return Identifier evaluated using the Arguments Processer.
     */
    private String getEvaluatedIdentifier(final String identifier)
    {
        return getParameters().getArguments().replaceArgumentsInString(identifier);
    }

    /**
     * @return - ManagedParameters casted to a ThresholdListParameters
     */
    private ThresholdListParameters getParameters()
    {
        return (ThresholdListParameters)getManagedParameters();
    }

    /**
     * Recreates the identifierToPanel hashmap. This is
     */
    @Override
    public void makePanelReflectParameters()
    {
    }

    @Override
    public void disableSubPanel()
    {
    }

    @Override
    public void enableSubPanel()
    {
    }

    private ThresholdParameterPanel buildThresholdParameterPanelWithIdentifier(final String identifier,
                                                                               final boolean tableClicked)
    {
        ThresholdParameterPanel panel = null;

        final ThresholdParameters defaultParameters = getDefaultParametersFromIdentifier(identifier);
        ThresholdParameters overRideParameters = getOverrideParameterFromIdentifier(identifier, tableClicked);

        if((overRideParameters == null) && (defaultParameters == null))
        {
            overRideParameters = new ThresholdParameters(identifier);
            getParameters().addThreshold(overRideParameters);
        }

        if((defaultParameters != null) && (overRideParameters == null))
        {
            overRideParameters = new ThresholdParameters(identifier);
        }

        if((defaultParameters == null) && (overRideParameters != null))
        {
            overRideParameters.setupDefaultValuesForNonOverriddenParameters();
        }

        boolean isNew;

        if(defaultParameters == null)
        {
            isNew = true;
        }
        else
        {
            isNew = false;
        }

        panel = new ThresholdParameterPanel(this,
                                            defaultParameters,
                                            overRideParameters,
                                            isNew,
                                            _subPlotIndexToRangeAxisTypes,
                                            _xAxisType);
        panel.addGraphicsGeneratorPlugInPanelListener(this);
        return panel;
    }

    /**
     * Called when an argument is changed
     */
    @Override
    public void argumentsChanged()
    {
        this.getParameters().rebuildMap(); //In case the location has changed, since that affects the map.
        final int row = _overRideThresholdListTable.getSelectedRow();
        if(row >= 0)
        {
            final String identifier = (String)_overRideThresholdListTable.getValueAt(row, 3);
            selectOverrideParameterInOverrideParametersTable(identifier);
            highlightMatchingDefaultThresholdParameters(identifier);
        }
    }

    @Override
    public void reactToVisibleChanged(final ThresholdParametersListPanelTableModel source, final String identifier)
    {
        this.removeTableListeners();

        //Remember the original selected rows and scroll position.
        final Point scrollPosition = this._selectedThresholdScrollPane.getViewport().getViewPosition();

        _overRideThresholdListTable.clearSelection();
        _defaultThresholdListTable.clearSelection();

        _overRideThresholdListTable.revalidate();
        _defaultThresholdListTable.revalidate();

        this.addTableListeners();

        final Integer overrideRowToSelect = this._overRideTableModel.getRowNumber(identifier);
        final Integer defaultRowToSelect = this._defaultTableModel.getRowNumber(identifier);

        //Reselect the original selected rows.
        if(overrideRowToSelect != null)
        {
            TableTools.selectRowAndScrollToIt(_overRideThresholdListTable, overrideRowToSelect);
        }
        else if(defaultRowToSelect != null)
        {
            TableTools.selectRowAndScrollToIt(_defaultThresholdListTable, defaultRowToSelect);
        }

        //Set the scroll position.
        this._selectedThresholdScrollPane.getViewport().setViewPosition(scrollPosition);

        fireParametersChanged();
    }

    @Override
    public void reactToIdentifierChange(final ThresholdParameters parameters, final String newIdentifier)
    {
        //If the old identifier had a default but the new one does not, copy over the default to the
        //working parameters before renaming.
        if(this._overRideTableModel.doesDefaultExist(parameters.getIdentifier()))
        {
            if(!this._overRideTableModel.doesDefaultExist(newIdentifier))
            {
                final ThresholdParameters newParms = (ThresholdParameters)_defaultTLParameters.retrieveThresholdParameters(parameters.getIdentifier())
                                                                                              .clone();
                newParms.copyOverriddenParameters(parameters);
                parameters.copyFrom(newParms);
            }
        }

        getParameters().renameThreshold(parameters.getIdentifier(), newIdentifier);
        parameterChanged(parameters);
    }

    @Override
    public Object getPanelStatus()
    {
        final List<Object> status = new ArrayList<Object>();
        status.add(_overRideThresholdListTable.getSelectedRow());
        status.add(this._selectedThresholdScrollPane.getViewport().getViewPosition());
        return status;
    }

    @Override
    public void resetPanelStatus(final Object status)
    {
        @SuppressWarnings("unchecked")
        final int indexOfSelectedOverrideThreshold = (Integer)(((List<Object>)status).get(0));
        @SuppressWarnings("unchecked")
        final Point pos = (Point)(((List<Object>)status).get(1));
        if(indexOfSelectedOverrideThreshold < _overRideThresholdListTable.getRowCount())
        {
            TableTools.selectRowAndScrollToIt(_overRideThresholdListTable, indexOfSelectedOverrideThreshold);
            _selectedThresholdScrollPane.getViewport().setViewPosition(pos);
        }
    }

    @Override
    public void processDeleteAction(final Object source, final int displayRowIndex)
    {
        final int row = _overRideThresholdListTable.convertRowIndexToModel(displayRowIndex);
        final String identifier = (String)_overRideThresholdListTable.getValueAt(row, 4);
        final int option = JOptionPane.showConfirmDialog(_overRideThresholdListTable,
                                                         "Are you sure you want to delete the overridden threshold parameters, so that defaults are used?",
                                                         "Delete Threshold",
                                                         JOptionPane.YES_NO_OPTION);
        if(option == JOptionPane.YES_OPTION)
        {
            getParameters().removeThreshold(identifier);
            _overRideThresholdListTable.updateUI();
            if(_overRideTableModel.doesDefaultExist(identifier))
            {
                final ThresholdParameterPanel panel = buildThresholdParameterPanelWithIdentifier(getEvaluatedIdentifier(identifier),
                                                                                                 true);
                _defaultThresholdListTable.updateUI();
                setThresholdPanel(panel);
            }
            else
            {
                setThresholdPanel(new JPanel());
            }
            _overRideThresholdListTable.clearSelection();
            fireParametersChanged();
        }
    }

    @Override
    public int getVerticalScrollBarPolicy()
    {
        return ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER;
    }

    @Override
    public int getHorizontalScrollBarPolicy()
    {
        return ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER;
    }

}
