package ohd.hseb.charter.plotter.instances;

import java.awt.BasicStroke;
import java.awt.Color;
import java.util.ArrayList;
import java.util.List;

import org.jfree.chart.plot.XYPlot;
import org.jfree.data.xy.XYDataset;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.charter.datasource.instances.TimeSeriesArraysXYDataSet;
import ohd.hseb.charter.jfreechartoverride.GraphGenXYDifferenceRenderer;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.plotter.XYChartPlotter;
import ohd.hseb.charter.plotter.XYChartPlotterException;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

public class AreaBetweenLinesXYChartPlotter implements XYChartPlotter
{

    /**
     * Constructs a new data set that includes the two series between which the area is filled.
     * 
     * @param rootSet The base set of time series.
     * @param series1 The first series.
     * @param series2 The second series.
     * @return A new {@link TimeSeriesArraysXYDataSet} that includes only the two time series.
     * @throws XYChartPlotterException If the rootSet is not a {@link TimeSeriesArraysXYDataSet} or {@link XYSeriesCollection} with equal sized series.
     */
    private XYDataset constructNewXYDataSet(final XYDataset rootSet, final int series1, final int series2) throws XYChartPlotterException
    {
        //Create difference renderer data set between two time series specified by series1 and series2.
        if(rootSet instanceof TimeSeriesArraysXYDataSet)
        {
            final TimeSeriesArraysXYDataSet base = (TimeSeriesArraysXYDataSet)rootSet;
            TimeSeriesArrays ts = new TimeSeriesArrays(base.getSeries(series1));
            ts.add(base.getSeries(series2));
            final List<TimeSeriesArray> newTS = TimeSeriesArraysTools.trimTimeSeriesToHaveSameStartAndEnd(TimeSeriesArraysTools.convertTimeSeriesArraysToList(ts));
            if(newTS != null)
            {
                ts = TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(newTS);
            }
            final List<String> seriesKeys = new ArrayList<String>();
            seriesKeys.add((String)base.getSeriesKey(series1));
            seriesKeys.add((String)base.getSeriesKey(series2));
            final TimeSeriesArraysXYDataSet newSet = new TimeSeriesArraysXYDataSet(ts,
                                                                                   base.getDomainTimeZone(),
                                                                                   seriesKeys);
            return newSet;
        }
        //Create difference renderer data set between two chart series specified by series1 and series2.
        else if (rootSet instanceof XYSeriesCollection)
        {
            final XYSeriesCollection baseSet = (XYSeriesCollection)rootSet;
            if (baseSet.getItemCount(series1) != baseSet.getItemCount(series2))
            {
                throw new XYChartPlotterException("For the numerical data set provided, all series must be of equal length, but are not.");
            }
            final XYSeriesCollection dataset = new XYSeriesCollection();
            final XYSeries newSeries1 = new XYSeries("", false);
            final XYSeries newSeries2 = new XYSeries("", false);
                for(int j = 0; j < baseSet.getItemCount(series1); j++)
                {
                    newSeries1.add(baseSet.getX(series1, j), baseSet.getY(series1,  j));
                    newSeries2.add(baseSet.getX(series2, j), baseSet.getY(series2,  j));
                }

                dataset.addSeries(newSeries1);
                dataset.addSeries(newSeries2);
            return dataset;
        }
        else
        {
            throw new XYChartPlotterException("The data to plot must be a time series or numerical data set for the AreaBetweenLines plot type, but is not.");
        }

    }

    @Override
    public void applyPlotterSettings(final XYPlot plot,
                                     final int datasetIndexWithinPlot,
                                     final DataSourceDrawingParameters parameters) throws XYChartPlotterException
    {
        final XYDataset dataset = plot.getDataset(datasetIndexWithinPlot);

        //First, plot the lines.  Pass in true to always display legend blocks.
        final LineAndScatterXYChartPlotter linePlotter = new LineAndScatterXYChartPlotter(true);
        linePlotter.applyPlotterSettings(plot, datasetIndexWithinPlot, parameters);

        //Next, construct one data set per pairing of series.  Plot it as a difference data set.
        final int datasetCount = plot.getDatasetCount();
        final int axisIndex = parameters.getYAxisIndex();
        final int originalSeriesCount = dataset.getSeriesCount();
        for(int i = 0; i < originalSeriesCount - 1; i++)
        {
            //Only create a difference renderer if the series fill boolean is true.  Otherwise, the series is just a line-and-scatter.
            if(parameters.getSeriesDrawingParameters().get(i).getShapeFilled())
            {
                final XYDataset newDataset = constructNewXYDataSet(dataset, i, i + 1);
                plot.setDataset(datasetCount + i, newDataset);
                plot.mapDatasetToRangeAxis(datasetCount + i, axisIndex);

                final GraphGenXYDifferenceRenderer diffRenderer = new GraphGenXYDifferenceRenderer();
                diffRenderer.setBaseCreateEntities(false);
                if(parameters.getSeriesDrawingParametersForSeriesIndex(i).getShapeFilled())
                {
                    final Color c = parameters.getSeriesDrawingParametersForSeriesIndex(i).getFillColor();
                    diffRenderer.setPositivePaint(c);
                    diffRenderer.setNegativePaint(c);
                }
                else
                {
                    diffRenderer.setPositivePaint(ColorTools.TRANSPARENT_WHITE);
                }

                diffRenderer.setShapesVisible(false);
                diffRenderer.setSeriesVisibleInLegend(0, false);
                for(int j = 0; j < newDataset.getSeriesCount(); j++)
                {
                    diffRenderer.setSeriesVisible(j, false);
                    diffRenderer.setSeriesStroke(j, new BasicStroke(0.0f));
                    diffRenderer.setSeriesPaint(j, Color.BLACK);
                }

                plot.setRenderer(datasetCount + i, diffRenderer);
            }
        }
    }

    @Override
    public SeriesDrawingParameters specifyApplicableParameters()
    {
        final LineAndScatterXYChartPlotter linePlotter = new LineAndScatterXYChartPlotter(true);
        final SeriesDrawingParameters dummyParms = linePlotter.specifyApplicableParameters();
        dummyParms.setShapeFilled(true);
        dummyParms.setFillColor(Color.white);
        return dummyParms;
    }
}
