package ohd.hseb.charter.plotter.instances;

import java.awt.BasicStroke;
import java.awt.Color;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.jfreechartoverride.GraphGenXYLineAndShapeRenderer;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.plotter.XYChartPlotter;
import ohd.hseb.charter.plotter.XYChartPlotterException;

import org.jfree.chart.plot.XYPlot;

public class LineAndScatterXYChartPlotter implements XYChartPlotter
{

    /**
     * This plotter is sometimes used in area plotters to draw the line and shape associated with the area border. As
     * such, this flag is used by those area plotters to indicate that if a shape is not specified, draw a square with
     * the area color in the legend. It is passed through to {@link GraphGenXYLineAndShapeRenderer}.
     */
    private boolean _alwaysDrawLegendColorSquare = false;

    public LineAndScatterXYChartPlotter()
    {
        super();
    }

    public LineAndScatterXYChartPlotter(final boolean alwaysDrawLegendColorSquare)
    {
        super();
        _alwaysDrawLegendColorSquare = alwaysDrawLegendColorSquare;
    }

    @Override
    public void applyPlotterSettings(final XYPlot plot,
                                     final int datasetIndexWithinPlot,
                                     final DataSourceDrawingParameters parameters) throws XYChartPlotterException
    {
        final GraphGenXYLineAndShapeRenderer renderer = new GraphGenXYLineAndShapeRenderer();
        renderer.setAlwaysDrawLegendColorSquare(_alwaysDrawLegendColorSquare);
        for(int i = 0; i < plot.getDataset(datasetIndexWithinPlot).getSeriesCount(); i++)
        {
            Boolean b = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShowInLegend();
            if(b != null)
            {
                renderer.setSeriesVisibleInLegend(i, b);
            }

            //Is line visible, based on width > 0?  If so, set width.
            final Float width = parameters.getSeriesDrawingParametersForSeriesIndex(i).getLineWidth();
            if(width <= 0)
            {
                renderer.setSeriesLinesVisible(i, false);
            }
            else
            {
                renderer.setSeriesLinesVisible(i, true);
                renderer.setSeriesStroke(i, new BasicStroke(width));
            }

            //Line color
            Color c = parameters.getSeriesDrawingParametersForSeriesIndex(i).getLineColor();
            renderer.setSeriesPaint(i, c);

            //Is filled?
            b = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShapeFilled();
            if(b != null)
            {
                renderer.setSeriesShapesFilled(i, b);

                //Fill color
                c = parameters.getSeriesDrawingParametersForSeriesIndex(i).getFillColor();
                renderer.setSeriesFillPaint(i, c);
                renderer.setUseFillPaint(true);
            }

            //Shape
            final String shape = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShapeName();
            final Double shapeSize = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShapeSize();

            if(ChartConstants.isNoShape(shape))
            {
                renderer.setSeriesShapesVisible(i, false);
            }
            else
            {
                renderer.setSeriesShapesVisible(i, true);
                renderer.setSeriesShape(i, ChartConstants.getShape(shape, shapeSize));
            }
        }

        plot.setRenderer(datasetIndexWithinPlot, renderer);
    }

    @Override
    public SeriesDrawingParameters specifyApplicableParameters()
    {
        final SeriesDrawingParameters dummyParms = new SeriesDrawingParameters(-1);
        dummyParms.setShowInLegend(true);
        dummyParms.setLineColor(Color.white);
        dummyParms.setLineWidth(1.0f);
        dummyParms.setFillColor(Color.black);
        dummyParms.setShapeName("HANK");
        dummyParms.setShapeSize(ChartConstants.DEFAULT_SHAPE_SIZE);
        dummyParms.setShapeFilled(true);
        dummyParms.setIncludeInPlot(true);
        dummyParms.setNameInLegend("HANK2");
        dummyParms.setBoxWidth(null);
        dummyParms.setBarWidth(null);
        return dummyParms;
    }
}
