package ohd.hseb.charter.plotter.instances;

import java.awt.BasicStroke;
import java.awt.Color;

import org.jfree.chart.plot.XYPlot;
import org.jfree.data.statistics.BoxAndWhiskerXYDataset;
import org.jfree.data.xy.XYDataset;

import ohd.hseb.charter.jfreechartoverride.ScatterBoxPlotDataset;
import ohd.hseb.charter.jfreechartoverride.ScatterBoxPlotItem;
import ohd.hseb.charter.jfreechartoverride.ScatterBoxPlotRenderer;
import ohd.hseb.charter.jfreechartoverride.XYBoxAndWhiskerRenderer;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.plotter.XYChartPlotter;
import ohd.hseb.charter.plotter.XYChartPlotterException;

/**
 * Time-based box-and-whisker plotter makes use of {@link BoxAndWhiskerXYDataset} and {@link XYBoxAndWhiskerRenderer}.
 * 
 * @author Hank.Herr
 */
public class ScatterBoxWhiskerXYChartPlotter implements XYChartPlotter
{
    /**
     * @return A {@link ScatterBoxPlotDataset} constructed from the data within the provided base data set.
     */
    private ScatterBoxPlotDataset constructBoxAndWhiskerDataSet(final XYDataset baseDataSet) throws XYChartPlotterException
    {
        if((baseDataSet.getSeriesCount() <= 1)
            || ((baseDataSet.getSeriesCount() > 2) && (baseDataSet.getSeriesCount() % 2 == 0)))
        {
            throw new XYChartPlotterException("XY box-whisker plot requires 2 or more series and, if more than two, it must be odd; the series count "
                + baseDataSet.getSeriesCount() + ", is not allowed.");
        }

        @SuppressWarnings("unchecked")
        final ScatterBoxPlotDataset bwDataSet = new ScatterBoxPlotDataset(baseDataSet.getSeriesKey(0));

        //For each box-and-whisker itemt to plot...
        for(int i = 0; i < baseDataSet.getItemCount(0); i++)
        {
            final int centerIndex = baseDataSet.getSeriesCount() / 2;
            final double[] whiskers = new double[baseDataSet.getSeriesCount() - 1];
            double middle = Double.NaN;
            int count = 0;

            //Loop over all series.  For each series, add the data value for the series and item to the whiskers or set it as the middle value in the
            //box and whisker.  
            for(int j = 0; j < baseDataSet.getSeriesCount(); j++)
            {
                if(j != centerIndex)
                {
                    whiskers[count] = baseDataSet.getYValue(j, i);
                    count ++;
                }
                else
                {
                    middle = baseDataSet.getYValue(j, i);
                }
            }

            final ScatterBoxPlotItem itemToAdd = new ScatterBoxPlotItem(whiskers, middle);
            bwDataSet.add(baseDataSet.getX(0, i), itemToAdd);
        }

        return bwDataSet;
    }

    @Override
    public void applyPlotterSettings(final XYPlot plot,
                                     final int datasetIndexWithinPlot,
                                     final DataSourceDrawingParameters parameters) throws XYChartPlotterException
    {
        //Get the data set.
        final XYDataset baseDataSet = plot.getDataset(datasetIndexWithinPlot);

        //Setup the new data set, which is a copy, unfortunately, because it requires adding items.
        final ScatterBoxPlotDataset boxSet = this.constructBoxAndWhiskerDataSet(baseDataSet);
        final ScatterBoxPlotRenderer renderer = new ScatterBoxPlotRenderer();
        plot.setDataset(datasetIndexWithinPlot, boxSet);
        plot.setRenderer(datasetIndexWithinPlot, renderer);

        //Show in legend.
        final Boolean b = parameters.getSeriesDrawingParametersForSeriesIndex(0).getShowInLegend();
        if(b != null)
        {
            renderer.setSeriesVisibleInLegend(0, b);
        }

        //Line width.
        final Float lineWidth = parameters.getSeriesDrawingParametersForSeriesIndex(0).getLineWidth();
        if(lineWidth != null)
        {
            renderer.setSeriesStroke(0, new BasicStroke(lineWidth));
            renderer.setSeriesOutlineStroke(0, new BasicStroke(lineWidth));
        }

        //Line color goes through as artifact paint.  Only the first of the 3 or 5 series matters.  The others
        //have no effect.
        Color c = parameters.getSeriesDrawingParametersForSeriesIndex(0).getLineColor();
        if(c != null)
        {
            renderer.setArtifactPaint(c);

            //These two combined define the legend item displayed.
            renderer.setSeriesPaint(0, parameters.getSeriesDrawingParametersForSeriesIndex(0).getFillColor());
            renderer.setSeriesOutlinePaint(0, c);
        }

        //Box filled?
        final Boolean shapeFilled = parameters.getSeriesDrawingParametersForSeriesIndex(0).getShapeFilled();
        if(shapeFilled != null)
        {
            renderer.setBoxFilled(shapeFilled);
        }

        //Fill color goes through base fill paint.
        c = parameters.getSeriesDrawingParametersForSeriesIndex(0).getFillColor();
        if(c != null)
        {
            renderer.setBaseFillPaint(parameters.getSeriesDrawingParametersForSeriesIndex(0).getFillColor());
        }

        //Width of box.
        final Double width = parameters.getSeriesDrawingParametersForSeriesIndex(0).getBoxWidth();
        if(width != null)
        {
            renderer.setBoxWidth(width);
        }
    }

    @Override
    public SeriesDrawingParameters specifyApplicableParameters()
    {
        final SeriesDrawingParameters dummyParms = new SeriesDrawingParameters(-1);
        dummyParms.setShowInLegend(true);
        dummyParms.setLineColor(Color.white);
        dummyParms.setLineWidth(1.0f);
        dummyParms.setFillColor(Color.green);
        dummyParms.setShapeName(null); //Not used
        dummyParms.setShapeSize(null); //Not used
        dummyParms.setShapeFilled(true);
        dummyParms.setIncludeInPlot(true);
        dummyParms.setNameInLegend("HANK2");
        dummyParms.setBoxWidth(-1d);
        dummyParms.setBarWidth(null); //Not used
        return dummyParms;
    }
}
