package ohd.hseb.charter.plotter.instances;

import java.awt.BasicStroke;
import java.awt.Color;

import ohd.hseb.charter.ChartConstants;
import ohd.hseb.charter.jfreechartoverride.GraphGenXYStepRenderer;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.plotter.XYChartPlotter;
import ohd.hseb.charter.plotter.XYChartPlotterException;

import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;

public class StepXYChartPlotter implements XYChartPlotter
{

    /**
     * This plotter is sometimes used in area plotters to draw the line and shape associated with the area border. As
     * such, this flag is used by those area plotters to indicate that if a shape is not specified, draw a square with
     * the area color in the legend. It is passed through to {@link GraphGenXYStepRenderer}.
     */
    private boolean _alwaysDrawLegendColorSquare = false;

    public StepXYChartPlotter()
    {
        super();
    }

    public StepXYChartPlotter(final boolean alwaysDrawLegendColorSquare)
    {
        super();
        _alwaysDrawLegendColorSquare = alwaysDrawLegendColorSquare;
    }

    @Override
    public void applyPlotterSettings(final XYPlot plot,
                                     final int datasetIndexWithinPlot,
                                     final DataSourceDrawingParameters parameters) throws XYChartPlotterException
    {
        //Add a copy of the dataset to the plot.
        final int newDatasetIndex = plot.getDatasetCount();
        plot.setDataset(newDatasetIndex, plot.getDataset(datasetIndexWithinPlot));
        plot.mapDatasetToRangeAxis(newDatasetIndex, parameters.getYAxisIndex());

        //Add a LineAndScatter, first, so shapes are on top, but without the lines
        final LineAndScatterXYChartPlotter lineAndScatterPlotter = new LineAndScatterXYChartPlotter();
        lineAndScatterPlotter.applyPlotterSettings(plot, datasetIndexWithinPlot, parameters);
        for(int i = 0; i < plot.getDataset(datasetIndexWithinPlot).getSeriesCount(); i++)
        {
            plot.getRenderer(datasetIndexWithinPlot).setSeriesVisibleInLegend(i, false);
            ((XYLineAndShapeRenderer)plot.getRenderer(datasetIndexWithinPlot)).setSeriesLinesVisible(i, false);
        }

        //Add steps
        final GraphGenXYStepRenderer renderer = new GraphGenXYStepRenderer();
        renderer.setAlwaysDrawLegendColorSquare(_alwaysDrawLegendColorSquare);
        renderer.setConnectedLineAndShapeRenderer(datasetIndexWithinPlot,
                                                  (XYLineAndShapeRenderer)plot.getRenderer(datasetIndexWithinPlot));
        for(int i = 0; i < plot.getDataset(newDatasetIndex).getSeriesCount(); i++)
        {
            Boolean b = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShowInLegend();
            if(b != null)
            {
                renderer.setSeriesVisibleInLegend(i, b);
            }

            //Line color
            Color c = parameters.getSeriesDrawingParametersForSeriesIndex(i).getLineColor();
            renderer.setSeriesPaint(i, c);

            //Is line visible, based on width > 0?  If so, set width.
            final Float width = parameters.getSeriesDrawingParametersForSeriesIndex(i).getLineWidth();
            if(width <= 0)
            {
                renderer.setSeriesLinesVisible(i, false);
                renderer.setSeriesPaint(i, new Color(255, 255, 255, 0));
            }
            else
            {
                renderer.setSeriesLinesVisible(i, true);
                renderer.setSeriesStroke(i, new BasicStroke(width));
            }

            //NOTE: I INCLUDE THE FILLED AND SHAPE STUFF BELOW SO THAT THE RENDERER KNOWS ABOUT THE
            //SHAPES EVEN THOUGH THE DRAWING OF THE SHAPES IS HANDLED BY THE LineAndScatterPlotter
            //ADDED ABOVE.  THIS IS REQUIRED FOR THE GraphGenXYStepAreaRenderer TO WORK!

            //Is filled?
            b = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShapeFilled();
            if(b != null)
            {
                renderer.setSeriesShapesFilled(i, b);

                //Fill color
                c = parameters.getSeriesDrawingParametersForSeriesIndex(i).getFillColor();
                renderer.setSeriesFillPaint(i, c);
                renderer.setUseFillPaint(true);
            }

            //Shape
            final String shape = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShapeName();
            final Double shapeSize = parameters.getSeriesDrawingParametersForSeriesIndex(i).getShapeSize();
            if(ChartConstants.isNoShape(shape))
            {
                renderer.setSeriesShapesVisible(i, false);
            }
            else
            {
                renderer.setSeriesShapesVisible(i, true);
                renderer.setSeriesShape(i, ChartConstants.getShape(shape, shapeSize));
            }
        }

        plot.setRenderer(newDatasetIndex, renderer);
    }

    @Override
    public SeriesDrawingParameters specifyApplicableParameters()
    {
        final SeriesDrawingParameters dummyParms = new SeriesDrawingParameters(-1);
        dummyParms.setShowInLegend(true);
        dummyParms.setLineColor(Color.white);
        dummyParms.setLineWidth(1.0f);
        dummyParms.setFillColor(Color.red);
        dummyParms.setShapeName("HANK");
        dummyParms.setShapeSize(ChartConstants.DEFAULT_SHAPE_SIZE);
        dummyParms.setShapeFilled(true);
        dummyParms.setIncludeInPlot(true);
        dummyParms.setNameInLegend("HANK2");
        dummyParms.setBoxWidth(null);
        dummyParms.setBarWidth(null);
        return dummyParms;
    }
}
