package ohd.hseb.charter.plotter.instances;

import java.awt.BasicStroke;
import java.awt.Color;
import java.util.ArrayList;
import java.util.Date;

import org.jfree.chart.plot.XYPlot;
import org.jfree.data.statistics.BoxAndWhiskerItem;
import org.jfree.data.statistics.BoxAndWhiskerXYDataset;
import org.jfree.data.statistics.DefaultBoxAndWhiskerXYDataset;
import org.jfree.data.xy.XYDataset;

import ohd.hseb.charter.jfreechartoverride.XYBoxAndWhiskerRenderer;
import ohd.hseb.charter.parameters.DataSourceDrawingParameters;
import ohd.hseb.charter.parameters.SeriesDrawingParameters;
import ohd.hseb.charter.plotter.XYChartPlotter;
import ohd.hseb.charter.plotter.XYChartPlotterException;

/**
 * Time-based box-and-whisker plotter makes use of {@link BoxAndWhiskerXYDataset} and {@link XYBoxAndWhiskerRenderer}.
 * Can only be used with 3 or 5 provided series (will throw exception during
 * {@link #applyPlotterSettings(XYPlot, int, DataSourceDrawingParameters)} if not) and it assumes that the x-values
 * (time) for all of the series are identical with each series being of the same length (i.e., number of times).
 * 
 * @author Hank.Herr
 */
public class XYBoxWhiskerXYChartPlotter implements XYChartPlotter
{

    /**
     * @param baseDataSet The base data set which will be processed into {@link BoxAndWhiskerItem} instances.
     * @return A {@link BoxAndWhiskerXYDataset} for use with the {@link XYBoxAndWhiskerRenderer}.
     */
    private BoxAndWhiskerXYDataset constructBoxAndWhiskerDataSet(final XYDataset baseDataSet)
    {
        final DefaultBoxAndWhiskerXYDataset bwDataSet = new DefaultBoxAndWhiskerXYDataset(baseDataSet.getSeriesKey(0));

        final boolean useFiveValues = (baseDataSet.getSeriesCount() == 5);

        for(int i = 0; i < baseDataSet.getItemCount(0); i++)
        {
            Number min = null;
            Number low = null;
            Number middle = null;
            Number hi = null;
            Number max = null;

            if(useFiveValues)
            {
                min = baseDataSet.getY(0, i);
                low = baseDataSet.getY(1, i);
                middle = baseDataSet.getY(2, i);
                hi = baseDataSet.getY(3, i);
                max = baseDataSet.getY(4, i);
            }
            else
            {
                min = baseDataSet.getY(0, i);
                middle = baseDataSet.getY(1, i);
                low = middle;
                hi = middle;
                max = baseDataSet.getY(2, i);
            }

            if((middle != null) && (!Double.isNaN(middle.doubleValue())))
            {
                @SuppressWarnings("rawtypes") //I don't know the type, since JFreeChart doesn't specify in the arguments.  Its just a List.
                final BoxAndWhiskerItem itemToAdd =
                                                  new BoxAndWhiskerItem(Float.NaN,
                                                                        middle,
                                                                        low,
                                                                        hi,
                                                                        min,
                                                                        max,
                                                                        null,
                                                                        null,
                                                                        new ArrayList());
                bwDataSet.add(new Date((Long)baseDataSet.getX(0, i)), itemToAdd);
            }
        }

        return bwDataSet;
    }

    @Override
    public void applyPlotterSettings(final XYPlot plot,
                                     final int datasetIndexWithinPlot,
                                     final DataSourceDrawingParameters parameters) throws XYChartPlotterException
    {
        //Check the base data set for series count (3 for whiskers only, 5 for box-and-whisker).
        final XYDataset baseDataSet = plot.getDataset(datasetIndexWithinPlot);
        if((baseDataSet.getSeriesCount() != 3) && (baseDataSet.getSeriesCount() != 5))
        {
            throw new XYChartPlotterException("XY box-whisker plot requires 3 or 5 series.");
        }

        //Construct the box set and setup the renderer.
        final BoxAndWhiskerXYDataset boxSet = this.constructBoxAndWhiskerDataSet(baseDataSet);
        final XYBoxAndWhiskerRenderer renderer = new XYBoxAndWhiskerRenderer();
        plot.setDataset(datasetIndexWithinPlot, boxSet);
        plot.setRenderer(datasetIndexWithinPlot, renderer);

        //Start passing through parameters.  In each case, if the flag or parameter is null, default is used.
        Boolean b = parameters.getSeriesDrawingParametersForSeriesIndex(0).getShowInLegend();
        if(b != null)
        {
            renderer.setSeriesVisibleInLegend(0, b);
        }

        b = parameters.getSeriesDrawingParametersForSeriesIndex(0).getShapeFilled();
        if(b != null)
        {
            renderer.setFillBox(b);
        }

        final Float lineWidth = parameters.getSeriesDrawingParametersForSeriesIndex(0).getLineWidth();
        if(lineWidth != null)
        {
            renderer.setSeriesStroke(0, new BasicStroke(lineWidth));
            renderer.setSeriesOutlineStroke(0, new BasicStroke(lineWidth));
        }

        Color c = parameters.getSeriesDrawingParametersForSeriesIndex(0).getLineColor();
        if(c != null)
        {
            renderer.setArtifactPaint(c);
            renderer.setSeriesPaint(0, parameters.getSeriesDrawingParametersForSeriesIndex(0).getLineColor());
            renderer.setSeriesOutlinePaint(0, c);
        }

        c = parameters.getSeriesDrawingParametersForSeriesIndex(0).getFillColor();
        if(c != null)
        {
            renderer.setBoxPaint(parameters.getSeriesDrawingParametersForSeriesIndex(0).getFillColor());
        }

        final Double width = parameters.getSeriesDrawingParametersForSeriesIndex(0).getBoxWidth();
        if(width != null)
        {
            renderer.setBoxWidth(width);
        }
    }

    @Override
    public SeriesDrawingParameters specifyApplicableParameters()
    {
        final SeriesDrawingParameters dummyParms = new SeriesDrawingParameters(-1);
        dummyParms.setShowInLegend(true);
        dummyParms.setLineColor(Color.white);
        dummyParms.setLineWidth(1.0f);
        dummyParms.setFillColor(Color.black);
        dummyParms.setShapeName(null); //not used
        dummyParms.setShapeSize(null); //not used
        dummyParms.setShapeFilled(true);
        dummyParms.setIncludeInPlot(true);
        dummyParms.setNameInLegend("HANK2");
        dummyParms.setBoxWidth(-1d);
        dummyParms.setBarWidth(null); //not used
        return dummyParms;
    }
}
