package ohd.hseb.charter.tools;

import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.font.FontRenderContext;
import java.awt.font.LineMetrics;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.TimeZone;

import org.jfree.chart.axis.AxisState;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.DateTick;
import org.jfree.chart.axis.DateTickMarkPosition;
import org.jfree.chart.axis.DateTickUnit;
import org.jfree.chart.axis.SegmentedTimeline;
import org.jfree.chart.axis.Tick;
import org.jfree.chart.axis.TickUnit;
import org.jfree.chart.axis.TickUnitSource;
import org.jfree.chart.axis.TickUnits;
import org.jfree.chart.axis.ValueTick;
import org.jfree.data.Range;
import org.jfree.data.time.DateRange;
import org.jfree.data.time.Month;
import org.jfree.data.time.RegularTimePeriod;
import org.jfree.data.time.Year;
import org.jfree.text.TextBlock;
import org.jfree.text.TextBlockAnchor;
import org.jfree.text.TextUtilities;
import org.jfree.ui.RectangleEdge;
import org.jfree.ui.RectangleInsets;
import org.jfree.ui.TextAnchor;

import com.google.common.base.CharMatcher;

import nl.wldelft.libx.jfreechart.CenteredTextBlock;
import ohd.hseb.util.misc.HCalendar;

public class DateAxisPlus extends DateAxis
{
    private static final long serialVersionUID = 1L;

    private Range unzoomedRange = null;
    private Range minimumRange = null;
    private Range maximumRange = null;

    private boolean centerLabelsBetweenTicks = false;
    private DateTickUnit minorTickUnit = null;
    private float minorTickMarkInsideLength = 2.0f;
    private float minorTickMarkOutsideLength = 0.0f;

    private long timeStepMillis;

    private int tickStartHour = -1;

    public DateAxisPlus(final long timeStepSize)
    {
        setTimeStepSize(timeStepSize);
    }

    public DateAxisPlus()
    {
        setTimeStepSize(0);
    }

    public int getTickStartHour()
    {
        return tickStartHour;
    }

    public void setTickStartHour(final int hourOfDay)
    {
        tickStartHour = hourOfDay;
    }

    public DateTickUnit getMinorTickUnit()
    {
        return minorTickUnit;
    }

    public void setMinorTickUnit(final DateTickUnit minorTickUnit)
    {
        this.minorTickUnit = minorTickUnit;
    }

    public float getMinorTickMarkInsideLength()
    {
        return minorTickMarkInsideLength;
    }

    public void setMinorTickMarkInsideLength(final float minorTickMarkInsideLength)
    {
        this.minorTickMarkInsideLength = minorTickMarkInsideLength;
    }

    public float getMinorTickMarkOutsideLength()
    {
        return minorTickMarkOutsideLength;
    }

    public void setMinorTickMarkOutsideLength(final float minorTickMarkOutsideLength)
    {
        this.minorTickMarkOutsideLength = minorTickMarkOutsideLength;
    }

    public Range getUnzoomedRange()
    {
        return unzoomedRange;
    }

    public void setUnzoomedRange(final Range unzoomedRange)
    {
        this.unzoomedRange = unzoomedRange;
    }

    public Range getMaximumRange()
    {
        return maximumRange;
    }

    public void setMaximumRange(final Range maximumRange)
    {
        this.maximumRange = maximumRange;
    }

    public Range getMinimumRange()
    {
        return minimumRange;
    }

    public void setMinimumRange(final Range minimumRange)
    {
        this.minimumRange = minimumRange;
    }

    private void setTimeStepSize(final long timeStepSize)
    {
        timeStepMillis = timeStepSize;
        setStandardTickUnits(createTimeStepDependantDateTickUnits(getTimeZone()));
    }

    private boolean isValidDateTickUnit(final DateTickUnit dateTickUnit)
    {
        if(timeStepMillis == 0)
            return true;

        final double tickSize = dateTickUnit.getSize();
        if(tickSize > timeStepMillis)
        {
            return tickSize % timeStepMillis == 0;
        }
        return timeStepMillis % tickSize == 0;

    }

    /**
     * Draws the axis line, tick marks and tick mark labels.
     * 
     * @param g2 the graphics device.
     * @param cursor the cursor.
     * @param plotArea the plot area.
     * @param dataArea the data area.
     * @param edge the edge that the axis is aligned with.
     * @return The width or height used to draw the axis.
     */
    @Override
    protected AxisState drawTickMarksAndLabels(final Graphics2D g2,
                                               final double cursor,
                                               final Rectangle2D plotArea,
                                               final Rectangle2D dataArea,
                                               final RectangleEdge edge)
    {
        if(minorTickUnit != null)
            drawMinorTickMarks(g2, cursor, dataArea, edge);

        final AxisState state = new AxisState(cursor);
        final List ticks = refreshTicks(g2, state, dataArea, edge);

//This used to be for single-line labels, but I decided to push everything through the multi-line algorithm.  To
//that algorithm I added a bounds check to ensure that labels are not cutoff.
//        if(getLabelLineCount(ticks) == 1)
//        {
//            return super.drawTickMarksAndLabels(g2, cursor, plotArea, dataArea, edge);
//        }

        //Multiline labels...        
        if(isAxisLineVisible())
        {
            drawAxisLine(g2, cursor, dataArea, edge);
        }

        final double ol = getTickMarkOutsideLength();
        final double il = getTickMarkInsideLength();

        state.setTicks(ticks);
        g2.setFont(getTickLabelFont());
        final Iterator iterator = ticks.iterator();
        while(iterator.hasNext())
        {
            final ValueTick tick = (ValueTick)iterator.next();
            if(isTickLabelsVisible())
            {
                g2.setPaint(getTickLabelPaint());
                final float[] anchorPoint = calculateAnchorPoint(tick, cursor, dataArea, edge);
                final TextBlock textBlock = new CenteredTextBlock();
                final String[] lines = tick.getText().split("\n");
                for(final String line: lines)
                {
                    textBlock.addLine(line, g2.getFont(), g2.getPaint());
                }

                //Compute the bounding box of the textBlock and draw it if its contained in the plot area.
                
                final Shape bounds = textBlock.calculateBounds(g2,
                                                               anchorPoint[0],
                                                               anchorPoint[1],
                                                               TextBlockAnchor.TOP_CENTER,
                                                               anchorPoint[0],
                                                               anchorPoint[1],
                                                               tick.getAngle());
                if(plotArea.contains(bounds.getBounds2D()))
                {
                    textBlock.draw(g2,
                                   anchorPoint[0],
                                   anchorPoint[1],
                                   TextBlockAnchor.TOP_CENTER,
                                   anchorPoint[0],
                                   anchorPoint[1],
                                   tick.getAngle());
                }

            }

            if(isTickMarksVisible())
            {
                final float xx = (float)valueToJava2D(tick.getValue(), dataArea, edge);
                g2.setStroke(getTickMarkStroke());
                g2.setPaint(getTickMarkPaint());
                Line2D mark = null;
                if(edge == RectangleEdge.LEFT)
                {
                    mark = new Line2D.Double(cursor - ol, xx, cursor + il, xx);
                }
                else if(edge == RectangleEdge.RIGHT)
                {
                    mark = new Line2D.Double(cursor + ol, xx, cursor - il, xx);
                }
                else if(edge == RectangleEdge.TOP)
                {
                    mark = new Line2D.Double(xx, cursor - ol, xx, cursor + il);
                }
                else if(edge == RectangleEdge.BOTTOM)
                {
                    mark = new Line2D.Double(xx, cursor + ol, xx, cursor - il);
                }
                g2.draw(mark);
            }
        }

        // need to work out the space used by the tick labels...
        // so we can update the cursor...
        if(isTickLabelsVisible())
        {
            double used = 0.0;
            if(edge == RectangleEdge.LEFT)
            {
                used += findMaximumTickLabelWidth(ticks, g2, plotArea, isVerticalTickLabels());
                state.cursorLeft(used);
            }
            else if(edge == RectangleEdge.RIGHT)
            {
                used = findMaximumTickLabelWidth(ticks, g2, plotArea, isVerticalTickLabels());
                state.cursorRight(used);
            }
            else if(edge == RectangleEdge.TOP)
            {
                used = findMaximumTickLabelHeight(ticks, g2, plotArea, isVerticalTickLabels());
                state.cursorUp(used);
            }
            else if(edge == RectangleEdge.BOTTOM)
            {
                used = findMaximumTickLabelHeight(ticks, g2, plotArea, isVerticalTickLabels());
                state.cursorDown(used);
            }
        }

        return state;
    }

    /**
     * Draws the minor tick marks.
     * 
     * @param g2 the graphics device.
     * @param cursor the cursor.
     * @param dataArea the data area.
     * @param edge the edge that the axis is aligned with.
     */
    protected void drawMinorTickMarks(final Graphics2D g2,
                                      final double cursor,
                                      final Rectangle2D dataArea,
                                      final RectangleEdge edge)
    {

        // minor tick can not have labels
        final double ol = minorTickMarkOutsideLength;
        final double il = minorTickMarkInsideLength;

        final List<Tick> ticks = refreshMinorTicksHorizontal();
        g2.setFont(getTickLabelFont());
        final Iterator<Tick> iterator = ticks.iterator();
        while(iterator.hasNext())
        {
            final ValueTick tick = (ValueTick)iterator.next();
            final float xx = (float)valueToJava2D(tick.getValue(), dataArea, edge);
            g2.setStroke(getTickMarkStroke());
            g2.setPaint(getTickMarkPaint());
            Line2D mark = null;
            if(edge == RectangleEdge.LEFT)
            {
                mark = new Line2D.Double(cursor - ol, xx, cursor + il, xx);
            }
            else if(edge == RectangleEdge.RIGHT)
            {
                mark = new Line2D.Double(cursor + ol, xx, cursor - il, xx);
            }
            else if(edge == RectangleEdge.TOP)
            {
                mark = new Line2D.Double(xx, cursor - ol, xx, cursor + il);
            }
            else if(edge == RectangleEdge.BOTTOM)
            {
                mark = new Line2D.Double(xx, cursor + ol, xx, cursor - il);
            }
            g2.draw(mark);
        }
    }

    /**
     * Recalculates the ticks for the date axis.
     * 
     * @param g2 the graphics device.
     * @param dataArea the area in which the data is to be drawn.
     * @param edge the location of the axis.
     * @return A list of ticks.
     */
    @Override
    protected List<Tick> refreshTicksHorizontal(final Graphics2D g2, final Rectangle2D dataArea, final RectangleEdge edge)
    {

        //XXX I'm not sure why this was included.  I commented out the lines below and don't see the exception they are talking about.
        //To avoid the FreeChart exception 'IllegalArgumentException: The 'year' argument must be in range 1900 to 9999.',
        //don't draw any time ticks  in the case of extreme zoom-out
//        if(super.getMinimumDate().getTime() < DateUtils.getTime(1901, 1, 1)
//            || super.getMaximumDate().getTime() > DateUtils.getTime(9000, 1, 1))
//        {
//            return new ArrayList();
//        }

        if(minorTickUnit != null)
            refreshMinorTicksHorizontal();
        return refreshTicksHorizontalOverride(g2, dataArea, edge);
    }

    /**
     * Recalculates the minor ticks for the date axis.
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public List<Tick> refreshMinorTicksHorizontal()
    {

        final List<Tick> result = new ArrayList<Tick>();

        final DateTickUnit unit = minorTickUnit;
        Date tickDate = calculateLowestVisibleTickValue(unit);
        final Date upperDate = calculateHighestVisibleTickValue(unit);
        while(tickDate.getTime() <= upperDate.getTime())
        {
            final Tick tick = new DateTick(tickDate, " ", TextAnchor.CENTER, TextAnchor.CENTER, 0);
            result.add(tick);
            tickDate = unit.addToDate(tickDate);
        }

        return result;
    }

    /**
     * Calculates the value of the lowest visible tick on the axis.
     * 
     * @param unit date unit to use.
     * @return The value of the lowest visible tick on the axis.
     */
    @Override
    public Date calculateLowestVisibleTickValue(final DateTickUnit unit)
    {
        final Date res = previousStandardDate(getMinimumDate(), unit);

        //After computing previousStandardDate, it is possible that the result is BEFORE getMinimumDate and would therefore
        //not be displayable on the axis. This may happen if the axis lower bound includes minutes and seconds, but the tick space
        //is in hours, for example.  If that is the case, then just push the tick mark one step forward and return
        //that instead.
        if(res.getTime() < getMinimumDate().getTime())
        {
            return nextStandardDate(getMinimumDate(), unit);
        }
        return res;
    }

    /**
     * Calculates the value of the highest visible tick on the axis.
     * 
     * @param unit date unit to use.
     * @return the value of the highest visible tick on the axis.
     */
    @Override
    public Date calculateHighestVisibleTickValue(final DateTickUnit unit)
    {
        final Date res = nextStandardDate(getMaximumDate(), unit);
        if(nextStandardDate(getMaximumDate(), unit).equals(getMaximumDate()))
            return res;
        return previousStandardDate(getMaximumDate(), unit);
    }

    public boolean isCenterLabelsBetweenTicks()
    {
        return centerLabelsBetweenTicks;
    }

    public void setCenterLabelsBetweenTicks(final boolean centerLabelsBetweenTicks)
    {
        this.centerLabelsBetweenTicks = centerLabelsBetweenTicks;
    }

    @Override
    protected double findMaximumTickLabelHeight(final List ticks,
                                                final Graphics2D g2,
                                                final Rectangle2D drawArea,
                                                final boolean vertical)
    {

        final RectangleInsets insets = getTickLabelInsets();
        final Font font = getTickLabelFont();
        double maxHeight = 0.0;
        if(vertical)
        {
            final FontMetrics fm = g2.getFontMetrics(font);
            final Iterator<?> iterator = ticks.iterator();
            while(iterator.hasNext())
            {
                final Tick tick = (Tick)iterator.next();
                final Rectangle2D labelBounds = TextUtilities.getTextBounds(tick.getText(), g2, fm);
                if(labelBounds.getWidth() + insets.getTop() + insets.getBottom() > maxHeight)
                {
                    maxHeight = labelBounds.getWidth() + insets.getTop() + insets.getBottom();
                }
            }
        }
        else
        {
            final LineMetrics metrics = font.getLineMetrics("ABCxyz", g2.getFontRenderContext());
            final int labelLineCount = getLabelLineCount(ticks);
            maxHeight = labelLineCount * metrics.getHeight() + (labelLineCount - 1) * metrics.getHeight() / 3
                + insets.getTop() + insets.getBottom();
        }
        return maxHeight;
    }

    protected static int getLabelLineCount(final List ticks)
    {
        int res = 1;
        final Iterator iterator = ticks.iterator();
        while(iterator.hasNext())
        {
            final Tick tick = (Tick)iterator.next();
            final String text = tick.getText();
//            final int lineCount = StringUtils.countMatches(text, "\n") + 1;  Apache commons solution.  Will not work in WRES due to version changes in commons.
            final int lineCount = CharMatcher.is('\n').countIn(text) + 1;
            if(lineCount > res)
                res = lineCount;
        }

        return res;
    }

    @Override
    protected float[] calculateAnchorPoint(final ValueTick tick,
                                           final double cursor,
                                           final Rectangle2D dataArea,
                                           final RectangleEdge edge)
    {

        if(!centerLabelsBetweenTicks)
        {
            return super.calculateAnchorPoint(tick, cursor, dataArea, edge);
        }

        final float[] result = new float[2];
        final double currentValue = tick.getValue();
        final double nextValue = getTickUnit().rollDate(new Date((long)currentValue)).getTime();
        final double value = (currentValue + nextValue) / 2;
        if(!getRange().contains(value))
        {
            result[0] = -1000;
            result[1] = -1000;
            return result;
        }

        if(edge == RectangleEdge.TOP)
        {
            result[0] = (float)valueToJava2D(value, dataArea, edge);
            result[1] = (float)(cursor - getTickLabelInsets().getBottom() - 2.0);
        }
        else if(edge == RectangleEdge.BOTTOM)
        {
            result[0] = (float)valueToJava2D(value, dataArea, edge);
            result[1] = (float)(cursor + getTickLabelInsets().getTop() + 2.0);
        }
        else if(edge == RectangleEdge.LEFT)
        {
            result[0] = (float)(cursor - getTickLabelInsets().getLeft() - 2.0);
            result[1] = (float)valueToJava2D(value, dataArea, edge);
        }
        else if(edge == RectangleEdge.RIGHT)
        {
            result[0] = (float)(cursor + getTickLabelInsets().getRight() + 2.0);
            result[1] = (float)valueToJava2D(value, dataArea, edge);
        }
        return result;
    }

    @Override
    protected void autoAdjustRange()
    {
        if(unzoomedRange != null)
        {
            setRange(unzoomedRange, false, false);
            return;
        }

        super.autoAdjustRange();
        final Range range = getRange();
        double lowerBound = range.getLowerBound();
        double upperBound = range.getUpperBound();
        if(minimumRange != null)
        {
            lowerBound = Math.min(lowerBound, minimumRange.getLowerBound());
            upperBound = Math.max(upperBound, minimumRange.getUpperBound());
        }

        if(maximumRange != null)
        {
            lowerBound = Math.max(lowerBound, maximumRange.getLowerBound());
            upperBound = Math.min(upperBound, maximumRange.getUpperBound());
        }

        setRange(new Range(lowerBound, upperBound), false, false);
    }

    /**
     * Returns the previous "standard" date, for a given date and tick unit. This has been significantly altered from
     * the original version.
     * 
     * @param date the reference date.
     * @param unit the tick unit.
     * @return The previous "standard" date.
     */
    @Override
    protected Date previousStandardDate(final Date date, final DateTickUnit unit)
    {
        final Calendar calendar = Calendar.getInstance(getTimeZone());
        calendar.setTime(date);
        final int count = unit.getCount();
        final int current = calendar.get(unit.getCalendarField());

        //tickStartHour, which is in GMT, needs to be adjusted to the time zone of the axis.  To do this, add
        //the time zone offset in hours + 24 (to ensure the number of hours is > 0) and then mod by 24.
        int adjustedTickStartHr = -1;
        if(tickStartHour >= 0)
        {
            adjustedTickStartHr = (tickStartHour
                + (int)(getTimeZone().getOffset(calendar.getTimeInMillis()) / HCalendar.MILLIS_IN_HR) + 24) % 24;
        }

        //TODO The old way of doing it yielded a prev std date that would be on a multiple of the 
        //count (7, for example for 1 week or 7 days).  So the std dates would be 7, 14, 21, 28, etc.
        //I don't like that.  So, I'm just going to use current.
        //int value = count * (current / count); 
        final int value = current;

        int years;
        int months;
        int days;
        int hours;
        int minutes;
        int seconds;
        int milliseconds;
        switch(unit.getUnit())
        {
            case DateTickUnit.MILLISECOND:
                years = calendar.get(Calendar.YEAR);
                months = calendar.get(Calendar.MONTH);
                days = calendar.get(Calendar.DATE);
                hours = calendar.get(Calendar.HOUR_OF_DAY);
                minutes = calendar.get(Calendar.MINUTE);
                seconds = calendar.get(Calendar.SECOND);
                calendar.set(years, months, days, hours, minutes, seconds);
                calendar.set(Calendar.MILLISECOND, value);
                Date mm = calendar.getTime();
                if(mm.getTime() > date.getTime())
                { // jfreechart-1.0.9 :  >=
                    calendar.set(Calendar.MILLISECOND, value - 1);
                    mm = calendar.getTime();
                }
                return mm;

            case DateTickUnit.SECOND:
                years = calendar.get(Calendar.YEAR);
                months = calendar.get(Calendar.MONTH);
                days = calendar.get(Calendar.DATE);
                hours = calendar.get(Calendar.HOUR_OF_DAY);
                minutes = calendar.get(Calendar.MINUTE);
                if(getTickMarkPosition() == DateTickMarkPosition.START)
                {
                    milliseconds = 0;
                }
                else if(getTickMarkPosition() == DateTickMarkPosition.MIDDLE)
                {
                    milliseconds = 500;
                }
                else
                {
                    milliseconds = 999;
                }
                calendar.set(Calendar.MILLISECOND, milliseconds);
                calendar.set(years, months, days, hours, minutes, value);
                Date dd = calendar.getTime();
                if(dd.getTime() > date.getTime())
                { // jfreechart-1.0.9 :  >=
                    calendar.set(Calendar.SECOND, value - 1);
                    dd = calendar.getTime();
                }
                return dd;

            case DateTickUnit.MINUTE:
                years = calendar.get(Calendar.YEAR);
                months = calendar.get(Calendar.MONTH);
                days = calendar.get(Calendar.DATE);
                hours = calendar.get(Calendar.HOUR_OF_DAY);
                if(getTickMarkPosition() == DateTickMarkPosition.START)
                {
                    seconds = 0;
                }
                else if(getTickMarkPosition() == DateTickMarkPosition.MIDDLE)
                {
                    seconds = 30;
                }
                else
                {
                    seconds = 59;
                }
                calendar.clear(Calendar.MILLISECOND);
                calendar.set(years, months, days, hours, value, seconds);
                Date d0 = calendar.getTime();
                if(d0.getTime() > date.getTime())
                { // jfreechart-1.0.9 :  >=
                    calendar.set(Calendar.MINUTE, value - 1);
                    d0 = calendar.getTime();
                }
                return d0;

            case DateTickUnit.HOUR:

                //Determine the adjusted value.  This starts with adjustedTickStartHr, which is the tick start hour adjusted for time zone.
                //It will then move it forward by the quantity/count value in the tick spacing in order to find the first step that is 
                //AFTER value (the current examined value) but within count steps of that value.  If count exceeds or equals 24, then the 
                //count part of the while clause will never be triggered and only one step at most will be performed in order to trigger the
                //first part of the while clause.  If count is less than 24, then there is guaranteed to be a solution to the while clause so
                //it should never be possible for it to go into an infinite loop.
                int adjustedValue = adjustedTickStartHr;
                while((adjustedValue < value) || (adjustedValue - value >= count))
                {
                    //If the adjusted value is after 24, then wrap it around and do the while checks again.
                    if(adjustedValue >= 24)
                    {
                        adjustedValue = adjustedValue % 24;
                    }
                    //Otherwise, add count to the adjusted value and do the while checks.
                    else
                    {
                        adjustedValue = adjustedValue + count;
                    }
                }
// Used to do this, which is just plain wrong... final int adjustedValue = value + (adjustedTickStartHr % count);

                years = calendar.get(Calendar.YEAR);
                months = calendar.get(Calendar.MONTH);
                days = calendar.get(Calendar.DATE);
                if(getTickMarkPosition() == DateTickMarkPosition.START)
                {
                    minutes = 0;
                    seconds = 0;
                }
                else if(getTickMarkPosition() == DateTickMarkPosition.MIDDLE)
                {
                    minutes = 30;
                    seconds = 0;
                }
                else
                {
                    minutes = 59;
                    seconds = 59;
                }
                calendar.clear(Calendar.MILLISECOND);

                //Wrap it around if necessary when the adjusted value made it to or after 24.  
                //This means we need to push to the next day; see if-clause below.
                boolean addOneToHandleWrap = false;
                if(adjustedValue >= 24)
                {
                    addOneToHandleWrap = true;
                    adjustedValue = adjustedValue % 24;
                }
                calendar.set(years, months, days, adjustedValue, minutes, seconds);
                if(addOneToHandleWrap)
                {
                    calendar.add(Calendar.DAY_OF_YEAR, 1);
                }
                final Date d1 = calendar.getTime();

                //System.err.println("####>> here -- " + HCalendar.buildDateTimeStr(calendar));
                //if(d1.getTime() > date.getTime())
                //{ // jfreechart-1.0.9 :  >=
                //    System.out.println("####>>     DOING THIS!  setting hour of day to " + (adjustedValue - 1));
                //    calendar.set(Calendar.HOUR_OF_DAY, adjustedValue - 1);
                //    d1 = calendar.getTime();
                //}
                return d1;

            case DateTickUnit.DAY:
                years = calendar.get(Calendar.YEAR);
                months = calendar.get(Calendar.MONTH);
                if(adjustedTickStartHr >= 0)
                {
                    hours = adjustedTickStartHr;
                    minutes = 0;
                    seconds = 0;
                }
                else if(getTickMarkPosition() == DateTickMarkPosition.START)
                {
                    hours = 0;
                    minutes = 0;
                    seconds = 0;
                }
                else if(getTickMarkPosition() == DateTickMarkPosition.MIDDLE)
                {
                    hours = 12;
                    minutes = 0;
                    seconds = 0;
                }
                else
                {
                    hours = 23;
                    minutes = 59;
                    seconds = 59;
                }
                calendar.clear(Calendar.MILLISECOND);
                calendar.set(years, months, value, hours, 0, 0);
                // long result = calendar.getTimeInMillis();
                // won't work with JDK 1.3
                Date d2 = calendar.getTime();
                if(d2.getTime() > date.getTime())
                { // jfreechart-1.0.9 :  >=
                    calendar.set(Calendar.DATE, value - 1);
                    d2 = calendar.getTime();
                }
                return d2;

            case DateTickUnit.MONTH:
                years = calendar.get(Calendar.YEAR);
                calendar.clear(Calendar.MILLISECOND);
                int startHour = 0;
                if(adjustedTickStartHr >= 0)
                {
                    startHour = adjustedTickStartHr;
                }
                calendar.set(years, value, 1, startHour, 0, 0);
                Month month = new Month(calendar.getTime(), getTimeZone());
                Date standardDate = calculateDateForPosition(month, getTickMarkPosition());
                final long millis = standardDate.getTime();
                if(millis > date.getTime())
                {
                    month = (Month)month.previous();
                    standardDate = calculateDateForPosition(month, getTickMarkPosition());
                }
                return standardDate;

            case DateTickUnit.YEAR:
                if(getTickMarkPosition() == DateTickMarkPosition.START)
                {
                    months = 0;
                    days = 1;
                }
                else if(getTickMarkPosition() == DateTickMarkPosition.MIDDLE)
                {
                    months = 6;
                    days = 1;
                }
                else
                {
                    months = 11;
                    days = 31;
                }
                calendar.clear(Calendar.MILLISECOND);
                startHour = 0;
                if(adjustedTickStartHr >= 0)
                {
                    startHour = adjustedTickStartHr;
                }
                calendar.set(value, months, days, startHour, 0, 0);
                Date d3 = calendar.getTime();
                if(d3.getTime() >= date.getTime())
                { // jfreechart-1.0.9 :  >=
                    calendar.set(Calendar.YEAR, value - 1);
                    d3 = calendar.getTime();
                }
                return d3;

            default:
                return null;

        }

    }

    /**
     * Returns a {@link Date} corresponding to the specified position within a {@link RegularTimePeriod}.
     * <p/>
     * NOTE: this is the same method as DateAxis.calculateDateForPosition, due to overruling of the method
     * previousStandardDate
     * 
     * @param period the period.
     * @param position the position (<code>null</code> not permitted).
     * @return A date.
     */
    private static Date calculateDateForPosition(final RegularTimePeriod period, final DateTickMarkPosition position)
    {

        if(position == null)
        {
            throw new IllegalArgumentException("Null 'position' argument.");
        }
        Date result = null;
        if(position == DateTickMarkPosition.START)
        {
            result = new Date(period.getFirstMillisecond());
        }
        else if(position == DateTickMarkPosition.MIDDLE)
        {
            result = new Date(period.getMiddleMillisecond());
        }
        else if(position == DateTickMarkPosition.END)
        {
            result = new Date(period.getLastMillisecond());
        }
        return result;

    }

    /**
     * Returns a collection of standard date tick units that either form a multiple of the TimeStepSize or of which the
     * TimeStepSize is a multiple.
     * 
     * @param zone the time zone (<code>null</code> not permitted).
     * @return A collection of standard date tick units.
     */
    public TickUnitSource createTimeStepDependantDateTickUnits(final TimeZone zone)
    {

        if(zone == null)
        {
            throw new IllegalArgumentException("Null 'zone' argument.");
        }
        final TickUnits units = new TickUnits();

        // date formatters
        final DateFormat f3 = new SimpleDateFormat("HH:mm");
        final DateFormat f4 = new SimpleDateFormat("d-MMM, HH:mm");
        final DateFormat f5 = new SimpleDateFormat("d-MMM");
        final DateFormat f6 = new SimpleDateFormat("MMM-yyyy");
        final DateFormat f7 = new SimpleDateFormat("yyyy");

        f3.setTimeZone(zone);
        f4.setTimeZone(zone);
        f5.setTimeZone(zone);
        f6.setTimeZone(zone);
        f7.setTimeZone(zone);

        // minutes
        DateTickUnit unit = new DateTickUnit(DateTickUnit.MINUTE, 1, DateTickUnit.SECOND, 5, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MINUTE, 2, DateTickUnit.SECOND, 10, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MINUTE, 5, DateTickUnit.MINUTE, 1, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MINUTE, 10, DateTickUnit.MINUTE, 1, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MINUTE, 15, DateTickUnit.MINUTE, 5, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MINUTE, 20, DateTickUnit.MINUTE, 5, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MINUTE, 30, DateTickUnit.MINUTE, 5, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);

        // hours
        unit = new DateTickUnit(DateTickUnit.HOUR, 1, DateTickUnit.MINUTE, 5, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.HOUR, 2, DateTickUnit.MINUTE, 10, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.HOUR, 4, DateTickUnit.MINUTE, 30, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.HOUR, 6, DateTickUnit.HOUR, 1, f3);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.HOUR, 12, DateTickUnit.HOUR, 1, f4);
        if(isValidDateTickUnit(unit))
            units.add(unit);

        // days
        unit = new DateTickUnit(DateTickUnit.DAY, 1, DateTickUnit.HOUR, 1, f5);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.DAY, 2, DateTickUnit.HOUR, 1, f5);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.DAY, 3, DateTickUnit.HOUR, 1, f5);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.DAY, 4, DateTickUnit.HOUR, 1, f5);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.DAY, 5, DateTickUnit.HOUR, 1, f5);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.DAY, 7, DateTickUnit.DAY, 1, f5);
        units.add(unit);
        unit = new DateTickUnit(DateTickUnit.DAY, 10, DateTickUnit.DAY, 1, f5);
        units.add(unit);
        unit = new DateTickUnit(DateTickUnit.DAY, 15, DateTickUnit.DAY, 1, f5);
        if(isValidDateTickUnit(unit))
            units.add(unit);

        // months
        unit = new DateTickUnit(DateTickUnit.MONTH, 1, DateTickUnit.DAY, 1, f6);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MONTH, 2, DateTickUnit.DAY, 1, f6);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MONTH, 3, DateTickUnit.MONTH, 1, f6);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MONTH, 4, DateTickUnit.MONTH, 1, f6);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.MONTH, 6, DateTickUnit.MONTH, 1, f6);
        if(isValidDateTickUnit(unit))
            units.add(unit);

        // years
        unit = new DateTickUnit(DateTickUnit.YEAR, 1, DateTickUnit.MONTH, 1, f7);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.YEAR, 2, DateTickUnit.MONTH, 3, f7);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.YEAR, 5, DateTickUnit.YEAR, 1, f7);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.YEAR, 10, DateTickUnit.YEAR, 1, f7);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.YEAR, 25, DateTickUnit.YEAR, 5, f7);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.YEAR, 50, DateTickUnit.YEAR, 10, f7);
        if(isValidDateTickUnit(unit))
            units.add(unit);
        unit = new DateTickUnit(DateTickUnit.YEAR, 100, DateTickUnit.YEAR, 20, f7);
        if(isValidDateTickUnit(unit))
            units.add(unit);

        return units;

    }

    /**
     * Override method for refreshing horizontal ticks that handles multiline tick labels.
     * 
     * @param g2 Graphics for drawing
     * @param dataArea Area of data.
     * @param edge Edge of the axis.
     * @return List of ticks.
     */
    @SuppressWarnings({"unchecked"})
    protected List<Tick> refreshTicksHorizontalOverride(final Graphics2D g2,
                                                  final Rectangle2D dataArea,
                                                  final RectangleEdge edge)
    {
        final List<Tick> result = new java.util.ArrayList<Tick>();

        final Font tickLabelFont = getTickLabelFont();
        g2.setFont(tickLabelFont);

        if(isAutoTickUnitSelection())
        {
            selectAutoTickUnit(g2, dataArea, edge);
        }

        final DateTickUnit unit = getTickUnit();

        Date tickDate = calculateLowestVisibleTickValue(unit);
        final Date upperDate = getMaximumDate();
        // float lastX = Float.MIN_VALUE;
        while(tickDate.before(upperDate))
        {
            if(!isHiddenValue(tickDate.getTime()))
            {
                // work out the value, label and position
                String tickLabel;
                final DateFormat formatter = getDateFormatOverride();
                if(formatter != null)
                {
                    formatter.setTimeZone(getTimeZone());

                    //TODO Turn off daylight savings time in axis tick marks.
                    //To turn off the daylight savings time for a time, just uncomment this code and use it in place
                    //of the above setTimeZone.  It creates a new SimpleTimeZone without any DST info in it.
                    //SimpleTimeZone simpleTZ = new SimpleTimeZone(getTimeZone().getRawOffset(), getTimeZone().getID());
                    //formatter.setTimeZone(simpleTZ);

                    tickLabel = formatter.format(tickDate);
                }
                else
                {
                    tickLabel = this.getTickUnit().dateToString(tickDate);
                }
                TextAnchor anchor = null;
                TextAnchor rotationAnchor = null;
                double angle = 0.0;
                if(isVerticalTickLabels())
                {
                    anchor = TextAnchor.CENTER_RIGHT;
                    rotationAnchor = TextAnchor.CENTER_RIGHT;
                    if(edge == RectangleEdge.TOP)
                    {
                        angle = Math.PI / 2.0;
                    }
                    else
                    {
                        angle = -Math.PI / 2.0;
                    }
                }
                else
                {
                    if(edge == RectangleEdge.TOP)
                    {
                        anchor = TextAnchor.BOTTOM_CENTER;
                        rotationAnchor = TextAnchor.BOTTOM_CENTER;
                    }
                    else
                    {
                        anchor = TextAnchor.TOP_CENTER;
                        rotationAnchor = TextAnchor.TOP_CENTER;
                    }
                }
                final Tick tick = new DateTick(tickDate, tickLabel, anchor, rotationAnchor, angle);
                result.add(tick);
                tickDate = unit.addToDate(tickDate, getTimeZone());
            }
            else
            {
                tickDate = unit.rollDate(tickDate, getTimeZone());
                continue;
            }

            //This appears to have been added by Deltares.
            //Hank (6/7/2010): Added getTimeZone() parameters to the Month and Year constructors.  Without it, the ticks
            //do not appropriate account for time zone.
            // could add a flag to make the following correction optional...
            switch(unit.getUnit())
            {
                case (DateTickUnit.MILLISECOND):
                case (DateTickUnit.SECOND):
                case (DateTickUnit.MINUTE):
                case (DateTickUnit.HOUR):
                case (DateTickUnit.DAY):
                    break;
                case (DateTickUnit.MONTH):
                    tickDate = calculateDateForPosition(new Month(tickDate, getTimeZone()), this.getTickMarkPosition());
                    break;
                case (DateTickUnit.YEAR):
                    tickDate = calculateDateForPosition(new Year(tickDate, getTimeZone()), this.getTickMarkPosition());
                    break;

                default:
                    break;

            }

        }
        return result;

    }

    /**
     * Override of method allows me to use my own estimateMaximumTickLabelWidthOverride method.
     */
    @Override
    protected void selectHorizontalAutoTickUnit(final Graphics2D g2,
                                                final Rectangle2D dataArea,
                                                final RectangleEdge edge)
    {

        long shift = 0;
        if(this.getTimeline() instanceof SegmentedTimeline)
        {
            shift = ((SegmentedTimeline)this.getTimeline()).getStartTime();
        }
        final double zero = valueToJava2D(shift + 0.0, dataArea, edge);
        double tickLabelWidth = estimateMaximumTickLabelWidthOverride(g2, getTickUnit());

        // start with the current tick unit...
        final TickUnitSource tickUnits = getStandardTickUnits();
        final TickUnit unit1 = tickUnits.getCeilingTickUnit(getTickUnit());
        final double x1 = valueToJava2D(shift + unit1.getSize(), dataArea, edge);
        final double unit1Width = Math.abs(x1 - zero);

        // then extrapolate...
        final double guess = (tickLabelWidth / unit1Width) * unit1.getSize();
        DateTickUnit unit2 = (DateTickUnit)tickUnits.getCeilingTickUnit(guess);
        final double x2 = valueToJava2D(shift + unit2.getSize(), dataArea, edge);
        final double unit2Width = Math.abs(x2 - zero);
        tickLabelWidth = estimateMaximumTickLabelWidthOverride(g2, unit2);
        if(tickLabelWidth > unit2Width)
        {
            unit2 = (DateTickUnit)tickUnits.getLargerTickUnit(unit2);
        }
        setTickUnit(unit2, false, false);
    }

    /**
     * Estimate the maximum width of any tick label. This is copied from DateAxis and calls calculateLargestLineWidth,
     * because the version in DateAxis does not handle multiline labels correctly (I ignores the new-line and treats it
     * as one long string).
     * 
     * @param g2
     * @param unit
     * @return
     */
    private double estimateMaximumTickLabelWidthOverride(final Graphics2D g2, final DateTickUnit unit)
    {

        final RectangleInsets tickLabelInsets = getTickLabelInsets();
        double result = tickLabelInsets.getLeft() + tickLabelInsets.getRight();

        final Font tickLabelFont = getTickLabelFont();
        final FontRenderContext frc = g2.getFontRenderContext();
        final LineMetrics lm = tickLabelFont.getLineMetrics("ABCxyz", frc);
        if(isVerticalTickLabels())
        {
            // all tick labels have the same width (equal to the height of
            // the font)...
            result += lm.getHeight();
        }
        else
        {
            // look at lower and upper bounds...
            final DateRange range = (DateRange)getRange();
            final Date lower = range.getLowerDate();
            final Date upper = range.getUpperDate();
            String lowerStr = null;
            String upperStr = null;
            final DateFormat formatter = getDateFormatOverride();
            if(formatter != null)
            {
                lowerStr = formatter.format(lower);
                upperStr = formatter.format(upper);
            }
            else
            {
                lowerStr = unit.dateToString(lower);
                upperStr = unit.dateToString(upper);
            }
            final FontMetrics fm = g2.getFontMetrics(tickLabelFont);
            final double w1 = calculateLargestLineWidth(fm, lowerStr);
            final double w2 = calculateLargestLineWidth(fm, upperStr);
            //double w1 = fm.stringWidth(lowerStr);
            //double w2 = fm.stringWidth(upperStr);

            result += Math.max(w1, w2);
        }

        return result;

    }

    /**
     * Return the width of the longest line of text using the given font metrics.
     * 
     * @param fm FontMetrics to use.
     * @param string String that may include \n.
     * @return
     */
    private int calculateLargestLineWidth(final FontMetrics fm, final String string)
    {
        final StringTokenizer tokens = new StringTokenizer(string, "\n");
        int maxWidth = 0;
        while(tokens.hasMoreElements())
        {
            maxWidth = Math.max(maxWidth, fm.stringWidth(tokens.nextToken()));
        }
        return maxWidth;
    }
}
