package ohd.hseb.charter.tools;

import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;
import java.text.NumberFormat;
import java.util.List;

import ohd.hseb.hefs.utils.dist.types.NormalDist;

import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberTick;
import org.jfree.chart.axis.NumberTickUnit;
import org.jfree.chart.axis.Tick;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.data.Range;
import org.jfree.ui.RectangleEdge;
import org.jfree.ui.TextAnchor;

public class NormalizedProbabilityAxis extends NumberAxis
{
    public final static double[] PROBABILITIES = {0.001, 0.005, 0.01, 0.02, 0.05, 0.10, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7,
        0.8, 0.9, 0.95, 0.98, 0.99, 0.995, 0.999};
    public final static double INITIAL_LOWER_BOUND = 0.01d;
    public final static double INITIAL_UPPER_BOUND = 0.99d;

    private static final long serialVersionUID = 1L;

    private final NormalDist _standardNormalDist = new NormalDist();

    public NormalizedProbabilityAxis(final String label)
    {
        super(label);

//        this.setAutoRange(false);
//        this.setUpperBound(INITIAL_UPPER_BOUND);
//        this.setLowerBound(INITIAL_LOWER_BOUND);
//        this.setTickUnit(new NumberTickUnit(0.01));
    }

    @Override
    protected void autoAdjustRange()
    {
        this.setUpperBound(INITIAL_UPPER_BOUND);
        this.setLowerBound(INITIAL_LOWER_BOUND);
    }

    @Override
    public double valueToJava2D(final double value, final Rectangle2D area, final RectangleEdge edge)
    {
        final double convertedValue = _standardNormalDist.functionInverseCDF(value);

        final Range range = getRange();
        final double axisMin = range.getLowerBound();
        final double axisMax = range.getUpperBound();

        final double convertedAxisMin = _standardNormalDist.functionInverseCDF(axisMin);
        final double convertedAxisMax = _standardNormalDist.functionInverseCDF(axisMax);

        //min and max store the plotted min and max pixels, I believe.
        double min = 0.0;
        double max = 0.0;
        if(RectangleEdge.isTopOrBottom(edge))
        {
            min = area.getX();
            max = area.getMaxX();
        }
        else if(RectangleEdge.isLeftOrRight(edge))
        {
            max = area.getMinY();
            min = area.getMaxY();
        }

        if(this.isInverted())
        {
            return max - ((convertedValue - convertedAxisMin) / (convertedAxisMax - convertedAxisMin)) * (max - min);
        }
        else
        {
            return min + ((convertedValue - convertedAxisMin) / (convertedAxisMax - convertedAxisMin)) * (max - min);
        }

    }

    @Override
    public double java2DToValue(final double java2DValue, final Rectangle2D area, final RectangleEdge edge)
    {
        final Range range = getRange();
        final double axisMin = range.getLowerBound();
        final double axisMax = range.getUpperBound();

        final double convertedAxisMin = _standardNormalDist.functionInverseCDF(axisMin);
        final double convertedAxisMax = _standardNormalDist.functionInverseCDF(axisMax);

        double min = 0.0;
        double max = 0.0;
        if(RectangleEdge.isTopOrBottom(edge))
        {
            min = area.getX();
            max = area.getMaxX();
        }
        else if(RectangleEdge.isLeftOrRight(edge))
        {
            min = area.getMaxY();
            max = area.getY();
        }

        double valueInNormalSpace;
        if(this.isInverted())
        {
            valueInNormalSpace = convertedAxisMax - (java2DValue - min) / (max - min)
                * (convertedAxisMax - convertedAxisMin);
        }
        else
        {
            valueInNormalSpace = convertedAxisMin + (java2DValue - min) / (max - min)
                * (convertedAxisMax - convertedAxisMin);
        }

        return _standardNormalDist.functionCDF(valueInNormalSpace);
    }
    
    @Override
    protected List<Tick> refreshTicksHorizontal(final Graphics2D g2, final Rectangle2D dataArea, final RectangleEdge edge)
    {
        final List<Tick> result = new java.util.ArrayList<Tick>();

        final Font tickLabelFont = getTickLabelFont();
        g2.setFont(tickLabelFont);

        if(isAutoTickUnitSelection())
        {
            selectAutoTickUnit(g2, dataArea, edge);
        }

        final int count = PROBABILITIES.length;

        if(count <= ValueAxis.MAXIMUM_TICK_COUNT)
        {
            for(int i = 0; i < count; i++)
            {
                final double currentTickValue = PROBABILITIES[i];

                //If the tick is outside of the bounds, do not show it.
                if((currentTickValue < this.getLowerBound()) || (currentTickValue > this.getUpperBound()))
                {
                    continue;
                }

                String tickLabel;
                final NumberFormat formatter = getNumberFormatOverride();
                if(formatter != null)
                {
                    tickLabel = formatter.format(currentTickValue);
                }
                else
                {
                    tickLabel = getTickUnit().valueToString(currentTickValue);
                }
                TextAnchor anchor = null;
                TextAnchor rotationAnchor = null;
                double angle = 0.0;
                if(isVerticalTickLabels())
                {
                    anchor = TextAnchor.CENTER_RIGHT;
                    rotationAnchor = TextAnchor.CENTER_RIGHT;
                    if(edge == RectangleEdge.TOP)
                    {
                        angle = Math.PI / 2.0;
                    }
                    else
                    {
                        angle = -Math.PI / 2.0;
                    }
                }
                else
                {
                    if(edge == RectangleEdge.TOP)
                    {
                        anchor = TextAnchor.BOTTOM_CENTER;
                        rotationAnchor = TextAnchor.BOTTOM_CENTER;
                    }
                    else
                    {
                        anchor = TextAnchor.TOP_CENTER;
                        rotationAnchor = TextAnchor.TOP_CENTER;
                    }
                }

                final Tick tick = new NumberTick(Double.valueOf(currentTickValue), tickLabel, anchor, rotationAnchor, angle);
                result.add(tick);
            }
        }
        return result;

    }

    @Override
    protected List<Tick> refreshTicksVertical(final Graphics2D g2, final Rectangle2D dataArea, final RectangleEdge edge)
    {
        final List<Tick> result = new java.util.ArrayList<Tick>();
        result.clear();

        final Font tickLabelFont = getTickLabelFont();
        g2.setFont(tickLabelFont);

        if(isAutoTickUnitSelection())
        {
            selectAutoTickUnit(g2, dataArea, edge);
        }

        final int count = PROBABILITIES.length;

        if(count <= ValueAxis.MAXIMUM_TICK_COUNT)
        {
            for(int i = 0; i < count; i++)
            {
                final double currentTickValue = PROBABILITIES[i];

                //If the tick is outside of the bounds, do not show it.
                if((currentTickValue < this.getLowerBound()) || (currentTickValue > this.getUpperBound()))
                {
                    continue;
                }

                String tickLabel;
                final NumberFormat formatter = getNumberFormatOverride();
                if(formatter != null)
                {
                    tickLabel = formatter.format(currentTickValue);
                }
                else
                {
                    tickLabel = getTickUnit().valueToString(currentTickValue);
                }

                TextAnchor anchor = null;
                TextAnchor rotationAnchor = null;
                double angle = 0.0;
                if(isVerticalTickLabels())
                {
                    if(edge == RectangleEdge.LEFT)
                    {
                        anchor = TextAnchor.BOTTOM_CENTER;
                        rotationAnchor = TextAnchor.BOTTOM_CENTER;
                        angle = -Math.PI / 2.0;
                    }
                    else
                    {
                        anchor = TextAnchor.BOTTOM_CENTER;
                        rotationAnchor = TextAnchor.BOTTOM_CENTER;
                        angle = Math.PI / 2.0;
                    }
                }
                else
                {
                    if(edge == RectangleEdge.LEFT)
                    {
                        anchor = TextAnchor.CENTER_RIGHT;
                        rotationAnchor = TextAnchor.CENTER_RIGHT;
                    }
                    else
                    {
                        anchor = TextAnchor.CENTER_LEFT;
                        rotationAnchor = TextAnchor.CENTER_LEFT;
                    }
                }

                final Tick tick = new NumberTick(Double.valueOf(currentTickValue), tickLabel, anchor, rotationAnchor, angle);
                result.add(tick);
            }
        }
        return result;

    }

    @Override
    protected void selectHorizontalAutoTickUnit(final Graphics2D g2,
                                                final Rectangle2D dataArea,
                                                final RectangleEdge edge)
    {
        selectVerticalAutoTickUnit(g2, dataArea, edge);
    }

    @Override
    protected void selectVerticalAutoTickUnit(final Graphics2D g2, final Rectangle2D dataArea, final RectangleEdge edge)
    {
        if((this.getLowerBound() < 0.01) || (this.getUpperBound() > 0.99))
        {
            this.setTickUnit(new NumberTickUnit(0.001), false, false);
        }
        else if((this.getLowerBound() < 0.1) || (this.getUpperBound() > 0.9))
        {
            this.setTickUnit(new NumberTickUnit(0.01), false, false);
        }
        else
        {
            this.setTickUnit(new NumberTickUnit(0.1), false, false);
        }
    }

}
