package ohd.hseb.charter.tools;

import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;

import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberTick;
import org.jfree.data.Range;
import org.jfree.ui.RectangleEdge;
import org.jfree.ui.TextAnchor;

import ohd.hseb.hefs.utils.xml.vars.XMLString;

/**
 * Overrides NumberAxis with these changes:<br>
 * <br>
 * 1. If the upper and lower bound are equal (or virtually equal.. within 0.000001 of each other), then the range is
 * artificially set to the lower bound +/- 0.5 so that the ticks are reasonable.<br>
 * <br>
 * 
 * @author hank.herr
 */
public class NumberAxisOverride extends NumberAxis
{
    private static final long serialVersionUID = 1L;

    /**
     * For WRES.  This will store a list of the category labels.  It is assumed that the number axis value corresponding to a
     * label is (index in this list + 1).  Thus, the list must be provided in appropriate order.
     */
    private final List<XMLString> _categorylabels = new ArrayList<>();

    public NumberAxisOverride(final String label)
    {
        super(label);
    }
    
    public boolean isCategorical()
    {
        return ((_categorylabels != null) && !_categorylabels.isEmpty());
    }

    public void setCategoricalLabels(final List<XMLString> categorylabels)
    {
        _categorylabels.clear();
        _categorylabels.addAll(categorylabels);
    }
    
    public void setCategoricalLabels(final String[] labels)
    {
        _categorylabels.clear();
        for (final String label : labels)
        {
            _categorylabels.add(new XMLString("label", label));
        }
    }
    
    public double getXValueForCategory(final String category)
    {
        for (int i = 0; i < _categorylabels.size(); i ++)
        {
            if (_categorylabels.get(i).get().equals(category))
            {
                return i + 1;
            }
        }
        return -1;
    }
    
    public String getCategoryForXValue(final double number)
    {
        final long catIndex = Math.round(number) - 1;
        if (catIndex >= 0 && catIndex < _categorylabels.size() )
        {
            return _categorylabels.get((int)catIndex).get();
        }
        return "No Category";
    }

    /**
     * Created for WRES to allow for using a number axis to display categories of a categorical plot.
     * 
     * @param edge Passed through from the refresh method calling this.
     * @return List of {@link NumberTick} instances, each with a label set according to the category labels. The list
     *         index is the numerical value inside the tick.
     */
    private List<NumberTick> buildCategoryLabelTicks(final RectangleEdge edge)
    {
        final List<NumberTick> ticks = new ArrayList<>();
        int index = 0;
        for(final XMLString label: _categorylabels)
        {
            index++;
            TextAnchor anchor = null;
            TextAnchor rotationAnchor = null;
            double angle = 0.0;
            if(isVerticalTickLabels())
            {
                anchor = TextAnchor.CENTER_RIGHT;
                rotationAnchor = TextAnchor.CENTER_RIGHT;
                if(edge == RectangleEdge.TOP)
                {
                    angle = Math.PI / 2.0;
                }
                else
                {
                    angle = -Math.PI / 2.0;
                }
            }
            else
            {
                if(edge == RectangleEdge.TOP)
                {
                    anchor = TextAnchor.BOTTOM_CENTER;
                    rotationAnchor = TextAnchor.BOTTOM_CENTER;
                }
                else
                {
                    anchor = TextAnchor.TOP_CENTER;
                    rotationAnchor = TextAnchor.TOP_CENTER;
                }
            }
            final NumberTick tick = new NumberTick(index, label.get(), anchor, rotationAnchor, angle);
            ticks.add(tick);
        }
        return ticks;
    }

    @Override
    protected List refreshTicksVertical(final Graphics2D g2, final Rectangle2D dataArea, final RectangleEdge edge)
    {
        if(isCategorical())
        {
            return buildCategoryLabelTicks(edge);
        }
        if(isAutoRange() && (this.getUpperBound() - this.getLowerBound() < 0.000001))
        {
            setRange(new Range(getLowerBound() - 0.5, getLowerBound() + 0.5), false, false);
        }
        return super.refreshTicksVertical(g2, dataArea, edge);
    }

    @Override
    protected List refreshTicksHorizontal(final Graphics2D g2, final Rectangle2D dataArea, final RectangleEdge edge)
    {
        if(isCategorical())
        {
            return buildCategoryLabelTicks(edge);
        }
        if(isAutoRange() && (this.getUpperBound() - this.getLowerBound() < 0.000001))
        {
            setRange(new Range(getLowerBound() - 0.5, getLowerBound() + 0.5), false, false);
        }
        return super.refreshTicksHorizontal(g2, dataArea, edge);
    }
}
