package ohd.hseb.charter.tools;

import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;
import java.util.List;

import org.jfree.chart.axis.AxisState;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.NumberTick;
import org.jfree.chart.axis.Tick;
import org.jfree.chart.axis.TickUnit;
import org.jfree.data.Range;
import org.jfree.ui.RectangleEdge;

import ohd.hseb.charter.parameters.AxisParameters;
import ohd.hseb.charter.translator.AxisTranslator;
import ohd.hseb.charter.translator.AxisTranslatorException;
import ohd.hseb.charter.translator.AxisTranslatorFactory;
import ohd.hseb.charter.translator.AxisTranslatorParameters;

/**
 * An instance of {@link NumberAxis} that applies an {@link AxisTranslator} to build tick labels both for axis spacing
 * and label drawing. The axis exactly mimicks the other axis for its bounds, redirecting calls related to lower and
 * upper bounds, and the range, to the other axis. Thus, this is an exact mirror of the other axis, except that the
 * labels for ticks are computed based on the translation.<br>
 * <br>
 * The flag {@link #_erroneousAxis} indicates if the axis is an error axis, implying that it should not be visible in
 * the plot. That is handled in the {@link AxisParameters} method {@link AxisParameters#applyParametersToChart(Object)}.
 * 
 * @author Hank.Herr
 */
public class TranslatedAxis extends NumberAxis
{

    private static final long serialVersionUID = 1L;

    private final NumberAxis _otherAxis;

    private final AxisTranslator _translator;

    private TickUnit _tickUnit = null;

    private boolean _erroneousAxis = false;

    public TranslatedAxis(final String label,
                          final NumberAxis otherAxis,
                          final AxisTranslatorParameters translatorParameters) throws AxisTranslatorException
    {
        super(label);
        this.setAutoRange(false);
        _otherAxis = otherAxis;

        _translator = AxisTranslatorFactory.loadTranslator(translatorParameters.getTranslatorName());
        _translator.setParameters(translatorParameters);
        _translator.checkParametersForValidity();
    }

    public void setErroneousAxis(final boolean b)
    {
        this._erroneousAxis = b;
    }

    public boolean getErroneousAxis()
    {
        return this._erroneousAxis;
    }

    public AxisTranslator getTranslator()
    {
        return this._translator;
    }

    @Override
    public double getLowerBound()
    {
        return _otherAxis.getLowerBound();
    }

    @Override
    public Range getRange()
    {
        // TODO Auto-generated method stub
        return _otherAxis.getRange();
    }

    @Override
    public double getUpperBound()
    {
        return _otherAxis.getUpperBound();
    }

    @Override
    public double valueToJava2D(final double value, final Rectangle2D area, final RectangleEdge edge)
    {

        return _otherAxis.valueToJava2D(value, area, edge);
    }

    @Override
    public double java2DToValue(final double java2DValue, final Rectangle2D area, final RectangleEdge edge)
    {
        final double originalValue = _otherAxis.java2DToValue(java2DValue, area, edge);
        return originalValue;
    }

    /**
     * Draws the translated ticks.
     */
    @SuppressWarnings("unchecked")
    @Override
    public List refreshTicks(final Graphics2D g2,
                             final AxisState state,
                             final Rectangle2D dataArea,
                             final RectangleEdge edge)
    {
        final List results = _otherAxis.refreshTicks(g2, state, dataArea, edge);
        final double[] originalValues = new double[results.size()];

        //Populate the original values array.
        for(int i = 0; i < originalValues.length; i++)
        {
            originalValues[i] = ((NumberTick)results.get(i)).getValue();
        }

        //Translate the original values array.
        double[] translatedTickValues = _translator.computeTranslatedValue(originalValues);
        /* Start FB 1835 */
//        double[] translatedValues = _translator.computeTranslatedRawValue(originalValues);

        if(translatedTickValues == null)
        {
            translatedTickValues = originalValues;
        }
        /* End FB 1835 */

//        if(translatedValues == null)
//        {
//            translatedValues = originalValues;
//        }

        //Setup the auto tick units.
        if(this._tickUnit == null)
        {
            _tickUnit = _otherAxis.getTickUnit();
        }

        String tickLabel;
        for(int i = 0; i < results.size(); i++)
        {
            final NumberTick originalTick = (NumberTick)results.get(i);

            //Process the number format override
            if(this.getNumberFormatOverride() != null)
            {
                tickLabel = this.getNumberFormatOverride().format(translatedTickValues[i]);
            }
            else
            {
                tickLabel = _tickUnit.valueToString(translatedTickValues[i]);
            }

            @SuppressWarnings("serial")
            final Tick tick = new NumberTick(originalValues[i],
                                             tickLabel,
                                             originalTick.getTextAnchor(),
                                             originalTick.getRotationAnchor(),
                                             originalTick.getAngle())
            {
                @Override
                public double getValue()
                {
                    return super.getValue();
                }
            };

            results.set(i, tick);
        }
        return results;
    }

    /**
     * @return A {@link TranslatedAxis} instance with its {@link TranslatedAxis#_erroneousAxis} flag set to true. The
     *         label on the axis will be set to an 'Error Loading Translator...' label, but will otherwise not appear
     *         anywhere.
     */
    public static TranslatedAxis buildDefaultErrorTranslatedAxis(final String label,
                                                                 final NumberAxis otherAxis,
                                                                 final AxisTranslatorParameters badParameters)
    {
        final AxisTranslatorParameters parms = new AxisTranslatorParameters();
        parms.setTranslatorName("Linear");
        try
        {
            final TranslatedAxis axis = new TranslatedAxis("Error Loading Translator "
                + badParameters.getTranslatorName(), otherAxis, parms);
            axis.setErroneousAxis(true);
            return axis;
        }
        catch(final AxisTranslatorException e)
        {
            //This should never happen.
            e.printStackTrace();
            return null;
        }
    }

}
