package ohd.hseb.charter.translator;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import ohd.hseb.hefs.utils.plugins.DefaultFactory;
import ohd.hseb.hefs.utils.plugins.DefaultFactoryException;

public class AxisTranslatorFactory
{

    private static HashMap<String, String> REGISTERED_NAME_TO_CLASS_NAME_MAP = null;

    public static void initializeAxisTranslatorFactoryIfNeeded() throws AxisTranslatorException
    {
        if(REGISTERED_NAME_TO_CLASS_NAME_MAP == null)
        {
            REGISTERED_NAME_TO_CLASS_NAME_MAP = new HashMap<String, String>();

            final DefaultFactory defaultFactory = new DefaultFactory("ohd.hseb.charter.translator.plugins",
                                                                     "AxisTranslator");
            try
            {
                @SuppressWarnings("unchecked")
                final List<String> classNames = defaultFactory.obtainListOfClassesWithSuffixInDefaultPackage();
                for(int i = 0; i < classNames.size(); i++)
                {
                    final AxisTranslator trans = (AxisTranslator)defaultFactory.loadInstance(classNames.get(i));
                    REGISTERED_NAME_TO_CLASS_NAME_MAP.put(trans.getName(), trans.getClass().getName());
                }
            }
            catch(final DefaultFactoryException e)
            {
                throw new AxisTranslatorException("Unable to load default translators: " + e.getMessage());
            }

        }
    }

    public static void clearRegistry()
    {
        REGISTERED_NAME_TO_CLASS_NAME_MAP = null;
    }

    public static void registerTranslator(final String name, final String className)
    {
        try
        {
            initializeAxisTranslatorFactoryIfNeeded();
        }
        catch(final AxisTranslatorException e)
        {
        }
        REGISTERED_NAME_TO_CLASS_NAME_MAP.put(name, className);
    }

    public static AxisTranslator loadTranslator(final String name) throws AxisTranslatorException
    {
        initializeAxisTranslatorFactoryIfNeeded();
        final String className = REGISTERED_NAME_TO_CLASS_NAME_MAP.get(name);
        if(className == null)
        {
            throw new AxisTranslatorException("Translator with name '" + name + "' is not registered as a translator.");
        }

        try
        {
            final Class<?> transClass = Class.forName(className);
            final AxisTranslator trans = (AxisTranslator)transClass.getDeclaredConstructor().newInstance();
            return trans;
        }
        catch(final ClassNotFoundException cnfe)
        {
            throw new AxisTranslatorException("Translator with name '" + name + "' is registered by the class name '"
                + className + "' is not found.");
        }
        catch(final InstantiationException e)
        {
            throw new AxisTranslatorException("Cannot instantiate plug-in class " + className + ": " + e.getMessage());
        }
        catch(final IllegalAccessException e)
        {
            throw new AxisTranslatorException("Cannot access plug-in class " + className + ": " + e.getMessage());
        } 
        catch (IllegalArgumentException e) 
        {
			throw new AxisTranslatorException("Cannot access plug-in class " + className + ": " + e.getMessage());
		} 
        catch (InvocationTargetException e) 
        {
			throw new AxisTranslatorException("Cannot access plug-in class " + className + ": " + e.getMessage());
		} 
        catch (NoSuchMethodException e) 
        {			
			throw new AxisTranslatorException("Cannot access plug-in class " + className + ": " + e.getMessage());
		} 
        catch (SecurityException e) 
        {
			throw new AxisTranslatorException("Cannot access plug-in class " + className + ": " + e.getMessage());			
		}
    }

    public static List<String> getListOfRegisteredTranslators()
    {
        try
        {
            initializeAxisTranslatorFactoryIfNeeded();
        }
        catch(final AxisTranslatorException e)
        {
        }

        final List<String> results = new ArrayList<String>();
        results.addAll(REGISTERED_NAME_TO_CLASS_NAME_MAP.keySet());
        return results;
    }

}
