package ohd.hseb.charter.translator;

import java.util.List;

import ohd.hseb.charter.ChartParameters;
import ohd.hseb.charter.parameters.ChartParametersException;
import ohd.hseb.charter.translator.plugins.LinearAxisTranslator;
import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.plugins.GeneralPlugInParameters;
import ohd.hseb.hefs.utils.plugins.GenericParameterList;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

/**
 * Specifies all parameters that are available to any axis translators. Specifically, this includes the translator name,
 * ArgumentsProcessor, and a list of parameter units identifying the units plotted along the other axis.
 * 
 * @author herrhd
 */
public class AxisTranslatorParameters extends GenericParameterList implements ChartParameters
{

    private String _translatorName = "Linear";

    private ArgumentsProcessor _arguments = null;

    private List<String> _otherAxisUnits = null;

    public AxisTranslatorParameters()
    {
        //Pass this tag name into _parameters, because I am going to use it to output parameters to XML.
        super();
        setupForDefaultLinearTranslator();
        setXMLTagName("axisTranslator");
    }

    private void setupForDefaultLinearTranslator()
    {
        _translatorName = "Linear";
        addParameter(LinearAxisTranslator.COEFF_PARAMETER, "1.0");
        addParameter(LinearAxisTranslator.CONSTANT_PARAMETER, "0.0");
    }

    public void setTranslatorName(final String name)
    {
        _translatorName = name;
    }

    public String getTranslatorName()
    {
        return _translatorName;
    }

    public void clearTranslatorName()
    {
        _translatorName = "Linear";
    }

    public void setOtherAxisUnits(final List<String> units)
    {
        this._otherAxisUnits = units;
    }

    public List<String> getOtherAxisUnits()
    {
        return this._otherAxisUnits;
    }

    public String getSingleOtherAxisUnitsIfOnlyOneDefined(final AxisTranslator translator)
    {
        if(_otherAxisUnits.size() == 0)
        {
            return null;
        }
        final String value = translator.translateUnitIfNecessary(_otherAxisUnits.get(0));
        for(int i = 1; i < _otherAxisUnits.size(); i++)
        {
            if(!translator.translateUnitIfNecessary(_otherAxisUnits.get(i)).equalsIgnoreCase(value))
            {
                return null;
            }
        }
        return value;
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        //For the main element, just process its text attribute.
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
            final String name = attr.getValue("name");
            if(name == null)
            {
                throw new XMLReaderException("Missing name attribute within " + getXMLTagName() + " tag.");
            }
            this._translatorName = name;
        }
        return super.readInPropertyFromXMLElement(elementName, attr);
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = super.writePropertyToXMLElement(request);
        mainElement.setAttribute("name", _translatorName);
        return mainElement;
    }

    @Override
    public String toString()
    {
        String result = "AxisTranslatorParameters: ";
        result += "translatorName = '" + _translatorName + "'; ";
        result += "xmlTag = '" + this.getXMLTagName() + "'; ";
        for(int i = 0; i < this.getParameters().size(); i++)
        {
            result += "[" + getParameters().get(i).toString() + "]";
        }
        result += ".";
        return result;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof AxisTranslatorParameters))
        {
            return false;
        }
        final AxisTranslatorParameters other = (AxisTranslatorParameters)obj;

        if(!super.equals(obj))
        {
            return false;
        }
        if(!this.getTranslatorName().equals(other.getTranslatorName()))
        {
            return false;
        }
        return true;
    }

    @Override
    public Object clone()
    {
        final AxisTranslatorParameters results = new AxisTranslatorParameters();
        results.copyFrom(this);
        return results;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        final AxisTranslatorParameters base = (AxisTranslatorParameters)parameters;
        this._translatorName = base.getTranslatorName();
        this.setArguments(base.getArguments());
        super.copyFrom(parameters);
    }

    @Override
    public void applyParametersToChart(final Object objectAppliedTo) throws ChartParametersException
    {
    }

    @Override
    public void copyOverriddenParameters(final ChartParameters override)
    {
        copyFrom(override);
    }

    @Override
    public void haveAllParametersBeenSet() throws ChartParametersException
    {
    }

    @Override
    public void setupDefaultParameters()
    {
        _translatorName = "Linear";
    }

    @Override
    public void setArguments(final ArgumentsProcessor arguments)
    {
        _arguments = arguments;
    }

    @Override
    public ArgumentsProcessor getArguments()
    {
        return _arguments;
    }

}
