package ohd.hseb.charter.translator.plugins;

import ohd.hseb.charter.parameters.AxisParameters;
import ohd.hseb.charter.parameters.panels.ChartParametersPanel;
import ohd.hseb.charter.translator.AxisTranslatorException;
import ohd.hseb.charter.translator.AxisTranslatorParameters;
import ohd.hseb.charter.translator.DefaultAxisTranslator;

public class LinearAxisTranslator extends DefaultAxisTranslator
{

    public final static String COEFF_PARAMETER = "coefficient";
    public final static String CONSTANT_PARAMETER = "constant";

    public final static String[] REQUIRED_PARAMETERS = {COEFF_PARAMETER, CONSTANT_PARAMETER};
    public final static String[] USED_PARAMETERS = {COEFF_PARAMETER, CONSTANT_PARAMETER};

    public LinearAxisTranslator()
    {
        this.initializeRequiredAndUsedParameters();
    }

    private double getCoeff()
    {
        return this.getParameters().getParameterWithName(COEFF_PARAMETER).getValueAsDouble();
    }

    private double getConstant()
    {
        return this.getParameters().getParameterWithName(CONSTANT_PARAMETER).getValueAsDouble();
    }

    @Override
    public void checkParametersForValidity() throws AxisTranslatorException
    {
        super.checkParametersForValidity();
        if(getCoeff() == 0.0D)
        {
            throw new AxisTranslatorException("Coefficient on a linear translator cannot be 0.");
        }
    }

    @Override
    public double computeTranslatedValue(final double originalValue)
    {
        return getCoeff() * originalValue + getConstant();
    }

    @Override
    public double computeOriginalValue(final double translatedValue)
    {
        return (translatedValue - getConstant()) / getCoeff();
    }

    @Override
    public void initializeRequiredAndUsedParameters()
    {
        setRequiredParameters(REQUIRED_PARAMETERS);
        setUsedParameters(USED_PARAMETERS);
    }

    @Override
    public String getName()
    {
        return "Linear";
    }

    @Override
    public ChartParametersPanel buildEditingPanel(final AxisParameters containingAxisParameters,
                                                  final AxisTranslatorParameters managedParameters,
                                                  final AxisTranslatorParameters defaultParameters)
    {
        return new LinearAxisTranslatorPanel(managedParameters, defaultParameters);
    }

}
