package ohd.hseb.grid;

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.OHDConstants;

final public class GridUtilities
{
    private GridUtilities()
    {
        /*
         * This class only contains static constants and methods, so an instance of this class should never be created.
         * If not providing this private constructor, users still can create objects of this class by using the
         * invisible default constructor provided by Java, which is wrong. Now, with this private constructor, the
         * default constructor by Java is destroyed. Now, nobody from outside of this class can create the object of
         * this class
         */
    }

    /**
     * Convert from HRAP Row number and Column number to a two elements double array[latitude, longitude].
     */
    public static double[] convertHRAPRowColToLatLon(final int hrapRow, final int hrapColumn)
    {
        final double latThenLonArray[] = new double[2];
        //---------------------------------
        // Define HRAP projection constants
        //---------------------------------
        final double stdlat = Math.toRadians(60);
        final double stdlon = -105;
        final double earthr = 6371200.0;
        final double mesh = NetcdfConstants.NETCDF_GRID_LENGTH_IN_METER;
        final int hrapxor = NetcdfConstants.NORTH_POLE_HRAP_X;
        final int hrapyor = NetcdfConstants.NORTH_POLE_HRAP_Y;

        final double x = (hrapColumn - hrapxor) * mesh;
        final double y = (hrapRow - hrapyor) * mesh;

        final double bigr = Math.pow((x * x) + (y * y), 0.5);
        final double arg = bigr / (earthr * (1 + Math.sin(stdlat)));
        double ang;

        latThenLonArray[0] = 90.0 - (2 * (Math.toDegrees(Math.atan(arg))));

        if((x < 0) && (y < 0))
        {
            ang = (((Math.toDegrees(Math.atan(y / x))) * (-1)) + 180) * (-1);
        }
        else if((x < 0) && (y > 0))
        {
            ang = Math.toDegrees(Math.atan(y / x)) + 180;
        }
        else
        {
            ang = Math.toDegrees(Math.atan(y / x));
        }

        if(y > 0)
        {
            ang = 270.0 - stdlon - ang;
        }
        else
        {
            ang = -90.0 - stdlon - ang;
        }

        if(ang < 180)
        {
            latThenLonArray[1] = -1 * ang;
        }
        else
        {
            latThenLonArray[1] = 360.0 - ang;
        }

        //trim the precision to 3 digits after the decimal point
        latThenLonArray[0] = MathHelper.roundToNDecimalPlaces(latThenLonArray[0], 3);
        latThenLonArray[1] = MathHelper.roundToNDecimalPlaces(latThenLonArray[1], 3);

        return latThenLonArray;
    }

    /** convert Polar Stereographic X coordinate with units of meters to national HRAP Column X */
    public static int getHrapColumnFromPolarStereoX(final float xster)
    {
        final int hrapCol = Math.round((xster / NetcdfConstants.NETCDF_GRID_LENGTH_IN_METER) - 0.5f)
            + NetcdfConstants.NORTH_POLE_HRAP_X;

        return hrapCol;
    }

    /** get Polar Stereographic X coordinate with units of meters based on the HRAP column number */
    public static int getPolarStereoXFromHrapColumn(final int hrapx)
    {
        final int xster = Math.round(((hrapx - NetcdfConstants.NORTH_POLE_HRAP_X) + 0.5f)
            * NetcdfConstants.NETCDF_GRID_LENGTH_IN_METER);

        return xster;
    }

    /** convert Polar Stereographic Y coordinate with units of meters to national HRAP Row Y */
    public static int getHrapRowFromPolarStereoY(final float yster)
    {
        final int hrapRow = Math.round((yster / NetcdfConstants.NETCDF_GRID_LENGTH_IN_METER) - 0.5f)
            + NetcdfConstants.NORTH_POLE_HRAP_Y;

        return hrapRow;
    }

    /** get Polar Stereographic Y coordinate with units of meters based on the HRAP row number */
    public static int getPolarStereoYFromHrapRow(final int hrapRowNum)
    {
        final int yster = Math.round((((hrapRowNum - NetcdfConstants.NORTH_POLE_HRAP_Y) + 0.5f) * NetcdfConstants.NETCDF_GRID_LENGTH_IN_METER));

        return yster;
    }

    /**
     * Check if the float value is a non-missing value in NetCDF file.
     */
    public static boolean isNonMissingValue(final float fValue)
    {
        if(fValue > -999.0 && fValue < 1E36)
        {//within reasonable range, non-missing value. NetCDF default missing value is 9.96921E36.

            return true;
        }

        //otherwise, it is missing value
        return false;
    }

    /**
     * Print out "FFG" or "THERSHOLD_RUNOFF" grid to separate ASCII ESRI format files.
     */
    public static void printOutFfgOrThreshRGridToAsciiEsriFiles(final RfcGridWithTime ffgThreshGrids,
                                                                final String outputDir) throws Exception
    {
        final StringBuffer strBuffer = new StringBuffer();

        double gridValue;

        String outputFileName;
        String durationHr;
        for(int i = 0; i < ffgThreshGrids.getTimeOrDurationDimSize(); i++)
        {
            strBuffer.append(getAsciiEsriHeader(ffgThreshGrids));

            for(int hrapRow = (ffgThreshGrids._rfcOrigHrapRow + ffgThreshGrids._rowNum - 1); hrapRow >= ffgThreshGrids._rfcOrigHrapRow; hrapRow--)
            {
                for(int hrapCol = ffgThreshGrids._rfcOrigHrapColumn; hrapCol < (ffgThreshGrids._rfcOrigHrapColumn + ffgThreshGrids._columnNum); hrapCol++)
                {

                    gridValue = ffgThreshGrids.getPixelValue(i, hrapCol, hrapRow);

                    if(GridUtilities.isNonMissingValue((float)gridValue) == false)
                    {
                        gridValue = OHDConstants.MISSING_DATA;//re-set it to -999.0
                    }

                    strBuffer.append(gridValue).append(" ");
                } //close inner loop

                strBuffer.append(OHDConstants.NEW_LINE);
            }//close middle loop

            switch(i)
            {
                case 0:
                    durationHr = "1hr";
                    break;
                case 1:
                    durationHr = "3hr";
                    break;
                case 2:
                    durationHr = "6hr";
                    break;
                case 3:
                    durationHr = "12hr";
                    break;
                case 4:
                    durationHr = "24hr";
                    break;
                default:
                    throw new Exception("The time/duration time step maximum number is 5.");
            }

            outputFileName = outputDir + "/threshold_runoff_" + durationHr + ".asc"; //hard-coded output file name
            final PrintWriter writer = new PrintWriter(new FileWriter(outputFileName));
            writer.print(strBuffer.toString());
            writer.close();

            strBuffer.setLength(0);

            System.out.println("Successfully generated Ascii Esri file: " + outputFileName);
        }//close outer loop

    }

    /**
     * Print out parameter or state individual variable to separate ASCII ESRI format files.
     * 
     * @param rfcGrids2D - for parameters, it is the object of class RfcGrids2D; for state case, it is the object of
     *            class RfcGrids3D.
     */
    public static void printOutGridToAsciiEsriFiles(final RfcGrid rfcGrids2D, final String outputDir) throws Exception
    {
        String prefixStr;

        final String title = rfcGrids2D.getTitle().toLowerCase();

        if(title.contains("api"))
        {
            prefixStr = "apicont";
        }
        else if(title.contains("snow17"))
        {
            prefixStr = "snow17";
        }
        else
        {
            throw new Exception(title + " does not contain api or snow17 and cannot be determined.");
        }

        if(rfcGrids2D instanceof RfcGridWithTime)
        {//states, need its time in the file name
            final long warmStateTime = ((RfcGridWithTime)rfcGrids2D).getStartTimeInMillis();
            final String dateStr = DateTime.getDateStringFromLong(warmStateTime, null); //GMT
            final String timeStr = DateTime.getTimeStringFromLong(warmStateTime, null).substring(0, 2); //GMT, only retrieve HH value

            prefixStr = prefixStr + "_state_" + dateStr + "_" + timeStr + "Z";
        }
        else
        {
            prefixStr = prefixStr + "_parameter";
        }

        String outputFileName;
        for(final String varName: rfcGrids2D._variablArrayMap.keySet())
        {
            outputFileName = outputDir + File.separator + prefixStr + "_" + varName.toLowerCase() + ".asc";
            if(rfcGrids2D.isVariableStringType(varName) == false)
            {//float type
                printOutFloatVariableToAsciiEsriFile(rfcGrids2D, varName, outputFileName);
            }
            else
            {//String type

                //check if it is double array type. If so, print to a series ASC files
                if(varName.equalsIgnoreCase(OHDConstants.EXLAGARRAY_TAG)
                    || varName.equalsIgnoreCase(OHDConstants.ADC_TAG))
                {//right now, only two variables are double array type, so hard-coded here
                    printOutDoubleArrayVariableToAsciiEsriFiles(rfcGrids2D, varName, outputFileName);
                }

                //do not print out other String type variables since not used
            }
        }//close for loop
    }

    private static void printOutFloatVariableToAsciiEsriFile(final RfcGrid rfcGrids2D,
                                                             final String varName,
                                                             final String outputFileName) throws Exception
    {
        final StringBuilder strBuilder = new StringBuilder();

        strBuilder.append(getAsciiEsriHeader(rfcGrids2D));

        double gridValue;

        for(int hrapRow = (rfcGrids2D._rfcOrigHrapRow + rfcGrids2D._rowNum - 1); hrapRow >= rfcGrids2D._rfcOrigHrapRow; hrapRow--)
        {
            for(int hrapCol = rfcGrids2D._rfcOrigHrapColumn; hrapCol < (rfcGrids2D._rfcOrigHrapColumn + rfcGrids2D._columnNum); hrapCol++)
            {

                gridValue = rfcGrids2D.getDoubleVariable(varName, hrapCol, hrapRow);

                if(GridUtilities.isNonMissingValue((float)gridValue) == false)
                {

                    gridValue = OHDConstants.MISSING_DATA;//re-set it to -999.0
                }

                strBuilder.append(gridValue).append(" ");
            } //close inner loop

            strBuilder.append(OHDConstants.NEW_LINE);
        }//close outer loop

        final PrintWriter writer = new PrintWriter(new FileWriter(outputFileName));
        writer.print(strBuilder.toString());
        writer.close();

        System.out.println("Successfully generated Ascii Esri file: " + outputFileName);
    }

    /**
     * Double array was stored as a line of String inside NETCDF file at each pixel. If no value at a pixel, the String
     * value is {@link NetcdfConstants#STRING_MISSING_VALUE}. Now, retrieve the line of String, convert back to double
     * array, generating a series of asc files, e.g.snow17_parameter_area_depletion_curve_0.asc,
     * snow17_parameter_area_depletion_curve_1.asc etc.
     */
    private static void printOutDoubleArrayVariableToAsciiEsriFiles(final RfcGrid rfcGrids2D,
                                                                    final String varName,
                                                                    final String outputFileName) throws Exception
    {
        double[] gridValues;

        int arraySize = 0;

        //find the first non-missing pixel, get the arraySize
        OUTERMOST: for(int hrapRow = (rfcGrids2D._rfcOrigHrapRow + rfcGrids2D._rowNum - 1); hrapRow >= rfcGrids2D._rfcOrigHrapRow; hrapRow--)
        {//one loop for one row
            for(int hrapCol = rfcGrids2D._rfcOrigHrapColumn; hrapCol < (rfcGrids2D._rfcOrigHrapColumn + rfcGrids2D._columnNum); hrapCol++)
            {//one loop for one pixel

                gridValues = rfcGrids2D.getDoubleArrayVariable(varName, hrapCol, hrapRow); //if missing at this pixel, it is null

                if(gridValues == null)
                {
                    continue; //empty pixel, move on to next pixel
                }

                //found a pixle with non missing values
                arraySize = gridValues.length;

                break OUTERMOST; //jump out two for loops

            }//close inner for loop
        }//close outer for loop

        if(arraySize == 0)
        {//all pixel are missing values

            System.out.println("There is no value for the double array " + varName);

            return;
        }

        final StringBuilder[] strBuilderArray = new StringBuilder[arraySize];

        for(int i = 0; i < strBuilderArray.length; i++)
        {
            strBuilderArray[i] = new StringBuilder();

            strBuilderArray[i].append(getAsciiEsriHeader(rfcGrids2D));
        }

        for(int hrapRow = (rfcGrids2D._rfcOrigHrapRow + rfcGrids2D._rowNum - 1); hrapRow >= rfcGrids2D._rfcOrigHrapRow; hrapRow--)
        {//one loop for one row
            for(int hrapCol = rfcGrids2D._rfcOrigHrapColumn; hrapCol < (rfcGrids2D._rfcOrigHrapColumn + rfcGrids2D._columnNum); hrapCol++)
            {//one loop for one pixel

                gridValues = rfcGrids2D.getDoubleArrayVariable(varName, hrapCol, hrapRow); //if missing at this pixel, it is null

                if(gridValues != null)
                {
                    for(int i = 0; i < gridValues.length; i++)
                    {
                        strBuilderArray[i].append(gridValues[i]).append(" ");
                    }
                }
                else
                {
                    for(int i = 0; i < arraySize; i++)
                    {
                        strBuilderArray[i].append(OHDConstants.MISSING_DATA).append(" ");
                    }
                }

            } //close inner loop(finished every pixel on this row)

            for(int i = 0; i < strBuilderArray.length; i++)
            {//move to next line
                strBuilderArray[i].append(OHDConstants.NEW_LINE);
            }
        }//close outer loop

        //dump out to a series of ASC files
        for(int i = 0; i < strBuilderArray.length; i++)
        {
            //set-up each ASC file name correctly
            String fileName = outputFileName.substring(0, outputFileName.length() - 4); //stripped ".asc" 4 chars

            fileName = fileName + "_" + String.valueOf(i) + ".asc";

            final PrintWriter writer = new PrintWriter(new FileWriter(fileName));
            writer.print(strBuilderArray[i].toString());
            writer.close();

            System.out.println("Successfully generated Ascii Esri file: " + fileName);
        }

    }

    /**
     * prints out the header to a String. "nodata_value " will be "-999.0". xllcorner and yllcorner are longitude and
     * latitude respectively.
     */
    private static String getAsciiEsriHeader(final RfcGrid rfcGrids)
    {
        final StringBuilder strBuilder = new StringBuilder();

        strBuilder.append("ncols ").append(rfcGrids._columnNum).append(OHDConstants.NEW_LINE);
        strBuilder.append("nrows ").append(rfcGrids._rowNum).append(OHDConstants.NEW_LINE);

        final double[] latLong = GridUtilities.convertHRAPRowColToLatLon(rfcGrids._rfcOrigHrapRow,
                                                                         rfcGrids._rfcOrigHrapColumn);

        strBuilder.append("xllcorner ")
//                  .append(NetcdfUtilities.getPolarStereoXFromHrapColumn(_rfcOrigHrapColumn))
                  .append(latLong[1])
                  .append(OHDConstants.NEW_LINE);
        strBuilder.append("yllcorner ")
//                  .append(NetcdfUtilities.getPolarStereoYFromHrapRow(_rfcOrigHrapRow))
                  .append(latLong[0])
                  .append(OHDConstants.NEW_LINE);
        strBuilder.append("cellsize ")
                  .append(NetcdfConstants.NETCDF_GRID_LENGTH_IN_METER)
                  .append(OHDConstants.NEW_LINE);
        strBuilder.append("nodata_value ");

        strBuilder.append(NetcdfConstants.FLOAT_MISSING_VALUE);
        strBuilder.append(OHDConstants.NEW_LINE);

        return strBuilder.toString();
    }
}
