package ohd.hseb.grid;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;

import javax.management.timer.Timer;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.fews.NwsrfsDataTypeMappingReader;
import ohd.hseb.util.fews.OHDConstants;
import ucar.ma2.Array;
import ucar.ma2.ArrayFloat;
import ucar.ma2.DataType;
import ucar.ma2.Index;
import ucar.ma2.IndexIterator;
import ucar.nc2.Dimension;
import ucar.nc2.NetcdfFileWriteable;
import ucar.nc2.Variable;

public class NetcdfTemplateCreator
{
    //this set contains String type variable names. Any variable name in it will be treated considered String type, not the float type
    private static final Set<String> _STRING_TYPE_VARIABLES_STATES = new HashSet<String>(Arrays.asList(OHDConstants.EXLAGARRAY_TAG));

    //this set contains String type variable names. Any variable name in it will be treated considered String type, not the float type
    private static final Set<String> _STRING_TYPE_VARIABLES_PARAMETERS = new HashSet<String>(Arrays.asList(OHDConstants.ADC_TAG));

    private final int _southRow;
    private final int _rowNum;
    private final int _westCol;
    private final int _colNum;

    private String _userFileName = null;
    private String _rfcName = null;

    //For FFG("time") and THRESHOLD_RUNOFF("duration"): 3 means that there are 1HR, 3HR and 6HR; 5 means that there are 1HR, 3HR, 6HR, 12HR and 24HR
    private int _timeSteps;

    public NetcdfTemplateCreator(final int southRow,
                                 final int rowNum,
                                 final int westCol,
                                 final int colNum,
                                 final int timeSteps)
    {
        _southRow = southRow;
        _rowNum = rowNum;
        _westCol = westCol;
        _colNum = colNum;
        _timeSteps = timeSteps;
    }

    /**
     * From User File, we can also set {@link #_westCol}, {@link #_colNum}, {@link #_southRow} and {@link #_rowNum} etc.
     * OptFileConverter uses this constructor.
     * 
     * @param userFileName
     * @param rfcName
     */
    public NetcdfTemplateCreator(final String userFileName, final String rfcName) throws IOException
    {
        _userFileName = userFileName;
        _rfcName = rfcName;

        this.setRfcName();

        // Read the define user control text file and store the information in a List.                       
        final List<String> inputFileLinesList = this.readinputFileLines(_userFileName); //read and store Opt file line by line in an array        
        // end read of user file.

        /*---------------- set origin values for this RFC based on information from user file --------------*/
        //367 335 263 159 <br> Card #9 in user file
        String line = inputFileLinesList.get(8);
        _westCol = Integer.valueOf(line.substring(0, 5).trim());
        _colNum = Integer.valueOf(line.substring(6, 11).trim());
        _southRow = Integer.valueOf(line.substring(12, 17).trim());
        _rowNum = Integer.valueOf(line.substring(18, 23).trim());

        /*---------------- set time steps for this RFC based on information from user file --------------*/
        _timeSteps = 3; //initialized to the default value(1HR, 3HR and 6HR)

        line = inputFileLinesList.get(4); //get line #5
        if(line.contains("-1.0") == false)
        {
            _timeSteps++; //1HR, 3HR, 6HR and 12HR

            //check if using 24HR
            line = inputFileLinesList.get(5); //get line #6
            if(line.contains("-1.0") == false)
            {
                _timeSteps++; //1HR, 3HR, 6HR, 12HR and 24HR
            }
        }

    }

    /**
     * Set the RFC Name if it is null from the user File Name
     */
    private void setRfcName()
    {
        if(_rfcName == null || (_rfcName != null && _rfcName.isEmpty()))
        {
            final File test = new File(_userFileName);
            _rfcName = test.getName();
            _rfcName = _rfcName.substring(0, _rfcName.indexOf("."));
        }
    }

    private List<String> readinputFileLines(final String fileName) throws IOException
    {
        final List<String> inputFileLinesList = new ArrayList<String>();
        String line = null;
        try
        {
            final BufferedReader inputStream = new BufferedReader(new FileReader(fileName));

            while((line = inputStream.readLine()) != null)
            {
                inputFileLinesList.add(line);
                /*
                 * can not trim the line here, because if there are " " in the beginning, trimming will shift the line
                 * and all the index later is wrong
                 */
            }

            inputStream.close();
        }
        catch(final IOException e)
        {
            throw e;
        }
        return inputFileLinesList;
    }

    /**
     * Create the Parameter NetCDF template files for {@link #_rfcName}.
     * 
     * @param variableNames - a String array holding all the parameters' names
     * @param outputFileName
     */
    public void createRfcParamsTemplateNCfile(final String[] variableNames, final String outputFileName) throws Exception
    {
        this.createRfcNetCDFTemplatefile(outputFileName,
                                         null,
                                         variableNames,
                                         0l,
                                         0,
                                         0,
                                         _STRING_TYPE_VARIABLES_PARAMETERS,
                                         false);
    }

    /**
     * Create the states NetCDF template files for {@link #_rfcName}.
     * 
     * @param variableNames - a String array holding all the states' names
     * @param stateTimeInMillis - warm state time in millis long
     * @param outputFileName
     */
    public void createRfcStatesTemplateNCfile(final String[] variableNames,
                                              final long stateTimeInMillis,
                                              final String outputFileName) throws Exception
    {

        this.createRfcNetCDFTemplatefile(outputFileName,
                                         NetcdfConstants.NETCDF_TIME_VAR_NAME,
                                         variableNames,
                                         stateTimeInMillis,
                                         0,
                                         1, //states do have "time" dimension, but only has 1 step(for warm state time)
                                         _STRING_TYPE_VARIABLES_STATES,
                                         false);
    }

    /**
     * Create FFG NETCDF template file for this RFC{@link #_rfcName}. The "time" dimension size is determined by
     * {@link #_timeSteps}. If 3: the time dimension only has the default 1HR, 3HR and 6HR; if 4: it has additional
     * 12HR; if 5: it has 24HR.
     * 
     * @param dateTimeInMillis - Based on this time, the time dimension will have 1HR, 3HR etc values. After the NetCDF
     *            template file created with this time value, it can be re-set to the correct value during real usage.
     */
    public void createRfcFfgTemplateNCfiles(final String outputFileName, final long dateTimeInMillis) throws Exception
    {
        this.createRfcNetCDFTemplatefile(outputFileName,
                                         NetcdfConstants.NETCDF_TIME_VAR_NAME,
                                         new String[]{ohd.hseb.util.fews.DataType.FFG_DATATYPE},
                                         dateTimeInMillis,
                                         0,
                                         _timeSteps,
                                         null,
                                         true);
    }

    /**
     * Create FFG NETCDF template file for this RFC{@link #_rfcName}. The "duration" dimension size is determined by
     * {@link #_timeSteps}. If 3: the time dimension only has the default 1HR, 3HR and 6HR; if 4: it has additional
     * 12HR; if 5: it has 24HR.<br>
     * Note: for THRESHOLD_RUNOFF template NetCDF file, since the 3rd dimension is "duration" and its values is 1(HR),
     * 3(HR), 6(HR) etc, so no need to set initial time(minutes since the Epoch time).
     */
    public void createRfcThresholdRunOffTemplateNCfiles(final String outputFileName) throws Exception
    {
        this.createRfcNetCDFTemplatefile(outputFileName,
                                         NetcdfConstants.NETCDF_DURATION_VAR_NAME,
                                         new String[]{NetcdfConstants.NETCDF_THRESHOLD_RUNOFF_VAR_NAME},
                                         0l,
                                         0,
                                         _timeSteps,
                                         null,
                                         true);
    }

    /**
     * Create time series NetCDF template file. Note: the number of time steps and interval are fixed once the NetCDF
     * template being created and can not be re-set. However, the initial time can be reset.
     * 
     * @param tsNameArray - String array holding all the time series names, e.g. ["MAP", "MAT"].
     * @param firstTimeStepsTimeInMillis - the first time step in milliseconds.
     * @param tsIntervalInHR - the time series interval in HR
     * @param numOfTimeSteps - the time steps of the time series
     */
    public void createRfcTimeSeriesTemplateNCfiles(final String outputFileName,
                                                   final String[] tsNameArray,
                                                   final long firstTimeStepsTimeInMillis,
                                                   final int tsIntervalInHR,
                                                   final int numOfTimeSteps) throws Exception
    {
        this.createRfcNetCDFTemplatefile(outputFileName,
                                         NetcdfConstants.NETCDF_TIME_VAR_NAME,
                                         tsNameArray,
                                         firstTimeStepsTimeInMillis,
                                         tsIntervalInHR,
                                         numOfTimeSteps,
                                         null,
                                         true);
    }

    /**
     * Create template NETCDF file for "FFG", "THERSHOLD_RUNOFF", time series containing MAP and MAT, states or
     * parameters. For the time series case, the interval is assumed to be 6HR for now.
     * 
     * @param outputFileName - the generated NetCDF template file name.
     * @param thirdDimName - either "time", or "duration" or null. For parameters, no 3rd dimension is needed, so this
     *            is null. For others, 3rd dimension is needed. For threshold runoff, this dimension is called
     *            "duration", and its value is 1, 3, 6, 12 etc. For others, e.g. States and Time Series, this dimension
     *            is called "time" and the value is minutes since 1970-01-01 00:00:00.
     * @param targetVarNameArray - a String array holding the variable names, like one element array("FFG" or
     *            "THRESHR"), or two elements array("MAP" & "MAT"), or many elements array(all the snow17 states' names)
     * @param initialTimeLong - needed for FFG, time series and States. For threshold runoff and Parameters, this can be
     *            0l.
     * @param tsIntervalHR - the time series interval in hours. Only used for creating time series template NetCDF file.
     * @param timeSteps - the number of steps in the third dimension. For both FFG and threshold runoff, if 3: the time
     *            dimension only has the default 1HR, 3HR and 6HR; if 4: it has additional 12HR; if 5: it has 24HR; For
     *            States, this is 1 since only warm state time is needed to be stored. For time series, this is ts
     *            steps. For Parameters, since no time dimension, this is 0 or be ignored.
     * @param STRING_TYPE_VARIABLES - a String[] containing some variable names which are String type or double array
     *            type. For these kind of variables, inside NetCDF file, they are stored as array of char. If none
     *            variable is this type, this parameter can be null.
     * @param varWithTimeDim - if true, the variables are (time, y, x) dimension; if false, (y,x) dimension only. States
     *            has one value in "time" dimension, but their variables all are (y, x) dimension.
     */
    private void createRfcNetCDFTemplatefile(final String outputFileName,
                                             final String thirdDimName,
                                             final String[] targetVarNameArray,
                                             final long initialTimeLong,
                                             final int tsIntervalHR,
                                             final int timeSteps,
                                             final Set<String> STRING_TYPE_VARIABLES,
                                             final boolean varWithTimeDim) throws Exception
    {
        final RfcHrapGrid hrapGrid = new RfcHrapGrid(_westCol, _southRow, _colNum, _rowNum);

//        System.out.println("Generating " + outputFileName);

        final NetcdfFileWriteable writableFile = NetcdfFileWriteable.createNew(outputFileName);

        /*------------------------Define the dimensions(X, Y, optional "time", "duration", "char_index" -------*/

        //first, 3rd dimension, if needed:
        Dimension timeDim = null;
        if(thirdDimName != null && timeSteps > 0)
        {//has 3rd dimension: "time" or "duration"
            timeDim = writableFile.addDimension(thirdDimName, timeSteps);

            writableFile.addVariable(thirdDimName, DataType.DOUBLE, new Dimension[]{timeDim});
            //use DataType.DOUBLE due to FEWS ncdump: double time(time), even though it seems should be DataType.INT

            if(thirdDimName == NetcdfConstants.NETCDF_TIME_VAR_NAME)
            {//FFG, States or TS
                writableFile.addVariableAttribute(thirdDimName, "standard_name", NetcdfConstants.NETCDF_TIME_VAR_NAME);
                writableFile.addVariableAttribute(thirdDimName, "long_name", NetcdfConstants.NETCDF_TIME_VAR_NAME);
                writableFile.addVariableAttribute(thirdDimName,
                                                  OHDConstants.UNITS_ATT_NAME,
                                                  "minutes since 1970-01-01 00:00:00.0 +0000");
                writableFile.addVariableAttribute(thirdDimName, "axis", "T");
            }
            else if(thirdDimName == NetcdfConstants.NETCDF_DURATION_VAR_NAME)
            {//THRESH-R
                writableFile.addVariableAttribute(thirdDimName, "standard_name", "interval value 1, 3, 6, 12 or 24 hr");
                writableFile.addVariableAttribute(thirdDimName, "long_name", "threshold runoff time interval value");
                writableFile.addVariableAttribute(thirdDimName, OHDConstants.UNITS_ATT_NAME, "hours interval");
            }
            else
            {//something not recognized
                throw new Exception("The 3rd dimensin name[" + thirdDimName + "] is not recognized.");
            }
        }

        //next, y dimension, then x dimension
        final String yVarName = NetcdfConstants.NETCDF_Y_VAR_NAME;
        final String xVarName = NetcdfConstants.NETCDF_X_VAR_NAME;

        final Dimension yDim = writableFile.addDimension(yVarName, _rowNum);
        final Dimension xDim = writableFile.addDimension(xVarName, _colNum);

        writableFile.addVariable(yVarName, DataType.FLOAT, new Dimension[]{yDim});
        writableFile.addVariableAttribute(yVarName, "standard_name", "projection_y_coordinate");
        writableFile.addVariableAttribute(yVarName, "long_name", "y coordinate according to Polar stereographic");
        writableFile.addVariableAttribute(yVarName, OHDConstants.UNITS_ATT_NAME, "");
        writableFile.addVariableAttribute(yVarName, NetcdfConstants.FILL_VALUE_ATT, NetcdfConstants.FLOAT_MISSING_VALUE);

        writableFile.addVariable(xVarName, DataType.FLOAT, new Dimension[]{xDim});
        writableFile.addVariableAttribute(xVarName, "standard_name", "projection_x_coordinate");
        writableFile.addVariableAttribute(xVarName, "long_name", "x coordinate according to Polar stereographic");
        writableFile.addVariableAttribute(xVarName, OHDConstants.UNITS_ATT_NAME, "");
        writableFile.addVariableAttribute(xVarName, NetcdfConstants.FILL_VALUE_ATT, NetcdfConstants.FLOAT_MISSING_VALUE);

        //last, char_index dimension, if needed:
        Dimension charIndexDim = null; //will remain null if no string or double array type variable
        for(final String varName: targetVarNameArray)
        {

            if(STRING_TYPE_VARIABLES != null && STRING_TYPE_VARIABLES.contains(varName))
            {//allow to have 100 chars, enough to hold double array of size 11 or enough to hold exlag array in snow17 state

                charIndexDim = writableFile.addDimension(NetcdfConstants.NETCDF_CHAR_INDEX_DIM_NAME,
                                                         NetcdfConstants.NETCDF_REGULAR_STRING_MAX_LENGTH);

                break; //jump out of for loop
            }
        }

        /*------------going through the array targetVarNameArray and add each one to the NetCDF file----------*/
        for(final String targetVarName: targetVarNameArray)
        {//for FFG and THRESHOLD_RUNOFF, only loop once; for TS, loop twice(MAP, MAT); for States & Params, loop many times

            if(varWithTimeDim == false)
            {
                /*
                 * Parameters does not have 3rd dimension. States do have 3rd dimension as "time", but only has one
                 * timeStep(warm state time), the vars only need to be 2D actually.
                 */

                if(STRING_TYPE_VARIABLES != null && STRING_TYPE_VARIABLES.contains(targetVarName))
                {//string type. charIndexDim should not be null
                    writableFile.addVariable(targetVarName, DataType.CHAR, new Dimension[]{yDim, xDim, charIndexDim});

                    //attributes:
                    writableFile.addVariableAttribute(targetVarName, "standard_name", targetVarName + " standard name");
                    writableFile.addVariableAttribute(targetVarName, "long_name", targetVarName + " long name");
                    writableFile.addVariableAttribute(targetVarName, OHDConstants.UNITS_ATT_NAME, "");
                    writableFile.addVariableAttribute(targetVarName,
                                                      NetcdfConstants.FILL_VALUE_ATT,
                                                      NetcdfConstants.STRING_MISSING_VALUE);
                }
                else
                {//does not add units information

                    writableFile.addVariable(targetVarName, DataType.FLOAT, new Dimension[]{yDim, xDim});

                    writableFile.addVariableAttribute(targetVarName, "standard_name", targetVarName);
                    writableFile.addVariableAttribute(targetVarName, "long_name", targetVarName);

                    writableFile.addVariableAttribute(targetVarName,
                                                      NetcdfConstants.FILL_VALUE_ATT,
                                                      NetcdfConstants.FLOAT_MISSING_VALUE);
                }
            }
            else
            {//varWithTimeDim == true; In this scenario, so far all variables are not String type

                writableFile.addVariable(targetVarName, DataType.FLOAT, new Dimension[]{timeDim, yDim, xDim});

                writableFile.addVariableAttribute(targetVarName, "standard_name", targetVarName);
                writableFile.addVariableAttribute(targetVarName, "long_name", targetVarName);

                final String unitStr = NwsrfsDataTypeMappingReader.getFewsUnit(targetVarName, OHDConstants.DUMMY_LOG);
                /*
                 * Two TSs have inconsistent units between the model outputs and FEWS expected: SNOW-17 full version
                 * output TS "SNSG" and SNOW-17 state "SNDPT". The model outputs are CM unit while FEWS want them to be
                 * in MM. Here, the NetCDF set them at FEWS' unit, MM. When inserting the TSs into the NetCDF file, the
                 * values need to be times 10.
                 */

                writableFile.addVariableAttribute(targetVarName, OHDConstants.UNITS_ATT_NAME, unitStr);

                writableFile.addVariableAttribute(targetVarName,
                                                  NetcdfConstants.FILL_VALUE_ATT,
                                                  NetcdfConstants.FLOAT_MISSING_VALUE);
            }
        }//for(final String targetVarName: targetVarNameArray)

        writableFile.addGlobalAttribute("title", "Netcdf data");
        writableFile.addGlobalAttribute("institution", "OHD");
        writableFile.addGlobalAttribute("source", "created by OHD CHPS team");
        writableFile.addGlobalAttribute("history",
                                        "Created at "
                                            + DateTime.getDateTimeStringFromLong(System.currentTimeMillis(),
                                                                                 TimeZone.getDefault())); //show the creation time at the local time zone
        writableFile.addGlobalAttribute("Conventions", "CF-1.4");
        writableFile.addGlobalAttribute("coordinate_system", "Polar stereographic");

        //new global attributes
        final double[] latLon = GridUtilities.convertHRAPRowColToLatLon(_southRow, _westCol);
        writableFile.addGlobalAttribute(NetcdfConstants.LATOFFIRSTGRIDPOINT, (int)(latLon[0] * 1000)); //times 1000, then round to int
        writableFile.addGlobalAttribute(NetcdfConstants.LONOFFIRSTGRIDPOINT, (int)(latLon[1] * 1000)); //times 1000, then round to int
        writableFile.addGlobalAttribute(NetcdfConstants.LONORIGIN, Integer.valueOf(NetcdfConstants.LONORIGIN_VAL)); //fixed for all RFCs
        writableFile.addGlobalAttribute(NetcdfConstants.XCELLSIZE, Integer.valueOf(NetcdfConstants.XCELLSIZE_VAL)); //fixed for all RFCs
        writableFile.addGlobalAttribute(NetcdfConstants.YCELLSIZE, Integer.valueOf(NetcdfConstants.YCELLSIZE_VAL)); //fixed for all RFCs

        writableFile.create();

        //re-set y values(row)
        final ArrayFloat yVarArray = new ArrayFloat.D1(hrapGrid.getRowNum());
        final Index yVarIndex = yVarArray.getIndex();

        for(int j = 0; j < hrapGrid.getRowNum(); j++)
        {
            yVarIndex.set(j);

            yVarArray.set(yVarIndex, hrapGrid.getYSter(j));
        }

        //re-set x values(column)
        final ArrayFloat xVarArray = new ArrayFloat.D1(hrapGrid.getColNum());
        final Index xVarIndex = xVarArray.getIndex();

        for(int j = 0; j < hrapGrid.getColNum(); j++)
        {
            xVarIndex.set(j);

            xVarArray.set(xVarIndex, hrapGrid.getXSter(j));
        }

        writableFile.write(yVarName, yVarArray);
        writableFile.write(xVarName, xVarArray);

        if(thirdDimName != null)
        {//FFG, THRESHOLD_RUNOFF, States or Time Series. In another words, exclude Parameters here

            final Variable timeVar = writableFile.findVariable(thirdDimName);
            final Array timeVarArray = timeVar.read();

            if(thirdDimName == NetcdfConstants.NETCDF_TIME_VAR_NAME)
            {//FFG, Time Series or States

                if(targetVarNameArray.length == 1
                    && targetVarNameArray[0].equalsIgnoreCase(ohd.hseb.util.fews.DataType.FFG_DATATYPE))
                {//creating FFG template

                    final long minutes = initialTimeLong / Timer.ONE_MINUTE;
                    timeVarArray.setDouble(0, minutes + 60); //1 HR                                                                                                    
                    timeVarArray.setDouble(1, minutes + 180); //3 HR                                                                                                   
                    timeVarArray.setDouble(2, minutes + 360); //6 HR                                                                                                   

                    if(timeSteps >= 4)
                    {//equal to 4 or 5
                        timeVarArray.setDouble(3, minutes + 720); //12 HR                                                                                              
                    }

                    if(timeSteps == 5)
                    {
                        timeVarArray.setDouble(4, minutes + 1440); //24 HR                                                                                             
                    }
                }
                else
                {//time series or states

                    long timeInLong = 0;

                    for(int i = 0; i < timeSteps; i++)
                    {//if states, only loop once, that time is the warm state time

                        timeInLong = initialTimeLong + i * tsIntervalHR * Timer.ONE_HOUR; //assuming 6HR interval
                        timeVarArray.setDouble(i, timeInLong / Timer.ONE_MINUTE); // "minutes since 1970-01-01 00:00:00.0 +0000"

                    }//close for loop
                }
            }
            else if(thirdDimName == NetcdfConstants.NETCDF_DURATION_VAR_NAME)
            {//THREH-R. targetVarNameArray contains "duration", not "time"

                timeVarArray.setDouble(0, 1); //1 HR
                timeVarArray.setDouble(1, 3); //3 HR
                timeVarArray.setDouble(2, 6); //6 HR

                if(timeSteps >= 4)
                {//equal to 4 or 5
                    timeVarArray.setDouble(3, 12); //12 HR
                }

                if(timeSteps == 5)
                {
                    timeVarArray.setDouble(4, 24); //24 HR
                }
            }

            writableFile.write(thirdDimName, timeVarArray);
        }

        final List<Variable> varList = writableFile.getVariables();

        //populate the missing values
        for(final Variable variable: varList)
        {
            final String varName = variable.getName();

            final Array varArray = variable.read();

            final IndexIterator varIndexItr = varArray.getIndexIterator();

            final List<Dimension> varDimList = variable.getDimensions();

            //ignore those variables like "x", "y", "time", "char_index" etc
            if(varDimList.size() > 1)
            {

                if(STRING_TYPE_VARIABLES != null && STRING_TYPE_VARIABLES.contains(varName))
                {//string type. 

                    //do nothing. Since this is char type(char array to represent String), there is no good missing char value to be used.
                }
                else
                {//float type variable

                    //set the value to the float missing value.
                    while(varIndexItr.hasNext())
                    {
                        varIndexItr.setFloatNext(NetcdfConstants.FLOAT_MISSING_VALUE);
                    }
                }

            }

            writableFile.write(varName, varArray);
        }//close for loop

        writableFile.flush();

        writableFile.close();
    }//close createRfcNetCDFTemplatefile
}
