package ohd.hseb.grid;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ucar.ma2.Array;
import ucar.ma2.DataType;
import ucar.ma2.Index;
import ucar.ma2.Index2D;
import ucar.ma2.Index3D;
import ucar.ma2.IndexIterator;
import ucar.nc2.Attribute;
import ucar.nc2.Dimension;
import ucar.nc2.NetcdfFileWriteable;
import ucar.nc2.Variable;

/**
 * Opens an existing RFC template NETCDF file which has 2 dimensions(row, column). Each pixel could have several
 * variables which could be data type of float, double array, boolean or String. Internally, boolean data type is
 * implemented by using float data type(1.0: ture, 0.0: false). String data type is implemented by using char array in
 * 3rd dimension. Double array data type internally is treated as a string containing all the numbers separated with
 * " ".
 */
public class RfcGrid
{
    protected final NetcdfFileWriteable _rfcNetcdfFile;
    protected final int _rfcOrigHrapColumn; //RFC's most south west pixel is its origin.
    protected final int _rfcOrigHrapRow;

    /*
     * Normally, there are only 2-D dimensions. When there are variables of String type or array of Strings, one more
     * dimension
     */
    protected int _columnNum; //x
    protected int _rowNum; //y
    protected int _charIndexLength; //optional dimension

    protected Variable _xVariable; //could be 'x' or 'lon'
    protected Variable _yVariable; //could be 'y' or 'lat'
    protected final Array _xVarArray;
    protected final Array _yVarArray;

    protected Map<String, Array> _variablArrayMap;
    private Map<String, String> _variableUnitsMap;
    private Map<String, String> _variableMissingValMap;

    protected Logger _log;

    private static final char _END_STRING_SYMBOL = '!'; //assuming '!' never be used

    private String _varName; //for FFG or THRESHOLD_RUNOFF, there is only one variable
    private Index _varIndex; //for FFG or THRESHOLD_RUNOFF, there is only one variable

    public RfcGrid(final String rfcNetcdfFile, final Logger log) throws Exception
    {
        _log = log;

        try
        {
            _rfcNetcdfFile = NetcdfFileWriteable.openExisting(rfcNetcdfFile);
        }
        catch(final IOException e)
        {//the exception message from netcdf library is "Not a netCDF file". Re-write it to be clearer.
            throw new IOException(rfcNetcdfFile + " was not found or is not a valid netcdf file.");
        }

        _rfcNetcdfFile.setTitle(rfcNetcdfFile);

        _rfcNetcdfFile.setRedefineMode(true);

        _yVariable = _rfcNetcdfFile.findVariable(NetcdfConstants.NETCDF_Y_VAR_NAME);

        if(_yVariable == null)
        {
            _yVariable = _rfcNetcdfFile.findVariable(NetcdfConstants.NETCDF_LAT_VAR_NAME);
        }

        _yVarArray = _yVariable.read();

        _xVariable = _rfcNetcdfFile.findVariable(NetcdfConstants.NETCDF_X_VAR_NAME);

        if(_xVariable == null)
        {
            _xVariable = _rfcNetcdfFile.findVariable(NetcdfConstants.NETCDF_LON_VAR_NAME);
        }

        _xVarArray = _xVariable.read();

        _rfcOrigHrapColumn = GridUtilities.getHrapColumnFromPolarStereoX(_xVarArray.getFloat(0));

        _rfcOrigHrapRow = GridUtilities.getHrapRowFromPolarStereoY(_yVarArray.getFloat(0));

        this.extractDimLength();

        this.setVariableMaps();

    }

    /**
     * Fill {@link #_variablArrayMap}, {@link #_variableUnitsMap} and {@link #_variableMissingValMap}.
     * 
     * @throws IOException
     */
    private void setVariableMaps() throws IOException
    {
        //fill _variablArrayMap(key: variable name, value: Variable object), so setter can insert values into Variable obj
        final List<Variable> varList = _rfcNetcdfFile.getVariables(); //including variable "x", "y", "time", etc

        _variablArrayMap = new HashMap<String, Array>(varList.size());
        _variableUnitsMap = new HashMap<String, String>(varList.size());
        _variableMissingValMap = new HashMap<String, String>(varList.size());

        List<Attribute> varAttList;
        for(final Variable variable: varList)
        {
            final String varName = variable.getName();

            final List<Dimension> varDimList = variable.getDimensions();

            /*
             * ignore those variables used as dimensions, e.g. "x", "y", "time", "duration", "char_index",
             * "analysis_time" and "realization"(marfc once in a while adds one more 1-d variable for them)
             */
            if(varDimList.size() > 1)
            {
                Array varArray = variable.read();

                for(int i = 0; i < varDimList.size(); i++)
                {
                    final Dimension varDim = varDimList.get(i);

                    if(!varDim.getName().equalsIgnoreCase(NetcdfConstants.NETCDF_TIME_VAR_NAME)
                        && varDim.getLength() == 1)
                    {//this dimension(which is not "time") only has one element, get rid of this dimension

                        varArray = varArray.reduce(i);
                    }
                }//close for loop

                _variablArrayMap.put(varName, varArray);

                /*--------find out "units" attribute of this variable and populate _variableUnitsMap ----*/
                varAttList = variable.getAttributes();

                String attName;
                String attValueStr;
                DataType attDatatype;
                for(final Attribute attribute: varAttList)
                {//going through each attribute of the variable, find "units" attribute(String type) and "FillValue" attribute(float type)

                    attName = attribute.getName();

                    attDatatype = attribute.getDataType();

                    if(attDatatype.isString())
                    {//attribute "units" is String type
                        attValueStr = attribute.getStringValue();
                    }
                    else if(attDatatype.isNumeric())
                    {//normally, attribute "FileValue" is numeric type, however, double array "FillValue" is String type
                     //final float attValueFloat = (Float)attribute.getNumericValue();
                        final double attValueDouble = attribute.getNumericValue().doubleValue();
                        attValueStr = String.valueOf(attValueDouble);
                    }
                    else
                    {
                        throw new IOException(varName + " attribute[" + attName
                            + "] is neither String type nor numeric type.");
                    }

                    if(attName.equals(OHDConstants.UNITS_ATT_NAME))
                    {//e.g.  MAP:units = "MM" ;
                        _variableUnitsMap.put(varName, attValueStr);
                    }
                    else if(attName.contains(NetcdfConstants.FILL_VALUE_ATT))
                    {//e.g. AEADJ:FillValue = -999.f 
                        _variableMissingValMap.put(varName, attValueStr);
                    }
                } //close inner for loop

                //fews 2012.02 version had changed netcdf schema.  To fix griddedFFG ArrayIndexOutOfBounds, it needed to add extra check condition.
                if(_varIndex == null
                    && (varName.equalsIgnoreCase(NetcdfConstants.NETCDF_FFG_VAR_NAME) || varName.equalsIgnoreCase(NetcdfConstants.NETCDF_THRESHOLD_RUNOFF_VAR_NAME)))
                {//only set once. For case of FFG or THRESHOLD_RUNOFF which only has one variable:

                    _varIndex = varArray.getIndex();

                    _varName = varName;
                }

            } //close if(varDimList.size() > 1)
        }//close outer for loop
    }//close the method

    protected int getVarNumber()
    {
        return _variablArrayMap.size();
    }

    /**
     * Assuming only one variable existing in this NetCDF file, e.g. "FFG" or "THRESHR". Then returns the variable
     * array. If there are many variables, this method returns the first variable array, whichever is the first.
     */
    public Array getVarArray()
    {
        return _variablArrayMap.get(_varName);
    }

    public Array getVarArray(final String varName)
    {
        return _variablArrayMap.get(varName);
    }

    /**
     * Assuming only one variable existing in this NetCDF file, e.g. "FFG" or "THRESHR". Then returns the variable's
     * index. If there are many variables, this method returns the first variable array index, whichever is the first.
     */
    protected Index getVarIndex()
    {
        return _varIndex;
    }

    /**
     * Returns the variable array index for the specific variable name.
     */
    protected Index getVarIndex(final String varName)
    {
        final Array varArray = _variablArrayMap.get(varName);
        return varArray.getIndex();
    }

    /**
     * Extract the following dimensions' length: <br>
     * {@link NetcdfConstants#NETCDF_X_VAR_NAME}<br>
     * {@link NetcdfConstants#NETCDF_Y_VAR_NAME}<br>
     * {@link NetcdfConstants#NETCDF_CHAR_INDEX_DIM_NAME}<br>
     */
    private void extractDimLength()
    {
        final List<Dimension> dimList = _rfcNetcdfFile.getDimensions();

        for(final Dimension dim: dimList)
        {
            final String dimName = dim.getName();

            if(dimName.equals(_xVariable.getName()))
            {
                _columnNum = dim.getLength(); //or:  _columnNum = (int)_xVarArray.getSize();
            }
            else if(dimName.equals(_yVariable.getName()))
            {
                _rowNum = dim.getLength(); //or:  _columnNum = (int)_xVarArray.getSize();
            }
            else if(dimName.equals(NetcdfConstants.NETCDF_CHAR_INDEX_DIM_NAME))
            {//optional
                _charIndexLength = dim.getLength();
            }
        }//close for loop
    }

    /**
     * Set the specific variable at specific HRAP Column and HRAP Row value(int). If the input HRAP Column and HRAP Row
     * is out of the range, throws an Exception. If this variable does not exist, throws an Exception too.
     */
    public void setIntVariable(final String variableName, final int hrapCol, final int hrapRow, final int varValue) throws Exception
    {
        this.setDoubleVariable(variableName, hrapCol, hrapRow, varValue);
    }

    /**
     * Returns the variable int value at the specified HRAP Column and HRAP Row. If out of range, throws an exception.
     */
    public double getIntVariable(final String variableName, final int hrapCol, final int hrapRow) throws Exception
    {
        return (int)getDoubleVariable(variableName, hrapCol, hrapRow);
    }

    /**
     * Set the specific variable at specific HRAP Column and HRAP Row value(double). If the input HRAP Column and HRAP
     * Row is out of the range, throws an Exception. If this variable does not exist, throws an Exception too.
     */
    public void setDoubleVariable(final String variableName, final int hrapCol, final int hrapRow, final double varValue) throws Exception
    {

        final Array varArray = _variablArrayMap.get(variableName);
        final Index2D varIndex = this.getVarIndex(variableName, hrapCol, hrapRow);

        varArray.setFloat(varIndex, (float)varValue);
    }

    /**
     * Returns the variable double value at the specified HRAP Column and HRAP Row. If out of range, throws an
     * exception.
     */
    public double getDoubleVariable(final String variableName, final int hrapCol, final int hrapRow) throws Exception
    {
        final Array varArray = _variablArrayMap.get(variableName);

        final Index2D varIndex = this.getVarIndex(variableName, hrapCol, hrapRow);
        return varArray.getFloat(varIndex);
    }

    /**
     * A helper method returning the NetCDF index for the specific variable name and the locations(hrapCol and hrapRow).
     * It also checks those location specifications within the RFC range or not.
     */
    private Index2D getVarIndex(final String variableName, final int hrapCol, final int hrapRow) throws Exception
    {
        final Index2D varIndex = (Index2D)this.getVarIndex(variableName);

        final int relativeColIndex = hrapCol - _rfcOrigHrapColumn; //starting from 0
        final int relativeRowIndex = hrapRow - _rfcOrigHrapRow; //starting from 0

        //if out of range, throws an Exception
        if(relativeColIndex < 0 || relativeColIndex >= _columnNum || relativeRowIndex < 0
            || relativeRowIndex >= _rowNum)
        {
            throw new Exception("It is out of range. West column= " + _rfcOrigHrapColumn + ", South row= "
                + _rfcOrigHrapRow + ", Colunm number= " + _columnNum + ", Row number= " + _rowNum);
        }

        //variable(y, x) 
        varIndex.set(relativeRowIndex, relativeColIndex);

        return varIndex;
    }

    /**
     * Stores double array at the specific pixel.
     */
    public void setDoubleArrayVariable(final String varNameOfStringType,
                                       final int hrapCol,
                                       final int hrapRow,
                                       final double[] doubleArray) throws Exception
    {
        this.setStringVariable(varNameOfStringType,
                               hrapCol,
                               hrapRow,
                               OHDUtilities.getStringLineFromDoubleArray(doubleArray));//convert double array to a string separated by " " and stored in netcdf file
    }

    /**
     * Get array of double at the specific pixel of the variable. If not specified at this pixel, returns null.
     */
    public double[] getDoubleArrayVariable(final String variableName, final int hrapCol, final int hrapRow) throws Exception
    {

        final String doubleArrayAsStr = getStringVariable(variableName, hrapCol, hrapRow);

        if(doubleArrayAsStr.equals(getMissingValueAsString(variableName)))
        {
            return null;
        }

        //retrieve the string which holding the double array separated by " ", then convert the string to double array
        return OHDUtilities.getDoubleArrayFromString(doubleArrayAsStr);
    }

    /**
     * Set the range of pixels(on the same row) to the specific value(double).
     */
    public void setDoubleVariableInRange(final String variableName,
                                         final int hrapRow,
                                         final int hrapStartColumn,
                                         final int hrapEndColunm,
                                         final double varValue) throws Exception
    {
        for(int i = hrapStartColumn; i <= hrapEndColunm; i++)
        {
            this.setDoubleVariable(variableName, i, hrapRow, varValue);
        }
    }

    /**
     * Set the specific variable at specific HRAP Column and HRAP Row value(boolean). If the input HRAP Column and HRAP
     * Row is out of the range, throws an Exception. If this variable does not exist, throws an Exception too.
     */
    public void setBoolVariable(final String variableName, final int hrapCol, final int hrapRow, final boolean boolValue) throws Exception
    {
        double varValue;
        if(boolValue)
        {
            varValue = 1.0;
        }
        else
        {
            varValue = 0.0;
        }

        this.setDoubleVariable(variableName, hrapCol, hrapRow, varValue);

    }

    /**
     * Returns the variable boolean value at the specified HRAP Column and HRAP Row. If out of range, throws an
     * exception.
     */
    public boolean getBoolVariable(final String variableName, final int hrapCol, final int hrapRow) throws Exception
    {
        final double result = this.getDoubleVariable(variableName, hrapCol, hrapRow);

        if(result > 0.0)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Set the variable(String type) to the specific string value at the specific pixel location. The String cannot
     * contain the char '!' or '=', because the two chars are used by the code inside.
     * 
     * @param varNameOfStringType - string type variable name
     * @param hrapCol
     * @param hrapRow
     * @param stringValue - the string value to be set at the specific location.
     */
    public void setStringVariable(final String varNameOfStringType,
                                  final int hrapCol,
                                  final int hrapRow,
                                  final String stringValue) throws Exception
    {
        if(_variablArrayMap.keySet().contains(varNameOfStringType) == false)
        {
            throw new Exception("The variable " + varNameOfStringType + " does not exist in the netcdf file");
        }

        final Array varArray = _variablArrayMap.get(varNameOfStringType);
        final Index3D var3DIndex = (Index3D)varArray.getIndex();

        final int relativeColIndex = hrapCol - _rfcOrigHrapColumn;
        final int relativeRowIndex = hrapRow - _rfcOrigHrapRow;

        //if out of range, throws an Exception
        if(relativeColIndex < 0 || relativeColIndex >= _columnNum || relativeRowIndex < 0
            || relativeRowIndex >= _rowNum)
        {
            throw new Exception("It is out of range. West column= " + _rfcOrigHrapColumn + ", South row= "
                + _rfcOrigHrapRow + ", Colunm number= " + _columnNum + ", Row number= " + _rowNum);
        }

        // newStringValue might be needed to defined a longer string
        // if so this means the template netcdf needs to be recrated
        // with a larger dimension
        String newStringValue = stringValue;

        if(stringValue.length() >= _charIndexLength)
        {

            newStringValue = stringValue.substring(0, _charIndexLength - 1);
            _log.log(Logger.WARNING,
                     "The variable " + varNameOfStringType + " of string type was truncated.  "
                         + "The current dimension  is " + _charIndexLength + " but the variable requires "
                         + stringValue.length());
        }

        for(int i = 0; i < newStringValue.length(); i++)
        {
            //variable(y, x, char_index) 
            var3DIndex.set(relativeRowIndex, relativeColIndex, i);

            varArray.setChar(var3DIndex, newStringValue.charAt(i));
        }

        final int lastIndex = newStringValue.length();
        var3DIndex.set(relativeRowIndex, relativeColIndex, lastIndex);

        varArray.setChar(var3DIndex, _END_STRING_SYMBOL);

    }

    /**
     * Get the String type variable value at the specified pixel. If not specified at this pixel, returns
     * {@link NetcdfConstants#STRING_MISSING_VALUE}
     * 
     * @param variableName
     */
    public String getStringVariable(final String variableName, final int hrapCol, final int hrapRow) throws Exception
    {
        final Array varArray = _variablArrayMap.get(variableName);
        final Index3D var3DIndex = (Index3D)varArray.getIndex();

        final int[] variableDims = var3DIndex.getShape(); //variable(y, x, STRING_LENGTH)

        final int relativeColIndex = hrapCol - _rfcOrigHrapColumn;
        final int relativeRowIndex = hrapRow - _rfcOrigHrapRow;

        //if out of range, throws an Exception
        if(relativeColIndex < 0 || relativeColIndex >= _columnNum || relativeRowIndex < 0
            || relativeRowIndex >= _rowNum)
        {
            throw new Exception("It is out of range. West column= " + _rfcOrigHrapColumn + ", South row= "
                + _rfcOrigHrapRow + ", Colunm number= " + _columnNum + ", Row number= " + _rowNum);
        }

        final List<Character> charList = new ArrayList<Character>();

        //variable(y, x, STRING_LENGTH) 
        for(int i = 0; i < variableDims[2]; i++)
        {
            var3DIndex.set(relativeRowIndex, relativeColIndex, i);

            final char letter = varArray.getChar(var3DIndex);

            if(letter == _END_STRING_SYMBOL)
            {//end of the string, jump out of the for loop
                break;
            }

            charList.add(letter);
        }

        //convert charList to charArray
        final char[] charArray = new char[charList.size()];

        for(int i = 0; i < charArray.length; i++)
        {
            charArray[i] = charList.get(i);
        }

        final String resultStr = new String(charArray); //convert char array to String
        return resultStr.trim();//if resultStr is 100 " ", after trimming, it is ""
    }

    /**
     * Set the range of pixels(on the same row) to the specific value(string).
     */
    public void setStringVariableInRange(final String varNameOfStringType,
                                         final int hrapStartColumn,
                                         final int hrapEndColunm,
                                         final int hrapRow,
                                         final String stringValue) throws Exception
    {
        for(int i = hrapStartColumn; i <= hrapEndColunm; i++)
        {
            this.setStringVariable(varNameOfStringType, i, hrapRow, stringValue);
        }
    }

    /**
     * Prints out every pixel with non-missing float values for the specific variable name. For example, for "LZPK"
     * variable:<br>
     * HRAP_ROW= 470 HRAP_COLUMN= 850 LZPK= 1.3<br>
     */
    public void printOutGridsFloatValuesOfVariable(final String varName)
    {

        final Array varArray = _variablArrayMap.get(varName);

        final IndexIterator indexItr = varArray.getIndexIterator();

        while(indexItr.hasNext())
        {
            final float varValue = indexItr.getFloatNext();

            if(GridUtilities.isNonMissingValue(varValue))
            {//this pixel has non-missing value(> -999.0  and != 9.9E36)

                final int[] counterArray = indexItr.getCurrentCounter();
                /*
                 * int[] counterArray depends on this object is RfcGrids2D or RfcGrids3D. If this obj is RfcGrids2D, the
                 * array has two elements: counterArray[0] + _rfcOrigHrapRow is HRAP_ROW; counterArray[1] +
                 * _rfcOrigHrapColumn is HRAP_COLUMN. If this object is RfcGrids3D, the array has three elements:
                 * counterArray[1] + _rfcOrigHrapRow is HRAP_ROW; counterArray[2] + _rfcOrigHrapColumn is
                 * HRAP_COLUMN(counterArray[0] is for the time, 3rd, dimension.
                 */

                if(counterArray.length == 2)
                {
                    System.out.println("HRAP_ROW= " + String.valueOf(counterArray[0] + _rfcOrigHrapRow)
                        + " HRAP_COLUMN= " + String.valueOf(counterArray[1] + _rfcOrigHrapColumn) + " " + varName
                        + "= " + varValue);
                }
                else
                {
                    System.out.println(counterArray[0] + " HRAP_ROW= "
                        + String.valueOf(counterArray[1] + _rfcOrigHrapRow) + " HRAP_COLUMN= "
                        + String.valueOf(counterArray[2] + _rfcOrigHrapColumn) + " " + varName + "= " + varValue);
                }

            }
        } //close while loop
    }

    /**
     * Prints out every pixel with valid String values for the specific variable name. For example, for
     * "OPERATION_CONTENTS" variable:<br>
     * HRAP_ROW= 470 HRAP_COLUMN= 850 OPERATION_CONTENTS= Hello World<br>
     */
    public void printOutGridsStringValuesOfVariable(final String varName) throws Exception
    {

        for(int hrapRow = _rfcOrigHrapRow; hrapRow < (_rfcOrigHrapRow + _rowNum); hrapRow++)
        {
            for(int hrapCol = _rfcOrigHrapColumn; hrapCol < (_rfcOrigHrapColumn + _columnNum); hrapCol++)
            {

                final String varValue = this.getStringVariable(varName, hrapCol, hrapRow).trim();

                if(varValue.equals(NetcdfConstants.STRING_MISSING_VALUE) == false)
                {//this pixel has valid String value

                    System.out.println("HRAP_ROW= " + String.valueOf(hrapRow) + " HRAP_COLUMN= "
                        + String.valueOf(hrapCol) + " " + varName + "= \"" + varValue + "\"");
                }

            } //close inner loop
        }//close outer loop

    }

    /**
     * Going through the whole netcdf grids and print out every grid with positive float values or string values, e.g.:<br>
     * HRAP_ROW= 470 HRAP_COLUMN= 850 LZPK= 1.3<br>
     * HRAP_ROW= 570 HRAP_COLUMN= 860 LZFPM= 7.6<br>
     * HRAP_ROW= 470 HRAP_COLUMN= 850 OPERATION_CONTENTS= Hello World<br>
     * ..
     * 
     * @throws Exception
     */
    public void printOutGridsValues() throws Exception
    {
        for(final String varName: _variablArrayMap.keySet())
        {
            if(this.isVariableStringType(varName))
            {
                this.printOutGridsStringValuesOfVariable(varName);
            }
            else
            {
                this.printOutGridsFloatValuesOfVariable(varName);
            }
        }

    }

    /**
     * Flush out all the variables to the outputFileName. After this method call, _rfcNetcdfFile define mode is back
     * false.
     */
    @SuppressWarnings("deprecation")
    //Netcdf community is considering removing "deprecate" label from setName() method
    public void writeOutNetcdfFile(final String outputFileName) throws Exception
    {
        _rfcNetcdfFile.setName(outputFileName);

        _rfcNetcdfFile.create();

        _rfcNetcdfFile.write(_yVariable.getName(), _yVarArray);
        _rfcNetcdfFile.write(_xVariable.getName(), _xVarArray);

        for(final Entry<String, Array> entry: _variablArrayMap.entrySet())
        {
            final String varName = entry.getKey();
            final Array array = entry.getValue();

            _rfcNetcdfFile.write(varName, array);
        }

        _rfcNetcdfFile.flush();

        this.close();

    }

    /**
     * After the method calling writeOutNetcdfFile(), need to use this method to re-set its define mode to true in order
     * to call writeOutNetcdfFile() again to write to a different file.
     * 
     * @param bool
     * @throws IOException
     */
    public void setRedefineMode(final boolean bool) throws IOException
    {
        _rfcNetcdfFile.setRedefineMode(bool);
    }

    /**
     * @return HRAP South Row number of this RFC.
     */
    public int getRfcOrigHrapRow()
    {
        return _rfcOrigHrapRow;
    }

    /**
     * @return HRAP West column number of this RFC.
     */
    public int getRfcOrigHrapColumn()
    {
        return _rfcOrigHrapColumn;
    }

    /**
     * @return the number of columns of this RFC.
     */
    public int getColumnNum()
    {
        return _columnNum;
    }

    /**
     * @return the number of rows of this RFC.
     */
    public int getRowNum()
    {
        return _rowNum;
    }

    public void printVariableRange() throws Exception
    {
        System.out.println("West column= " + _rfcOrigHrapColumn);
        System.out.println("South row= " + _rfcOrigHrapRow);
        System.out.println("Colunm number= " + _columnNum);
        System.out.println("Row number= " + _rowNum);
    }

    /**
     * Without calling this method, there will be too many files open in FEWS.
     */
    public void close() throws IOException
    {
        _rfcNetcdfFile.close();
    }

    /**
     * Returns true if this variable is data type of String or double array; Returns false if it is data type of float,
     * int or boolean.
     */
    protected boolean isVariableStringType(final String varName)
    {
        final Variable var = _rfcNetcdfFile.findVariable(varName);

        final DataType dataType = var.getDataType();

        return dataType.isString();

    }

    /**
     * Return the variable's unit as a String.
     */
    public String getUnitAsString(final String varName)
    {
        return _variableUnitsMap.get(varName);
    }

    /**
     * Returns the value of the attribute "FillValue" in the NetCDF file for the specific variable, which stands for the
     * missing value. If the variable is float type(absolutely most cases), the missing value is float type. However, a
     * few cases, the variable is String type(including double array case), the missing value is String type. In all
     * cases, this method returns the value in String format.
     */
    public String getMissingValueAsString(final String varName)
    {
        return _variableMissingValMap.get(varName);
    }

    /**
     * Check if the pixel has missing value, assuming float type, not String type.
     */
    public boolean isValueMissing(final String variableName, final int hrapCol, final int hrapRow) throws Exception
    {
        final double value2D = this.getDoubleVariable(variableName, hrapCol, hrapRow);

        final double missingValue = Double.valueOf(getMissingValueAsString(variableName)).doubleValue();

        return (value2D == missingValue);
    }

    /**
     * Returns all the variable names as a String array, excluding the variables used as dimensions, e.g. "x", "y",
     * "time", "duration" and "char_index".
     */
    public String[] getVariableNames()
    {
        final Set<String> nameSet = _variablArrayMap.keySet();

        //convert variable name set to array
        final String[] nameArray = new String[nameSet.size()];

        int index = 0;
        for(final String string: nameSet)
        {
            nameArray[index] = string;

            index++;
        }

        return nameArray;
    }

    /**
     * Returns the NetCDF file name
     */
    public String getTitle()
    {
        return _rfcNetcdfFile.getTitle();
    }
}
