package ohd.hseb.grid;

import java.io.IOException;
import java.util.List;
import java.util.Map.Entry;

import javax.management.timer.Timer;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DominantNumberFinder;
import ohd.hseb.util.fews.LineSeg;
import ohd.hseb.util.fews.ParameterType.Row;
import ucar.ma2.Array;
import ucar.ma2.Index;
import ucar.ma2.Index3D;
import ucar.nc2.Variable;

/**
 * Opens an existing RFC template NETCDF file which has 3 dimensions(3rd dimension, row, column). Its dimension size can
 * not be changed. Only its variable("FFG" or other name) values and time(3rd dimension, the minutes since 1970-01-01
 * 00:00) can be modified. After that, all the contents can be dumped out to another NETCDF file by calling
 * "writeOutNetcdfFile(final String outputFileName)". The original template NETCDF file was not changed.<br>
 * <br>
 * There are two scenarios to use this class: one is directly for holding "FFG"(3rd dim is time) or "THRESHR"(3rd
 * dimension is duration) data(3rd dimension size is 3, 4 or 5 to represent 1HR, 3HR, 6HR, including 12HR, including
 * 24HR respectively). In this case, {@link #_targetVarName} etc are used and {@link RfcGrid#_variablArrayMap} are not
 * used; another scenario is to be parent class of *StatesGrids.java(3rd dimension is time and size is 1). In this case,
 * {@link #_targetVarName} etc are not used and {@link RfcGrid#_variablArrayMap} are used;
 */
public class RfcGridWithTime extends RfcGrid
{
    private String _thirdDimName; //either "time" or "duration"
    protected Array _thirdVarArray; //DataType.DOUBLE based on FEWS' NC file ncdump results, even though seems should be DataType.INT
    private int _thirdDimSize;

    private long _startTimeInMillis;//used in cases of TS or States or FFG, not THRESHOLD_RUNOFF("duration")
    private long _tsIntValInMillis; //used in cases of TS, not States

    /**
     * Opens an existing RFC netcdf file. Retrieves each variable(FFG/or something else, Y, X and TIME). Calculates the
     * RFC West Column and South Row. Does not change the time axis origin and the time steps.
     * 
     * @param rfcNetcdfFile - netcdf file for the RFC.
     * @throws IOException
     */
    public RfcGridWithTime(final String rfcNetcdfFile, final Logger log) throws Exception
    {
        super(rfcNetcdfFile, log);

        final List<Variable> listVar = _rfcNetcdfFile.getVariables();

        //find the 3rd dimension name
        for(final Variable var: listVar)
        {
            final String varName = var.getName();

            if(varName.equals(NetcdfConstants.NETCDF_TIME_VAR_NAME)
                || varName.equals(NetcdfConstants.NETCDF_DURATION_VAR_NAME))
            {
                _thirdDimName = varName;

                final Variable thirdVariable = _rfcNetcdfFile.findVariable(_thirdDimName);
                _thirdVarArray = thirdVariable.read();
                _thirdDimSize = (int)_thirdVarArray.getSize();

                if(varName.equals(NetcdfConstants.NETCDF_TIME_VAR_NAME))
                {//only useful for "time" case; not useful for THRESHOLD_RUNOFF "duration" case

                    _startTimeInMillis = this.getTimeInMinutesByIndex(0) * Timer.ONE_MINUTE;

                    if(_thirdDimSize > 1 && getVarNumber() > 1)
                    {//time series, not State case or FFG case

                        final long timeStep2InMillis = this.getTimeInMinutesByIndex(1) * Timer.ONE_MINUTE;
                        _tsIntValInMillis = timeStep2InMillis - _startTimeInMillis;
                    }
                }

                break;
            }

        }//close for loop

        //check if "time" or "duration" dimension was present in the rfcNetcdfFile, as 3rd dim.
        if(_thirdDimName == null)
        {
            throw new Exception("The NETCDF file[" + rfcNetcdfFile + "] does not contain either time or duration.");
        }

        this.close();
    }

    /**
     * Opens an existing FFG or THRESHOLD_RUNOFF template netcdf file. In addition, change the time axis origin and time
     * steps.<br>
     * Note: this constructor is not suitable for any *StatesGrids.java
     * 
     * @param rfcTemplateNetcdfFile - pre-created netcdf template file for the RFC.
     * @param timeZeroInLong - minutes since 1970-01-01 00:00, used to re-set time variable values.
     */
    public RfcGridWithTime(final String rfcTemplateNetcdfFile, final long timeZeroInLong, final Logger log) throws Exception
    {
        this(rfcTemplateNetcdfFile, log);

        final long minutesSinceEpochTime = timeZeroInLong / Timer.ONE_MINUTE;

        final Index timeIndex = _thirdVarArray.getIndex();

        _thirdVarArray.setDouble(timeIndex.set(0), minutesSinceEpochTime + 60); //1HR
        _thirdVarArray.setDouble(timeIndex.set(1), minutesSinceEpochTime + 180); //3HR
        _thirdVarArray.setDouble(timeIndex.set(2), minutesSinceEpochTime + 360); //6HR

        if((int)_thirdVarArray.getSize() == 4)
        {
            _thirdVarArray.setDouble(timeIndex.set(3), minutesSinceEpochTime + 720); //12HR
        }
        else if((int)_thirdVarArray.getSize() == 5)
        {
            _thirdVarArray.setDouble(timeIndex.set(3), minutesSinceEpochTime + 720); //12HR
            _thirdVarArray.setDouble(timeIndex.set(4), minutesSinceEpochTime + 1440); //24HR
        }

        this.close();

    }

    /**
     * @return - the value on the third dimension("time" or "duration") of the index specified. Index starts from 0.
     */
    public int getTimeInMinutesByIndex(final int indexNum) throws Exception
    {
        if((indexNum + 1) > _thirdDimSize)
        {
            throw new Exception("The index number(starting from 0) requested[" + indexNum
                + "] is greater than available[" + _thirdDimSize + "]");
        }

        final Index timeIndex = _thirdVarArray.getIndex();

        timeIndex.set(indexNum);

        final int result = (int)_thirdVarArray.getDouble(timeIndex);

        return result;
    }

    /**
     * @return the "time" dimension 1st value, in milliseconds.
     */
    public long getStartTimeInMillis() throws Exception
    {
        return _startTimeInMillis;
    }

    /**
     * Flush out all the variables to the outputFileName. After this method call, _rfcNetcdfFile define mode is back
     * false.
     */
    @Override
    @SuppressWarnings("deprecation")
    //Netcdf community is considering removing "deprecate" label from setName() method
    public void writeOutNetcdfFile(final String outputFileName) throws Exception
    {
        _rfcNetcdfFile.setName(outputFileName);

        _rfcNetcdfFile.create();

        _rfcNetcdfFile.write(super._yVariable.getName(), _yVarArray);
        _rfcNetcdfFile.write(super._xVariable.getName(), _xVarArray);
        _rfcNetcdfFile.write(_thirdDimName, _thirdVarArray);

        for(final Entry<String, Array> entry: _variablArrayMap.entrySet())
        {
            final String varName = entry.getKey();
            final Array array = entry.getValue();

            _rfcNetcdfFile.write(varName, array);
        }

        _rfcNetcdfFile.flush();

        this.close();

    }

    /**
     * Set the specific variable at specific time steps, HRAP Column and HRAP Row value(double). If the time steps or
     * the input HRAP Column and HRAP Row is out of the range, throws an Exception. If this variable does not exist,
     * throws an Exception too.
     */
    public void setDoubleVariable(final String variableName,
                                  final int timeSteps,
                                  final int hrapCol,
                                  final int hrapRow,
                                  final double varValue) throws Exception
    {

        final Array varArray = _variablArrayMap.get(variableName);

        final Index3D varIndex = this.getVarIndex(variableName, timeSteps, hrapCol, hrapRow);

        varArray.setFloat(varIndex, (float)varValue);
    }

    /**
     * Returns the variable double value at the specified time steps, HRAP Column and HRAP Row. If out of range, throws
     * an exception.
     */
    public double getDoubleVariable(final String variableName, final int timeSteps, final int hrapCol, final int hrapRow) throws Exception
    {
        final Array varArray = _variablArrayMap.get(variableName);

        final Index3D varIndex = this.getVarIndex(variableName, timeSteps, hrapCol, hrapRow);

        return varArray.getFloat(varIndex);
    }

    /**
     * Returns the variable double value at the specified time in milliseconds, HRAP Column and HRAP Row. If out of
     * range, throws an exception.
     */
    public double getDoubleVariable(final String variableName,
                                    final long timeInMillis,
                                    final int hrapCol,
                                    final int hrapRow) throws Exception
    {
        final int thirdDimIndex = (int)((timeInMillis - _startTimeInMillis) / _tsIntValInMillis); //could be 0, when timeInMillis equal to _startTimeInMillis

        return this.getDoubleVariable(variableName, thirdDimIndex, hrapCol, hrapRow);
    }

    /**
     * A helper method returning the NetCDF index for the specific variable name and the locations(timeSteps, hrapCol
     * and hrapRow). It also checks those location specifications within the RFC range or not.<br>
     * 
     * @param timeSteps - starting from 0.
     */
    private Index3D getVarIndex(final String variableName, final int timeSteps, final int hrapCol, final int hrapRow) throws Exception
    {
        if(_variablArrayMap.keySet().contains(variableName) == false)
        {
            throw new Exception("The variable " + variableName + " does not exist in the netcdf file");
        }

        final Index3D varIndex = (Index3D)super.getVarIndex(variableName);

        final int relativeColIndex = hrapCol - _rfcOrigHrapColumn; //starting from 0
        final int relativeRowIndex = hrapRow - _rfcOrigHrapRow; //starting from 0

        //if out of range, throws an Exception
        if(relativeColIndex < 0 || relativeColIndex >= _columnNum || relativeRowIndex < 0
            || relativeRowIndex >= _rowNum || timeSteps >= _thirdDimSize)
        {
            throw new Exception("Out of range: available HRAP West column= " + _rfcOrigHrapColumn
                + ", HRAP South row= " + _rfcOrigHrapRow + ", Colunm number= " + _columnNum + ", Row number= "
                + _rowNum + " third dimension size=" + _thirdDimSize + ". But trying to access HRAP_Col[" + hrapCol
                + "], HRAP_Row[" + hrapRow + "] and " + _thirdDimName + "(starting from 0)[" + timeSteps + "]");
        }

        //variable(timeSteps, y, x) 
        varIndex.set(timeSteps, relativeRowIndex, relativeColIndex);

        return varIndex;
    }

    /**
     * Set the value at the specified pixel. This method does not specify which variable to set so it only works when
     * there is only one variable, like "FFG" or "THRESHR". It does not work with cases like states, parameters and time
     * series.
     * 
     * @param durationIndex - 0(1 HR), 1(3 HR), 2(6 HR), etc
     * @param hrapCol
     * @param hrapRow
     * @param gridValue
     */
    public void setPixelVariableValue(final int durationIndex,
                                      final int hrapCol,
                                      final int hrapRow,
                                      final float gridValue)
    {
        this.setPixelVariableValue(durationIndex, hrapCol, hrapRow, gridValue, false);
    }

    /**
     * Get the target variable value at the specified pixel. This method does not specify which variable to retrieve so
     * it only works when there is only one variable, like "FFG" or "THRESHR". It does not work with cases like states,
     * parameters and time series.
     * 
     * @param durationIndex - 0(1 HR), 1(3 HR), 2(6 HR), etc
     * @param hrapCol
     * @param hrapRow
     */
    public float getPixelValue(final int durationIndex, final int hrapCol, final int hrapRow)
    {
        final int relativeColIndex = hrapCol - _rfcOrigHrapColumn;
        final int relativeRowIndex = hrapRow - _rfcOrigHrapRow;

        //float targetVariable(time, y, x) 
        getVarIndex().set(durationIndex, relativeRowIndex, relativeColIndex);

        final float gridValue = getVarArray().getFloat(getVarIndex());

        return gridValue;
    }

    /**
     * Set the grid values of the lineseg's pixels. If lineseg has several parts(rows), set each row's ffg value. This
     * method does not specify which variable to set so it only works when there is only one variable, like "FFG" or
     * "THRESHR". It does not work with cases like states, parameters and time series.
     * 
     * @param durationIndex - 0, 1, 2, 3 etc.
     * @param lineSeg - object of {@link LineSeg} which contains many {@link Row}s.
     * @param logger
     */
    public void setFfgGridValuesFromLineSeg(final int durationIndex, final LineSeg lineSeg)
    {
        for(final Row row: lineSeg.getRowList())
        {
            if(row.getF() == null)
            {//some rows, because its threshold runoff value is missing value, so its ffg value(F column) was not set, move on to next row
                continue;
            }

            this.setGridValuesFromRow(durationIndex, row);
        }
    }

    /**
     * Each row, defined from a table in paramsGridded.xml, has same threshold runoff value, so every pixel on this row
     * has the same calculated ffg value.
     * 
     * @param durationIndex - 0, 1, 2, 3 etc.
     * @param row - its F column has ffg value as String; its C columns value is start HRAP Column; its D column value
     *            is end HRAP Column. Its C and D value could be changed due to grid filling.
     */
    private void setGridValuesFromRow(final int durationIndex, final Row row)
    {
        final int hrapRow = Integer.valueOf(row.getB());

        final int startHrapColumnNum = Integer.valueOf(row.getC());

        final int endHrapColumnNum = Integer.valueOf(row.getD());

        final float ffgValue = Float.valueOf(row.getF());

        //original columns, not considering grid filling
        final boolean isGridFilling = false;
        for(int colNum = startHrapColumnNum; colNum <= endHrapColumnNum; colNum++)
        {
            this.setPixelVariableValue(durationIndex, colNum, hrapRow, ffgValue, isGridFilling);
        }

    }

    /**
     * Fill left side pixels' ffg value with the lineseg's first pixel's ffg value and right side pixels' ffg values
     * with the lineseg's last pixel value. If the lineseg's first pixel's ffg value is missing value, don't fill left
     * side's pixels. If the lineseg's last pixel's ffg value is missing, don't fill right side pixels.<br>
     * Note: after filling, expand the lineseg's starting column and ending column.
     * 
     * @param durationIndex
     * @param lineSeg
     * @param gridFill
     */
    public void fillFfgLineSegLeftAndRightPixels(final int durationIndex, final LineSeg lineSeg, final int gridFill)
    {
        //grid filling left and right columns
        final boolean isGridFilling = true;
        int startHrapColumnNum = lineSeg.getHrapStartColunm() - gridFill;
        if(startHrapColumnNum < _rfcOrigHrapColumn)
        {
            startHrapColumnNum = _rfcOrigHrapColumn;
        }

        int endHrapColumnNum = lineSeg.getHrapEndColunm() + gridFill;

        final int rfcEastColumn = _rfcOrigHrapColumn + _columnNum - 1;
        if(endHrapColumnNum > rfcEastColumn)
        {
            endHrapColumnNum = rfcEastColumn;
        }

        //grid filling left side columns if 1st pixel ffg value is not missing
        final int hrapRow = lineSeg.getHrapRowNum();
        float ffgValue = lineSeg.getFirstPixelFfgValue();

        if(ffgValue >= 0.0)
        {
            for(int colNum = startHrapColumnNum; colNum < lineSeg.getHrapStartColunm(); colNum++)
            {
                this.setPixelVariableValue(durationIndex, colNum, hrapRow, ffgValue, isGridFilling);
            }

            lineSeg.setHrapStartColunm(startHrapColumnNum); //expand the row due to grid filling
        }

        //grid filling right side columns if last pixel ffg value is not missing
        ffgValue = lineSeg.getLastPixelFfgValue();
        if(ffgValue >= 0.0)
        {
            for(int colNum = lineSeg.getHrapEndColunm() + 1; colNum <= endHrapColumnNum; colNum++)
            {
                this.setPixelVariableValue(durationIndex, colNum, hrapRow, ffgValue, isGridFilling);
            }

            lineSeg.setHrapEndColunm(endHrapColumnNum); //expand the row due to grid filling
        }

    }

    /**
     * @param durationIndex - 0, 1, 2, 3 etc.
     * @param hrapCol - HRAP Column number
     * @param hrapRow - HRAP Row number
     * @param ffgValue - ffg value in inch
     * @param isGridFilling - if true, this pixel is grid filling, needs to consider the original value is missing or
     *            not; if false, this pixel is not grid filling.
     */
    private void setPixelVariableValue(final int durationIndex,
                                       final int hrapCol,
                                       final int hrapRow,
                                       final float ffgValue,
                                       final boolean isGridFilling)
    {
        final int relativeColIndex = hrapCol - _rfcOrigHrapColumn;
        final int relativeRowIndex = hrapRow - _rfcOrigHrapRow;

        //if out of range, do nothing
        if(relativeColIndex < 0 || relativeColIndex >= _columnNum)
        {
            return;
        }

        if(relativeRowIndex < 0 || relativeRowIndex >= _rowNum)
        {
            return;
        }

        //float FFG(time, y, x) 
        getVarIndex().set(durationIndex, relativeRowIndex, relativeColIndex);

        final float oldFfgValue = getVarArray().getFloat(getVarIndex());

        if(isGridFilling)
        {//for left or right side pixels due to grid filling, only fill it when it has missing value
            if(oldFfgValue < 0.0)
            {
                getVarArray().setFloat(getVarIndex(), ffgValue);

                _log.log(Logger.DEBUG, "Grid filling the pixel[" + durationIndex + "," + relativeRowIndex + ","
                    + relativeColIndex + "], ffg=" + ffgValue);
            }
            else
            {
                _log.log(Logger.DEBUG, "Do not grid fill the pixel[" + durationIndex + "," + relativeRowIndex + ","
                    + relativeColIndex + "], because it has positive value already.");
            }
        }
        else
        {//for original pixels
            getVarArray().setFloat(getVarIndex(), ffgValue);
        }

    }

    /**
     * If being parent class of *StatesGrids.java, print out warm state time; if holding "FFG" or "THRESHR", print out
     * "time" or "duration" size.
     */
    @Override
    public void printVariableRange() throws Exception
    {
        if(_thirdDimSize > 1)
        {//just print out how many they are
            System.out.println(_thirdDimName + " steps= " + _thirdDimSize);
        }
        else
        {//_thirdDimSize == 1, print out the real value(converted to easy read format)
            System.out.println(_thirdDimName + " = "
                + DateTime.getDateTimeStringFromLong(this.getStartTimeInMillis(), null));
        }
        super.printVariableRange();
    }

    /**
     * Do FFG grid filling with its control be 6. Going through every pixel in the whole RFC. If a pixel with missing
     * ffg value has 5 or more than 5 pixels with non-negative ffg values(including 0.0), this pixel will be filled with
     * ffg values which is the dominant value among the surrounding pixels.
     */
    public void ffgGridFill6()
    {

        final DominantNumberFinder dominatFinder = new DominantNumberFinder();

        for(int timeStepCount = 0; timeStepCount < _thirdDimSize; timeStepCount++)
        {

            for(int rowCount = 0; rowCount < _rowNum; rowCount++)
            {

                for(int colCount = 0; colCount < _columnNum; colCount++)
                {
                    dominatFinder.reSet();

                    final int[] targetCount = new int[]{timeStepCount, rowCount, colCount};

                    getVarIndex().set(targetCount); //set the index to point to the target pixel

                    if(getVarArray().getFloat(getVarIndex()) < 0.0)
                    {//this target pixel has missing value

                        //check surrounding pixels around the target
                        for(int jj = 1; jj <= 3; jj++)
                        {
                            final int j = rowCount + jj - 2;

                            if(j < 0 || j >= _rowNum)
                            {
                                continue;
                            }

                            for(int ii = 1; ii <= 3; ii++)
                            {
                                final int i = colCount + ii - 2;

                                if(i < 0 || i >= _columnNum)
                                {
                                    continue;
                                }

                                final int[] newCounter = new int[]{timeStepCount, j, i};

                                getVarIndex().set(newCounter); //re-set the index to point to the neighbor pixel

                                final float neighborFfg = getVarArray().getFloat(getVarIndex());

                                if(neighborFfg >= 0.0)
                                {//this neighbor pixel has valid value

                                    dominatFinder.add(neighborFfg);
                                }

                            }

                        } //finished checking surrounding pixels

                        if(dominatFinder.getTotalCount() >= 5)
                        {
                            final float predominatValue = dominatFinder.getPredominatValueAsFloat(); //at least 5 pixels surrounding have values

                            getVarIndex().set(targetCount); //set the index to point to the target pixel

                            getVarArray().setFloat(getVarIndex(), predominatValue);
                        }

                    } //close if(ffgDataArray.getFloat(ffgIndex3D) < 0.0)

                } //close column for loop
            } //close row for loop
        } //close timeStep for loop

    }//close method

    /**
     * @return the third dimension(e.g. "time", "duration") size.
     */
    public int getTimeOrDurationDimSize()
    {
        return _thirdDimSize;
    }

    /**
     * Check if the pixel has missing value or not, within specified time period. If it has one missing value, return
     * true; if all values are present, not missing, returns false. This method only applies to RfcGrids3D representing
     * time series case.
     */
    public boolean isTsValueMissing(final String variableName,
                                    final int hrapCol,
                                    final int hrapRow,
                                    final long startTime,
                                    final long endTime) throws Exception
    {

        final int startIndex = (int)((startTime - _startTimeInMillis) / _tsIntValInMillis);
        final int endIndex = (int)((endTime - _startTimeInMillis) / _tsIntValInMillis);

        boolean result;
        for(int i = startIndex; i <= endIndex; i++)
        {
            result = this.isValueMissing(variableName, i, hrapCol, hrapRow);

            if(result)
            {
                return true;
            }
        }

        //if reaching here, all values within specified period must be non-missing
        return false;
    }

    /**
     * Check if the pixel at specific time step has missing value, assuming float type, not String type.
     */
    public boolean isValueMissing(final String variableName, final int timeSteps, final int hrapCol, final int hrapRow) throws Exception
    {
        final double missingValue = Double.valueOf(getMissingValueAsString(variableName)).doubleValue();

        final double value3D = this.getDoubleVariable(variableName, timeSteps, hrapCol, hrapRow);

        return (value3D == missingValue);
    }

    /**
     * Return time series interval in hours. This method can only be used when this is time series grid.
     * 
     * @throws Exception
     */
    public int getTimeSeriesIntervalInHR() throws Exception
    {
        if(_tsIntValInMillis <= 0)
        {
            throw new Exception("Error: Grid time series interval is not valid.  At least two time steps "
                + "are needed for computing a time series interval");
        }
        return (int)(_tsIntValInMillis / Timer.ONE_HOUR);
    }

    public Array getTimeArray()
    {
        return _thirdVarArray;
    }

}
