/*
 * Created on Sep 26, 2003
 */
package ohd.hseb.measurement;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.model.ForecastAdjusterParams;
import ohd.hseb.model.ForecastInterpolationMethod;

/**
 * @author Chip Gobs
 */
final public class IrregularTimeSeries
{

    private final double _missingValue = -999.0;
    private List<Measurement> _measurementList = new ArrayList<Measurement>();
    private MeasuringUnit _measuringUnit = null;
    private String _name = null;

    /**
     * represents the timeseries type such as MAT, MAP, RAIM etc.
     */
    private String _timeSeriesType;

    /*
     * added for FEWS; used to uniquely define a timeseries
     */
    private String _qualifierId = "";

    //---------------------------------------------------------------------

    //---------------------------------------------------------------------

    public IrregularTimeSeries(final MeasuringUnit unit)
    {
        _measuringUnit = unit;
    }

//	---------------------------------------------------------------------

    /**
     * @param newTimeSeries This add method perfoms mathematical addition, not insertion to the set of values. See
     *            insertXXX methods for insertion.
     */

    public static IrregularTimeSeries add(final IrregularTimeSeries ts1, final IrregularTimeSeries ts2)
    {
        //algorithm

        //1.  find out all the times for which either time series has a measurement.
        //   store these times in an array
        //2. iterate over the list to drive the addition of the 2 time series.
        //  There is a getMeasurementByTime  method that allows for filling in zeroes
        // and for interpolation adn for exact Measurement retrieval.

        final MeasuringUnit unit1 = ts1.getMeasuringUnit();
        final MeasuringUnit unit2 = ts2.getMeasuringUnit();

        final IrregularTimeSeries combinedTs = new IrregularTimeSeries(unit1);
        final IrregularTimeSeries newTs = new IrregularTimeSeries(unit1);

        // make sure that ts2 contains the same units;
        // this might create an unpleasant side effect, so I convert it back at the end

        if(unit1 != unit2)
        {
            ts2.convert(unit1);
        }

        // insert all of the measurements into the combined time series	  
        combinedTs.insertTimeSeries(ts1);
        // System.out.println("combined with ts1 \n " + combinedTs);

        combinedTs.insertTimeSeries(ts2);
        // System.out.println("combined with ts1 and ts2 \n" + combinedTs);

        // get the array out of the time series
        final AbsTimeMeasurement[] combinedArray = combinedTs.getMeasurementArray();

        //for now, assume that any values out of range for a particular time series
        // = 0.0;

        final boolean allowInterpolation = true;
        final boolean fillInEndsWithZero = true;

        long desiredTime = -1;
        long previousTime = -2;

        for(int i = 0; i < combinedArray.length; i++)
        {
            final AbsTimeMeasurement currentMeasurement = combinedArray[i];

            desiredTime = currentMeasurement.getTime();

            if(desiredTime != previousTime) //don't want duplicate entries, so checking
            {

                final AbsTimeMeasurement m1 = ts1.getAbsTimeMeasurementByTime(desiredTime,
                                                                              allowInterpolation,
                                                                              fillInEndsWithZero);

                final AbsTimeMeasurement m2 = ts2.getAbsTimeMeasurementByTime(desiredTime,
                                                                              allowInterpolation,
                                                                              fillInEndsWithZero);
                final double value = m1.getValue() + m2.getValue();
                final AbsTimeMeasurement newM = new AbsTimeMeasurement(value, desiredTime, unit1);
                newTs.insertMeasurement(newM);
            }

            previousTime = desiredTime;

        } //end for i

        //convert back - might be needed
        ts2.convert(unit2);

        return newTs;

    } // end add

    //---------------------------------------------------------------------
    public void convert(final MeasuringUnit newUnit)
    {
//        final String header = "TimeSeries.convert(): ";
        //  System.out.println(header + "is being called for ts = " + _name );

        //  trace();

        final List<Measurement> newMeasurementList = new ArrayList<Measurement>();

        if(_measuringUnit != newUnit)
        {
            for(int i = 0; i < _measurementList.size(); i++)
            {
                final Measurement origMeasurement = _measurementList.get(i);
                final Measurement convertedMeasurement = Measurement.getConvertedCopy(origMeasurement, newUnit);

                newMeasurementList.add(convertedMeasurement);

            }

            _measuringUnit = newUnit;
            _measurementList = newMeasurementList;
        }

    }

    //---------------------------------------------------------------------

    public long getEndTime()
    {
        long endTime = -1;
        if(_measurementList.size() > 0)
        {
            endTime = getAbsTimeMeasurementByIndex(_measurementList.size() - 1).getTime();
        }
        return endTime;
    }

    //---------------------------------------------------------------------
    public AbsTimeMeasurement findClosestMeasurementByTime(final long targetTime)
    {
        AbsTimeMeasurement curMeasurement = null;

        AbsTimeMeasurement closestMeasurement = null;
        long smallestTimeDiff = -1;

        long previousTimeDiff = -1;
        long timeDiff = 0;

        boolean done = false;

        for(int i = 0; !done && i < _measurementList.size(); i++)
        {
            curMeasurement = (AbsTimeMeasurement)_measurementList.get(i);

            timeDiff = Math.abs(curMeasurement.getTime() - targetTime);

            if(closestMeasurement == null) //first time
            {
                smallestTimeDiff = timeDiff;
                closestMeasurement = curMeasurement;
            }
            else
            //not first time
            {
                if(timeDiff < smallestTimeDiff)
                {
                    smallestTimeDiff = timeDiff;
                    closestMeasurement = curMeasurement;
                }
                else if(timeDiff > previousTimeDiff) //gone too far
                {
                    done = true;
                }
            }

            previousTimeDiff = timeDiff;

        }

        return closestMeasurement;
    }

    //	---------------------------------------------------------------------
    public AbsTimeMeasurement getMinMeasurement(final long startTime, final long endTime)
    {
        //could keep track of this internally when items are added or deleted from the list, but then it would have to
        // be called everytime something is deleted, or else set a boolean marker to note that something would have to be recalculated in this
        // method the next time it is called
        AbsTimeMeasurement measurement = null;

        AbsTimeMeasurement minMeasurement = null;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            measurement = (AbsTimeMeasurement)_measurementList.get(i);
            final long time = measurement.getTime();

            if((time >= startTime) && (time <= endTime))
            {
                if((measurement.getValue() != _missingValue)
                    && ((minMeasurement == null) || (measurement.getValue() < minMeasurement.getValue())))
                {
                    minMeasurement = measurement;
                }

                if(measurement.getValue() < minMeasurement.getValue())
                {
                    minMeasurement = measurement;
                }
            }
            else if(time > endTime)
            {
                break;
            }
        } //end for i

        return minMeasurement;

    }

    //  ---------------------------------------------------------------------

    public AbsTimeMeasurement getMinMeasurement()
    {

        return getMinMeasurement(getStartTime(), getEndTime());

    }

    //---------------------------------------------------------------------
    public AbsTimeMeasurement getMaxMeasurement(final long startTime, final long endTime)
    {
        AbsTimeMeasurement measurement = null;
        AbsTimeMeasurement maxMeasurement = null;

//        final double value = 0;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            measurement = (AbsTimeMeasurement)_measurementList.get(i);
            final long time = measurement.getTime();

            if((time >= startTime) && (time <= endTime))
            {
                if((measurement.getValue() != _missingValue) &&

                ((maxMeasurement == null) || (measurement.getValue() > maxMeasurement.getValue())))
                {
                    maxMeasurement = measurement;
                }
            }
            else if(time > endTime)
            {
                break;
            }
        } //end for

        return maxMeasurement;
    }

    //---------------------------------------------------------------------

    public AbsTimeMeasurement getMaxMeasurement()
    {
        return getMaxMeasurement(getStartTime(), getEndTime());
    }

    //---------------------------------------------------------------------

    public AbsTimeMeasurement[] getMeasurementArray()
    {
        final AbsTimeMeasurement[] measurementArray = new AbsTimeMeasurement[_measurementList.size()];

        for(int index = 0; index < _measurementList.size(); index++)
        {
            measurementArray[index] = getAbsTimeMeasurementByIndex(index);
        }
        return measurementArray;

    } //getMeasurementArray

    //---------------------------------------------------------------------

    public AbsTimeMeasurement getAbsTimeMeasurementByIndex(final int index)
    {
        AbsTimeMeasurement measurement = null;

        measurement = (AbsTimeMeasurement)_measurementList.get(index);

        return measurement;

    } //getMeasurement

    //---------------------------------------------------------------------

    public AbsTimeMeasurement getAbsTimeMeasurementByTime(final long desiredTime,
                                                          final boolean interpolate,
                                                          final boolean fillEndsWithZero)
    {
        AbsTimeMeasurement measurement = null;
        AbsTimeMeasurement curMeasurement = null;

        AbsTimeMeasurement previousMeasurement = null;
        AbsTimeMeasurement nextMeasurement = null;

        if((desiredTime < this.getStartTime()) || (desiredTime > this.getEndTime()))
        {
            if(fillEndsWithZero)
            {
                measurement = new AbsTimeMeasurement(0.0, desiredTime, this._measuringUnit);
                measurement.setIsDefaulted(true);
            }
        }

        else
        // the desired time is in the range of the measurements contained in the
        // TimeSeries
        {
            for(int i = 0; i < _measurementList.size(); i++)
            {
                curMeasurement = getAbsTimeMeasurementByIndex(i);
                if(curMeasurement.getTime() == desiredTime)
                {
                    //found it exactly
                    measurement = curMeasurement;
                    break;
                }
                else if(curMeasurement.getTime() < desiredTime)
                {
                    previousMeasurement = curMeasurement;
                    //don't break, keep looking
                }
                else
                //curMeasurement.getTime() > time)
                {
                    nextMeasurement = curMeasurement;

                    if(interpolate)
                    {
                        measurement = AbsTimeMeasurement.interpolate(previousMeasurement, nextMeasurement, desiredTime);
                    }

                    break; // it is not in here, since we are past where it
                    // should be
                }
            } //end for

        } //end else

        return measurement;

    } //getMeasurement

    //---------------------------------------------------------------------

    public int getMeasurementCount()
    {
        return _measurementList.size();
    }

    //---------------------------------------------------------------------

    public double[] getMeasurementValueArray() throws Exception
    {
        return getMeasurementValueArray(_measuringUnit);
    } //end getMeasurementValueArray

    //---------------------------------------------------------------------

    public double[] getMeasurementValueArray(final MeasuringUnit unit) throws Exception
    {
        final double[] valueArray = new double[_measurementList.size()];

        if(_measuringUnit != unit)
        {
            convert(unit);

        }

        for(int index = 0; index < _measurementList.size(); index++)
        {

            valueArray[index] = getAbsTimeMeasurementByIndex(index).getValue();
        }

        return valueArray;

    } //end getMeasurementValueArray

    //---------------------------------------------------------------------

    public MeasuringUnit getMeasuringUnit()
    {
        return _measuringUnit;
    }

    //---------------------------------------------------------------------

    public long getStartTime()
    {
        long startTime = -1;
        if(_measurementList.size() > 0)
        {
            startTime = getAbsTimeMeasurementByIndex(0).getTime();
        }
        return startTime;
    }

    //---------------------------------------------------------------------

    public void insertMeasurement(final AbsTimeMeasurement newMeasurement)
    {
        //insert the measurement in the list in ascending order of time
        boolean inserted = false;

        //convert to the units of the TimeSeries
        //try
        {
//            final AbsTimeMeasurement convertedMeasurement = 
            AbsTimeMeasurement.getConvertedCopy(newMeasurement, _measuringUnit);
        }
        //catch (MeasuringUnitConversionException e)
        //{
        //    e.printStackTrace();
        //    throw new Error (e.getMessage());
        //}

        // if the first measurement in the list
        if(_measurementList.size() == 0)
        {
            _measurementList.add(0, newMeasurement);
            inserted = true;
        }
        else if(newMeasurement.getTime() >= this.getEndTime())
        {

            _measurementList.add(newMeasurement);
            inserted = true;
        }

        // if the proper location was not found
        if(!inserted)
        {
            //find the location to insert the measurement within the list
            // and insert it
            for(int i = 0; i < _measurementList.size() && (!inserted); i++)
            {
                final AbsTimeMeasurement currentMeasurement = (AbsTimeMeasurement)_measurementList.get(i);
                if(newMeasurement.getTime() < currentMeasurement.getTime())
                {
                    _measurementList.add(i, newMeasurement);
                    inserted = true;
                }
            } //end for i
        }

        return;

    } //insertMeasurement

    //---------------------------------------------------------------------

    public void insertTimeSeries(final IrregularTimeSeries ts)
    {
        if(ts != null)
        {
            try
            {
                for(int i = 0; i < ts._measurementList.size(); i++)
                {
                    this.insertMeasurement((ts.getAbsTimeMeasurementByIndex(i)));
                }
            }
            catch(final Exception e)
            {
                //can't really happen, since the TimeSeries can't
                //hold a measurement of the wrong type
                throw new Error("woh! somehow a bad measurement got in here ");

            }
        }
    } //end insertTimeSeries
//	---------------------------------------------------------------------

    public IrregularTimeSeries getSubTimeSeries(final long startTime, final long endTime)
    {
        /**
         * Return the time series that is within this time window, inclusively.
         */
        final IrregularTimeSeries newTs = new IrregularTimeSeries(this._measuringUnit);

        for(int i = 0; i < this.getMeasurementCount(); i++)
        {
            final AbsTimeMeasurement m = this.getAbsTimeMeasurementByIndex(i);
            final long mTime = m.getTime();
            if((mTime >= startTime) && (mTime <= endTime))
            {
                newTs.insertMeasurement(m);
            }
        }

        return newTs;

    }

//---------------------------------------------------------------------

    public void setName(final String name)
    {
        _name = name;
    }

//	---------------------------------------------------------------------

    public String getName()
    {
        return _name;
    }

    //---------------------------------------------------------------------

    public void setTimeSeriesType(final String timeseriesType)
    {
        _timeSeriesType = timeseriesType;

    }

    public String getTimeSeriesType()
    {
        return _timeSeriesType;

    }

    public void setQualifierId(final String qualifierId)
    {
        _qualifierId = qualifierId;

    }

    public String getQualifierId()
    {
        return _qualifierId;
    }

    @Override
    public String toString()
    {

        final StringBuffer buffer = new StringBuffer();

        if(getName() != null)
        {
            buffer.append("Name = " + getName() + " ");
        }

        buffer.append("Units: " + _measuringUnit + " ");

        final NumberFormat f = new DecimalFormat("###.##");

        for(int i = 0; i < _measurementList.size(); i++)
        {
            final double value = getAbsTimeMeasurementByIndex(i).getValue();
            final String valueString = f.format(value);
            buffer.append(valueString + " ");
        }

        return buffer.toString();
    }

//	---------------------------------------------------------------------
    public static void main(final String[] argStringArray)
    {
        final MeasuringUnit dischargeUnit = MeasuringUnit.cfs;
        final ForecastAdjusterParams params = new ForecastAdjusterParams();
        params.setBlendingHours(5);
        params.setBlendingMethod(ForecastInterpolationMethod.DIFFERENCE);
        params.setShouldDoAdjustment(true);

//        ForecastAdjuster adjuster = new ForecastAdjuster(params);

        final long fcstStartTime = System.currentTimeMillis();

        final long MILLIS_PER_HOUR = 1000 * 3600;

        final long obsStartTime = fcstStartTime - (3 * 24 * MILLIS_PER_HOUR); //3 days before fcst start
        final long obsEndTime = fcstStartTime + (7 * MILLIS_PER_HOUR); // 7 hours after fcst start

        final IrregularTimeSeries observedTs = new IrregularTimeSeries(dischargeUnit);

        final double[] obsMeasurementValueArray = {18, 19, 20, 22, 24, 28, 33, 28, 39, 40, 41, 43, 53, 57, 41, 29, 29,
            28, 27, 26, 26, 24, 28};

        int index = 0;

        // load up the observed time series
        index = 0;
        for(long time = obsStartTime; time <= obsEndTime; time += MILLIS_PER_HOUR)
        {
            final AbsTimeMeasurement m = new AbsTimeMeasurement(obsMeasurementValueArray[index], time, dischargeUnit);

            observedTs.insertMeasurement(m);

            index++;
            if(index == obsMeasurementValueArray.length)
            {
                index = 0;
            }
        }

        final long subStartTime = obsStartTime + (2 * MILLIS_PER_HOUR);
        final long subEndTime = obsEndTime - (2 * MILLIS_PER_HOUR);

        final IrregularTimeSeries subTimeSeries = observedTs.getSubTimeSeries(subStartTime, subEndTime);

        System.out.println("origObsTimeSeries = " + observedTs);
        System.out.println("subObsTimeSeries = " + subTimeSeries);

    }

} //end class TimeSeries
