/*
 * Created on Jul 2, 2003 This class is a fundamental class to be used as the basis for time series and other
 * measurements. TEST
 */

package ohd.hseb.measurement;

import java.util.Date;

/**
 * @author Chip Gobs
 */
public class Measurement
{

    //--------------------------------------------------------------- 
    public static Measurement getConvertedCopy(final Measurement origMeasurement, final MeasuringUnit toUnit)
    {
        Measurement newMeasurement = null;

        if(origMeasurement != null)
        {
            newMeasurement = new Measurement(origMeasurement);
            newMeasurement.convert(toUnit);
        }
        return newMeasurement;
    }

    public static void main(final String[] args) throws Exception
    {
        final AbsTimeMeasurement m1 = new AbsTimeMeasurement(3.0, new Date().getTime(), MeasuringUnit.feet);

        System.out.println(m1);

        final AbsTimeMeasurement m2 = AbsTimeMeasurement.getConvertedCopy(m1, MeasuringUnit.inches);

        System.out.println(m2);

    } //end main

    private MeasuringUnit _unit;
    private double _value;

    private boolean _flag; //_flag to hold quality flag from FEWS.

    private String _comment;

    private boolean _interpolated = false;
    private boolean _extrapolated = false;
    private boolean _defaulted = false;

    private boolean _isMissing = false;

    private final double _missingValue = -999.0;

    //consider a separate valid time class

    //---------------------------------------------------------------

    public Measurement(final double value, final MeasuringUnit unit)
    {
        _value = value;
        _unit = unit;

        _isMissing = false;

        if(_unit == null)
        {
            throw new Error("ERROR: Measurement.Measurement() MeasuringUnit may not be null");
        }
        _flag = true;

        return;
    }

    public Measurement(final double value, final MeasuringUnit unit, final boolean flag)
    {
        this(value, unit);
        _flag = flag;
    }

    //add by deng temp

    public Measurement(final double value, final MeasuringUnit unit, final boolean flag, final String comment)
    {
        this(value, unit, flag);
        _comment = comment;
    }

    //---------------------------------------------------------------

    public Measurement(final Measurement measurement)
    {
        _value = measurement.getValue();
        _unit = measurement.getUnit();

        _isMissing = measurement.isMissing();
        _flag = measurement._flag;
        _comment = measurement._comment;

        return;
    }

    //---------------------------------------------------------------
    public void addMeasurement(final Measurement measurement)
    {
        measurement.convert(_unit);

        this.setValue(this.getValue() + measurement.getValue());

    }

    //--------------------------------------------------------------- 

    public void convert(final MeasuringUnit toUnit)
    {
        final MeasuringUnit fromUnit = this.getUnit();

        //since MeasueringUnit.mm etc and MeasureingUnitType length etc are public static variables, "==" is ok:
        if(fromUnit == toUnit)
        {
            // no conversion needed
            return;
        }

        if(fromUnit.getType() == toUnit.getType())
        {
            //when switching temperature, MeasuringUnit.getConversionFactor cannot do it
            if(fromUnit == MeasuringUnit.degreesCelsius)
            {//oC --> oF
                _value = 9.0 * _value / 5.0 + 32.0;
            }
            else if(fromUnit == MeasuringUnit.degreesFahrenheit)
            {//oF --> oC
                _value = 5.0 * (_value - 32) / 9.0;
            }
            else
            {
                final double factor = MeasuringUnit.getConversionFactor(fromUnit, toUnit);
                _value = _value * factor;

            }

            _unit = toUnit;
        }
        else
        //can't convert these types
        {
            throw new Error("Bogus Unit conversion attempt from " + fromUnit.getName() + " to " + toUnit.getName());
        }
    } //end convert

    public boolean equals(Measurement measurement)
    {
        boolean result = false;

        if(this._unit != measurement.getUnit())
        {
            measurement = this.getCopy(this._unit);
        }

        if(this.getValue() == measurement.getValue())
        {
            result = true;
        }

        if(this._flag == measurement.getFlag())
        {
            result = true;
        }

        return result;

    }

    //--------------------------------------------------------------- 

    //--------------------------------------------------------------- 
    @Override
    public boolean equals(final Object object)
    {
        boolean result = false;

        if(object instanceof Measurement)
        {
            final Measurement m = (Measurement)object;
            result = this.equals(m);
        }

        return result;

    }

    //--------------------------------------------------------------- 

    public String getComment()
    {
        return _comment;
    }

    //---------------------------------------------------------------

    /**
     * returns an independent copy.
     */
    public Measurement getCopy()
    {
        return new Measurement(this);
    }

    //---------------------------------------------------------------

    public Measurement getCopy(final MeasuringUnit unit)
    {
        return Measurement.getConvertedCopy(this, unit);
    }

    public boolean getFlag()
    {
        return _flag;
    }

    public MeasuringUnit getUnit()
    {
        return _unit;
    }

    public double getValue()
    {
        if(_isMissing)
        {
            _value = _missingValue;
        }

        return _value;
    }

    //---------------------------------------------------------------   
    public double getValue(final MeasuringUnit unit)
    {
        //make an independent copy so that will not be altered by convert()
        final Measurement tempM = new Measurement(_value, _unit);

        if(_unit != unit)
        {
            tempM.convert(unit);
        }

        return tempM.getValue();
    }

    @Override
    public int hashCode()
    {
        int value = 0;

        final MeasuringUnit standardUnit = MeasuringUnit.getStandardUnitForUnit(this.getUnit());
        value = (int)(this.getValue(standardUnit));

        return value;
    }

    //---------------------------------------------------------------

    public boolean isDefaulted()
    {
        return _defaulted;
    }

    //---------------------------------------------------------------

    public boolean isExtrapolated()
    {
        return _extrapolated;
    }

    //---------------------------------------------------------------

    public boolean isInterpolated()
    {
        return _interpolated;
    }

    //---------------------------------------------------------------

    //---------------------------------------------------------------
    public boolean isMissing()
    {
        return _isMissing;
    }

    //---------------------------------------------------------------

    public void setComment(final String comment)
    {
        _comment = comment;
    }

    //---------------------------------------------------------------

    public void setFlag(final boolean flag)
    {
        _flag = flag;
    }

    //---------------------------------------------------------------

    public void setIsDefaulted(final boolean defaulted)
    {
        this._defaulted = defaulted;
    }

    //---------------------------------------------------------------

    public void setIsExtrapolated(final boolean extrapolated)
    {
        this._extrapolated = extrapolated;
    }

    //---------------------------------------------------------------

    public void setIsInterpolated(final boolean interpolated)
    {
        this._interpolated = interpolated;
    }

    public void setIsMissing(final boolean isMissing)
    {
        _isMissing = isMissing;

        _value = _missingValue;
    }

    //---------------------------------------------------------------

    //---------------------------------------------------------------
    public void setValue(final double value)
    {
        _value = value;
    }

    //---------------------------------------------------------------

    @Override
    public String toString()
    {
        String string = null;

        if(!_isMissing)
        {
            string = _value + " in " + _unit.getName();
        }
        else
        {
            string = "MISSING";
        }

        return string;
    }

} //end class Measurement
