/*
 * Created on Jul 14, 2004
 */
package ohd.hseb.measurement;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

/**
 * @author Chip Gobs This class contains a collection of MeasurementPoint objects and can operate on the collection.
 */
public class MeasurementPointSeries
{
    // ------------------------------------------------------------------------

    private List<MeasurementPoint> _measurementPointList = null;

    private MeasuringUnit _xMeasuringUnit = null;
    private MeasuringUnit _yMeasuringUnit = null;

    private Comparator<MeasurementPoint> _comparator;

    // ------------------------------------------------------------------------

    public MeasurementPointSeries(final MeasuringUnit xMeasuringUnit,
                                  final MeasuringUnit yMeasuringUnit,
                                  final Comparator<MeasurementPoint> comparator)
    {
        _xMeasuringUnit = xMeasuringUnit;
        _yMeasuringUnit = yMeasuringUnit;

        _comparator = comparator;

        _measurementPointList = new ArrayList<MeasurementPoint>();
    }

    // ------------------------------------------------------------------------

    public void setComparator(final Comparator<MeasurementPoint> comparator)
    {
        _comparator = comparator;
    }

    // ------------------------------------------------------------------------

    public Comparator<MeasurementPoint> getComparator()
    {
        return _comparator;
    }

    //  ------------------------------------------------------------------------  

    // ------------------------------------------------------------------------
    public void addPoint(MeasurementPoint newPoint)
    {
        newPoint = new MeasurementPoint(newPoint, _xMeasuringUnit, _yMeasuringUnit);

        boolean added = false;
        for(int i = 0; i < _measurementPointList.size() && !added; i++)
        {
            final MeasurementPoint existingPoint = _measurementPointList.get(i);

            final int result = _comparator.compare(newPoint, existingPoint);

            if((result == -1) || (result == 0))
            {
                _measurementPointList.add(i, newPoint);
                added = true;
            }
        }

        if(!added)
        {
            _measurementPointList.add(newPoint);
        }

        return;
    }

    // ------------------------------------------------------------------------
    public void removePoint(final MeasurementPoint point)
    {
        _measurementPointList.remove(point);
    }

    // ------------------------------------------------------------------------
    public void clear()
    {
        _measurementPointList.clear();
    }

    // ------------------------------------------------------------------------

    public int size()
    {
        int result = 0;

        if(_measurementPointList != null)
        {
            result = _measurementPointList.size();
        }

        return result;
    }

    // ------------------------------------------------------------------------

    public MeasurementPoint getPoint(final int index)
    {
        return _measurementPointList.get(index);
    }

    //  ------------------------------------------------------------------------

    public Measurement getMaxXMeasurement()
    {
        return getMaxMeasurement(true);
    }

    //  ------------------------------------------------------------------------

    public Measurement getMinXMeasurement()
    {
        return getMinMeasurement(true);
    }

    //  ------------------------------------------------------------------------

    public Measurement getMaxYMeasurement()
    {
        return getMaxMeasurement(false);

    }

    //  ------------------------------------------------------------------------

    public Measurement getMinYMeasurement()
    {
        return getMinMeasurement(false);
    }

    //  ------------------------------------------------------------------------

    public Measurement getMaxMeasurement(final boolean isXOrdinate)
    {
        Measurement maxMeasurement = null;
        Measurement pointMeasurement = null;

        MeasuringUnit unit = null;

        // determine which units we are dealing with
        if(isXOrdinate)
        {
            unit = _xMeasuringUnit;
        }
        else
        {
            unit = _yMeasuringUnit;
        }

        // look at each measurement point to determine the max
        for(int i = 0; i < size(); i++)
        {
            final MeasurementPoint point = _measurementPointList.get(i);

            if(isXOrdinate)
            {
                pointMeasurement = point.getXMeasurement();
            }
            else
            {
                pointMeasurement = point.getYMeasurement();
            }

            if(maxMeasurement == null)
            {
                maxMeasurement = pointMeasurement;
            }
            else
            //maxMeasurement != null
            {
                if(pointMeasurement != null)
                {
                    if(pointMeasurement.getValue(unit) > maxMeasurement.getValue(unit))
                    {
                        maxMeasurement = pointMeasurement;
                    }
                }
            }

        }

        return maxMeasurement;

    }

    //  ------------------------------------------------------------------------
    public Measurement getMinMeasurement(final boolean isXOrdinate)
    {
        Measurement minMeasurement = null;
        Measurement pointMeasurement = null;

        MeasuringUnit unit = null;

        // determine which units we are dealing with
        if(isXOrdinate)
        {
            unit = _xMeasuringUnit;
        }
        else
        {
            unit = _yMeasuringUnit;
        }

        // look at each measurement point to determine the max
        for(int i = 0; i < size(); i++)
        {
            final MeasurementPoint point = _measurementPointList.get(i);

            if(isXOrdinate)
            {
                pointMeasurement = point.getXMeasurement();
            }
            else
            {
                pointMeasurement = point.getYMeasurement();
            }

            if(minMeasurement == null)
            {
                minMeasurement = pointMeasurement;
            }
            else
            //minMeasurement != null
            {
                if(pointMeasurement != null)
                {
                    if(pointMeasurement.getValue(unit) < minMeasurement.getValue(unit))
                    {
                        minMeasurement = pointMeasurement;
                    }
                }
            }

        }

        return minMeasurement;

    }

    //  ------------------------------------------------------------------------ 
    @Override
    public String toString()
    {
        final StringBuffer buffer = new StringBuffer();

        for(int i = 0; i < size(); i++)
        {
            final MeasurementPoint point = getPoint(i);

            final Measurement xMeasurement = point.getXMeasurement();
            final Measurement yMeasurement = point.getYMeasurement();

            buffer.append("x = " + xMeasurement + " y = " + yMeasurement + "\n");

        }

        return buffer.toString();

    }

    //  ------------------------------------------------------------------------
    @SuppressWarnings("unchecked")
    public static void main(final String[] argStringArray)
    {
        final MeasuringUnit xUnit = MeasuringUnit.hours;
        final MeasuringUnit yUnit = MeasuringUnit.cfs;

        final boolean compareXValues = true;
        final boolean ascendingOrder = true;

        final Comparator comparator = new MeasurementPointComparator(compareXValues, ascendingOrder);

        final MeasurementPointSeries measurementPointSeries = new MeasurementPointSeries(xUnit, yUnit, comparator);

        final int measurementCount = 10;

        for(int i = 0; i < measurementCount; i++)
        {
            final Measurement m1 = new Measurement(measurementCount - i - 1, xUnit);
            final Measurement m2 = new Measurement(2 * i, yUnit);

            final MeasurementPoint point = new MeasurementPoint(m1, m2);

            measurementPointSeries.addPoint(point);
        }

        for(int i = 0; i < measurementPointSeries.size(); i++)
        {
            final MeasurementPoint point = measurementPointSeries.getPoint(i);

            final Measurement xMeasurement = point.getXMeasurement();
            final Measurement yMeasurement = point.getYMeasurement();

            System.out.println("x = " + xMeasurement + " y = " + yMeasurement);
        }

        final Measurement minXMeasurement = measurementPointSeries.getMinXMeasurement();
        final Measurement minYMeasurement = measurementPointSeries.getMinYMeasurement();

        final Measurement maxXMeasurement = measurementPointSeries.getMaxXMeasurement();
        final Measurement maxYMeasurement = measurementPointSeries.getMaxYMeasurement();

        System.out.println("minX = " + minXMeasurement + " minY = " + minYMeasurement);
        System.out.println("maxX = " + maxXMeasurement + " maxY = " + maxYMeasurement);
    }
    //  ------------------------------------------------------------------------

} //end MeasurementPointSeries
