/*
 * Created on Jul 2, 2003 To change the template for this generated file go to
 */
package ohd.hseb.measurement;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;

/**
 * @author gobsc This is basically an enum of all possible units of measure, but it is also smart enough to help handle
 *         unit conversions.
 */
final public class MeasuringUnit
{

    //conversion factor map
    private static Map<String, Double> _conversionFactorMap = new HashMap<String, Double>();

    //conversion standard map
    private static Map<MeasuringUnitType, MeasuringUnit> _conversionStandardMap = new HashMap<MeasuringUnitType, MeasuringUnit>();

    // Note: the standard unit for a particular MeasuringUnitType, MUST be the first
    //one of that type created here. Otherwise, a circular lookup dependency will be created when
    //when the code tries to set up Maps used to convert first to the standard unit (which wouldn't exist yet)
    // and then to the desired unit.
    // In the private constructor, having a conversion factor of 1.0 is used to identify the unit
    // as the standard of that MeasuringUnitType.

    /**
     * Maps unit name to the {@link MeasuringUnit} instance that uses its name. The name must be in all caps!
     */
    private static HashMap<String, MeasuringUnit> _mapOfUnits = new HashMap<String, MeasuringUnit>();

    /**
     * Maps a unit to the english version of the unit. Used by {@link #getEnglishUnit(MeasuringUnit)} and populated
     * during construction of the static measuring units.
     */
    private static HashMap<MeasuringUnit, String> _unitToEnglishMap = new HashMap<>();

    /**
     * Maps a unit to the metric version of the unit. Used by {@link #getMetricUnit(MeasuringUnit)} and populated during
     * construction of the static measuring units.
     */
    private static HashMap<MeasuringUnit, String> _unitToMetricMap = new HashMap<>();

    //discharge types
    public static final MeasuringUnit cfs = new MeasuringUnit("CFS", MeasuringUnitType.discharge, 1.0, "CMS", null);
    public static final MeasuringUnit kcfs = new MeasuringUnit("KCFS", MeasuringUnitType.discharge, 1000.0, "CMS", null);
    public static final MeasuringUnit cms = new MeasuringUnit("CMS",
                                                              MeasuringUnitType.discharge,
                                                              35.314666572,
                                                              null,
                                                              "CFS");
    public static final MeasuringUnit m3ps = new MeasuringUnit("M3/S",
                                                               MeasuringUnitType.discharge,
                                                               35.314666572,
                                                               null,
                                                               "CFS");

    //length types
    public static final MeasuringUnit meters = new MeasuringUnit("M", MeasuringUnitType.length, 1.0, null, "FT");
    public static final MeasuringUnit cm = new MeasuringUnit("CM", MeasuringUnitType.length, 0.01, null, "IN");
    public static final MeasuringUnit mm = new MeasuringUnit("MM", MeasuringUnitType.length, 0.001, null, "IN");
    public static final MeasuringUnit inches = new MeasuringUnit("IN", MeasuringUnitType.length, 0.0254, "MM", null);
    public static final MeasuringUnit feet = new MeasuringUnit("FT", MeasuringUnitType.length, 0.3048, "M", null);

    //relative time types
    public static final MeasuringUnit hours = new MeasuringUnit("HR", MeasuringUnitType.elapsedTime, 1.0, null, null);
    public static final MeasuringUnit days = new MeasuringUnit("DAY", MeasuringUnitType.elapsedTime, 24.0, null, null);

    //temperature types
    // The conversion factor for degreesFahrenheit and degreesCelsius are not used
    // the Measurement convert() method handles these internally 
    public static final MeasuringUnit degreesFahrenheit = new MeasuringUnit("DEGF",
                                                                            MeasuringUnitType.temperature,
                                                                            1.0,
                                                                            "DEGC",
                                                                            null);
    public static final MeasuringUnit degreesCelsius = new MeasuringUnit("DEGC",
                                                                         MeasuringUnitType.temperature,
                                                                         0,
                                                                         null,
                                                                         "DEGF");

    //speed types
    public static final MeasuringUnit mph = new MeasuringUnit("MI/H", MeasuringUnitType.speed, 1.0, "KM/H", null);
    public static final MeasuringUnit knots = new MeasuringUnit("KNOT",
                                                                MeasuringUnitType.speed,
                                                                1.150779448,
                                                                "KM/H",
                                                                null);
    public static final MeasuringUnit kph = new MeasuringUnit("KM/H",
                                                              MeasuringUnitType.speed,
                                                              0.621371192,
                                                              null,
                                                              "MI/H");
    public static final MeasuringUnit mps = new MeasuringUnit("M/S", MeasuringUnitType.speed, 2.23693629, null, "MI/H");
    //direction
    public static final MeasuringUnit degrees = new MeasuringUnit("DEG",
                                                                  MeasuringUnitType.elapsedTime,
                                                                  1.0,
                                                                  null,
                                                                  "DEGF");

    //volume -  
    public static final MeasuringUnit m3 = new MeasuringUnit("M3", MeasuringUnitType.volume, 1.0, null, "FT3");
    public static final MeasuringUnit ft3 = new MeasuringUnit("FT3", MeasuringUnitType.volume, 0.0283168466, "M3", null);
    public static final MeasuringUnit cmsd = new MeasuringUnit("CMSD", MeasuringUnitType.volume, 86400, null, "CFSD");
    public static final MeasuringUnit cfsd = new MeasuringUnit("CFSD", MeasuringUnitType.volume, 2446.6, "CMSD", null);
    public static final MeasuringUnit acft = new MeasuringUnit("ACFT", MeasuringUnitType.volume, 1233.48, "M3", null);
    public static final MeasuringUnit tcum = new MeasuringUnit("TCUM", MeasuringUnitType.volume, 1000.0, null, null);

    //unitless type
    public static final MeasuringUnit unitless = new MeasuringUnit("UNITLESS",
                                                                   MeasuringUnitType.unitless,
                                                                   1.0,
                                                                   null,
                                                                   null);
    public static final MeasuringUnit unitlessReal = new MeasuringUnit("REAL",
                                                                       MeasuringUnitType.unitless,
                                                                       1.0,
                                                                       null,
                                                                       null);
    public static final MeasuringUnit unitlessInt = new MeasuringUnit("INT",
                                                                      MeasuringUnitType.unitless,
                                                                      1.0,
                                                                      null,
                                                                      null);
    public static final MeasuringUnit percentDecimal = new MeasuringUnit("PCTD",
                                                                         MeasuringUnitType.unitless,
                                                                         1.0,
                                                                         null,
                                                                         null);
    public static final MeasuringUnit percent = new MeasuringUnit("PCT", MeasuringUnitType.unitless, 100.0, null, null);

    // member variables
    private final MeasuringUnitType _type;
    private final String _name;

    //note, we simply need to convert to a standard and convert from the standard,
    // so that we don't have (N*N) -N conversions, just 2*N conversions

    // --------------------------------------------------------------------
    private MeasuringUnit(final String unitName,
                          final MeasuringUnitType type,
                          final double conversionFactor,
                          final String metricUnitName,
                          final String englishUnitName)
    {
        _name = unitName.toUpperCase(); //make sure the string representing unit is all UpperCase
        _type = type;

        String toStandardKey = null;
        String fromStandardKey = null;
        _mapOfUnits.put(_name, this);

        if(englishUnitName != null)
        {
            _unitToEnglishMap.put(this, englishUnitName);
        }

        if(metricUnitName != null)
        {
            _unitToMetricMap.put(this, metricUnitName);
        }

        if(conversionFactor == 1.0)
        {
            // this is the standard unit
            _conversionStandardMap.put(type, this);

            toStandardKey = getConversionKey(_name, _name);
            fromStandardKey = getConversionKey(_name, _name);

            _conversionFactorMap.put(toStandardKey, conversionFactor);
            _conversionFactorMap.put(fromStandardKey, 1.0 / conversionFactor);
        }
        else
        {
            final MeasuringUnit standardUnit = getStandardUnitForType(type);
            toStandardKey = getConversionKey(_name, standardUnit.getName());
            fromStandardKey = getConversionKey(standardUnit.getName(), _name);

            _conversionFactorMap.put(toStandardKey, conversionFactor);
            _conversionFactorMap.put(fromStandardKey, 1.0 / conversionFactor);
        }

    } //end constructor

    public static MeasuringUnit getMeasuringUnit(final String units)
    {
        return (_mapOfUnits.get(units.trim().toUpperCase())); //since all keys in _map are UpperCase
    }

    public String getName()
    {
        return _name;
    }

    public MeasuringUnitType getType()
    {
        return _type;
    }

    @Override
    public String toString()
    {
        return _name;
    }

    /**
     * @return {@link String}s identifying the units compatible to this.
     */
    public List<String> getCompatibleUnits()
    {
        final List<String> compatibleUnits = Lists.newArrayList();
        for(final String key: _mapOfUnits.keySet())
        {
            try
            {
                if(!_mapOfUnits.get(key).equals(this))
                {
                    getConversionFactor(this, _mapOfUnits.get(key));
                    compatibleUnits.add(key);
                }
            }
            catch(final Error e)
            {
                //Do nothing... the units are not compatible.
            }
        }
        return compatibleUnits;
    }

    public boolean equals(final MeasuringUnit unit)
    {
        if(_type.getName().equalsIgnoreCase(unit.getType().getName()) && _name.equalsIgnoreCase(unit.getName()))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    public static double getConversionFactor(final MeasuringUnit fromUnit, final MeasuringUnit toUnit)
    {
        final MeasuringUnit standardUnit = getStandardUnitForType(fromUnit.getType());
        double conversionFactor = 1.0;

        if(fromUnit == toUnit)
        {
            conversionFactor = 1.0;
        }
        else
        {

            final String conversionKey1 = getConversionKey(fromUnit, standardUnit);
            final Double conversionFactor1 = _conversionFactorMap.get(conversionKey1);

            final String conversionKey2 = getConversionKey(standardUnit, toUnit);
            final Double conversionFactor2 = _conversionFactorMap.get(conversionKey2);
            if(conversionFactor1 != null && conversionFactor2 != null)
            {
                conversionFactor = conversionFactor1.doubleValue() * conversionFactor2.doubleValue();
            }
            else
            {
                throw new Error("Bogus Unit conversion attempt from " + fromUnit.getName() + " to " + toUnit.getName());
            }

        }

        return conversionFactor;
    }

    protected static MeasuringUnit getStandardUnitForType(final MeasuringUnitType type)
    {
        final MeasuringUnit unit = _conversionStandardMap.get(type);

        return unit;

    } //end getStandardUnitForType

    protected static MeasuringUnit getStandardUnitForUnit(final MeasuringUnit unit)
    {
        final MeasuringUnitType type = unit.getType();
        final MeasuringUnit standardUnit = _conversionStandardMap.get(type);

        return standardUnit;

    } //end getStandardUnitForType

    private static String getConversionKey(final MeasuringUnit fromUnit, final MeasuringUnit toUnit)
    {
        return getConversionKey(fromUnit.getName(), toUnit.getName());
    }

    private static String getConversionKey(final String fromUnitName, final String toUnitName)
    {
        return fromUnitName + "|" + toUnitName;
    }

    /**
     * @return The English unit that corresponds to the provided unit or null if either (1) the provided unit did not
     *         result in an English unit being found or (2) the unit is already English.
     */
    public static MeasuringUnit getEnglishUnit(final MeasuringUnit unit)
    {
        final String englishUnitName = _unitToEnglishMap.get(unit);
        if(englishUnitName == null)
        {
            return null;
        }
        return getMeasuringUnit(englishUnitName);
    }

    /**
     * @return The metric unit that corresponds to the provided unit or null if either (1) the provided unit did not
     *         result in an metric unit being found or (2) the unit is already metric.
     */
    public static MeasuringUnit getMetricUnit(final MeasuringUnit unit)
    {
        final String metricUnitName = _unitToMetricMap.get(unit);
        if(metricUnitName == null)
        {
            return null;
        }
        return getMeasuringUnit(metricUnitName);
    }

} //end MeasuringUnit
