/*
 * Created on Oct 15, 2003
 */
package ohd.hseb.measurement;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import javax.management.timer.Timer;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.TimeHelper;
import ohd.hseb.util.fews.OHDConstants;

/**
 * @author Chip Gobs
 */
public class RegularTimeSeries implements Cloneable
{
    private double _missingValue = OHDConstants.MISSING_DATA;

    public static final String NO_ENSEMBLE_ID = "main";

    private static final String _missingString = "MSG";

    private String _name = null;
    private long _startTime = 0; //all the time in this class is GMT time zone
    private long _endTime = 0; //GMT time zone
    private int _intervalInHours = 1;
    private String _intervalTimes = null;
    private List<Measurement> _measurementList = null;
    //assuming all Measurement obj have the same unit

    private MeasuringUnit _measuringUnit;

    /**
     * represents the timeseries type such as MAT, MAP, RAIM etc.
     */
    protected String _timeSeriesType;

    /**
     * represents the locationId received from FEWS
     */
    private String _locationId;

    //part of key for timeseries (equivalent to nwsrfs timeseriesid)
    private List<String> _qualifierId = null;

    /**
     * The ensemble id. This will never be null or empty.
     */
    private String _ensembleId = NO_ENSEMBLE_ID;

    /**
     * The ensembleMemberIndex. Defaults to 0.
     */
    private int _ensembleMemberIndex = 0;

    // -----------------------------------------------------------------------------
    public RegularTimeSeries(final long startTime,
                             final long endTime,
                             final int intervalInHours,
                             final MeasuringUnit unit)
    {
        // defaults to initialValue = -999.0;
        this(startTime, endTime, intervalInHours, unit, -999.0);
    }

    //  -----------------------------------------------------------------------------

    public RegularTimeSeries(final long startTime,
                             final long endTime,
                             final int intervalInHours,
                             final MeasuringUnit unit,
                             final double initialValue)
    {
        _intervalInHours = intervalInHours;

        _startTime = startTime;
        _endTime = endTime;

        _measuringUnit = unit; //each unit is a final public static variable, no need of clone()

        long numberOfIntervals = (_endTime - _startTime) / getIntervalInMillis();
        numberOfIntervals++; //if endTime == startTime, that is 1 interval

        _measurementList = new ArrayList<Measurement>((int)numberOfIntervals);

        for(int i = 0; i < numberOfIntervals; i++)
        {
            final Measurement newMeasurement = new Measurement(initialValue, unit, true);           
            _measurementList.add(i, newMeasurement);
        }

    } //end RegularTimeSeries() 

    //  ---------------------------------------------------------------------------

    /**
     * Add RegularTimeSeries ts to "this" RegularTimeSeries: expand "this" time period to cover both "this" and ts time
     * period. Sum the measurement values in the same time step. Assuming ts has the same timeseries type and interval
     * as "this", but can have different start time, end time and unit.
     * 
     * @param ts
     */
    public void addToRegularTimeSeries(final RegularTimeSeries ts)
    {
        Measurement m1 = null;
        Measurement m2 = null;
        double value = 0.0;

        ts.convert(getMeasuringUnit());

        //new RTS has the longest time period
        final long startTime = Math.min(getStartTime(), ts.getStartTime());
        final long endTime = Math.max(getEndTime(), ts.getEndTime());

        stretchTimeSeries(startTime, endTime, 0);
        ts.stretchTimeSeries(startTime, endTime, 0);

        for(long time = getStartTime(); time <= getEndTime(); time += getIntervalInMillis())
        {
            m1 = getMeasurementByTime(time);
            m2 = ts.getMeasurementByTime(time);

            value = 0.0;

            if(m1.isMissing() == false)
            {
                value += m1.getValue();
            }

            if(m2.isMissing() == false)
            {
                value += m2.getValue();
            }

            //final Measurement newMeasurement = new Measurement(value, getMeasuringUnit());
            setMeasurementByTime(new Measurement(value, getMeasuringUnit()), time);
        }

    }

    //assuming all the RTS in the array have the same interval, same unit, same type, start time and end time can be different
    public static RegularTimeSeries sumArrayOfRTS(final RegularTimeSeries[] rtsArray)
    {

        final RegularTimeSeries newTs = rtsArray[0];

        if(rtsArray.length > 1)
        {
            for(int i = 1; i < rtsArray.length; i++)
            {
                newTs.addToRegularTimeSeries(rtsArray[i]);
            }
        }

        return newTs;
    }

    /**
     * scales each value inside the RTS
     */
    public void scaleRegularTimeSeries(final double d)
    {
        for(int i = 0; i < _measurementList.size(); i++)
        {
            //final Measurement newM = new Measurement(_measurementList.get(i).getValue() * d, getMeasuringUnit());

            this.setMeasurementByIndex(new Measurement(_measurementList.get(i).getValue() * d, getMeasuringUnit()), i);
        }
    }

    /**
     * add a value to each value inside the RTS
     */
    public void addToRegularTimeSeries(final double d)
    {
        for(int i = 0; i < _measurementList.size(); i++)
        {
            //final Measurement newM = new Measurement(_measurementList.get(i).getValue() + d, getMeasuringUnit());

            this.setMeasurementByIndex(new Measurement(_measurementList.get(i).getValue() + d, getMeasuringUnit()), i);
        }
    }

    //  ---------------------------------------------------------------------------

    public void convert(final MeasuringUnit newUnit)
    {
        this.setMeasuringUnit(newUnit);
    }

    //  -----------------------------------------------------------------------------
    public void stretchTimeSeries(final long newStartTime, final long endStartTime, final double fillInValue)
    {
        prependTimeSeries(newStartTime, fillInValue);
        extendTimeSeries(endStartTime, fillInValue);

        return;
    }

    //  -----------------------------------------------------------------------------

    /**
     * if newStartTime is earlier than current start time, do nothing; otherwise, delete time steps so that the new
     * start time is the newStartTime. If the newStartTime is not one of the time steps, throw an Exception
     */
    public void trimTimeSeriesAtStart(final long newStartTime)
    {
        // round off the newEndTime to the nearest hour
        // if it is not already rounded, then add an hour to the rounded result
        long newRoundedStartTime = TimeHelper.truncateTimeInMillisToNearestHour(newStartTime, 1);

        if(newRoundedStartTime != newStartTime)
        {
            newRoundedStartTime += Timer.ONE_HOUR;
        }

        //if newRoundedStartTime is earlier than or equal to the current start time, return and do nothing
        if(newRoundedStartTime <= _startTime)
        {
            return;
        }

        // delete measurements from the beginning of the list
        final int numDataPointsTrimmed = ((Long)((newRoundedStartTime - _startTime) / getIntervalInMillis())).intValue();

        _measurementList.subList(0, numDataPointsTrimmed).clear();

        _startTime = newRoundedStartTime;

        return;
    }

    /**
     * This method is similar to {@link #trimTimeSeriesAtStart(long)}, except that making sure newStartTime is one of
     * the time steps. If not, throw an Exception.
     * 
     * @param newStartTime
     * @throws Exception
     */
    public void trimTimeSeriesAtStartWithCheck(final long newStartTime) throws Exception
    {
        //check newRoundedStartTime falls on one of the time steps
        if(this.isInTimeSteps(newStartTime) == false)
        {
            throw new Exception("Error: when trimming the " + _timeSeriesType
                + " timeseries start time, the expected new start time ("
                + DateTime.getDateTimeStringFromLong(newStartTime,OHDConstants.GMT_TIMEZONE) + ") is not one of the time steps");
        }

        this.trimTimeSeriesAtStart(newStartTime);
    }

    //  -----------------------------------------------------------------------------

    /**
     * if newEndTime is later than current end time, do nothing; otherwise, delete time steps so that the new end time
     * is the newEndTime.
     */
    public void trimTimeSeriesAtEnd(final long newEndTime)
    {
        //if newEndTime is later than or equal to the current end time, return and do nothing
        if(newEndTime >= _endTime)
        {
            return;
        }

        // delete measurements from the back of the list
        final int numDataPointsTrimmed = ((Long)((_endTime - newEndTime) / getIntervalInMillis())).intValue();

        _measurementList.subList(_measurementList.size() - numDataPointsTrimmed, _measurementList.size()).clear();

        _endTime = newEndTime;

        return;
    }

    /**
     * This method is similar to {@link #trimTimeSeriesAtEnd(long)}, except that making sure newEndTime is one of the
     * time steps. If not, throw an Exception.
     * 
     * @param newEndTime
     * @throws Exception
     */
    public void trimTimeSeriesAtEndWithCheck(final long newEndTime) throws Exception
    {
        //check newEndTime falls on one of the time steps
        if(this.isInTimeSteps(newEndTime) == false)
        {
            throw new Exception("Error: when trimming the TS end time, the expected new end time ("
                + DateTime.getDateTimeStringFromLong(newEndTime, OHDConstants.GMT_TIMEZONE) + ") is not one of the time steps");
        }

        this.trimTimeSeriesAtEnd(newEndTime);
    }

    /**
     * If the long belongs to one of the time steps, returns true; otherwise, returns false.
     */
    private boolean isInTimeSteps(final long time)
    {

        if(Math.abs(time - _startTime) % getIntervalInMillis() != 0)
        {
            return false;
        }

        if(time > _endTime)
        {
            return false;
        }

        return true;
    }

    /**
     * If this TS runs from 12Z, 18Z, 0Z, 6Z, 12Z, etc, and the parameter is one of them, returns true; if not, returns
     * false
     * 
     * @param timeStepHour
     */
    public boolean isInTimeSteps(final int timeStepHour)
    {
        final String timeStr = DateTime.getTimeStringFromLong(_startTime, OHDConstants.GMT_TIMEZONE);
        final String hourStr = timeStr.substring(0, 2);

        final int hour = Integer.valueOf(hourStr);

        if(Math.abs(hour - timeStepHour) % _intervalInHours == 0)
        {
            return true;
        }

        return false;
    }

    //  -----------------------------------------------------------------------------
    public void prependTimeSeries(final long newStartTime)
    {
        prependTimeSeries(newStartTime, 0.0);

    }

    public void prependTimeSeries(final long newStartTime, final double fillInValue)
    {
        // round off the newEndTime to the nearest hour
        // if it is not already rounded, then add an hour to the rounded result
        long roundedStartTime = TimeHelper.truncateTimeInMillisToNearestHour(newStartTime, 1);

        if(roundedStartTime != newStartTime)
        {
            roundedStartTime += Timer.ONE_HOUR;
        }

        // add measurements to the beginning of the list
        if(roundedStartTime < _startTime)
        {
            final long numDataPointsAdded = (_startTime - roundedStartTime) / getIntervalInMillis();

            for(int i = 0; i < numDataPointsAdded; i++)
            {
                //add to the beginning of the list
                _measurementList.add(0, new Measurement(fillInValue, getMeasuringUnit()));
            }

            _startTime = roundedStartTime;
        }

    } //end prependTimeSeries()

    // ---------------------------------------------------------------------------
    public void extendTimeSeries(final long newEndTime)
    {
        extendTimeSeries(newEndTime, 0.0);
    }

    public void extendTimeSeries(final long newEndTime, final double fillInValue)
    {

        // round off the newEndTime to the nearest hour
        // if it is not already rounded, then add an hour to the rounded result
        long roundedEndTime = TimeHelper.truncateTimeInMillisToNearestHour(newEndTime, 1);

        if(roundedEndTime != newEndTime)
        {
            roundedEndTime += Timer.ONE_HOUR;
        }

        // add measurements to the end of the list
        if(roundedEndTime > _endTime)
        {
            final long numDataPointsAdded = (roundedEndTime - _endTime) / getIntervalInMillis();
            for(int i = 0; i < numDataPointsAdded; i++)
            {
                //add to the end of the list
                _measurementList.add(new Measurement(fillInValue, getMeasuringUnit()));
            }

            _endTime = roundedEndTime;
        }

    } //end extendTimeSeries()

    public void appendMeasurement(final Measurement m)
    {
        _measurementList.add(m);
    }

    //	---------------------------------------------------------------------------
    public void setName(final String name)
    {
        _name = name;
    }

    //	---------------------------------------------------------------------------

    public String getName()
    {
        return _name;
    }

    //	---------------------------------------------------------------------------

    public int getMeasurementCount()
    {
        return _measurementList.size();
    }

    //	--------------------------------------------------------    

    //Throws IndexOutOfBoundsException - if the index is out of range (index < 0 || index >= size())
    //Returned is an independant copy, not an alias to the measurement element in the measurementList
    public Measurement getMeasurementByIndex(final int index)
    {
        return _measurementList.get(index).getCopy(); //returns an independant copy 
    }

    //  -------------------------------------------------------- 

    //  Returned is an independant copy, not an alias to the measurement element in the measurementList
    public double getMeasurementValueByIndex(final int index, final MeasuringUnit unit)
    {
        final Measurement m = getMeasurementByIndex(index);

        if(m.getValue() == getMissingValue())
        {
            return m.getValue();
        }

        if(m.getUnit().equals(unit) == false)
        { //diff. unit
            m.convert(unit);
        }

        return m.getValue();
    }

    /**
     * Not considering unit conversion, just truncate the big values(if any) in the RTS to the maxValue.
     */
    public void setMaxValue(final double maxValue)
    {
        for(int i = 0; i < _measurementList.size(); i++)
        {
            final double origValue = _measurementList.get(i).getValue();

            if(origValue > maxValue)
            {
                _measurementList.get(i).setValue(maxValue);
            }

        }
    }

    /**
     * Not considering unit conversion, just increase the small values -- ignoring the missing values -- in the RTS to
     * the minimum value.
     */
    public void setMinValue(final double minValue)
    {
        for(int i = 0; i < _measurementList.size(); i++)
        {
            final double origValue = _measurementList.get(i).getValue();

            if(origValue != getMissingValue() && origValue < minValue)
            {
                _measurementList.get(i).setValue(minValue);
            }

        }
    }

    //	-------------------------------------------------------- 
    public double getMeasurementValueByTime(final long time, final MeasuringUnit unit)
    {
        final Measurement m = getMeasurementByTime(time);

        if(m.getUnit().equals(unit) == false)
        { //diff. unit
            m.convert(unit);
        }

        return m.getValue();
    }

    //  -------------------------------------------------------- 
    /**
     * If the timing is wrong, returns {@link #_missingValue}; Do not throw an Exception.
     */
    public double getMeasurementValueByTimeNoException(final long time, final MeasuringUnit unit)
    {
        final Measurement m = getMeasurementByTime(time);
        if(m != null)
        {
            return m.getValue(unit);
        }
        else
        {
            return _missingValue;
        }
    }

    //  -------------------------------------------------------- 

    public long getMeasurementTimeByIndex(final int index)
    {
        final long newTime = _startTime + index * getIntervalInMillis();

        return newTime;
    }

    //	-------------------------------------------------------- 

    /**
     * If time is not between(including) {@link #_startTime} and {@link #_endTime}, returns -1; otherwise, returns the
     * index corresponding to the time.
     * <p>
     * If the parameter time is not exactly on the time step, it will be rounded and the most nearby index will be
     * returned. This could be a problem.
     */
    private int getMeasurementIndexByTime(final long time)
    {
        int index = -1;

        if(time >= _startTime && time <= _endTime)
        {
            index = (int)((time - _startTime) / getIntervalInMillis()); //if not divided to even, will be rounded
        }

        return index;
    }

    //	-------------------------------------------------------- 

    public void setMeasurementByIndex(final Measurement newMeasurement, final int index)
    {
        if(newMeasurement.getUnit() != _measuringUnit)
        {
            newMeasurement.convert(_measuringUnit);
        }

        _measurementList.set(index, newMeasurement);

        return;
    }

    /**
     * Comparing to setMeasurementByIndex(Measurement newMeasurement, int index), this method is more convenient and
     * more useful, using the RTS MeasuringUnit.
     */
    public void setMeasurementByIndex(final double measurementValue, final int index)
    {
        setMeasurementByIndex(new Measurement(measurementValue, _measuringUnit), index);
    }

    // --------------------------------------------------------

    public void setMeasurementByTime(final Measurement measurement, final long time)
    {

        if(time > _endTime)
        {
            extendTimeSeries(time);
        }
        else if(time < _startTime)
        {
            prependTimeSeries(time);
        }

        final int index = getMeasurementIndexByTime(time);

        if(index != -1)
        {
            setMeasurementByIndex(measurement, index);
        }

        return;
    }

    /**
     * Compaing to setMeasurementByTime(Measurement measurement, long time), this method is more convenient and more
     * useful, using the RTS MeasuringUnit.
     * 
     * @param measurementValue
     * @param time
     */
    public void setMeasurementByTime(final double measurementValue, final long time)
    {
        setMeasurementByTime(new Measurement(measurementValue, getMeasuringUnit()), time);

        return;
    }

    public Measurement getMeasurementByTime(final long time)
    {
        Measurement measurement = null;
        int index = -1;

        index = getMeasurementIndexByTime(time);

        if(index != -1)
        {
            measurement = getMeasurementByIndex(index);
        }

        return measurement;
    }

    public AbsTimeMeasurement getAbsTimeMeasurementByTime(final long time)
    {
        Measurement measurement = null;
        AbsTimeMeasurement absTimeMeasurement = null;

        measurement = getMeasurementByTime(time);
        if(measurement != null)
        {
            absTimeMeasurement = new AbsTimeMeasurement(measurement, time);
        }

        return absTimeMeasurement;
    }

    //	-------------------------------------------------------- 

    public AbsTimeMeasurement getAbsTimeMeasurementByIndex(final int index)
    {
        final Measurement measurement = _measurementList.get(index);
        final long time = getMeasurementTimeByIndex(index);

        final AbsTimeMeasurement absTimeMeasurement = new AbsTimeMeasurement(measurement, time);

        return absTimeMeasurement;

    }

    //	-------------------------------------------------------- 

    /**
     * if startTime is earlier than current start time, does NOT stretch start time to new start time; similar with end
     * time. If startTime or endTime is not one of the time steps, an Exception will be thrown.
     */
    public RegularTimeSeries getSubTimeSeries(final long startTime, final long endTime)
    {
        this.trimTimeSeriesAtStart(startTime);

        this.trimTimeSeriesAtEnd(endTime);

        return this;

    }

    // --------------------------------------------------------    

    public void setStartTime(final long startTime)
    {
        _startTime = startTime;
    }

    //----------------------------------------------------------

    public long getStartTime()
    {
        return _startTime;
    }

    //--------------------------------------------------------
    public void setEndTime(final long endTime)
    {
        _endTime = endTime;
    }

    //--------------------------------------------------------

    public long getEndTime()
    {
        return _endTime;
    }

    public MeasuringUnit getMeasuringUnit()
    {
        return _measuringUnit; //each unit is a final public static variable
    }

    /**
     * Besides setting {@link #_measuringUnit}, also converts each Measurement obj inside {@link #_measurementList}.
     */
    public void setMeasuringUnit(final MeasuringUnit newUnit)
    {
        _measuringUnit = newUnit;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            _measurementList.get(i).convert(newUnit);
        }
    }

    /**
     * Different from {@link #setMeasuringUnit(MeasuringUnit)}, this method only re-sets the unit, does not change the
     * time step values. Use with cautiousness.
     */
    public void setMeasuringUnitNoChangeData(final MeasuringUnit newUnit)
    {
        _measuringUnit = newUnit;
    }

    //	--------------------------------------------------------

    //---------------------------------------------------------------------
    public Measurement getMinMeasurement(final long startTime, final long endTime)
    {
        Measurement measurement = null;

        Measurement minMeasurement = null;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            measurement = getMeasurementByIndex(i);
            final long time = getMeasurementTimeByIndex(i);

            if(time >= startTime && time <= endTime)
            {
                if(measurement.getValue() != _missingValue &&

                (minMeasurement == null || measurement.getValue() < minMeasurement.getValue()))
                {
                    minMeasurement = measurement;
                }
            }
            else if(time > endTime)
            {
                break;
            }
        } //end for i

        // System.out.println("RegularTimeSeries.getMinMeasurement(): result = " + minMeasurement); 

        return minMeasurement;

    }

    //  ---------------------------------------------------------------------

    public Measurement getMinMeasurement()
    {

        return getMinMeasurement(getStartTime(), getEndTime());

    }

    //---------------------------------------------------------------------
    public Measurement getMaxMeasurement(final long startTime, final long endTime)
    {
        Measurement measurement = null;
        Measurement maxMeasurement = null;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            measurement = getMeasurementByIndex(i);
            final long time = getMeasurementTimeByIndex(i);

            if(time >= startTime && time <= endTime)
            {

                if(measurement.getValue() != _missingValue &&

                (maxMeasurement == null || measurement.getValue() > maxMeasurement.getValue()))
                {
                    maxMeasurement = measurement;
                }
            }
            else if(time > endTime)
            {
                break;
            }
        } //end for

        return maxMeasurement;
    }

    public long getMaxMeasurementTime(final long startTime, final long endTime)
    {
        Measurement measurement = null;
        Measurement maxMeasurement = null;
        int maxValue_index = (int)_missingValue;
        long maxValue_time = (long)_missingValue;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            measurement = getMeasurementByIndex(i);
            final long time = getMeasurementTimeByIndex(i);

            if(time >= startTime && time <= endTime)
            {

                if(measurement.getValue() != _missingValue &&

                (maxMeasurement == null || measurement.getValue() > maxMeasurement.getValue()))
                {
                    maxMeasurement = measurement;
                    maxValue_index = i;
                }
            }
            else if(time > endTime)
            {
                break;
            }
        } //end for
        if(maxValue_index != (int)_missingValue)
            maxValue_time = getMeasurementTimeByIndex(maxValue_index);

        return maxValue_time;
    }

    // ---------------------------------------------------------------------

    public Measurement getMaxMeasurement()
    {
        return getMaxMeasurement(getStartTime(), getEndTime());
    }

    // --------------------------------------------------------
    public static RegularTimeSeries concatenate(final RegularTimeSeries ts1, final RegularTimeSeries ts2)
    {
        final long startTime = Math.min(ts1.getStartTime(), ts2.getStartTime());
        final long endTime = Math.max(ts1.getEndTime(), ts2.getEndTime());

        final RegularTimeSeries newTs = new RegularTimeSeries(startTime,
                                                              endTime,
                                                              ts1.getIntervalInHours(),
                                                              ts1.getMeasuringUnit());

        // note: the ts1 timeseries takes precedence during overlapping periods

        // insert the ts2 data
        for(int i = 0; i < ts2.getMeasurementCount(); i++)
        {
            newTs.setMeasurementByTime(ts2.getMeasurementByIndex(i), ts2.getMeasurementTimeByIndex(i));
        }

        // insert the ts1 data
        for(int i = 0; i < ts1.getMeasurementCount(); i++)
        {
            newTs.setMeasurementByTime(ts1.getMeasurementByIndex(i), ts1.getMeasurementTimeByIndex(i));
        }

        return newTs;
    }

    //	--------------------------------------------------------

    public void setIntervalInHours(final int intervalInHours)
    {
        _intervalInHours = intervalInHours;
    }

    //--------------------------------------------------------

    public int getIntervalInHours()
    {
        return _intervalInHours;
    }

    /**
     * @return A string representing the Time step Intervals in hours for instance: 00:00 06:00 12:00 18:00.
     */
    public String getIntervalTimeStepTimes()
    {
        return _intervalTimes;
    }

    /**
     * @param The time steps interval in Times to be set, for instance: 00:00 06:00 12:00 18:00
     */
    public void setIntervalTimeStepTimes(final String intervalTimes)
    {
        _intervalTimes = intervalTimes;
    }

    public long getIntervalInMillis()
    {
        return _intervalInHours * Timer.ONE_HOUR;
    }

    // --------------------------------------------------------

    @Override
    public String toString()
    {
        final StringBuffer buffer = new StringBuffer();

        if(_name != null)
        {
            buffer.append("Name:  " + getName());
        }
        if(getTimeSeriesType() != null)
        {
            buffer.append(" Type: " + getTimeSeriesType());
        }

        if(getLocationId() != null)
        {
            buffer.append(" LocationId: " + getLocationId());
        }

        if(getMeasuringUnit().toString() != null)
        {
            buffer.append(" Units: " + getMeasuringUnit().toString() + " ");
        }
        buffer.append(" Measurement Count = " + getMeasurementCount());

        buffer.append(" Start Time = " + DateTime.getDateTimeStringFromLong(_startTime,OHDConstants.GMT_TIMEZONE));
        buffer.append(" End Time = " + DateTime.getDateTimeStringFromLong(_endTime,OHDConstants.GMT_TIMEZONE));
        buffer.append(" Interval = " + _intervalInHours + "HR");

        buffer.append(" ");

        final NumberFormat f = new DecimalFormat("##0.00");

        for(int i = 0; i < _measurementList.size(); i++)
        {
            final Measurement measurement = getMeasurementByIndex(i);

            if(!measurement.isMissing())
            {
                final double value = measurement.getValue();
                final String valueString = f.format(value);

                buffer.append(valueString + " ");
            }
            else
            {// it is missing
                buffer.append(RegularTimeSeries._missingString + " ");
            }
        }
        return buffer.toString();

    }

    /**
     * Print out locationId, type, qualifierId and interval, then print out time and values in two columns.
     */
    public String toStringWithTimeStamp()
    {
        final StringBuffer buffer = new StringBuffer();

        buffer.append("locationId= " + this.getLocationId() + "\n");
        buffer.append("type= " + this.getTimeSeriesType() + "\n");
        //buffer.append("qualifierId= " + this.getQualifierId() + "\n");
        if(this.getQualifierIds() != null)
        {
            for(final String qualifierId: this.getQualifierIds())
            {
                buffer.append(qualifierId + " ");
            }
            buffer.append("\n");
        }
        buffer.append("interval= " + this.getIntervalInHours() + "\n");

        for(int i = 0; i < this.getMeasurementCount(); i++)
        {
            buffer.append(DateTime.getDateTimeStringFromLong(this.getMeasurementTimeByIndex(i),OHDConstants.GMT_TIMEZONE) + "\t"
                + this.getMeasurementValueByIndex(i, this.getMeasuringUnit()) + "\t"
                + this.getMeasuringUnit().toString() + "\n");
        }

        return buffer.toString();
    }

    // --------------------------------------------------------

    public String getLocationId()
    {
        return _locationId;
    }

    public void setLocationId(final String id)
    {
        _locationId = id;
    }

    // --------------------------------------------------------
    public double getMissingValue()
    {
        return _missingValue;
    }

    public void setMissingValue(final double missingValue)
    {
        _missingValue = missingValue;
    }

    public String getTimeSeriesType()
    {
        return _timeSeriesType;
    }

    public void setTimeSeriesType(final String t)
    {
        _timeSeriesType = t;
    }

    //check to see if there is any missing value inside RTS
    public boolean hasMissingValue()
    {
        for(int i = 0; i < _measurementList.size(); i++)
        {
            if(_measurementList.get(i).getValue() == getMissingValue())
            {
                return true;
            }
        }

        //if reach here, all the values inside RTS must be not missingValue
        return false;
    }

    /**
     * For example, if the RTS obj is 24 hr interval, using this method to convert to 6 hr interval, then all the 6hr
     * time steps are assigned with the value of the next 24hr time step value. In another words, one value of 24hr time
     * steps are used 4 times. Note: 1)this works for instantaneous and mean type RTS, not accumulative RTS. Right now,
     * that info is stored in FewsRegularTimeSeries, so not available here. 2)this method changes the RTS start time to
     * several time steps earlier. The end time remains fixed.
     * 
     * @param smallInterval - the current interval must be multiple of the new interval. If not, an Exception will be
     *            thrown.
     */
    public void disIntegrateToIntervalInst(final int smallInterval) throws Exception
    {
        //only need to change _intervalInHours, _intervalInMillis, _measurementList and _startTime; 
        //nothing else are changed

        if((_intervalInHours % smallInterval) != 0)
        {
            throw new Exception("Cannot disintegrate the RegularTimeSeries(" + getTimeSeriesType() + ") interval from "
                + _intervalInHours + " to " + smallInterval + ". Because the former must be multiple of the later.");
        }

        final int ratio = _intervalInHours / smallInterval; //suppose old interval is multiple of finer interval !!!

        //update interval now
        _intervalInHours = smallInterval;

        _startTime = _startTime - (ratio - 1) * getIntervalInMillis();

        final List<Measurement> finerMeasurementList = new ArrayList<Measurement>();

        for(int i = 0; i < _measurementList.size(); i++)
        {
            //one timestep value of old interval will be used multiple times for new finer interval TS
            for(int j = 0; j < ratio; j++)
            {
                finerMeasurementList.add(_measurementList.get(i));
            } //close inner loop

        } //close outer loop

        //re-assign _measurementList
        _measurementList = finerMeasurementList;
    }

    /**
     * Make the RTS interval to smaller interval. The original values is evened up to several sub time steps. For
     * example, input MAPE TS is always 24hr interval. During the SAC-SMA computation, it has to be converted to the
     * same interval as precip TS, say 6hr. Each finer time step value is a quarter of the original day value.
     * 
     * @param smallInterval - current interval must be multiple of the new interval. If not, an Exception will be
     *            thrown.
     */
    public void disIntegrateToIntervalAccum(final int smallInterval) throws Exception
    {
        final int ratio = this._intervalInHours / smallInterval; //store the ratio before interval is changed

        this.disIntegrateToIntervalInst(smallInterval); //each value is repeatedly used

        this.scaleRegularTimeSeries(1.0 / ratio); //scales down the value
    }

    /**
     * Comparing to {@link #disIntegrateToIntervalInst(int)}, this method fills the new measurement values, due to
     * disintegration, with {@link #_missingValue}(-999.0).
     * 
     * @param smallInterval - the current interval must be multiple of the new interval
     */
    public void disIntegrateToIntervalWithMissing(final int smallInterval) throws Exception
    {
        //only need to change _intervalInHours, _intervalInMillis, _measurementList and _startTime; 
        //nothing else are changed

        if((_intervalInHours % smallInterval) != 0)
        {
            throw new Exception("Cannot disintegrate the RegularTimeSeries(" + getTimeSeriesType() + ") interval from "
                + _intervalInHours + " to " + smallInterval + ". Because the former must be multiple of the later.");
        }

        final int ratio = _intervalInHours / smallInterval; //suppose old interval is multiple of finer interval !!!

        //update interval now
        _intervalInHours = smallInterval;

        _startTime = _startTime - (ratio - 1) * getIntervalInMillis();

        final List<Measurement> finerMeasurementList = new ArrayList<Measurement>();

        for(int i = 0; i < _measurementList.size(); i++)
        {
            //-999.0 is used for new time steps
            for(int j = 0; j < (ratio - 1); j++)
            {
                finerMeasurementList.add(new Measurement(-999.0, getMeasuringUnit()));
            } //close inner loop

            //the original values at the original times are kept un-changed
            finerMeasurementList.add(_measurementList.get(i));

        } //close outer loop

        //re-assign _measurementList
        _measurementList = finerMeasurementList;
    }

    /**
     * For example, if the RTS obj is 6 hr interval, using this method to convert to 24 hr interval, then every 4th
     * measurement remains while all the others are deleted. In another words, the 1st, 2nd and 3rd measurements are
     * deleted, and 4th one becomes the 1st one in the new RTS; 5th, 6th and 7th measurements are deleted, and 8th one
     * becomes the 2nd in the new RTS .... Note: this works for instantaneous and mean type RTS, not accumulative type
     * RTS. Right now, that info is stored in FewsRegularTimeSeries. For accumulative type RTS, data needs to be
     * aggregated.
     * 
     * @param greatInterval - it must be multiple of the current interval
     */
    public void integrateToIntervalInst(final int greatInterval) throws Exception
    {
        if((greatInterval % _intervalInHours) != 0)
        {
            throw new Exception("Cannot integrate the RegularTimeSeries(" + getTimeSeriesType() + ") interval from "
                + _intervalInHours + " to " + greatInterval + ". Because the later must be multiple of the former.");
        }

        final int ratio = greatInterval / _intervalInHours; //suppose new interval is multiple of old interval !!!

        _startTime = _startTime + (ratio - 1) * getIntervalInMillis();

        //update interval now
        _intervalInHours = greatInterval;

        final List<Measurement> newMeasurementList = new ArrayList<Measurement>();

        for(int i = 0; i < _measurementList.size(); i++)
        {
            //extract those elements that needs to remain
            if((i + 1) % ratio == 0)
            { //keep 3rd, 7th, 11th, ...

                newMeasurementList.add(_measurementList.get(i));
            }

        } //close outer loop

        //re-assign _measurementList
        _measurementList = newMeasurementList;
    }

    /**
     * This method works for accumulative RTS. The new RTS, with bigger interval, is with values equal to the sum of
     * those smaller time step values.
     * 
     * @param greatInterval - it must be multiple of the current interval
     */
    public void integrateToIntervalAccum(final int greatInterval) throws Exception
    {
        if((greatInterval % _intervalInHours) != 0)
        {
            throw new Exception("Cannot integrate the RegularTimeSeries(" + getTimeSeriesType() + ") interval from "
                + _intervalInHours + " to " + greatInterval + ". Because the later must be multiple of the former.");
        }

        final int ratio = greatInterval / _intervalInHours; //suppose new interval is multiple of old interval !!!

        _startTime = _startTime + (ratio - 1) * getIntervalInMillis();

        //update interval now
        _intervalInHours = greatInterval;

        final List<Measurement> newMeasurementList = new ArrayList<Measurement>();

        double sumValues = 0.0;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            //sum up all the measurement values in that small period and put in the new measurement list
            if((i + 1) % ratio != 0)
            {
                sumValues += _measurementList.get(i).getValue(_measuringUnit);//should not be needed to pass in the unit
            }
            else
            { //pick 3rd, 7th, 11th, ...

                sumValues += _measurementList.get(i).getValue(_measuringUnit);//should not be needed to pass in the unit

                //put the aggregated measuremet into a new list
                newMeasurementList.add(new Measurement(sumValues, _measuringUnit));

                //re-set it for next period
                sumValues = 0.0;
            }

        } //close for loop

        //re-assign _measurementList
        _measurementList = newMeasurementList;
    }

    // --------------------------------------------------------

    @Override
    public RegularTimeSeries clone()
    {
        try
        {
            final RegularTimeSeries copyRTS = (RegularTimeSeries)super.clone();

            //deep copy _measurementList
            copyRTS._measurementList = new ArrayList<Measurement>();
            for(int i = 0; i < this.getMeasurementCount(); i++)
            {
                copyRTS._measurementList.add(this._measurementList.get(i).getCopy());
            }

            //MeasureingUnit is a static object. No need to worry about that

            return copyRTS;
        }
        catch(final CloneNotSupportedException e)
        { //This should not happern
            return null; //To keep the compiler happy
        }
    }

    /**
     * You should use List<String> getQualifierIds() instead of this method to comply with xml schema
     * 
     * @return
     */
    @Deprecated
    public String getQualifierId()
    {
        if(_qualifierId != null && _qualifierId.size() > 0)
            return _qualifierId.get(0);
        else
            return "";
    }

    /**
     * You should use setQualifierIds(final List<String> ids) instead of this method to comply with xml schema.
     * 
     * @param id
     */
    @Deprecated
    public void setQualifierId(final String id)
    {
        if(_qualifierId != null && _qualifierId.size() > 0)
            _qualifierId.remove(0);
        else
            _qualifierId = new ArrayList<String>();
        _qualifierId.add(id);
    }

    public List<String> getQualifierIds()
    {
        return _qualifierId;
    }

    public void setQualifierIds(final List<String> ids)
    {
        _qualifierId = ids;
    }

    public String getEnsembleId()
    {
        return _ensembleId;
    }

    /**
     * @param id Any null passed in will be converted to NO_ENSEMBLE_ID.
     */
    public void setEnsembleId(final String id)
    {
        if((id == null) || (id.length() == 0))
        {
            _ensembleId = NO_ENSEMBLE_ID;
        }
        else
        {
            _ensembleId = id;
        }
    }

    public int getEnsembleMemberIndex()
    {
        return this._ensembleMemberIndex;
    }

    public void setEnsembleMemberIndex(final int index)
    {
        this._ensembleMemberIndex = index;
    }

    /**
     * Based on the input parameters, check if this RegularTimeSeries instance has enough data. If not, throws an
     * Exception.
     */
    public void checkHasEnoughData(final long initStateTime, final long endTime) throws Exception
    {
        final long rtsCoverStartTime = getStartTime() - getIntervalInMillis();

        String message = "Input time series " + getTimeSeriesType() + " does not have enough data.";

        //its first time step should cover a period including the initial state time; otherwise, it starts too late
        if(rtsCoverStartTime > initStateTime)
        { //rts start time is too late

            message += " It starts at " + DateTime.getDateTimeStringFromLong(rtsCoverStartTime,OHDConstants.GMT_TIMEZONE)
                + ". But initial state time is " + DateTime.getDateTimeStringFromLong(initStateTime,OHDConstants.GMT_TIMEZONE);

            throw new Exception(message);
        }
        else if(getEndTime() < endTime)
        {//rts ends before the specified model run end time, not enough data

            message += " It ends at " + DateTime.getDateTimeStringFromLong(getEndTime(),OHDConstants.GMT_TIMEZONE)
                + ". But the required end time is " + DateTime.getDateTimeStringFromLong(endTime,OHDConstants.GMT_TIMEZONE);

            throw new Exception(message);
        }

    } //close method

    /**
     * Check if "this" RTS instance is in sync with the parameter RTS. If not, throw an Exception. Assuming both RTS are
     * equi-distant time series.
     * <p>
     * 
     * @param aRTS - should not be null
     */
    public void checkSyncness(final RegularTimeSeries aRTS) throws Exception
    {
        final String errorMessage = "Input time series " + getTimeSeriesType() + " and " + aRTS.getTimeSeriesType()
            + " are out of sync.";

        int smallestIntervalInHours = this.getIntervalInHours();

        if(smallestIntervalInHours > aRTS.getIntervalInHours())
        {
            smallestIntervalInHours = aRTS.getIntervalInHours();
        }

        //make sure the two intervals are even multiple or equal; if not, out of sync
        if((getIntervalInHours() % smallestIntervalInHours != 0)
            || (aRTS.getIntervalInHours() % smallestIntervalInHours != 0))
        {
            throw new Exception(errorMessage);
        }

        /*
         * The following code can handle two scenarios: 1)if both RTSs have the same start time, due to their intervals
         * are checked to be equal or even multiple, they are in sync. 2)if they have different start time, then check
         * the difference between the two RTS start time, if it is even multiple of smallest interval. If so, they are
         * in sync; if not, out of sync.
         */
        if((Math.abs(getStartTime() - aRTS.getStartTime()) % (smallestIntervalInHours * Timer.ONE_HOUR)) == 0)
        {
            //they are in sync
            return;
        }

        //if reach here, must be out of sync
        throw new Exception(errorMessage);

    }

    public List<Measurement> getMeasurementList()
    {
        return _measurementList;
    }

    public void setMeasurementList(final List<Measurement> measurementList)
    {
        _measurementList = measurementList;
    }

    /**
     * Returns true if all the values are {@link #_missingValue}; otherwise, returns false.
     */
    public boolean isEmpty()
    {
        for(final Measurement m: _measurementList)
        {
            if(m.getValue() != _missingValue)
            {
                return false;
            }
        }

        //if reach here, all the values must be -999.0
        return true;
    }

    public static void main(final String[] args)
    {
        final long millisPerHour = 1000 * 60 * 60;
        final int hoursCount = 6;
        long startTime = System.currentTimeMillis();
        startTime = TimeHelper.truncateTimeInMillisToNearestHour(startTime, 1);

        final long endTime = startTime + millisPerHour * hoursCount;

        final int hoursPerInterval = 1;

        final RegularTimeSeries ts1 = new RegularTimeSeries(startTime, endTime, hoursPerInterval, MeasuringUnit.mm);

        final long startTime2 = endTime - 3 * millisPerHour;
        final long endTime2 = startTime2 + millisPerHour * hoursCount;
        final RegularTimeSeries ts2 = new RegularTimeSeries(startTime2, endTime2, hoursPerInterval, MeasuringUnit.mm);

        long m1Time = startTime;
        long m2Time = startTime2;
        for(int i = 0; i <= hoursCount; i++)
        {
            final Measurement measurement = new Measurement(13.0 + i, MeasuringUnit.mm);
            ts1.setMeasurementByTime(measurement, m1Time);

            final Measurement measurement2 = new Measurement(47.0 + i, MeasuringUnit.mm);
            ts2.setMeasurementByTime(measurement2, m2Time);

            m1Time += millisPerHour;
            m2Time += millisPerHour;
        }

        System.out.println("ts1 = " + ts1);
        System.out.println("ts2 = " + ts2);

        // test out sub time series
        final RegularTimeSeries subTimeSeries = ts1.getSubTimeSeries(startTime2, endTime2);

        System.out.println("new start time = " + DateTime.getDateTimeStringFromLong(startTime2,OHDConstants.GMT_TIMEZONE)
            + " new end time = " + DateTime.getDateTimeStringFromLong(endTime2,OHDConstants.GMT_TIMEZONE));
        System.out.println("subTimeSeries = " + subTimeSeries);

        // test out adding measurements that occur before and after the way the time series
        // is currently set up
        final Measurement measurement = new Measurement(999.0, MeasuringUnit.mm);
        ts1.setMeasurementByTime(measurement, ts1.getStartTime() - millisPerHour);

        final Measurement measurement2 = new Measurement(1000.0, MeasuringUnit.mm);
        ts1.setMeasurementByTime(measurement2, ts1.getEndTime() + millisPerHour);

        System.out.println("altered ts1 = " + ts1);

        // test out adding measurements before and after the current start and end times
        // of a RegularTimeSeries

        long newStartTime = ts1.getStartTime() - 5 * millisPerHour;
        final long newEndTime = ts1.getEndTime() + 5 * millisPerHour;

        ts1.stretchTimeSeries(newStartTime, newEndTime, -99999);

        System.out.println("stretched ts1 = " + ts1);

        newStartTime += 10 * millisPerHour;
        ts1.trimTimeSeriesAtStart(newStartTime);
        System.out.println("trimmed ts1 = " + ts1);

        /*
         * // index testing section int index = ts1.getMeasurementIndexByTime(startTime); long time =
         * ts1.getMeasurementTimeByIndex(index); int index2 = ts1.getMeasurementIndexByTime(time); long time2 =
         * ts1.getMeasurementTimeByIndex(index2); System.out.println("ts1 index = " + index); System.out.println("ts1
         * time = " + time); System.out.println("ts1 index = " + index2); System.out.println("ts1 time2 = " + time2);
         */

        final RegularTimeSeries concatTs = RegularTimeSeries.concatenate(ts1, ts2);

        System.out.println("concat Ts = " + concatTs);

        System.out.println("ts1:\n" + ts1.toStringWithTimeStamp());
        System.out.println("ts2:\n" + ts2.toStringWithTimeStamp());

        System.out.println("concatTs:\n" + concatTs.toStringWithTimeStamp());

    }
} //end  RegularTimeSeries
