/*
 * Created on Aug 27, 2004
 */
package ohd.hseb.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.AbsTimeMeasurement;
import ohd.hseb.measurement.IrregularTimeSeries;
import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.CodeTimer;
import ohd.hseb.util.fews.OHDConstants;

/**
 * @author Chip Gobs The job of this class is to manage the blending of observed and forecast data such that the
 *         forecast time series is adjusted to match better with the observed data.
 */
public class ForecastAdjuster
{
    private static final long MILLIS_PER_MINUTE = 60 * 1000;
    private static final long MILLIS_PER_HOUR = 60 * MILLIS_PER_MINUTE;

    private ForecastAdjusterParams _params = null;
    private List<ObservedForecastMeasurementPair> _observedForecastPairList = null;
    private Map<AbsTimeMeasurement, AbsTimeMeasurement> _observedForecastMap = null;
    private long _lastObservedTimeToUseAsInput = 0;

    private final static boolean _debug = false;

    // ----------------------------------------------------------------------------------

    public ForecastAdjuster(final ForecastAdjusterParams params)
    {
//        final String header = "ForecastAdjuster()";

        //  System.out.println("inside " + header);

        _params = params;

        _observedForecastPairList = new ArrayList<ObservedForecastMeasurementPair>();
        _observedForecastMap = new HashMap<AbsTimeMeasurement, AbsTimeMeasurement>();
    }

    // ----------------------------------------------------------------------------------
    public RegularTimeSeries getAdjustedTimeSeries(final IrregularTimeSeries observedTs,
                                                   final RegularTimeSeries origFcstTs)
    {
        // String header = "ForecastAdjuster.getAdjustTimeSeries():" ;
        /*
          *  
          */
        RegularTimeSeries adjustedFcstTs = null;

        //trim the observedTimeSeries as needed
        IrregularTimeSeries trimmedObsTs = null;

        trimmedObsTs = observedTs.getSubTimeSeries(observedTs.getStartTime(), getLastObservedTimeToUseAsInput());

        //  trimmedObsTs = observedTs;

        final CodeTimer findTimer = new CodeTimer();
        //CodeTimer latestTimer = new CodeTimer();
        final CodeTimer buildTimer = new CodeTimer();

        final int allowableBlendingHours = _params.getBlendingHours();

        final ForecastBlender forwardBlender = new ForecastBlender(allowableBlendingHours);
        final ForecastBlender backwardBlender = new ForecastBlender(allowableBlendingHours);

        if(_params.shouldDoAdjustment())
        {
            // System.out.println(header + "Actually doing the adjustment");
            //create the data structure for helping go from a forecast point to its matching
            //observed point (if any)
            buildTimer.restart();
            buildObsAndForecastPairList(trimmedObsTs, origFcstTs, _params.getPairingTimeMinutes() * MILLIS_PER_MINUTE);
            buildTimer.stop();

            final long firstFcstTime = origFcstTs.getStartTime();
            final long lastFcstTime = origFcstTs.getEndTime();

            // create new forecast time series
            adjustedFcstTs = new RegularTimeSeries(firstFcstTime,
                                                   lastFcstTime,
                                                   origFcstTs.getIntervalInHours(),
                                                   origFcstTs.getMeasuringUnit());

//            ObservedForecastMeasurementPair currentPair = null;
            ObservedForecastMeasurementPair previousPair = null;
            ObservedForecastMeasurementPair nextPair = null;

            AbsTimeMeasurement previousObs = null;
            AbsTimeMeasurement nextObs = null;

            AbsTimeMeasurement origFcstMeasurement = null;
            AbsTimeMeasurement adjustedMeasurement = null;

//            boolean usedDirectAdjustment = false;
//            boolean mayInterpolate = false;
            boolean doBlending = false;

            //for each forecast value
            for(long time = firstFcstTime; time <= lastFcstTime; time += MILLIS_PER_HOUR)
            {
                origFcstMeasurement = origFcstTs.getAbsTimeMeasurementByTime(time);

                final String fcstTimeString = DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE);

                /*
                 * if (_debug) { System.out.println("fcstTime = " + DbTimeHelper.getDateTimeStringFromLongTime(time) );
                 * }
                 */

                adjustedMeasurement = new AbsTimeMeasurement(origFcstMeasurement);

                //re-init all the pairing variables
//                currentPair = null;
                previousPair = null;
                nextPair = null;

                previousObs = null;
                nextObs = null;

//                usedDirectAdjustment = false;
//                mayInterpolate = false;
                doBlending = false;

                findTimer.restart();

                //try to find the origFcstMeasurement in the hashMap                
                final AbsTimeMeasurement matchingObsMeasurement = _observedForecastMap.get(origFcstMeasurement);

                if(matchingObsMeasurement != null) //there is a matching observedMeasurement, use direct adjustment
                {
                    //set value to current pairing observed value
                    adjustedMeasurement = new AbsTimeMeasurement(origFcstMeasurement);
                    adjustedMeasurement.setValue(matchingObsMeasurement.getValue());

                    if(_debug)
                    {
                        System.out.println(fcstTimeString + " using direct correction by pairing");
                    }
                    forwardBlender.clearBlender();
                    backwardBlender.clearBlender();

                }
                else
                // can't do direct adjustment, try interpolation and blending
                {
                    previousPair = findBestPreviousPairByTime(time);
                    nextPair = findBestNextPairByTime(time);

                    if((previousPair != null) && (nextPair != null)) //
                    {

                        previousObs = previousPair.getObservedMeasurement();
                        nextObs = nextPair.getObservedMeasurement();

                        final long prevTime = previousObs.getTime();
                        final long nextTime = nextObs.getTime();

                        //consider adding a method to AbsTimeMeasurement to
                        // allow for time window comparisons

                        final long timeDiff = nextTime - prevTime;
                        if(timeDiff <= _params.getInterpolationHours() //can do interpolation
                            * MILLIS_PER_HOUR)
                        {

                            if(_debug)
                            {
                                System.out.print(fcstTimeString + " using interpolation by");
                            }
                            adjustedMeasurement = adjustByInterpolation(origFcstMeasurement, previousPair, nextPair);

                            forwardBlender.clearBlender();
                            backwardBlender.clearBlender();
                            doBlending = false;
                        }
                        else
                        //can't do interpolation, need to try blending
                        {
                            doBlending = true;
                        }
                    } //end if previousPair and nextPair not null

                    else
                    //can't do interpolation, need to try blending
                    {
                        doBlending = true;
                    }

                    if(doBlending) //pairs not close enough to each other, or at least
                    // one pair is missing, so try to use blending
                    {

                        final AbsTimeMeasurement forwardAdjustment = forwardBlender.getBlendAdjustment(previousPair,
                                                                                                       origFcstMeasurement);
                        final AbsTimeMeasurement backwardAdjustment = backwardBlender.getBlendAdjustment(nextPair,
                                                                                                         origFcstMeasurement);

                        final AbsTimeMeasurement totalAdjustment = new AbsTimeMeasurement(forwardAdjustment);
                        totalAdjustment.addMeasurement(backwardAdjustment);
                        adjustedMeasurement.addMeasurement(totalAdjustment);

                        if(totalAdjustment.getValue() != 0.0)
                        {
                            if(_debug)
                            {
                                System.out.println(fcstTimeString + " Total blending adjustment =  " + totalAdjustment);
                            }
                        }
                        else
                        {
                            if(_debug)
                            {
                                System.out.println(fcstTimeString + " No adjustment.");
                            }
                        }

                    }

                } //end else for unmatched fcst measurements
                findTimer.stop();

                if(adjustedMeasurement.getValue() < 0.0)
                {
                    adjustedMeasurement.setValue(0.0);
                }

                adjustedFcstTs.setMeasurementByTime(adjustedMeasurement, time);

            } //end for each forecast value

        }
        else
        //don't adjust anything
        {
            if(_debug)
            {
                System.out.println("Not doing the adjustment");
            }
            adjustedFcstTs = origFcstTs;
        }

        return adjustedFcstTs;

    }

    // -------------------------------------------------------------------------------------------------
//    private AbsTimeMeasurement getLatestObservedBeforeTime(final IrregularTimeSeries obsTs, final long targetTime)
//    {
//        AbsTimeMeasurement currentMeasurement = null;
//        AbsTimeMeasurement latestMeasurement = null;
//        //     String targetTimeString = DbTimeHelper.getDateTimeStringFromLongTime(targetTime);
//
//        boolean done = false;
//        long time = 0;
//
//        // the measurements returned from getAbsTimeMeasurementByIndex are in ascending order
//        // from earliest to latest in time
//        for(int i = 0; (!done && (i < obsTs.getMeasurementCount())); i++)
//        {
//            currentMeasurement = obsTs.getAbsTimeMeasurementByIndex(i);
//            time = currentMeasurement.getTime();
//
//            //     String timeString = DbTimeHelper.getDateTimeStringFromLongTime(time);
//
//            if(time <= targetTime)
//            {
//                latestMeasurement = currentMeasurement;
//            }
//
//            if(time > targetTime)
//            {
//                done = true;
//            }
//        }
//        return latestMeasurement;
//    }

    // ----------------------------------------------------------------------------------
//    private AbsTimeMeasurement getEarliestObservedAfterTime(final IrregularTimeSeries obsTs, final long targetTime)
//    {
//        AbsTimeMeasurement currentMeasurement = null;
//        AbsTimeMeasurement latestMeasurement = null;
//
//        boolean done = false;
//        long time = 0;
//        // the measurements returned from getAbsTimeMeasurementByIndex are in ascending order
//        // from earliest to latest in time
//        for(int i = 0; (!done && (i < obsTs.getMeasurementCount())); i++)
//        {
//            currentMeasurement = obsTs.getAbsTimeMeasurementByIndex(i);
//            time = currentMeasurement.getTime();
//
//            if(time <= targetTime)
//            {
//                //do nothing
//            }
//
//            if(time >= targetTime)
//            {
//                latestMeasurement = currentMeasurement;
//                done = true;
//            }
//        }
//        return latestMeasurement;
//    }

    // ----------------------------------------------------------------------------------

    // ----------------------------------------------------------------------------------
//    private boolean isTimeBetween(final long time, final long startTime, final long endTime)
//    {
//        boolean result = false;
//
//        if((time >= startTime) && (time <= endTime))
//        {
//            result = true;
//        }
//
//        return result;
//    }

    // ----------------------------------------------------------------------------------
    private AbsTimeMeasurement adjustByInterpolation(final AbsTimeMeasurement origFcstMeasurement,
                                                     final ObservedForecastMeasurementPair prevPair,
                                                     final ObservedForecastMeasurementPair nextPair)
    {

        final AbsTimeMeasurement adjustedFcstMeasurement = new AbsTimeMeasurement(origFcstMeasurement);

        final ForecastInterpolationMethod blendingMethod = _params.getBlendingMethod();

        final AbsTimeMeasurement prevObs = prevPair.getObservedMeasurement();
        final AbsTimeMeasurement prevFcst = prevPair.getForecastMeasurement();

        final AbsTimeMeasurement nextObs = nextPair.getObservedMeasurement();
        final AbsTimeMeasurement nextFcst = nextPair.getForecastMeasurement();

        final long origFcstTime = origFcstMeasurement.getTime();

        if((blendingMethod != null) && (blendingMethod.equals(ForecastInterpolationMethod.RATIO)))
        {
            final double ratio1 = prevObs.getValue() / prevFcst.getValue();
            final double ratio2 = nextObs.getValue() / nextFcst.getValue();

            final double ratioDiff = Math.abs(ratio1 - ratio2);

            //if out of sanity range, then do DIFFERENCES anyway
            if((ratio1 > 5.0) || (ratio2 > 5.0) || (ratioDiff > 2.0))
            {
                System.out.println(" differences ");
                final double adjustment = getAdjustmentUsingDifferences(origFcstTime, prevPair, nextPair);
                final double oldValue = adjustedFcstMeasurement.getValue();
                adjustedFcstMeasurement.setValue(oldValue + adjustment);
            }
            else
            // it is OK to use ratio method
            {
                System.out.println(" ratios ");
                final double adjustment = getAdjustmentUsingRatios(origFcstTime, prevPair, nextPair);
                final double oldValue = adjustedFcstMeasurement.getValue();
                adjustedFcstMeasurement.setValue(oldValue * adjustment);
            }
        }
        else
        //use differences
        {
            System.out.println(" differences ");
            final double adjustment = getAdjustmentUsingDifferences(origFcstTime, prevPair, nextPair);
            final double oldValue = adjustedFcstMeasurement.getValue();
            adjustedFcstMeasurement.setValue(oldValue + adjustment);

        }
        return adjustedFcstMeasurement;
    }

    // ----------------------------------------------------------------------------------
    private double getAdjustmentUsingDifferences(final long origFcstTime,
                                                 final ObservedForecastMeasurementPair prevPair,
                                                 final ObservedForecastMeasurementPair nextPair)
    {
        final AbsTimeMeasurement prevObs = prevPair.getObservedMeasurement();
        final AbsTimeMeasurement prevFcst = prevPair.getForecastMeasurement();

        final AbsTimeMeasurement nextObs = nextPair.getObservedMeasurement();
        final AbsTimeMeasurement nextFcst = nextPair.getForecastMeasurement();

        final double diff1 = prevObs.getValue() - prevFcst.getValue();
        final double diff2 = nextObs.getValue() - nextFcst.getValue();

        final AbsTimeMeasurement diff1Measurement = new AbsTimeMeasurement(diff1, prevObs.getTime(), prevObs.getUnit());

        final AbsTimeMeasurement diff2Measurement = new AbsTimeMeasurement(diff2, nextObs.getTime(), nextObs.getUnit());

        final AbsTimeMeasurement adjustmentMeasurement = interpolate(diff1Measurement, diff2Measurement, origFcstTime);

        final double adjustmentValue = adjustmentMeasurement.getValue();

        return adjustmentValue;

    }

    // ----------------------------------------------------------------------------------

    private double getAdjustmentUsingRatios(final long origFcstTime,
                                            final ObservedForecastMeasurementPair prevPair,
                                            final ObservedForecastMeasurementPair nextPair)
    {
        final AbsTimeMeasurement prevObs = prevPair.getObservedMeasurement();
        final AbsTimeMeasurement prevFcst = prevPair.getForecastMeasurement();

        final AbsTimeMeasurement nextObs = nextPair.getObservedMeasurement();
        final AbsTimeMeasurement nextFcst = nextPair.getForecastMeasurement();

        final double ratio1 = prevObs.getValue() / prevFcst.getValue();
        final double ratio2 = nextObs.getValue() / nextFcst.getValue();

        final AbsTimeMeasurement ratio1Measurement = new AbsTimeMeasurement(ratio1,
                                                                            prevObs.getTime(),
                                                                            prevObs.getUnit());

        final AbsTimeMeasurement ratio2Measurement = new AbsTimeMeasurement(ratio2,
                                                                            nextObs.getTime(),
                                                                            nextObs.getUnit());

        final AbsTimeMeasurement adjustmentMeasurement = AbsTimeMeasurement.interpolate(ratio1Measurement,
                                                                                        ratio2Measurement,
                                                                                        origFcstTime);

        final double adjustmentValue = adjustmentMeasurement.getValue();

        return adjustmentValue;
    }

    // ----------------------------------------------------------------------------------

//    private ObservedForecastMeasurementPair oldFindBestPreviousPairByTime(final IrregularTimeSeries observedTs,
//                                                                          final RegularTimeSeries forecastTs,
//                                                                          final long targetTime,
//                                                                          final long maxTimeDifferenceForPairing)
//    {
//        ObservedForecastMeasurementPair pair = null;
//
//        final IrregularTimeSeries choppedObservedTs = observedTs.getSubTimeSeries(observedTs.getStartTime(), targetTime);
//
//        AbsTimeMeasurement currentObsMeasurement = null;
//        AbsTimeMeasurement forecastMeasurement = null;
//
//        boolean done = false;
//        long obsTime = 0;
//        // the measurements returned from getAbsTimeMeasurementByIndex(index)
//        // are in ascending order
//        // from earliest to latest in time
//
//        // start near or at targetTime and work backwards in time
//        for(int i = choppedObservedTs.getMeasurementCount() - 1; (!done && i >= 0); i--)
//        {
//            currentObsMeasurement = choppedObservedTs.getAbsTimeMeasurementByIndex(i);
//            obsTime = currentObsMeasurement.getTime();
//
//            forecastMeasurement = findClosestForecastByObsTimeAtOrBeforeTargetTime(forecastTs, obsTime, targetTime);
//
//            if(areMeasurementsCloseEnoughInTime(forecastMeasurement, currentObsMeasurement, maxTimeDifferenceForPairing))
//            {
//
//                pair = new ObservedForecastMeasurementPair(currentObsMeasurement, forecastMeasurement);
//
//                done = true;
//            }
//
//            else
//            //keep looking
//            {
//
//            }
//        } //end for
//
//        return pair;
//
//    }

    // -----------------------------------------------------------------------------------------------
//    private AbsTimeMeasurement findClosestForecastByObsTimeAtOrBeforeTargetTime(final RegularTimeSeries forecastTs,
//                                                                                final long obsTime,
//                                                                                final long targetTime)
//    {
//        //find the closest forecast (by time) to the obstime.
//        //The forecast found must be at or before the targetTime
//
//        boolean done = false;
//        final boolean firstTime = true;
//        long smallestTimeDiff = 0;
//        long timeDiff = 0;
//        AbsTimeMeasurement bestFcstMeasurement = null;
//
//        final String obsTimeString = DbTimeHelper.getDateTimeStringFromLongTime(obsTime);
//        final String targetTimeString = DbTimeHelper.getDateTimeStringFromLongTime(targetTime);
//
//        for(int i = 0; !done && i < forecastTs.getMeasurementCount(); i++)
//        {
//
//            final AbsTimeMeasurement fcstMeasurement = forecastTs.getAbsTimeMeasurementByIndex(i);
//            final long fcstTime = fcstMeasurement.getTime();
//
//            final String fcstTimeString = DbTimeHelper.getDateTimeStringFromLongTime(fcstTime);
//
//            if(fcstTime <= targetTime)
//            {
//                if(i == 0) //first time, do some initialization
//                {
//                    smallestTimeDiff = Math.abs(obsTime - fcstTime);
//                    bestFcstMeasurement = fcstMeasurement;
//                }
//                else
//                //not the first iteration
//                {
//                    timeDiff = Math.abs(obsTime - fcstTime);
//
//                    if(timeDiff < smallestTimeDiff)
//                    {
//                        smallestTimeDiff = timeDiff;
//                        bestFcstMeasurement = fcstMeasurement;
//                    }
//                    else
//                    //we are getting farther from the obsTime, so stop
//                    {
//                        done = true;
//                    }
//                }
//            }
//            else
//            //fcstTime > targetTime, so don't look any more
//            {
//                done = true;
//            }
//
//        }
//
//        return bestFcstMeasurement;
//    }

    // -----------------------------------------------------------------------------------------------
    @SuppressWarnings({"unchecked", "rawtypes"})
    private void buildObsAndForecastPairList(final IrregularTimeSeries observedTs,
                                             final RegularTimeSeries forecastTs,
                                             final long maxTimeDifferenceForPairing)

    {
//        final String header = "ForecastAdjuster.buildObsAndForecastPairList(): ";

        _observedForecastPairList = new ArrayList();
        _observedForecastMap = new HashMap();

        int startingObsIndex = 0;

        long forecastTime = 0;
        long obsTime = 0;
        long timeDiff = 0;

        // for each forecast point
        for(int fcstIndex = 0; fcstIndex < forecastTs.getMeasurementCount(); fcstIndex++)
        {
            final AbsTimeMeasurement forecastMeasurement = forecastTs.getAbsTimeMeasurementByIndex(fcstIndex);

            forecastTime = forecastMeasurement.getTime();

            //set up inner loop variables
            boolean foundFirstLegalMatch = false;
            long smallestTimeDifference = -1;
//            final int bestIndexSoFar = -1;
            AbsTimeMeasurement bestObsMeasurement = null;
            boolean done = false;

            // System.out.println(header + "forecastMeasurement = " + forecastMeasurement);

//          look at each observed to find the best match (if any) for this forecast value

            // System.out.println(header + "obsTs.getMeasurementCount = " + observedTs.getMeasurementCount());

            for(int obsIndex = startingObsIndex; (!done && (obsIndex < observedTs.getMeasurementCount())); obsIndex++)
            {
                final AbsTimeMeasurement obsMeasurement = observedTs.getAbsTimeMeasurementByIndex(obsIndex);

                //     System.out.println(header + "obsMeasurement = " + obsMeasurement);
                //     System.out.println(header + "obsIndex = " + obsIndex);

                obsTime = obsMeasurement.getTime();

                timeDiff = Math.abs(forecastTime - obsTime);

                //  observed is in valid range for pairing
                if(timeDiff <= maxTimeDifferenceForPairing)
                {
                    if(!foundFirstLegalMatch)
                    {
                        foundFirstLegalMatch = true;
                        smallestTimeDifference = timeDiff;
                        bestObsMeasurement = obsMeasurement;
                    }
                    else
                    //have already found a valid match for this forecast point
                    {
                        if(timeDiff < smallestTimeDifference)
                        {
                            smallestTimeDifference = timeDiff;
                            bestObsMeasurement = obsMeasurement;
                        }
                        else
                        //we have gone past the best-matching observed
                        //we have already found smallest time difference 
                        //(list is sorted in ascending order by time)
                        {
                            done = true;
                            startingObsIndex = obsIndex; //next time we do the inner loop, start here

                        }
                    }
                }
                else
                // observed is not within valid range for pairing
                {
                    if(foundFirstLegalMatch) // we now know the best match
                    {
                        done = true;
                        startingObsIndex = obsIndex; //next time we do the inner loop, start here

                    }
                    else
                    // (! foundFirstLegalMatch) 
                    {
                        if(obsTime > forecastTime)
                        {
                            // there is no match for this forecast measurement,
                            // since we have passed the forecastTime and
                            // are out of the time window
                            done = true;
                            startingObsIndex = obsIndex;
                        }
                        else
                        // obsTime <= forecastTime, actually, if it were =, the time window would match 
                        {
                            //System.out.println("obsTime < forecastTime");
                            // we need to keep looking, because we have not 
                            // yet gotten in range
                        }
                    } //end else ! foundFirstLegalMatch

                } //end else observed not within valid range for printing

            } //end for each observed

            //check to see if any matches were found
            // if so, then add the pair of points to the list and map
            if(foundFirstLegalMatch)
            {
                final ObservedForecastMeasurementPair pair = new ObservedForecastMeasurementPair(bestObsMeasurement,
                                                                                                 forecastMeasurement);

                _observedForecastPairList.add(pair);
                _observedForecastMap.put(forecastMeasurement, bestObsMeasurement);

                if(_debug)
                {
                    System.out.println("pair = " + pair);
                }
            }
            else
            {

                //System.out.println("did not find a pair for " + forecastMeasurement);
            }

        } //end for each forecast value

        return;

    } // buildObsAndForecastPairList

    // -----------------------------------------------------------------------------------------------
    /*
     * private ObservedForecastMeasurementPair findBestNextPairByTime(long targetTime, List obsForecastPairList) {
     * ObservedForecastMeasurementPair pair = null; for (int i = 0; i < obsForecastPairList.size(); i++ ) { } return
     * pair; }
     */
    //  -----------------------------------------------------------------------------------------------
    private ObservedForecastMeasurementPair findBestNextPairByTime(final long targetTime)
    {
        ObservedForecastMeasurementPair nextPair = null;
        ObservedForecastMeasurementPair currentPair = null;
        boolean done = false;

        for(int i = 0; !done && i < _observedForecastPairList.size(); i++)
        {
            currentPair = _observedForecastPairList.get(i);
            if(currentPair.getForecastMeasurement().getTime() > targetTime)
            {
                nextPair = currentPair;
                done = true;
            }
        }

        return nextPair;
    }

//  -----------------------------------------------------------------------------------------------
    private ObservedForecastMeasurementPair findBestPreviousPairByTime(final long targetTime)
    {
        ObservedForecastMeasurementPair previousPair = null;
        ObservedForecastMeasurementPair currentPair = null;
        boolean done = false;

        for(int i = 0; !done && i < _observedForecastPairList.size(); i++)
        {
            currentPair = _observedForecastPairList.get(i);
            if(currentPair.getForecastMeasurement().getTime() < targetTime)
            {
                previousPair = currentPair;
            }
            else
            // m.time >= targetTime
            {
                done = true;
            }
        }

        return previousPair;
    }

//  -----------------------------------------------------------------------------------------------

//    private ObservedForecastMeasurementPair oldFindBestNextPairByTime(final IrregularTimeSeries observedTs,
//                                                                      final RegularTimeSeries forecastTs,
//                                                                      final long targetTime,
//                                                                      final long maxTimeDifferenceForPairing)
//    {
//        ObservedForecastMeasurementPair pair = null;
//
//        final IrregularTimeSeries choppedObservedTs = observedTs.getSubTimeSeries(targetTime, observedTs.getEndTime());
//
//        AbsTimeMeasurement currentObsMeasurement = null;
//        AbsTimeMeasurement forecastMeasurement = null;
//
//        boolean done = false;
//        long obsTime = 0;
//        // the measurements returned from getAbsTimeMeasurementByIndex(index)
//        // are in ascending order
//        // from earliest to latest in time
//
//        // start near or at targetTime and work forward in time
//        for(int i = 0; (!done && i < choppedObservedTs.getMeasurementCount()); i++)
//        {
//            currentObsMeasurement = choppedObservedTs.getAbsTimeMeasurementByIndex(i);
//            obsTime = currentObsMeasurement.getTime();
//
//            forecastMeasurement = findClosestForecastByObsTimeAtOrAfterTargetTime(forecastTs, obsTime, targetTime);
//
//            if(areMeasurementsCloseEnoughInTime(forecastMeasurement, currentObsMeasurement, maxTimeDifferenceForPairing))
//            {
//
//                pair = new ObservedForecastMeasurementPair(currentObsMeasurement, forecastMeasurement);
//
//                done = true;
//            }
//
//            else
//            //keep looking
//            {
//
//            }
//        } //end for
//
//        return pair;
//
//    }

    // ---------------------------------------------------------------------------------------------------
//    private AbsTimeMeasurement findClosestForecastByObsTimeAtOrAfterTargetTime(final RegularTimeSeries forecastTs,
//                                                                               final long obsTime,
//                                                                               final long targetTime)
//    {
//        //find the closest forecast (by time) to the obstime.
//        //The forecast found must be at or after the targetTime
//
//        boolean done = false;
//        boolean firstTime = true;
//        long smallestTimeDiff = 0;
//        long timeDiff = 0;
//        AbsTimeMeasurement bestFcstMeasurement = null;
//
//        for(int i = 0; !done && i < forecastTs.getMeasurementCount(); i++)
//        {
//
//            final AbsTimeMeasurement fcstMeasurement = forecastTs.getAbsTimeMeasurementByIndex(i);
//            final long fcstTime = fcstMeasurement.getTime();
//
//            if(fcstTime >= targetTime)
//            {
//                if(firstTime) //first time, do some initialization
//                {
//                    smallestTimeDiff = Math.abs(obsTime - fcstTime);
//                    bestFcstMeasurement = fcstMeasurement;
//                    firstTime = false;
//                }
//                else
//                //not the first iteration
//                {
//                    timeDiff = Math.abs(obsTime - fcstTime);
//
//                    if(timeDiff < smallestTimeDiff)
//                    {
//                        smallestTimeDiff = timeDiff;
//                        bestFcstMeasurement = fcstMeasurement;
//                    }
//                    else
//                    //we are getting farther from the obsTime, so stop
//                    {
//                        done = true;
//                    }
//                }
//
//            }
//            else
//            //fcstTime < targetTime, so we need to just keep iterating
//            {
//                // do nothing
//            }
//
//        }
//
//        return bestFcstMeasurement;
//    }

    // -----------------------------------------------------------------------------------------------

//    private boolean areMeasurementsCloseEnoughInTime(final AbsTimeMeasurement m1,
//                                                     final AbsTimeMeasurement m2,
//                                                     final long maxMillisForCloseness)
//    {
//        boolean result = false;
//
//        if((m1 != null) && (m2 != null))
//        {
//            final long timeDiff = m1.getTime() - m2.getTime();
//
//            if(Math.abs(timeDiff) <= maxMillisForCloseness)
//            {
//                result = true;
//            }
//        }
//
//        return result;
//    }

    // ----------------------------------------------------------------------------------
    public static AbsTimeMeasurement interpolate(final AbsTimeMeasurement m1,
                                                 final AbsTimeMeasurement m2,
                                                 final long desiredTime)
    {
        final long t1 = m1.getTime();
        final long t2 = m2.getTime();

        final double value1 = m1.getValue();
        final double value2 = m2.getValue();

        final double slope = (value2 - value1) / (t2 - t1);
        final double intercept = (value1) - (slope * t1);

        final double value = (slope * desiredTime) + intercept;
        final AbsTimeMeasurement measurement = new AbsTimeMeasurement(value, desiredTime, m1.getUnit());

        measurement.setIsInterpolated(true);

        return measurement;
    }

    // ----------------------------------------------------------------------------------

    public ForecastAdjusterParams getParams()
    {
        return _params;
    }

    // ----------------------------------------------------------------------------------

    public void setParams(final ForecastAdjusterParams params)
    {
        _params = params;
    }

    // ----------------------------------------------------------------------------------
    public static void main(final String[] argStringArray)
    {
        final MeasuringUnit dischargeUnit = MeasuringUnit.cfs;
        final ForecastAdjusterParams params = new ForecastAdjusterParams();
        params.setBlendingHours(5);
        params.setBlendingMethod(ForecastInterpolationMethod.DIFFERENCE);
        params.setShouldDoAdjustment(true);

        final ForecastAdjuster adjuster = new ForecastAdjuster(params);

        final long fcstStartTime = System.currentTimeMillis();
        final long fcstEndTime = fcstStartTime + (7 * 24 * MILLIS_PER_HOUR); //7 days

        final long obsStartTime = fcstStartTime - (3 * 24 * MILLIS_PER_HOUR); //3 days before fcst start
        final long obsEndTime = fcstStartTime + (7 * MILLIS_PER_HOUR); // 7 hours after fcst start

        final IrregularTimeSeries observedTs = new IrregularTimeSeries(dischargeUnit);
        final RegularTimeSeries forecastTs = new RegularTimeSeries(fcstStartTime, fcstEndTime, 1, dischargeUnit);

        // load up the forecast time series
        final double[] fcstMeasurementValueArray = {20, 21, 22, 24, 26, 30, 35, 40, 41, 42, 43, 45, 55, 67, 43, 32, 31,
            30, 29, 28, 27, 26, 25};

        int index = 0;

        for(long time = fcstStartTime; time <= fcstEndTime; time += MILLIS_PER_HOUR)
        {
            final Measurement m = new Measurement(fcstMeasurementValueArray[index], dischargeUnit);
            forecastTs.setMeasurementByTime(m, time);

            index++;
            if(index == fcstMeasurementValueArray.length)
            {
                index = 0;
            }
        }

        // load up the observed time series
        index = 0;
        final double[] obsMeasurementValueArray = {18, 19, 20, 22, 24, 28, 33, 28, 39, 40, 41, 43, 53, 57, 41, 29, 29,
            28, 27, 26, 26, 24, 28};

        final long MILLIS_PER_MINUTE = 1000 * 60;
        for(long time = obsStartTime; time <= obsEndTime; time += 11 * MILLIS_PER_MINUTE)
        {
            final AbsTimeMeasurement m = new AbsTimeMeasurement(obsMeasurementValueArray[index], time, dischargeUnit);

            observedTs.insertMeasurement(m);

            index++;
            if(index == obsMeasurementValueArray.length)
            {
                index = 0;
            }
        }

        RegularTimeSeries adjustedForecastTs = null;

        adjustedForecastTs = adjuster.getAdjustedTimeSeries(observedTs, forecastTs);

        //print results
        System.out.println("observedTs = " + observedTs + "\n");

        System.out.println("original forecast Ts = " + forecastTs + "\n");
        System.out.println("adjusted forecast Ts = " + adjustedForecastTs);

    }

    // ----------------------------------------------------------------------------------

    /**
     * @param lastObservedTimeToUseAsInput The lastObservedTimeToUseAsInput to set.
     */
    public void setLastObservedTimeToUseAsInput(final long lastObservedTimeToUseAsInput)
    {
        //    String header = " ForecastAdjuster.setLastObservedTimeToUseAsInput()";
        //   System.out.println(header + "lastObservedTimeToUseAsInput = " + 
        //            DbTimeHelper.getDateTimeStringFromLongTime(lastObservedTimeToUseAsInput));
        _lastObservedTimeToUseAsInput = lastObservedTimeToUseAsInput;
    }

    // ----------------------------------------------------------------------------------

    /**
     * @return Returns the lastObservedTimeToUseAsInput.
     */
    public long getLastObservedTimeToUseAsInput()
    {
        //  String header = " ForecastAdjuster.getLastObservedTimeToUseAsInput()";
        //  System.out.println(header + "lastObservedTimeToUseAsInput = " + 
        //          DbTimeHelper.getDateTimeStringFromLongTime(_lastObservedTimeToUseAsInput));
        return _lastObservedTimeToUseAsInput;
    }

    // ----------------------------------------------------------------------------------

//    private class BlendingData
//    {
//
//        boolean _usedBlend = false;
//        int _blendHoursCount = 0;
//        double _blendAmount = 0.0;
//
//    }
    // ----------------------------------------------------------------------------------

} //end ForecastAdjuster
