/*
 * Created on Jan 20, 2004
 */
package ohd.hseb.model;

import java.util.Calendar;
import java.util.GregorianCalendar;

import javax.management.timer.Timer;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;

/**
 * @author GobsC This class encapsulates the double precision floating point values that are stored for each month.
 */
public class MonthlyValues
{

    private String _basinId = null;
    private String _pe = null;
    private short _dur = 0;
    private String _ts = null;
    private String _extremum = null;
    private boolean _adjustment = false;
    private long _postingTime = 0;
    private final double[] _valueArray = new double[12];
    private final int _monthsInYear = 12;

    final GregorianCalendar _cal = new GregorianCalendar(OHDConstants.GMT_TIMEZONE); //always in GMT, tz in-sensitive, 
    private int _startOfDay = 0; //used to set start of day for MAPE caculations;

    //jan feb mar apr   may  jun  jul  aug sep oct Nov Dec
    private static final int[] _daysInMonthArray = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

    //contains the number of days in the year prior to the start of the month
    private static int[] _totalDaysPriorToMonthArray = new int[12]; //leap years will be handled elsewhere

    static
    {
        _totalDaysPriorToMonthArray[0] = 0;
        for(int i = 1; i < _daysInMonthArray.length; i++)
        {
            _totalDaysPriorToMonthArray[i] = _totalDaysPriorToMonthArray[i - 1] + _daysInMonthArray[i - 1];
        }
    }

    // ------------------------------------------------------------

    public MonthlyValues()
    {
    } //MonthlyValuesFaster()

    // ------------------------------------------------------------
    public MonthlyValues(final double[] valueArray)
    {
        setValues(valueArray);
    } //MonthlyValuesFaster()

    // ------------------------------------------------------------
    public void setValues(final double[] valueArray)
    {
        for(int i = 0; i < valueArray.length && i < _monthsInYear; i++)
        {
            _valueArray[i] = valueArray[i];
        }

    } //setValues()

    // ------------------------------------------------------------

    /**
     * Based on the 12 monthly values stored inside(each one corresponding to 16th of the month) and the parameter
     * "time", calculate which day the time is and use the interpolation method to find the corresponding value.<br>
     * Note: 1)the returned value is the same for various parameter "time" if they are within the same day.<br>
     * 2)the Java code inside the method tries to achieve the same effect of Fortran code(NWSRFS). In another words,
     * 01-02 00:00 is treated as 1st day of the month, not 2nd day of the month. The Java code is translation of the
     * Fortran code MDYH1.h.
     * 
     * @param time -- milliseconds from 1970-01-01 00:00 GMT
     */
    public double getValue(final long time)
    {

        /*
         * JAVA convention is everyday counts from [00:00, 24:00), which means, e.g. "2008-01-01 00:00:00" belongs to
         * 1st day of the month, "2008-01-02 00:00:00"(i.e. "2008-01-01 24:00:00") belongs to 2nd day of the month. But
         * NWSRFS convention(by Fortran code) is (00:00, 24:00], which means "2008-01-01 00:00:00"(i.e.
         * "2007-12-31 24:00") belongs to last day of the last December, not 1st day of this January. By deducting one
         * millisecond, Java will consider the original time "2008-01-02 00:00:00" as 1st day of the month now when
         * calling get(Calendar.DAY_OF_MONTH). When _startOfDay is not zero, e.g. 12, deducting it from time is
         * necessary. In this way, Java code achieves the effect of Fortran code, which is what we want, since the
         * original code is Fortran and the Java code is the translation from it. The subtraction not only affects
         * dayOfMonth, it could affects month and year as well.
         */
        _cal.setTimeInMillis(time - _startOfDay * Timer.ONE_HOUR - 1);

        //extract needed info based on the current time
        final int monthIndex = _cal.get(Calendar.MONTH);
        final int year = _cal.get(Calendar.YEAR);
        final int dayOfMonth = _cal.get(Calendar.DAY_OF_MONTH);

        int year1 = year;
        int year2 = year;

        final int currentMonth = monthIndex;

        double month1Value = 0;
        double month2Value = 0;

        double weight1 = 0;
        double weight2 = 0;

        double monthToMonthDayDistance = 0.0;
        double month1ToDayDistance = 0.0;
        double month2ToDayDistance = 0.0;

        //determine which months to interpolate against
        if(dayOfMonth == 16)
        {
            return _valueArray[currentMonth];//the day is 16th of the month, no interpolation is needed
        }
        else if(dayOfMonth < 16)
        {
            int previousMonth = monthIndex - 1;
            if(previousMonth < Calendar.JANUARY)
            {
                previousMonth = Calendar.DECEMBER;
                year1 = year - 1;
            }

            month1Value = _valueArray[previousMonth];
            month2Value = _valueArray[currentMonth];

            monthToMonthDayDistance = getDayDistance(year1, previousMonth, 16, year2, currentMonth, 16);

            month1ToDayDistance = getDayDistance(year1, previousMonth, 16, year2, currentMonth, dayOfMonth);
            month2ToDayDistance = getDayDistance(year2, currentMonth, dayOfMonth, year2, currentMonth, 16);

        }
        else
        //dayOfMonth > 16
        {

            int nextMonth = currentMonth + 1;
            if(nextMonth > Calendar.DECEMBER)
            {
                nextMonth = Calendar.JANUARY;
                year2 = year + 1;
            }

            month1Value = _valueArray[currentMonth];
            month2Value = _valueArray[nextMonth];

            monthToMonthDayDistance = getDayDistance(year1, currentMonth, 16, year2, nextMonth, 16);

            month1ToDayDistance = getDayDistance(year1, currentMonth, 16, year1, currentMonth, dayOfMonth);
            month2ToDayDistance = getDayDistance(year1, currentMonth, dayOfMonth, year2, nextMonth, 16);

        }

        //calculate the weights based on the date's distance from the 16th of the current and adjacent month
        weight1 = (1.0 - (month1ToDayDistance / monthToMonthDayDistance));
        weight2 = (1.0 - (month2ToDayDistance / monthToMonthDayDistance));

        final double interpolatedValue = (weight1 * month1Value) + (weight2 * month2Value);

        return interpolatedValue;
    }

    // ------------------------------------------------------------
    /**
     * This is a helper method. Returns the number of days between the two dates. Date1(year1, month1 and day1) is older
     * than Date2(year2, month2, and day2).
     * <p>
     * Note: the two parameters(month1 and month2) are using Java convention: 0 for January, 1 for Feb. etc, because in
     * the method {@link #getValue(long)}, the month is using Java convention. Also note that year1 and year2 must be
     * the same or differ by only 1 year.
     */
    public double getDayDistance(final int year1,
                                 final int month1,
                                 final int day1,
                                 final int year2,
                                 final int month2,
                                 final int day2)
    {

        // Note: this algorithm precalculates the leap year so that it can avoid recalculating it.
        // It would be prettier code if getDayOfYear calculated it, but then it would have to be
        // calculated 2 to 3 times, instead of 1 to 2 times.
        final boolean isLeapYear1 = OHDUtilities.isLeapYear(year1);
        boolean isLeapYear2 = isLeapYear1;

        if(year2 != year1)
        {
            isLeapYear2 = OHDUtilities.isLeapYear(year2);
        }

        final int dayOfYear1 = getDayOfYear(year1, month1, day1, 12, 0, isLeapYear1);
        int dayOfYear2 = getDayOfYear(year2, month2, day2, 12, 0, isLeapYear2);

        /*
         * if there is a year change, dayOfYear2 needs to be boosted by 365 or 366(leap year) so that dayOfYear2 -
         * dayOfYear1 is always positive
         */
        if(year2 != year1)
        {
            if(isLeapYear1)
            {
                dayOfYear2 += 366;
            }
            else
            {
                dayOfYear2 += 365;
            }
        }

        return dayOfYear2 - dayOfYear1;

    } //end getDayDistance()

    // --------------------------------------------------------------------------------------
    private int getDayOfYear(final int year,
                             final int month,
                             final int day,
                             final int hour,
                             final int min,
                             final boolean isLeapYear)
    {
        int dayOfYear = 0;

        int totalPriorMonthlyTotalDays = _totalDaysPriorToMonthArray[month];

        if(month > 1) //if February or later - remember Jan == 0 and Feb == 1
        {
            if(isLeapYear)
            {
                totalPriorMonthlyTotalDays++;
            }
        }

        dayOfYear = totalPriorMonthlyTotalDays + day;

        return dayOfYear;
    }

    // ------------------------------------------------------------

    public String getBasinId()
    {
        return _basinId;
    }

    public long getPostingTime()
    {
        return _postingTime;
    }

    public void setBasinId(final String basinId)
    {
        _basinId = basinId;
    }

    public void setPostingTime(final long postingTime)
    {
        _postingTime = postingTime;
    }

    public double[] getValueArray()
    {
        final double[] valueArray = new double[_valueArray.length];

        for(int i = 0; i < _valueArray.length; i++)
        {
            valueArray[i] = _valueArray[i];
        }
        return valueArray;
    }

    public void setPe(final String pe)
    {
        _pe = pe;
    }

    public String getPe()
    {
        return _pe;
    }

    public void setDur(final short dur)
    {
        _dur = dur;
    }

    public short getDur()
    {
        return _dur;
    }

    public void setTs(final String ts)
    {
        _ts = ts;
    }

    public String getTs()
    {
        return _ts;
    }

    public void setExtremum(final String extremum)
    {
        _extremum = extremum;
    }

    public String getExtremum()
    {
        return _extremum;
    }

    public void setAdjustment(final boolean adjustment)
    {
        _adjustment = adjustment;
    }

    public boolean isAdjustment()
    {
        return _adjustment;
    }

    // ------------------------------------------------------------
    public static void main(final String[] args)
    {

        final double[] valuesArray = {1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0};

        final MonthlyValues values = new MonthlyValues(valuesArray);

        final long millisPerDay = 1000 * 60 * 60 * 24;
        final long currentTime = System.currentTimeMillis();
        long todayTime = (currentTime / millisPerDay) * millisPerDay;

        todayTime += 1000 * 3600 * 12; //add 12  hours

        long loopTime = todayTime;
        final int daysInYear = 365;

        for(int i = 0; i < daysInYear; i++)
        {
            final String dateString = DbTimeHelper.getDateStringFromLongTime(loopTime);

            final String dateTimeString = DateTime.getDateTimeStringFromLong(loopTime,OHDConstants.GMT_TIMEZONE);

            final double value = values.getValue(loopTime);
            System.out.println("On " + dateString + " at " + dateTimeString + ", the value is " + value);

            loopTime += millisPerDay;
        }

    } //end main

    // ------------------------------------------------------------

    public void setHourForStartOfDay(final int hourOfDay)
    {
        _startOfDay = hourOfDay;

    }

}
