/*
 * Created on Jun 25, 2003 This class is designed to compute the flow from runoff.
 */
package ohd.hseb.model;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import ohd.hseb.measurement.AbsTimeMeasurement;
import ohd.hseb.measurement.IrregularTimeSeries;
import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.measurement.RelTimeMeasurement;

/**
 * @author Chip Gobs The UnitHydrograph class maps runoff to the amount of marginal flow generated by that runoff. It
 *         returns a RegularTimeSeries object that contains flow measurements.
 */

public class UnitHydrograph
{

    //input:   inches of runoff
    //a starting time
    //output: a time series of flow in CFS

    private static final long MILLIS_PER_HOUR = 1000 * 60 * 60;

    private MeasuringUnit _measuringUnit = MeasuringUnit.cfs;

    private String _locationId = null;
    private String _areaId = null;
    private String _rainfallRunoffModelTypeName = null;

    private int _interval = -1; // in hours

    private final List _measurementList = new ArrayList();

    //----------------------------------------------------------------------

    public UnitHydrograph(final MeasuringUnit measuringUnit, final int intervalInHours)
    {
        _measuringUnit = measuringUnit;
        _interval = intervalInHours;
    }

    //----------------------------------------------------------------------

    public void setMeasuringUnit(final MeasuringUnit measuringUnit)
    {
        _measuringUnit = measuringUnit;
    }

    //----------------------------------------------------------------------

    public MeasuringUnit getMeasuringUnit()
    {
        return _measuringUnit;
    }

    //----------------------------------------------------------------------

    public void setLocationId(final String locationId)
    {
        _locationId = locationId;
    }

    //----------------------------------------------------------------------

    public String getLocationId()
    {
        return _locationId;
    }

    //----------------------------------------------------------------------

    public void setAreaId(final String areaId)
    {
        _areaId = areaId;
    }

    //----------------------------------------------------------------------

    public String getAreaId()
    {
        return _areaId;
    }

    //----------------------------------------------------------------------

    public int getOrdinateCount()
    {
        return _measurementList.size();
    }

    //----------------------------------------------------------------------

    public List getMeasurementList()
    {
        return _measurementList;
    }

    //----------------------------------------------------------------------

    @SuppressWarnings("unchecked")
    public void addMeasurement(final RelTimeMeasurement origMeasurement)
    {
        final RelTimeMeasurement newMeasurement = (RelTimeMeasurement)origMeasurement.getConvertedCopy(_measuringUnit);

        _measurementList.add(newMeasurement);

        return;
    }

    //  ----------------------------------------------------------------------

    public void setInterval(final int interval)
    {
        _interval = interval;
    }

    //----------------------------------------------------------------------

    public int getIntervalInHours()
    {
        return _interval;
    }

    //----------------------------------------------------------------------

    public static UnitHydrograph getTestUnitHydrograph()
    {
        final double[] valueArray = {100.0, 200.0, 350.0, 500.0, 300.0, 250.0, 200.0, 150.0, 100.0, 50.0, 25.0, 10};

        final UnitHydrograph uhg = new UnitHydrograph(MeasuringUnit.cfs, 1);

        for(int i = 0; i < valueArray.length; i++)
        {
            final RelTimeMeasurement measurement = new RelTimeMeasurement(valueArray[i], i, MeasuringUnit.cfs);
            uhg.addMeasurement(measurement);
        }

        return uhg;

    }

    //----------------------------------------------------------------------

    public RelTimeMeasurement getPeakFlowMeasurement()
    {
        RelTimeMeasurement peak = null;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            final RelTimeMeasurement measurement = (RelTimeMeasurement)_measurementList.get(i);
            if((peak == null) || (measurement.getValue() > peak.getValue()))
            {
                peak = measurement;
            }
        }

        return peak;
    }

    //----------------------------------------------------------------------

    public IrregularTimeSeries getFlowTimeSeries(final AbsTimeMeasurement runoffMeasurement)
    {
        final MeasuringUnit runoffUnit = MeasuringUnit.inches;
        final MeasuringUnit dischargeUnit = MeasuringUnit.cfs;

        //convert to inches
        final AbsTimeMeasurement convertedRunoffMeasurement = AbsTimeMeasurement.getConvertedCopy(runoffMeasurement,
                                                                                                  runoffUnit);

        final double scaleFactor = convertedRunoffMeasurement.getValue();

        final IrregularTimeSeries ts = new IrregularTimeSeries(MeasuringUnit.cfs);

        final long startTime = runoffMeasurement.getTime();
        long measurementTime = 0;
        double discharge = 0.0;

        for(int i = 0; i < _measurementList.size(); i++)
        {
            measurementTime = startTime + (i) * (MILLIS_PER_HOUR);

            final Measurement unitGraphValue = (Measurement)_measurementList.get(i);

            discharge = scaleFactor * unitGraphValue.getValue();
            final AbsTimeMeasurement newMeasurement = new AbsTimeMeasurement(discharge, measurementTime, dischargeUnit);
            ts.insertMeasurement(newMeasurement);
        }

        return ts;
    }

    //----------------------------------------------------------------------

    public RegularTimeSeries getFlowRegularTimeSeries(final AbsTimeMeasurement runoffMeasurement)
    {
        final MeasuringUnit runoffUnit = MeasuringUnit.inches;
        final MeasuringUnit dischargeUnit = MeasuringUnit.cfs;

        //convert to inches
        final AbsTimeMeasurement convertedRunoffMeasurement = AbsTimeMeasurement.getConvertedCopy(runoffMeasurement,
                                                                                                  runoffUnit);

        final double scaleFactor = convertedRunoffMeasurement.getValue();

        final int hourCount = _measurementList.size();
        final int intervalInHours = 1;

        final long intervalInMillis = intervalInHours * MILLIS_PER_HOUR;

        final long startTime = runoffMeasurement.getTime();
        final long endTime = startTime + ((hourCount - 1) * MILLIS_PER_HOUR);

        final RegularTimeSeries ts = new RegularTimeSeries(startTime, endTime, intervalInHours, dischargeUnit);

        double discharge = 0.0;

        int i = 0;
        for(long time = startTime; time <= endTime; time += intervalInMillis)
        {

            final Measurement unitgraphValue = (Measurement)_measurementList.get(i);

            discharge = scaleFactor * unitgraphValue.getValue();

            final Measurement newMeasurement = new Measurement(discharge, dischargeUnit);

            ts.setMeasurementByTime(newMeasurement, time);

            i++;
        }

        return ts;

    } //end getFlowRegularTimeSeries  

    //----------------------------------------------------------------------

    @Override
    public String toString()
    {
        String outString = null;

        final StringBuffer buffer = new StringBuffer();

        for(int i = 0; i < _measurementList.size(); i++)
        {
            final Measurement m = (Measurement)_measurementList.get(i);
            buffer.append("ordinal " + i + " = " + m.getValue() + "\n");
        }

        outString = buffer.toString();

        return outString;
    }

    //------------------------------------------------------------------------

    public RegularTimeSeries calculateTotalDischargeTs(final String status,
                                                       final RegularTimeSeries interimFlowTimeSeries,
                                                       final RegularTimeSeries runoffTs,
                                                       final long startTime,
                                                       final long endTime,
                                                       final long intervalInMillis)

    {

//          String header = "UnitHydrograph.calculateTotalDischargeTs(): " + status;

        RegularTimeSeries unitgraphFlowTimeSeries = null;
        RegularTimeSeries totalFlowTimeSeries = interimFlowTimeSeries;

        // convert the runoff to discharge through the unithydrograph                                                
        for(long time = startTime; time <= endTime; time += intervalInMillis)
        {
            Measurement runoffMeasurement = null;

            if(runoffTs != null)
            {
                runoffMeasurement = runoffTs.getMeasurementByTime(time);
            }
            else
            {
                //System.out.println(header + " runoffTs is null");    
            }

            if(runoffMeasurement == null)
            {
                // System.out.println(header + " runoffMeasurement is null when time = " + 
                //                   DbTimeHelper.getDateTimeStringFromLongTime(time));    
            }
            else
            {

                final AbsTimeMeasurement absTimeRunoffMeasurement = new AbsTimeMeasurement(runoffMeasurement, time);

                unitgraphFlowTimeSeries = this.getFlowRegularTimeSeries(absTimeRunoffMeasurement);

                if((totalFlowTimeSeries != null) && (unitgraphFlowTimeSeries != null))
                {
                    totalFlowTimeSeries.addToRegularTimeSeries(unitgraphFlowTimeSeries);
                }
                else if(unitgraphFlowTimeSeries != null)
                {
                    totalFlowTimeSeries = unitgraphFlowTimeSeries;
                }
            }
        } //end for

        return totalFlowTimeSeries;
    }

//     ---------------------------------------------------------------------     

    public static void main(final String[] args) throws Exception
    {
        final UnitHydrograph uhg = UnitHydrograph.getTestUnitHydrograph();
        AbsTimeMeasurement runoff = null;
        IrregularTimeSeries ts1 = null;
        IrregularTimeSeries ts2 = null;
        IrregularTimeSeries ts3 = null;

        long desiredTime = new Date().getTime();
        final long roundingFactor = 1000 * 60 * 60;
        desiredTime /= roundingFactor;
        desiredTime *= roundingFactor;

        runoff = new AbsTimeMeasurement(1.0, desiredTime, MeasuringUnit.inches);
        ts1 = uhg.getFlowTimeSeries(runoff);
        System.out.println("ts1 = " + ts1);

        runoff = new AbsTimeMeasurement(1.0, desiredTime + (1000 * 60 * 60), MeasuringUnit.inches);
        ts2 = uhg.getFlowTimeSeries(runoff);
        System.out.println("ts2 = " + ts2);

        ts3 = IrregularTimeSeries.add(ts1, ts2);
        System.out.println("ts3 = " + ts3);

    } //main

    /**
     * @param rainfallRunoffModel The rainfallRunoffModel to set.
     */
    public void setRainfallRunoffModel(final String rainfallRunoffModelTypeName)
    {
        _rainfallRunoffModelTypeName = rainfallRunoffModelTypeName;
    }

    /**
     * @return Returns the rainfallRunoffModel.
     */
    public String getRainfallRunoffModelTypeName()
    {
        return _rainfallRunoffModelTypeName;
    }

} //end UnitHydrograph
