/*
 * Created on Jul 18, 2003
 */
package ohd.hseb.model.sacsma;

import ohd.hseb.measurement.IrregularTimeSeries;
import ohd.hseb.measurement.IrregularTimeSeriesHolder;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.measurement.RegularTimeSeriesHolder;
import ohd.hseb.model.ForecastAdjuster;
import ohd.hseb.model.RainfallRunoffModel;
import ohd.hseb.model.RainfallToStageModel;
import ohd.hseb.model.RatingCurve;
import ohd.hseb.model.UnitHydrograph;

/**
 * @author Chip Gobs This class combines the RainfallRunoffModel with a UnitHydrograph and a Rating Curve to output a
 *         TimeSeries object.
 */
final class SacRainfallToStageModel implements RainfallToStageModel
{
    private RainfallRunoffModel _rainfallRunoffModel = null;
    private ForecastAdjuster _forecastAdjuster = null;

    private RatingCurve _ratingCurve = null;
    private UnitHydrograph _unitgraph = null;

    private RegularTimeSeriesHolder _priorRunoffTimeSeriesHolder = null;
    private IrregularTimeSeriesHolder _observedStageTimeSeriesHolder = null;

    private final static long _millisPerHour = 60 * 60 * 1000;

//  -------------------------------------------------------------	

    public SacRainfallToStageModel(final RainfallRunoffModel rainfallRunoffModel,
                                   final UnitHydrograph unitgraph,
                                   final RatingCurve ratingCurve,
                                   final RegularTimeSeriesHolder priorRunoffTimeSeriesHolder,
                                   final IrregularTimeSeriesHolder observedStageTimeSeriesHolder,
                                   final ForecastAdjuster forecastAdjuster)
    {

        _rainfallRunoffModel = rainfallRunoffModel;

        _ratingCurve = ratingCurve;
        _unitgraph = unitgraph;

        _priorRunoffTimeSeriesHolder = priorRunoffTimeSeriesHolder;

        // forecast adjustment variables
        _observedStageTimeSeriesHolder = observedStageTimeSeriesHolder;
        _forecastAdjuster = forecastAdjuster;

    } //end RainfallToStageModel()

    //	-------------------------------------------------------------

    public RegularTimeSeries calculateStageTimeSeries(final long startTime,
                                                      final long endTime,
                                                      final RegularTimeSeries precipTimeSeries,
                                                      final RegularTimeSeriesHolder runoffTimeSeriesHolder) throws Exception

    {

        final RegularTimeSeries priorRunoffTimeSeries = _priorRunoffTimeSeriesHolder.getTimeSeries();

        final int intervalInHours = 1;
        final long intervalInMillis = _millisPerHour * intervalInHours;

        final long tsStartTime = startTime + intervalInMillis;

        final RegularTimeSeries runoffTs = _rainfallRunoffModel.calculateRunoff(startTime, endTime, precipTimeSeries);

        runoffTimeSeriesHolder.setTimeSeries(runoffTs);

        RegularTimeSeries totalFlowTimeSeries = new RegularTimeSeries(tsStartTime,
                                                                      endTime,
                                                                      intervalInHours,
                                                                      MeasuringUnit.cfs);

        //note: the previous totalFlowTimeSeries will be ADDED(math) to the flow output
        // from the unit hydrograph

        //use prior runoff

        final long priorRunoffStartTime = startTime - (_unitgraph.getOrdinateCount() * _millisPerHour);

        final long priorRunoffEndTime = startTime;
        totalFlowTimeSeries = _unitgraph.calculateTotalDischargeTs("prior ",
                                                                   totalFlowTimeSeries,
                                                                   priorRunoffTimeSeries,
                                                                   priorRunoffStartTime,
                                                                   priorRunoffEndTime,
                                                                   intervalInMillis);

        //    System.out.println(header + "after _unitgraph.calculateTotalDischargeTs(), prior total flow time series\n " + totalFlowTimeSeries);

        //use newly calculated runoff
        totalFlowTimeSeries = _unitgraph.calculateTotalDischargeTs("new ",
                                                                   totalFlowTimeSeries,
                                                                   runoffTs,
                                                                   tsStartTime,
                                                                   endTime,
                                                                   intervalInMillis);

        //    System.out.println(header + "after _unitgraph.calculateTotalDischargeTs(), new total flow time series\n " + totalFlowTimeSeries);

        //lop off the hours at the beginning that are before the model run start time
        totalFlowTimeSeries.trimTimeSeriesAtStartWithCheck(tsStartTime);

        // forecast adjustment
        if(_forecastAdjuster.getParams().shouldDoAdjustment())
        {
            // CodeTimer timer = new CodeTimer();
            //timer.start();
            final IrregularTimeSeries observedStageTimeSeries = _observedStageTimeSeriesHolder.getTimeSeries();
            final IrregularTimeSeries observedDischargeTimeSeries = _ratingCurve.getDischargeIrregularTimeSeries(observedStageTimeSeries);
            totalFlowTimeSeries = _forecastAdjuster.getAdjustedTimeSeries(observedDischargeTimeSeries,
                                                                          totalFlowTimeSeries);
            // timer.stop(" the forecast adjustment took ");
        }

        //    System.out.println(header + "after totalFlowTimeSeries.trimTimeSeriesAtStart(), new total flow time series\n " + totalFlowTimeSeries);

        // put through the rating curve to get the stages
        final RegularTimeSeries stageTimeSeries = _ratingCurve.getStageRegularTimeSeries(totalFlowTimeSeries);

        return stageTimeSeries;

    } //end getStageTimeSeries

//	---------------------------------------------------------------------

}
