package ohd.hseb.model.sacsma;

import java.util.EnumMap;
import java.util.List;

import net.jcip.annotations.Immutable;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.fews.OHDConstants;

/**
 * Immutable class holding SacSma_MODS corresponding time series if present.
 */

@Immutable
public class SacSmaModsTSHolder
{
    private final EnumMap<SacSma_MODS, RegularTimeSeries> _map = new EnumMap<SacSma_MODS, RegularTimeSeries>(SacSma_MODS.class);

    public SacSmaModsTSHolder(final List<RegularTimeSeries> modsTsList)
    {
        for(final RegularTimeSeries rts: modsTsList)
        {
            final String tyType = rts.getTimeSeriesType();

            final SacSma_MODS mod = SacSma_MODS.valueOf(tyType);

            _map.put(mod, rts.clone());
        }
    }

    /**
     * Returns false if no any mod at all.
     */
    boolean hasMods()
    {
        if(_map.size() == 0)
        {
            return false;
        }

        return true;
    }

    boolean hasMod(final SacSma_MODS mod)
    {
        return _map.containsKey(mod);
    }

    /**
     * get MOD value at this time step, after checking this mod is effective. The unit is MM for the MODS UZTWC, UZFWC,
     * LZTWC, LZFPC, LZFSC and ADIMC(SACCO MODS). The unit is DEGC for MOD FGIX. The unit for SACBASEF is unitlessReal.
     * Returns the missing value(-999.0) when no MOD at this time.
     */
    double getModValueAtTime(final SacSma_MODS mod, final long time)
    {
        double modValue = OHDConstants.MISSING_DATA;

        try
        {
            modValue = _map.get(mod).getMeasurementByTime(time).getValue();
        }
        catch(final NullPointerException e)
        {//no MODS at this time, return the missing value.

        }

        return modValue;

    }

}
