package ohd.hseb.model.sacsma;

import java.util.Map;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.model.MonthlyValues;
import ohd.hseb.model.RainfallRunoffModel;
import ohd.hseb.model.RainfallRunoffModelType;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.RUNOFF_COMPONENT;
import ohd.hseb.util.misc.HNumber;

/**
 * @author Chip Gobs
 * @version 0.5
 *          <p>
 *          History:
 *          <p>
 *          ChamP 12/02/2014 - FB131 Added logic to handle U/LZTWD and U/LZTWCADJ modifiers.
 */
@SuppressWarnings("unused")
//there are some unused variables causing warnings. they maybe useful so keeping them and supress warnings
final public class SacSmaRainfallRunoffModel implements RainfallRunoffModel
{
    private double _subsurfaceRunoff;
    private double _surfaceRunoff;
    private SacSmaParameters _params;
    private SacSmaState _state;

    private boolean _useStaticMape = false;
    private MonthlyValues _monthlyMapeAdjustmentValues = null;
    private MonthlyValues _monthlyMapeValues = null;

    private RegularTimeSeries _sascTS = null; //optional input TS: null or not null will be used as a flag
    private RegularTimeSeries _matTS = null; //required input TS for FRZE
    private RegularTimeSeries _weTS = null; //optional input TS for FRZE
    private RegularTimeSeries _fgixTS = null; //optional output for FRZE in NWSRFS, but here is required output

    /* -------------------6 State variable TS ------------------------ */
    private RegularTimeSeries _uztwcTS = null;
    private RegularTimeSeries _uzfwcTS = null;
    private RegularTimeSeries _lztwcTS = null;
    private RegularTimeSeries _lzfscTS = null;
    private RegularTimeSeries _lzfpcTS = null;
    private RegularTimeSeries _adimcTS = null;

    private RegularTimeSeries _potentialEvapTimeSeries = null;

    /* -------------------2 ET variable TS ------------------------ */
    private RegularTimeSeries _etDemandTS = null;
    private RegularTimeSeries _etActualTS = null;

    /* -------------------6 runoff component TS ------------------------ */
    /*
     * The 7 runoff components (ROCL) data values per time interval are: (1) total channel inflow (2) impervious runoff
     * (3) direct runoff (4) surface runoff (5) interflow (6) supplemental baseflow (7) primary baseflow
     */
    private RegularTimeSeries _impRTS = null;
    private RegularTimeSeries _dirRTS = null;
    private RegularTimeSeries _surRTS = null;
    private RegularTimeSeries _intRTS = null;
    private RegularTimeSeries _supRTS = null;
    private RegularTimeSeries _priRTS = null;

    /* ---------- LZDEFR timeseries of Lower Zone Deficiency Ration used by calibration */
    private RegularTimeSeries _lzdefrTS = null;

    private double _peadj = 1.0;
    private double _pxadj = 1.0;

    private double _perc;

    //private double[] _fgco = new double[6]; //unused
    //note: I found _rsum array and a lot of other variables(_srot, etc) are not used at all.
    private final double[] _rsum = new double[7];

    // common block sums
    private double _srot = 0.0;
    private double _simpvt = 0.0;
    private double _srodt = 0.0;
    private double _srost = 0.0;
    private double _sintft = 0.0;
    private double _sgwfp = 0.0;
    private double _sgwfs = 0.0;
    private double _srecht = 0.0;
    private double _sett = 0.0;

    private double _se1 = 0.0;
//	private double _se2 = 0.0;
    private double _se3 = 0.0;
    private double _se4 = 0.0;
    private double _se5 = 0.0;

    private double _dlzp = 0.0;
    private double _dlzs = 0.0;
    private double _duz = 0.0;

    private double _sbf = 0.0;
    private double _ssur = 0.0; //surface runoff, SUR(component #3)
    private double _sif = 0.0; //interflow, INT(component #4)
    private double _sperc = 0.0;
    private double _sdro = 0.0; //direct runoff, DIR(component #2)
    private double _spbf = 0.0;

    private double _pinc = 0.0;

    //variables that could have been local, but belong to the object so that results of the operation can be verified
    private double _bfp = 0.0; //primary baseflow PRI(component #6, last one)
    private double _bfs = 0.0; //supplemental baseflow SUP(component #5)
    private double _roimp = 0.0; //impervious runoff IMP(component #1)

    private final MeasuringUnit _precipUnit = MeasuringUnit.mm;
    private final MeasuringUnit _runoffUnit = MeasuringUnit.mm;

    private Logger _logger; //get assigned through setter

    private boolean _printLog = false;

    private final StringBuilder _tableLogMessage = new StringBuilder();//for printing out table like log when PRINTSMA technique is on

    private boolean _printSmaTechnique = false; //TECHNIQUE PRINTSMA: if true, print table log. Default is false

    private final DateTime _dateTime = new DateTime(); //for SAC-SMA, no need to consider starting of day at 12Z; we use 0Z here

    private long _initStateTime;

    private String _fgixStr = "";

    private double _tet;

    private long _lstcmpdy = System.currentTimeMillis(); //used to not applying MODS, since SACSMA all MODS are ignored after LSTCMPDY. Initialized to a default value.

    private SacSmaModsTSHolder _modsTsHolder; //immutable object holding all SacSma MODS time series if any present

    /*
     * * Original Constructor
     */
    public SacSmaRainfallRunoffModel(final SacSmaState state,
                                     final SacSmaParameters params,
                                     final double peadj,
                                     final double pxadj,
                                     final boolean useStaticMape,
                                     final MonthlyValues monthlyValues,
                                     final RegularTimeSeries potentialEvapTimeSeries)
    {
        _peadj = peadj;
        _pxadj = pxadj;

        // says whether using MAPE time series or static monthly MAPE values
        _useStaticMape = useStaticMape;

        if(_useStaticMape)
        {
            _monthlyMapeValues = monthlyValues;
        }
        else
        {
            _monthlyMapeAdjustmentValues = monthlyValues;
        }

        // can be null
        _potentialEvapTimeSeries = potentialEvapTimeSeries;

        _params = params;

        _state = state;

    }

    /**
     * FEWS Constructor. Comparing to the original constructor
     * {@link SacSmaRainfallRunoffModel#SacSmaRainfallRunoffModel(SacSmaState, SacSmaParameters, double, double, boolean, MonthlyValuesUseNwsCal, RegularTimeSeries)}
     * some redundant parameters has been removed.
     * <p>
     * 
     * @param listOfInputSacSmaMods - a List object containing all the input MOD TSs. It cannot be null. Its size() can
     *            be zero, meaning that there is no any input MOD TSs.
     */

    public SacSmaRainfallRunoffModel(final SacSmaState state,
                                     final SacSmaParameters params,
                                     final boolean useStaticMape,
                                     final MonthlyValues monthlyValues,
                                     final RegularTimeSeries potentialEvapTimeSeries,
                                     final int startOfDay,
                                     final long lstcmpdy,
                                     final Logger logger)
    {
        //use original constructor first
        this(state, params, params.getPeadj(), params.getPxadj(), useStaticMape, monthlyValues, potentialEvapTimeSeries);

        _lstcmpdy = lstcmpdy;

        if(useStaticMape)
        {
            _monthlyMapeValues.setHourForStartOfDay(startOfDay);
        }
        else
        {
            _monthlyMapeAdjustmentValues.setHourForStartOfDay(startOfDay);
        }

        _logger = logger;

        if(_logger != null && _logger.getPrintDebugInfo() > Logger.FATAL)
        {
            _printLog = true;
        }

    }

    /**
     * This is the interface for CHPS' SAC-SMA model.
     * 
     * @param precipTimeSeries - input TS, its start time and end time must be equal to model run start time and model
     *            run end time.
     * @param resultMap - the Map holding all the output TSs. Besides, {@link #_outputStateRtsMap} holds all the 6 state
     *            TSs.
     * @param modsTsHolder - the immutable object holding SacSma MODS time series if any present.
     */
    public void runSacSmaModel(final RegularTimeSeries precipTimeSeries,
                               final long initStateTime,
                               final Map<String, RegularTimeSeries> resultMap,
                               final SacSmaModsTSHolder modsTsHolder)
    {
        final long startTime = precipTimeSeries.getStartTime();
        final long endTime = precipTimeSeries.getEndTime();

        this._modsTsHolder = modsTsHolder;

        _initStateTime = initStateTime;

        /** ----------------------initialize 6 Runoff component TS ------------------- */

        //All TS are set up with the input RAIM TS interval now; After returned to SacSmaModelDriver, if the specified
        //interval is different, (e.g. bigger interval), convert to new interval
        _impRTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _impRTS.setLocationId(precipTimeSeries.getLocationId());
        _impRTS.setTimeSeriesType(RUNOFF_COMPONENT.IMP.getTypeName());

        _dirRTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _dirRTS.setLocationId(precipTimeSeries.getLocationId());
        _dirRTS.setTimeSeriesType(RUNOFF_COMPONENT.DIR.getTypeName());

        _surRTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _surRTS.setLocationId(precipTimeSeries.getLocationId());
        _surRTS.setTimeSeriesType(RUNOFF_COMPONENT.SUR.getTypeName());

        _intRTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _intRTS.setLocationId(precipTimeSeries.getLocationId());
        _intRTS.setTimeSeriesType(RUNOFF_COMPONENT.INT.getTypeName());

        _supRTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _supRTS.setLocationId(precipTimeSeries.getLocationId());
        _supRTS.setTimeSeriesType(RUNOFF_COMPONENT.SUP.getTypeName());

        _priRTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _priRTS.setLocationId(precipTimeSeries.getLocationId());
        _priRTS.setTimeSeriesType(RUNOFF_COMPONENT.PRI.getTypeName());

        /** ----------------------initialize 6 State variable TS ------------------- */
        _uztwcTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _uztwcTS.setLocationId(precipTimeSeries.getLocationId());
        _uztwcTS.setTimeSeriesType(OHDConstants.SAC_STATE_UZTWC);

        _uzfwcTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _uzfwcTS.setLocationId(precipTimeSeries.getLocationId());
        _uzfwcTS.setTimeSeriesType(OHDConstants.SAC_STATE_UZFWC);

        _lztwcTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _lztwcTS.setLocationId(precipTimeSeries.getLocationId());
        _lztwcTS.setTimeSeriesType(OHDConstants.SAC_STATE_LZTWC);

        _lzfscTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _lzfscTS.setLocationId(precipTimeSeries.getLocationId());
        _lzfscTS.setTimeSeriesType(OHDConstants.SAC_STATE_LZFSC);

        _lzfpcTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _lzfpcTS.setLocationId(precipTimeSeries.getLocationId());
        _lzfpcTS.setTimeSeriesType(OHDConstants.SAC_STATE_LZFPC);

        _adimcTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _adimcTS.setLocationId(precipTimeSeries.getLocationId());
        _adimcTS.setTimeSeriesType(OHDConstants.SAC_STATE_ADIMC);

        /** ---------------------initialize ET variables ----------------------- */
        _etDemandTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _etDemandTS.setLocationId(precipTimeSeries.getLocationId());
        _etDemandTS.setTimeSeriesType(DataType.ET_DEMAND);

        _etActualTS = new RegularTimeSeries(startTime, endTime, precipTimeSeries.getIntervalInHours(), MeasuringUnit.mm);
        _etActualTS.setLocationId(precipTimeSeries.getLocationId());
        _etActualTS.setTimeSeriesType(DataType.ET_ACTUAL);

        /** ---------------------initialize FGIX TS if use FRZE ----------------------- */
        if(_params.useFronzeGroundCalc())
        {
            //FGIX TS is always outputted when use Frozen Ground Calculation
            _fgixTS = new RegularTimeSeries(startTime,
                                            endTime,
                                            precipTimeSeries.getIntervalInHours(),
                                            MeasuringUnit.degreesCelsius);
            _fgixTS.setLocationId(precipTimeSeries.getLocationId());
            _fgixTS.setTimeSeriesType(DataType.FGIX_DATATYPE);
        }

        /** ---------------------initialize LZDEFR TS (cp calibration) --------------------- */
        _lzdefrTS = new RegularTimeSeries(startTime,
                                          endTime,
                                          precipTimeSeries.getIntervalInHours(),
                                          MeasuringUnit.unitlessReal);
        _lzdefrTS.setLocationId(precipTimeSeries.getLocationId());
        _lzdefrTS.setTimeSeriesType(OHDConstants.SACSMA_LZDEFR);

        /** --------------------------Apply MODS at initial state time --------------------- */
        this.applySacSmaMods(_initStateTime);

        /** -----------------modify states as in fckco1.f at initial state time -------------- */
        this.checkStateValues(_initStateTime);

        /** --------------------------Run SAC-SMA Model --------------------- */
        final RegularTimeSeries tciRTS = calculateRunoff(precipTimeSeries.getStartTime(),
                                                         precipTimeSeries.getEndTime(),
                                                         precipTimeSeries);

        /** ------------------------Add all the result TS into the map --------------------------- */
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(tciRTS), tciRTS);

        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_impRTS), _impRTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_dirRTS), _dirRTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_surRTS), _surRTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_intRTS), _intRTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_supRTS), _supRTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_priRTS), _priRTS);

        //add FGIX output TS if use FRZE
        if(_params.useFronzeGroundCalc())
        {
            resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_fgixTS), _fgixTS);
        }

        /** --------------------Add all the 6 state variable TS into the map --------------------- */
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_uztwcTS), _uztwcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_uzfwcTS), _uzfwcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_lztwcTS), _lztwcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_lzfscTS), _lzfscTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_lzfpcTS), _lzfpcTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_adimcTS), _adimcTS);

        /** --------------------Add the lower zone deficiency ratio variable TS into the map (cp - calibration)- */
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_lzdefrTS), _lzdefrTS);

        /** --------------------Add the ET variables into the map --------------------- */
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_etDemandTS), _etDemandTS);
        resultMap.put(OHDUtilities.getCompositeKeyInResultMap(_etActualTS), _etActualTS);
    }

    //implements interface
    public RegularTimeSeries calculateRunoff(final long startTime,
                                             final long endTime,
                                             final RegularTimeSeries precipTimeSeries)
    {

        final RegularTimeSeries runoffRTS = new RegularTimeSeries(startTime,
                                                                  endTime,
                                                                  precipTimeSeries.getIntervalInHours(),
                                                                  _runoffUnit);
        runoffRTS.setLocationId(precipTimeSeries.getLocationId());
        runoffRTS.setTimeSeriesType(DataType.CHANNEL_INFLOW_DATATYPE);

        final double intervalPortionOfDay = precipTimeSeries.getIntervalInHours() * 1.0 / OHDConstants.HOURS_PER_DAY;

        double evaporationAmount = 0.0;
        double precipAmount = 0.0;

        final long precipIntervalInMillis = precipTimeSeries.getIntervalInMillis();

        for(long time = startTime; time <= endTime; time += precipIntervalInMillis)
        {
            precipAmount = precipTimeSeries.getMeasurementValueByTime(time, _precipUnit);

            if(precipAmount < 0.0)
            {
                precipAmount = 0.0;
            }
            else
            {
                precipAmount *= _pxadj;
            }

            if(_useStaticMape)
            {
                evaporationAmount = _monthlyMapeValues.getValue(time) * intervalPortionOfDay * _peadj;
            }
            else
            //using time series
            {
                evaporationAmount = _potentialEvapTimeSeries.getMeasurementValueByTime(time, _precipUnit)
                    * _monthlyMapeAdjustmentValues.getValue(time) * _peadj;

            }

            //if using the optional input SASC TS, further adjust evaporationAmount
            if(_sascTS != null)
            {
                final double aesc = _sascTS.getMeasurementValueByTime(time, MeasuringUnit.percentDecimal);

                evaporationAmount = _params.getEfc() * evaporationAmount + (1.0 - _params.getEfc()) * (1.0 - aesc)
                    * evaporationAmount;
            }

            /** ---------------------calculate runoff in each time step---------------------- */
            final double totalChannelInflow = calculate(intervalPortionOfDay, precipAmount, evaporationAmount, time);

            runoffRTS.setMeasurementByTime(totalChannelInflow, time);

            //save the state after being updated in calculate()
            _uztwcTS.setMeasurementByTime(_state.getUztwc(), time);
            _uzfwcTS.setMeasurementByTime(_state.getUzfwc(), time);
            _lztwcTS.setMeasurementByTime(_state.getLztwc(), time);
            _lzfscTS.setMeasurementByTime(_state.getLzfsc(), time);
            _lzfpcTS.setMeasurementByTime(_state.getLzfpc(), time);
            _adimcTS.setMeasurementByTime(_state.getAdimc(), time);

            //save the ET Demand
            _etDemandTS.setMeasurementByTime(evaporationAmount, time);
            _etActualTS.setMeasurementByTime(_tet, time);

            if(_params.useFronzeGroundCalc())
            {
                //set fgix value in the output TS. The value could be from FGIX MOD.
                _fgixTS.setMeasurementByTime(new Measurement(_state.getFgix(), MeasuringUnit.degreesCelsius), time);
            }

            // add to sums of runoff components
            _rsum[0] += totalChannelInflow;
            _rsum[1] += _roimp;
            _rsum[2] += _sdro;
            _rsum[3] += _ssur;
            _rsum[4] += _sif;
            _rsum[5] += _bfs;
            _rsum[6] += _bfp;

            //store the 6 runoff component values into their TS
            _impRTS.setMeasurementByTime(_roimp, time);
            _dirRTS.setMeasurementByTime(_sdro, time);
            _surRTS.setMeasurementByTime(_ssur, time);
            _intRTS.setMeasurementByTime(_sif, time);
            _supRTS.setMeasurementByTime(_bfs, time);
            _priRTS.setMeasurementByTime(_bfp, time);

            //store the lower zone deficiency ratio (LZDEFR or DEFR) values to TS (need by calibration - percolation analysis CP 4/18/13)
            final double lzdefr = 1.0 - ((_state.getLztwc() + _state.getLzfpc() + _state.getLzfsc()) / (_params.getLztwm()
                + _params.getLzfpm() + _params.getLzfsm()));

            _lzdefrTS.setMeasurementByTime(lzdefr, time);

        } //close for loop

        // reset the valid time                                
        _state.setValidTime(endTime);

        //print out table log 
        this.printOutTableLog();

        return runoffRTS;
    }

    /**
     * When the technique PRINTSMA is on, print the table like message into diag.xml
     */
    public void printOutTableLog()
    {
        if(_printSmaTechnique)
        {
            _logger.log(Logger.INFO, _tableLogMessage.toString());

            //print out to console for visual inspection; may not be needed at some point
//            System.out.println(_tableLogMessage.toString());
        }
    }

    public double calculate(final double dt /* dt */,
                            final double rainfallPlusMelt /* pxv */,
                            final double evaporation /* ep */,
                            final long time)
    {
        final double evapTransDemand = evaporation;

        double e1;
        double e2;
        double e3;

        double residualEvapTransDemand = 0.0;

        // old code does this:
        //evapTransDemand = evaporation * evaporationDistribution(Kint)

        // compute et from upper zone

        e1 = evapTransDemand * (_state.getUztwc() / _params.getUztwm());

        residualEvapTransDemand = evapTransDemand - e1;

        //subtract off e1
        _state.setUztwc(_state.getUztwc() - e1);

        // introduce e2
        e2 = 0.0;

        boolean perform220Check = false;

        if(_state.getUztwc() < 0.0)
        {
            e1 += _state.getUztwc();
            _state.setUztwc(0.0);
            residualEvapTransDemand = evapTransDemand - e1;

            if(_state.getUzfwc() < residualEvapTransDemand)
            {
                //e2 is evap from UZFWC
                e2 = _state.getUzfwc();
                //trace("setting _uzfwc to 0.0");
                _state.setUzfwc(0.0);

                residualEvapTransDemand -= e2;
                perform220Check = false; //simulates goto 225
            }
            else
            //_state.getUzfwc() >= residualEvapTransDemand //simulates 221
            {
                //trace("221");
                e2 = residualEvapTransDemand;

                _state.setUzfwc(_state.getUzfwc() - e2);

                //trace("near 221, _uzfwc = " + _uzfwc);
                residualEvapTransDemand = 0.0;

                perform220Check = true;
            }

        }
        else
        //(state.getUztwc() >= 0.0) //simulates goto 220
        {
            perform220Check = true;
        }

        if(perform220Check)
        {
            //trace("220");
            if((_state.getUztwc() / _params.getUztwm()) < (_state.getUzfwc() / _params.getUzfwm()))
            {
                // upper zone free water ratio exceeds upper zone tension
                // water ratio, this transfer free water to tension

                final double uzrat = ((_state.getUztwc() + _state.getUzfwc()) / (_params.getUztwm() + _params.getUzfwm()));

                _state.setUztwc((_params.getUztwm() * uzrat));

                _state.setUzfwc((_params.getUzfwm() * uzrat));
                //trace("near 220, _uzfwc = " + _uzfwc);
            }
        }

        // line 225 equiv
        //trace("225");
        if(_state.getUztwc() < 0.00001)
        {
            _state.setUztwc(0.0);
        }

        if(_state.getUzfwc() < 0.00001)
        {

            _state.setUzfwc(0.0);
            //trace("near 225, _uzfwc set to 0.0");
        }

        // compute et from the lower zone
        // compute et from LZTWC (E3)

        e3 = residualEvapTransDemand * (_state.getLztwc() / (_params.getUztwm() + _params.getLztwm()));

        _state.setLztwc(_state.getLztwc() - e3);

        if(_state.getLztwc() < 0.0) //simulates the goto 226 stuff
        {
            // e3 cannot exceed lztwc
            e3 += _state.getLztwc();

            _state.setLztwc(0.0);
        }

        //trace("226");

        //simulates line 226
        final double ratlzt = _state.getLztwc() / _params.getLztwm();
        final double ratlz = (_state.getLztwc() + _state.getLzfpc() + _state.getLzfsc() - getSaved())
            / (_params.getLztwm() + _params.getLzfpm() + _params.getLzfsm() - getSaved());

        if(ratlzt < ratlz)
        {
            // resupply lower zone tension water from lower
            // zone free water if more water available there

            final double del = (ratlz - ratlzt) * _params.getLztwm();

            // transfer from lzfwc to lztwc

            _state.setLztwc(_state.getLztwc() + del);
            _state.setLzfsc(_state.getLzfsc() - del);

            if(_state.getLzfsc() < 0.0)
            {
                //if transfer exceeds lzfsc then remainder
                // comes from lzfpc
                _state.setLzfpc(_state.getLzfpc() + _state.getLzfsc());
                _state.setLzfsc(0.0);
            }
        }

        //line 230 stuff
        //trace("230");
        if(_state.getLztwc() < 0.00001)
        {
            _state.setLztwc(0.0);
        }

        // compute et from _adimp area - e5
        double e5 = e1 + (residualEvapTransDemand + e2)
            * ((_state.getAdimc() - e1 - _state.getUztwc()) / (_params.getUztwm() + _params.getLztwm()));

        //adjust adminc, additional impervious area storage, for evaporation

        _state.setAdimc(_state.getAdimc() - e5);

        if(_state.getAdimc() < 0.0)
        {
            // e5 cannot exceed adminc
            e5 += _state.getAdimc();

            _state.setAdimc(0.0);
        }

        //simulates line 231
        //trace("231");
        e5 *= _params.getAdimp();
        // e5 is et from the area _adimp

        //--------------------------
        // compute percolation and runoff amounts

        double twx = rainfallPlusMelt + _state.getUztwc() - _params.getUztwm();
        // twx is the time interval  available moisture in excess of 
        // uztw requirements

        if(twx < 0.0)
        {
            //all moisture held in uztw - no excess
            _state.setUztwc(_state.getUztwc() + rainfallPlusMelt);
            twx = 0.0;

        }
        else
        //(twc >= 0.0) //simulates line 232
        {
            _state.setUztwc(_params.getUztwm());
        }

        //line 233
        //trace("233");

        _state.setAdimc(_state.getAdimc() + (rainfallPlusMelt - twx));

        //compute impervious area runoff
        _roimp = rainfallPlusMelt * _params.getPctim();
        // roimp is runoff from the minimum impervious area

        _simpvt += _roimp;

        //initialize time interval sums

        _sbf = 0.0;
        _ssur = 0.0;
        _sif = 0.0;
        _sperc = 0.0;
        _sdro = 0.0;
        _spbf = 0.0;

        //determine computational time increments for the basic time interval

        //make sure that the rounding here is done the same way that fortran does 
        // it
        //incrementCount = ninc
        final long ninc = (long)Math.floor(1.0 + (0.2 * (_state.getUzfwc() + twx)));

        //ninc = number of time increments that the time interval is
        // is divided into for further soil-moisture accounting. no one increment
        // will exceed 5.0 mm of uzfwc + pav

        //dinc = incrementLength
        final double dinc = (1.0 / ninc) * dt;

        //trace("dinc RESULT = " + dinc + "with inputs:");
        //trace("ninc = " + ninc);
        //trace("timeInterval = " + dt);
        //trace("END dinc block");

        // dinc = length of each increment in days

        //_pinc = _moisturePerIncrement
        _pinc = twx / ninc;
        //  moisturePerIncrement = amount of available moisture per increment
        // compute free water depletion fractions for the time increment being use
        // basic depletions are for one day

        _duz = 1.0 - Math.pow((1.0 - _params.getUzk()), dinc);
        _dlzp = 1.0 - Math.pow((1.0 - _params.getLzpk()), dinc);
        _dlzs = 1.0 - Math.pow((1.0 - _params.getLzsk()), dinc);

        //trace("_dlzp result = " + _dlzp + " with inputs"); 
        //trace("_lzpk = " + _lzpk);
        //trace("dinc = " + dinc);

        //trace("_dlzs result = " + _dlzs + " with inputs"); 
        //trace("_lzsk = " + _lzsk);
        //trace("dinc = " + dinc);

        // start incremental do loop for the time interval

        for(int i = 0; i < ninc; i++)
        {
            //trace("doIncrements loop, iteration # " + i);
            doIncrements();
        }

        //compute sums and adjust runoff amounts by the area over
        // which they are generated

        double eUsed = e1 + e2 + e3;
        //eUsed is the et from pArea which is 1.0-_adimp-pctim
        _sif *= getParea();

        //separate channel component of baseflow
        //from the non-channel component

        final double tbf = _sbf * getParea();
        //tbf is toal baseflow

        final double bfcc = tbf * (1.0 / (1.0 + _params.getSide()));
        //bfcc is baseflow, channel component

        _bfp = _spbf * getParea() / (1.0 + _params.getSide());

        // error #2, had _bfs = bfcc = bfp;
        _bfs = bfcc - _bfp;

        if(_bfs < 0.0)
        {
            _bfs = 0.0;
        }

        final double bfncc = tbf - bfcc;
        // bfncc is baseflow, non-channel component

        //add to monthly sums
        _sintft += _sif;
        _sgwfp += _bfp;
        _sgwfs += _bfs;
        _srecht += bfncc;
        _srost += _ssur;
        _srodt += _sdro;

        //compute total channel inflow for the time interval
        double tci = _roimp + _sdro + _ssur + _sif + bfcc;
        _surfaceRunoff = _roimp + _sdro + _ssur;
        _subsurfaceRunoff = _sif + bfcc;

        //compute e4-et from riparian vegetarian 
        double e4 = (evapTransDemand - eUsed) * _params.getRiva();

        //subtract e4 from channel inflow
        tci = tci - e4;

        if(tci < 0.0)
        {
            e4 += tci;
            tci = 0.0;
        }

        _subsurfaceRunoff = _subsurfaceRunoff - e4;

        if(_subsurfaceRunoff < 0.)
        {
            _surfaceRunoff = _surfaceRunoff + _subsurfaceRunoff;
            _subsurfaceRunoff = 0.;
        }
        if(_surfaceRunoff < 0.)
        {
            _surfaceRunoff = 0;
        }

        //trace("250");
        _srot += tci;

        //compute total evapotranspiration - tet   
        eUsed *= getParea();
        _tet = eUsed + e5 + e4;

        _sett += _tet;

        _se1 += e1 * getParea();
        _se3 += e3 * getParea();

        _se4 += e4;
        _se5 += e5;

        //check that adimc is >= uztwc
        if(_state.getAdimc() < _state.getUztwc())
        {
            _state.setAdimc(_state.getUztwc());
        }

        /** -----------------------------frost1.f ----------------------------- */
        if(_params.useFronzeGroundCalc())
        {
            final double csoil = 4.0 * dt * _params.getCsoil();
            final double ghc = dt * _params.getGhc();
            final double findx1 = _state.getFgix();

            double water = 0.0;
            if(_state.getFgix() < 0.0)
            {
                water = rainfallPlusMelt - _ssur - _sdro;

                if(water > 0.0)
                {
                    _state.setFgix(_state.getFgix() + _params.getRthaw() * water);

                    if(_state.getFgix() > 0.0)
                    {
                        _state.setFgix(0.0);
                    }
                }
            }

            //MAT TS is required for FRZE for modifying state fgix
            final double airTemp = _matTS.getMeasurementValueByTime(time, MeasuringUnit.degreesCelsius);

            double C = csoil; //transfer coefficient

            if(_state.getFgix() < 0.0 || airTemp < 0.0)
            {

                /** -------------Compute Transfer Coefficient-------------------- */

                //check if WE TS available, since it is optional
                if(_weTS != null)
                {
                    final double we = _weTS.getMeasurementValueByTime(time, MeasuringUnit.mm);

                    if(we != 0.0)
                    {
                        double cover;
                        //check if SASC TS available, since it is optional
                        if(_sascTS != null)
                        {
                            cover = _sascTS.getMeasurementValueByTime(time, MeasuringUnit.percentDecimal);
//                           if(cover == 0.0) GOTO 124(no need to modify C)
                        }
                        else
                        {
                            cover = 1.0;
                        }

                        //only modify C if cover != 0.0
                        if(cover != 0.0)
                        {
                            final double twe = we / cover;

                            C = csoil * (1.0 - cover) + csoil * Math.pow((1.0 - _params.getCsnow()), twe) * cover;

//                         GOTO 125;
                        }

                    } //close if(we != 0.0)

                } //close if(_weTS != null)

                /** ---------------Compute Change in FROST index ----------------- */
                //125
                if(airTemp >= 0.0)
                {
                    _state.setFgix(_state.getFgix() + C * airTemp + ghc);
                }
                else
                {
                    final double cfi = -1 * C * Math.sqrt(airTemp * airTemp + _state.getFgix() * _state.getFgix()) - C
                        * _state.getFgix() + ghc;

                    _state.setFgix(_state.getFgix() + cfi);
                }

            } //close if(_state.getFindex() < 0.0 || airTemp < 0.0)

            /** ------------------Check FROST index ------------------ */
            if(_state.getFgix() >= 0.0)
            {
                _state.setFgix(0.0);
            }

            //get real value if using FRZE, must before FGIX MOD, because this value is the value used 
            _fgixStr = OHDUtilities.getFormatString(_state.getFgix(), 3, 9);

            //modify the state by MOD, after the computation has been done at this step
            this.applySacSmaMods(time);

            if(_printLog)
            {
                _logger.log(Logger.DEBUG, "FGIX1=" + findx1 + "  FGIX=" + _state.getFgix() + "  C=" + C + "  WATER="
                    + water + "  TA=" + airTemp); //FGIX and FINDX are the same, we decided to use FGIX
            }

        } //close if(_params.useFronzeGroundCalc())
        else
        { // not using FRZE

            //modify the state by MOD, after the computation has been done at this step
            this.applySacSmaMods(time);
        }
        /** --------------------------end of frost1.f ----------------------------------- */

        if(_printSmaTechnique)
        {
            _dateTime.setTime(time);
            _tableLogMessage.append(OHDUtilities.getFormatString(_dateTime.dayOfMonth(), 4))
                            .append(OHDUtilities.getFormatString(_dateTime.getNwsrfsHour(), 4))
                            .append(OHDUtilities.getFormatString(_state.getUztwc(), 2, 7))
                            .append(OHDUtilities.getFormatString(_state.getUzfwc(), 3, 7))
                            .append(OHDUtilities.getFormatString(_state.getLztwc(), 2, 7))
                            .append(OHDUtilities.getFormatString(_state.getLzfsc(), 3, 7))
                            .append(OHDUtilities.getFormatString(_state.getLzfpc(), 2, 7))
                            .append(OHDUtilities.getFormatString(_state.getAdimc(), 2, 7))
                            .append(_fgixStr)
                            .append(OHDUtilities.getFormatString(_perc, 3, 7))
                            .append(OHDUtilities.getFormatString(_roimp, 3, 7))
                            .append(OHDUtilities.getFormatString(_sdro, 3, 7))
                            .append(OHDUtilities.getFormatString(_ssur, 3, 7))
                            .append(OHDUtilities.getFormatString(_sif, 3, 7))
                            .append(OHDUtilities.getFormatString(_bfs, 3, 7))
                            .append(OHDUtilities.getFormatString(_bfp, 3, 7))
                            .append(OHDUtilities.getFormatString(tci, 3, 7))
                            .append(OHDUtilities.getFormatString(evaporation, 3, 8))
                            .append(OHDUtilities.getFormatString(_tet, 3, 8))
                            .append(OHDUtilities.getFormatString(rainfallPlusMelt, 2, 7))
                            .append(OHDConstants.NEW_LINE);
        }

        return tci;
    } //end calculate

//  --------------------------------------------------------------------------  

    /**
     * Apply the MODS modifications. It takes the effect of SACCO MOD and FGIX MOD first, then SACBASEF MOD in the end.
     * SACBASEF MOD will be ignored if LZFPC or LZFSC MOD happened at this time step.
     * <p>
     * Note: the state period is from (time - interval) to time. So modify the state after the computation has been done
     * at this step. Get ready for next time step.
     */
    private void applySacSmaMods(final long time)
    {

        if(time > _lstcmpdy || _modsTsHolder.hasMods() == false)
        {
            return;
        }

        boolean baseflowStatesModified = false;
        boolean isLowerTensionWaterMod = false;
        boolean isUpperTensionWaterMod = false;
        boolean isLFirst = false;
        boolean isLSecond = false;
        boolean isUFirst = false;
        boolean isUSecond = false;

        final double oldStateUZTWC = _state.getUztwc();
        final double oldStateLZTWC = _state.getLztwc();
        int numUZTWCMod = 1;
        int numLZTWCMod = 1;
        /*
         * This loop is for SACCO MOD or FGIX MOD, SACBASEF mod will be treated separately, later
         */

        for(final SacSma_MODS saccoMod: SacSma_MODS.SACCO_MODS)
        {
            if(_modsTsHolder.hasMod(saccoMod) == false)
            {
                continue; //move to next mod
            }

            final double modValue = _modsTsHolder.getModValueAtTime(saccoMod, time);

            if(modValue == OHDConstants.MISSING_DATA)
            {//no MODS at this time, loop to next mod
                continue;
            }

            /* FB131 11/12/14- Added deficit and adjustment tension water Mods */
            // UZTWD wins UZTWCADJ and UZTWC ==========================
            if(saccoMod.equals(SacSma_MODS.UZTWD))
            {
                isUpperTensionWaterMod = true;
                isUFirst = true;

                double uztwc = _params.getUztwm() - modValue;

                if(uztwc < 0.0)
                {
                    uztwc = 0.0;

                    _logger.log(Logger.WARNING,
                                "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + " - Mod UZTWD (="
                                    + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " IN) tried to adjust tension water beyond the limit of tension water bucket ("
                                    + HNumber.roundDouble(_params.getUztwm() / OHDConstants.MM_PER_INCH, 2)
                                    + " IN).  Reset State Mod UZTWC to 0.0");
                }

                // Update state UZTWC based on adjusted tension water
                _state.setUztwc(uztwc);

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "Mod UZTWD = " + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN); Parameter UZTWM= "
                                    + HNumber.roundDouble(_params.getUztwm() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) >>> Apply State MODS = "
                                    + HNumber.roundDouble(_state.getUztwc() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) at time " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
                }
            }
            // UZTWCADJ wins UZTWC
            if((time != _initStateTime && saccoMod.equals(SacSma_MODS.UZTWCADJ)) && isUpperTensionWaterMod == false)
            {
                double newState = oldStateUZTWC + modValue;

                isUpperTensionWaterMod = true;
                isUSecond = true;

                if(newState < 0.0)
                {
                    newState = 0.0;
                    _logger.log(Logger.WARNING, "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                        + " - Mod UZTWCADJ (=" + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                        + " IN) tried to adjust tension water below 0 then reset State Mod UZTWC to 0.0.");
                }

                if(newState > _params.getUztwm())
                {
                    newState = _params.getUztwm();

                    _logger.log(Logger.WARNING,
                                "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + " - Mod UZTWCADJ (="
                                    + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " IN) tried to adjust tension water beyond the limit of tension water bucket ("
                                    + HNumber.roundDouble(_params.getUztwm() / OHDConstants.MM_PER_INCH, 2)
                                    + " IN) then reset State Mod UZTWC to the limit.");
                }
                // Update state UZTWC based on adjusted tension water
                _state.setUztwc(newState);

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "Mod UZTWCADJ = " + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN); oldState = "
                                    + HNumber.roundDouble(oldStateUZTWC / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) >>> Apply State MODS = "
                                    + HNumber.roundDouble(_state.getUztwc() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) at time " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
                }
            }

            if(saccoMod.equals(SacSma_MODS.UZTWC) && isUpperTensionWaterMod == false)
            {
                _state.setUztwc(modValue);

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "Mod UZTWC >>> Apply UZTWC State Mod = "
                                    + HNumber.roundDouble(_state.getUztwc() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) at time " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
                }
            }
            // Check if multiple mods are defined for one segment then send out WARN message which mod wins.
            if(saccoMod.equals(SacSma_MODS.UZTWD) || saccoMod.equals(SacSma_MODS.UZTWCADJ)
                || saccoMod.equals(SacSma_MODS.UZTWC))
            {
                String ModWin = null;
                if(isUFirst)
                    ModWin = "UZTWD";
                if(isUSecond)
                    ModWin = "UZTWCADJ";

                if(numUZTWCMod > 1 && isUFirst)
                {
                    _logger.log(Logger.WARNING, "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                        + " - There are more than one" + " Tension Water Mods tried to apply.  " + ModWin
                        + " Mod wins.");
                    isUFirst = false;
                }
                else if(numUZTWCMod == 2 && isUSecond)
                {
                    _logger.log(Logger.WARNING, "At " + DateTime.getDateTimeStringFromLong(time, OHDConstants.GMT_TIMEZONE)
                        + " - there are " + numUZTWCMod + " Tension Water Mods tried to apply.  " + ModWin
                        + " Mod wins.");
                }
                numUZTWCMod++;
                continue; // go to next Mod
            }

            // LZTWD wins LZTWCADJ, LZTWC ==================
            if(saccoMod.equals(SacSma_MODS.LZTWD))
            {
                isLowerTensionWaterMod = true;
                isLFirst = true;

                double lztwc = _params.getLztwm() - modValue;

                if(lztwc < 0.0)
                {
                    lztwc = 0.0;
                    _logger.log(Logger.WARNING,
                                "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + " - Mod LZTWD (="
                                    + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " IN) tried to adjust tension water beyond the limits of tension water bucket ("
                                    + HNumber.roundDouble(_params.getLztwm() / OHDConstants.MM_PER_INCH, 2)
                                    + " IN).  Reset State Mod LZTWC to 0.0");
                }
                // Update state LZTWC based on adjusted tension water
                _state.setLztwc(lztwc);

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "Mod LZTWD = " + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN); Parameter LZTWM = "
                                    + HNumber.roundDouble(_params.getLztwm() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) >>> Apply State MODS = "
                                    + HNumber.roundDouble(_state.getLztwc() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) at time " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
                }
            }
            // LZTWCADJ wins LZTWC
            if((time != _initStateTime && saccoMod.equals(SacSma_MODS.LZTWCADJ)) && isLowerTensionWaterMod == false)
            {
                double newState = oldStateLZTWC + modValue;

                isLowerTensionWaterMod = true;
                isLSecond = true;

                if(newState < 0.0)
                {
                    newState = 0.0;
                    _logger.log(Logger.WARNING, "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                        + " - Mod LZTWCADJ (=" + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                        + " IN) tried to adjust tension water below 0.0 then reset State Mod LZTWC to 0.0");
                }

                if(newState > _params.getLztwm())
                {
                    newState = _params.getLztwm();

                    _logger.log(Logger.WARNING,
                                "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + " - Mod LZTWCADJ (="
                                    + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " IN) tried to adjust tension water beyond the limit of tension water bucket ("
                                    + HNumber.roundDouble(_params.getLztwm() / OHDConstants.MM_PER_INCH, 2)
                                    + " IN) then reset State Mod LZTWC to the limit.");
                }
                // Update state LZTWC based on adjusted tension water
                _state.setLztwc(newState);

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "Mod LZTWCADJ = " + HNumber.roundDouble(modValue / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN); oldState = "
                                    + HNumber.roundDouble(oldStateLZTWC / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) >>> Apply LZTWC State Mod = "
                                    + HNumber.roundDouble(_state.getLztwc() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) at time " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
                }
            }

            if(saccoMod.equals(SacSma_MODS.LZTWC) && isLowerTensionWaterMod == false)
            {
                _state.setLztwc(modValue);

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "Mod LZTWC >>> Apply State MODS LZTWC = "
                                    + HNumber.roundDouble(_state.getLztwc() / OHDConstants.MM_PER_INCH, 2)
                                    + " (IN) at time " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
                }
            }
            // Check if multiple mods are defined for one segment then send out WARN message which mod wins.
            if(saccoMod.equals(SacSma_MODS.LZTWD) || saccoMod.equals(SacSma_MODS.LZTWCADJ)
                || saccoMod.equals(SacSma_MODS.LZTWC))
            {
                String ModWin = null;
                if(isLFirst)
                    ModWin = "LZTWD";
                if(isLSecond)
                    ModWin = "LZTWCADJ";

                if(numLZTWCMod > 1 && isLFirst)
                {
                    _logger.log(Logger.WARNING, "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                        + " - There are more than one" + " Tension Water Mods tried to apply.  " + ModWin
                        + " Mod wins.");
                    isLFirst = false;
                }
                else if(numLZTWCMod == 2 && isLSecond)
                {
                    _logger.log(Logger.WARNING, "At " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                        + " - there are " + numLZTWCMod + " Tension Water Mods tried to apply.  " + ModWin
                        + " Mod wins.");
                }
                numLZTWCMod++;
                continue; // go to next Mod
            } //End FB131 ------------------------------------

            //update the corresponding state with new state value
            _state.updateState(saccoMod.getDataType(), modValue);

            if(_printLog)
            {
                _logger.log(Logger.DEBUG,
                            "Applying state MODS " + saccoMod.getDataType() + " at time "
                                + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
            }

            //SACBASEF MOD will be ignored if LZFPC or LZFSC MOD happened at this time step.
            if(saccoMod == SacSma_MODS.LZFPC || saccoMod == SacSma_MODS.LZFSC)
            {
                baseflowStatesModified = true;
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG,
                                "SACBASEF MOD is ignored at time " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                                    + " since LZFPC or LZFSC is changed by SACCO MOD");
                }
            }
        } //close for loop, sacco MODS

        /*
         * Apply SACBASEF MOD after SACCO MOD. If the baseflow has been explicitly changed by LZFPC MOD or LZFSC MOD
         * above, ignore SACBASE MOD now. (SACBASE MOD cannot be at the initial state time either)
         */
        if(time != _initStateTime && _modsTsHolder.hasMod(SacSma_MODS.SACBASEF) && baseflowStatesModified == false)
        {

            final double multiplier = _modsTsHolder.getModValueAtTime(SacSma_MODS.SACBASEF, time);

            if(multiplier != OHDConstants.MISSING_DATA)
            {
                _state.setLzfsc(_state.getLzfsc() * multiplier);
                _state.setLzfpc(_state.getLzfpc() * multiplier);

                if(_state.getLzfsc() > _params.getLzfsm())
                {
                    _state.setLzfsc(_params.getLzfsm());
                }

                if(_state.getLzfpc() > _params.getLzfpm())
                {
                    _state.setLzfpc(_params.getLzfpm());
                }

                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "Applying state MODS " + OHDConstants.SAC_MOD_BASEF + " at time "
                        + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE));
                }
            }

        }

    }//close method applySacSmaMods(final long time)

    public SacSmaParameters getParameters()
    {
        return new SacSmaParameters(_params);
    } //end getParameters

//--------------------------------------------
    public SacSmaState getState()
    {
        return _state;
    }

//  --------------------------------------------------------------------------  

    public double getSaved()
    { //calculated parameter
      //
        return (_params.getRserv() * (_params.getLzfpm() + _params.getLzfsm()));
    }

//  --------------------------------------------------------------------------  

    public double getParea()
    { //calculated parameter
        return (1.0 - _params.getPctim() - _params.getAdimp());
    }

    //---------------------------------------------------------------------

    private void doIncrements()
    {
        //put in lines from pages 2b to 3b
        double adsur = 0.0;

        //compute direct runoff (from _adimp area)
        double ratio = (_state.getAdimc() - _state.getUztwc()) / _params.getLztwm();

        if(ratio < 0.0)
        {
            ratio = 0.0;
        }

        double addro = _pinc * Math.pow(ratio, 2);
        //addro is the amount of direct runoff from the area _adimp

        //compute baseflow and keep track of time interval sum

        double bf = _state.getLzfpc() * _dlzp;

        _state.setLzfpc(_state.getLzfpc() - bf);

        if(_state.getLzfpc() <= 0.0001)
        {
            bf += _state.getLzfpc();
            _state.setLzfpc(0.0);
        }

        //simulates line 234
        //trace("234");
        _sbf += bf;
        _spbf += bf;
        bf = _state.getLzfsc() * _dlzs;

        _state.setLzfsc(_state.getLzfsc() - bf);
        if(_state.getLzfsc() <= 0.0001)
        {
            bf += _state.getLzfsc();
            _state.setLzfsc(0.0);
        }

        //line 235
        //trace("235");
        _sbf += bf;

        //computer percolation - if no water available then skip

        if((_pinc + _state.getUzfwc()) <= 0.01)
        {

            _state.setUzfwc(_state.getUzfwc() + _pinc);
            //trace("near 235, _uzfwc = " + _uzfwc);

            //skip most of the rest//need to goto 249
        }
        else
        //water available line  // 251 equiv
        {
            //trace("251");
            final double percm = (_params.getLzfpm() * _dlzp) + (_params.getLzfsm() * _dlzs);
            _perc = percm * (_state.getUzfwc() / _params.getUzfwm());

            final double defr = 1.0 - ((_state.getLztwc() + _state.getLzfpc() + _state.getLzfsc()) / (_params.getLztwm()
                + _params.getLzfpm() + _params.getLzfsm()));

            //defr is the lower zone moisture deficiency ratio

            double fr = 1.0;
            //fr is the change in percolation withdrawal due to frozen ground

            double fi = 1.0;
            // fi is the change in interflow withdrawal due to frozen ground

            if(_params.useFronzeGroundCalc())
            {

                /** ----------------------fgfr1.f ----------------------- */
                if(_state.getFgix() < _params.getFrtemp())
                {
                    final double exp = _params.getFrtemp() - _state.getFgix();
                    final double fsat = Math.pow((1.0 - _params.getSatr()), exp);

                    final double fdry = 1.0;

                    if(defr > 0.0)
                    {
                        fr = fsat + (fdry - fsat) * (Math.pow(defr, _params.getFrexp()));
                        fi = fr;
                    }
                    else
                    {
                        fr = fsat;
                        fi = fr;
                    }
                }

            } //close if (_params.useFronzeGroundCalc())

            _perc = _perc * (1.0 + (_params.getZperc() * Math.pow(defr, _params.getRexp()))) * fr;

            //Note: percolation occurs from uzfwc before pav is added
            if(_perc >= _state.getUzfwc())
            {
                //percolation rate exceeds uzfwc

                _perc = _state.getUzfwc();
            }

            _state.setUzfwc(_state.getUzfwc() - _perc);

            // check to see if percolation exceeds lower zone deficiency

            final double check = _state.getLztwc() + _state.getLzfpc() + _state.getLzfsc() + _perc - _params.getLztwm()
                - _params.getLzfpm() - _params.getLzfsm();
            if(check > 0.0)
            {
                _perc -= check;

                _state.setUzfwc(_state.getUzfwc() + check);

            }

            _sperc += _perc;
            //sperc is the time interval summation of perc

            //compute interflow and keep track of time interval sum
            // note: moisturePerIncrement has not yet been added

            final double del = _state.getUzfwc() * _duz * fi;
            _sif += del;

            _state.setUzfwc(_state.getUzfwc() - del);

            //distribute percolated water into the lower zones
            //tension water must be filled first except for the pfree area
            //perct is percolation to tension water and perfc is percolation 
            //going to free water
            double percf = 0.0;

            final double perct = _perc * (1.0 - _params.getPfree());
            if((perct + _state.getLztwc()) <= _params.getLztwm())
            {
                _state.setLztwc(_state.getLztwc() + perct);
                percf = 0.0;
            }
            else
            // (perct + _lztwc) >  _lztwm) // simulates line 234
            {
                //trace("234");
                percf = perct + _state.getLztwc() - _params.getLztwm();

                _state.setLztwc(_params.getLztwm());
            }

            //distribute percolation in excess tension requirements
            //among the free water storages

            //line 244 equiv
            //trace("244");
            percf += (_perc * _params.getPfree());

            if(percf != 0.0) // may have problem with exact equivalence
            {
                final double hpl = _params.getLzfpm() / (_params.getLzfpm() + _params.getLzfsm());
                //hpl is the relative size of the primary storage
                //as compared with toal lower zone free water storage

                final double ratlp = _state.getLzfpc() / _params.getLzfpm();
                final double ratls = _state.getLzfsc() / _params.getLzfsm();
                //ratlp and ratls are content to capacity rations, or
                //in other words, the realtive fullness of each storage

                double fracp = (hpl * 2.0 * (1.0 - ratlp)) / ((1.0 - ratlp) + (1.0 - ratls));

                //fracp is the fraction going to primary
                if(fracp > 1.0)
                {
                    fracp = 1.0;
                }

                final double percp = percf * fracp;
                double percs = percf - percp;

                //percp and percs are the amount of the excess
                //percolation going to primary and supplemental
                //storages, respectively

                _state.setLzfsc(_state.getLzfsc() + percs);

                if(_state.getLzfsc() > _params.getLzfsm())
                {
                    percs = percs - _state.getLzfsc() + _params.getLzfsm();
                    _state.setLzfsc(_params.getLzfsm());
                }

                //line 246 equiv
                //trace("246");

                _state.setLzfpc(_state.getLzfpc() + (percf - percs));

                //check to make sure _lzfpc does not exceed _lzfpm

                if(_state.getLzfpc() > _params.getLzfpm())
                {
                    final double excess = _state.getLzfpc() - _params.getLzfpm();

                    _state.setLztwc(_state.getLztwc() + excess);

                    _state.setLzfpc(_params.getLzfpm());

                }
            }

            //line 245 equiv
            //trace("245");
            //distribute moisturePerIncrement between _uzfwc and surface runoff
            if(_pinc != 0.0)
            {
                //check if moisturePerIncrement exceeds _uzfwm

                if((_pinc + _state.getUzfwc()) <= _params.getUzfwm())
                {
                    //no surface runoff
                    _state.setUzfwc(_state.getUzfwc() + _pinc);
                    //trace("near 245, _uzfwc = " + _uzfwc);
                    //simulate goto 249 - it will flow through correctly as is
                }
                else
                {
                    //simulate line 248
                    //trace("248");
                    //compute surface runoff (sur) and keep track of time
                    // interval sum
                    final double sur = _pinc + _state.getUzfwc() - _params.getUzfwm();

                    _state.setUzfwc(_params.getUzfwm());
                    //trace("near 248, _uzfwc = " + _uzfwc);

                    _ssur += (sur * getParea());
                    adsur = sur * (1.0 - (addro / _pinc));
                    //adsur is the amount of surface runoff which comes
                    //from that portion of _adimp which is not
                    //currently generating direct runoff. addro/moisturePerIncrement
                    // is the fraction of _adimp currently generating
                    // direct runoff

                    _ssur += (adsur * _params.getAdimp());

                }
            }
        }

        //line 249 equiv
        //trace("249");

        _state.setAdimc(_state.getAdimc() + ((_pinc) - addro) - adsur);
        if(_state.getAdimc() > _params.getUztwm() + _params.getLztwm())
        {
            addro += (_state.getAdimc()) - (_params.getUztwm() + _params.getLztwm());

            _state.setAdimc(_params.getUztwm() + _params.getLztwm());

        }

        //trace("247");
        _sdro += (addro * _params.getAdimp());
        if(_state.getAdimc() < 0.00001)
        {
            _state.setAdimc(0.0);
        }

    } //end doIncrements

    // --------------------------------------------------------------------------

    @Override
    public String toString()
    {
        final String stateString = getState().toString();
        final String paramString = getParameters().toString();
        final String modelString = stateString + " " + paramString;

        return modelString;
    }

    // --------------------------------------------------------------------------

    public RainfallRunoffModelType getModelType()
    {
        return RainfallRunoffModelType.SAC_SMA;
    }

    public double getSurfaceRunoff()
    {
        return _surfaceRunoff;
    }

    public double getSubsurfaceRunoff()
    {
        return _subsurfaceRunoff;
    }

    // --------------------------------------------------------------------------

    public void setSascTs(final RegularTimeSeries rts)
    {
        _sascTS = rts;
    }

    public void setMatTS(final RegularTimeSeries matTS)
    {
        _matTS = matTS;
    }

    public void setWeTS(final RegularTimeSeries weTS)
    {
        _weTS = weTS;
    }

    /**
     * Set the first line of the table log. It has locationId, time zone info(always Z time). Also automatically set the
     * boolean {@link #_printSmaTechnique} to true.
     */
    public void setPrintTableLogHeader(final String headerLine)
    {
        _printSmaTechnique = true;

        _tableLogMessage.append(headerLine);
        _tableLogMessage.append(OHDConstants.NEW_LINE);

        if(_params.useFronzeGroundCalc())
        {
            _tableLogMessage.append("DAY HR  UZTWC  UZFWC  LZTWC  LZFSC  LZFPC  ADIMC  FGIX     PERC   IMP    DIR    SUR   "
                + " INT    SUP    PRI   TOT-RO  ET-DMD  ACT-ET  RAIN+MELT")
                            .append(OHDConstants.NEW_LINE); //with FRZE, the heading contains FGIX
        }
        else
        {
            _tableLogMessage.append("DAY HR  UZTWC  UZFWC  LZTWC  LZFSC  LZFPC  ADIMC  PERC   IMP    DIR    SUR   "
                + " INT    SUP    PRI   TOT-RO  ET-DMD  ACT-ET  RAIN+MELT").append(OHDConstants.NEW_LINE);
        }
    }

    public void setSacSmaState(final SacSmaState sacSmaState)
    {
        _state = sacSmaState;
    }

    public SacSmaState getSacSmaState()
    {
        return _state;
    }

    public void setSacSmaParams(final SacSmaParameters sacSmaParams)
    {
        _params = sacSmaParams;
    }

    public SacSmaParameters getSacSmaParams()
    {
        return _params;
    }

    private void checkStateValues(final long time)
    {
        /** -------------modify states as in fckco1.f ------------------ */
        if(_state.getUztwc() > _params.getUztwm())
        {
            _logger.log(Logger.WARNING,
                        "The value for Upper Zone Tension Water Contents was changed from " + _state.getUztwc()
                            + " to " + _params.getUztwm() + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            _state.setUztwc(_params.getUztwm());
        }

        if(_state.getUzfwc() > _params.getUzfwm())
        {
            _logger.log(Logger.WARNING,
                        "The value for Upper Zone Free Water Contents was changed from " + _state.getUzfwc() + " to "
                            + _params.getUzfwm() + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            _state.setUzfwc(_params.getUzfwm());
        }

        if(_state.getLztwc() > _params.getLztwm())
        {
            _logger.log(Logger.WARNING,
                        "The value for Lower Zone Tension Water Contents was changed from " + _state.getLztwc()
                            + " to " + _params.getLztwm() + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                            + ".");
            _state.setLztwc(_params.getLztwm());
        }

        if(_state.getLzfsc() > _params.getLzfsm())
        {
            _logger.log(Logger.WARNING,
                        "The value for Lower Zone Free Secondary Contents was changed from " + _state.getLzfsc()
                            + " to " + _params.getLzfsm() + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE)
                            + ".");
            _state.setLzfsc(_params.getLzfsm());
        }

        if(_state.getLzfpc() > _params.getLzfpm())
        {
            _logger.log(Logger.WARNING,
                        "The value for Lower Zone Free Primary Contents was changed from " + _state.getLzfpc() + " to "
                            + _params.getLzfpm() + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            _state.setLzfpc(_params.getLzfpm());
        }

        if(_state.getAdimc() > (_params.getUztwm() + _params.getLztwm()))
        {
            _logger.log(Logger.WARNING,
                        "The value for additional impervious contents was changed from " + _state.getAdimc() + " to "
                            + (_params.getUztwm() + _params.getLztwm()) + " on "
                            + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            _state.setAdimc(_params.getUztwm() + _params.getLztwm());
        }

        if(_state.getAdimc() < _state.getUztwc())
        {
            _logger.log(Logger.WARNING,
                        "The value for additional impervious contents was changed from " + _state.getAdimc() + " to "
                            + _state.getUztwc() + " on " + DateTime.getDateTimeStringFromLong(time,OHDConstants.GMT_TIMEZONE) + ".");
            _state.setAdimc(_state.getUztwc());
        }
    }
} //end class SacSmaModel
